"use strict";
/*********************************************************************
 * Copyright (c) 2020 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionHelper = exports.CHECTL_DEVELOPMENT_VERSION = void 0;
const tslib_1 = require("tslib");
const axios_1 = require("axios");
const execa = require("execa");
const fs = require("fs-extra");
const https = require("https");
const path = require("path");
const semver = require("semver");
const constants_1 = require("../constants");
const che_1 = require("../tasks/che");
const util_1 = require("../util");
const context_1 = require("./context");
const kube_1 = require("./kube");
exports.CHECTL_DEVELOPMENT_VERSION = '0.0.2';
const UPDATE_INFO_FILENAME = 'update-info.json';
const A_DAY_IN_MS = 24 * 60 * 60 * 1000;
var VersionHelper;
(function (VersionHelper) {
    VersionHelper.MINIMAL_OPENSHIFT_VERSION = '3.11';
    VersionHelper.MINIMAL_K8S_VERSION = '1.9';
    VersionHelper.MINIMAL_HELM_VERSION = '2.15';
    VersionHelper.CHE_POD_MANIFEST_FILE = '/home/user/eclipse-che/tomcat/webapps/ROOT/META-INF/MANIFEST.MF';
    VersionHelper.CHE_PREFFIX_VERSION = 'Implementation-Version: ';
    function getOpenShiftCheckVersionTask(flags) {
        return {
            title: 'Check OpenShift version',
            task: (ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                const actualVersion = yield getOpenShiftVersion();
                if (actualVersion) {
                    task.title = `${task.title}: ${actualVersion}.`;
                }
                else if (ctx.isOpenShift4) {
                    task.title = `${task.title}: 4.x`;
                }
                else {
                    task.title = `${task.title}: Unknown`;
                }
                if (!flags['skip-version-check'] && actualVersion) {
                    const checkPassed = checkMinimalVersion(actualVersion, VersionHelper.MINIMAL_OPENSHIFT_VERSION);
                    if (!checkPassed) {
                        throw getMinimalVersionError(actualVersion, VersionHelper.MINIMAL_OPENSHIFT_VERSION, 'OpenShift');
                    }
                }
            })
        };
    }
    VersionHelper.getOpenShiftCheckVersionTask = getOpenShiftCheckVersionTask;
    function getK8sCheckVersionTask(flags) {
        return {
            title: 'Check Kubernetes version',
            task: (_ctx, task) => tslib_1.__awaiter(this, void 0, void 0, function* () {
                let actualVersion;
                switch (flags.platform) {
                    case 'minishift':
                    case 'openshift':
                    case 'crc':
                        actualVersion = yield getK8sVersionWithOC();
                        break;
                    default:
                        actualVersion = yield getK8sVersionWithKubectl();
                }
                if (actualVersion) {
                    task.title = `${task.title}: Found ${actualVersion}.`;
                }
                else {
                    task.title = `${task.title}: Unknown.`;
                }
                if (!flags['skip-version-check'] && actualVersion) {
                    const checkPassed = checkMinimalVersion(actualVersion, VersionHelper.MINIMAL_K8S_VERSION);
                    if (!checkPassed) {
                        throw getMinimalVersionError(actualVersion, VersionHelper.MINIMAL_K8S_VERSION, 'Kubernetes');
                    }
                }
            })
        };
    }
    VersionHelper.getK8sCheckVersionTask = getK8sCheckVersionTask;
    function getOpenShiftVersion() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return getVersionWithOC('openshift ');
        });
    }
    VersionHelper.getOpenShiftVersion = getOpenShiftVersion;
    function getK8sVersionWithOC() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return getVersionWithOC('kubernetes ');
        });
    }
    VersionHelper.getK8sVersionWithOC = getK8sVersionWithOC;
    function getK8sVersionWithKubectl() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            return getVersionWithKubectl('Server Version: ');
        });
    }
    VersionHelper.getK8sVersionWithKubectl = getK8sVersionWithKubectl;
    function checkMinimalK8sVersion(actualVersion) {
        return checkMinimalVersion(actualVersion, VersionHelper.MINIMAL_K8S_VERSION);
    }
    VersionHelper.checkMinimalK8sVersion = checkMinimalK8sVersion;
    function checkMinimalOpenShiftVersion(actualVersion) {
        return checkMinimalVersion(actualVersion, VersionHelper.MINIMAL_OPENSHIFT_VERSION);
    }
    VersionHelper.checkMinimalOpenShiftVersion = checkMinimalOpenShiftVersion;
    function checkMinimalHelmVersion(actualVersion) {
        return checkMinimalVersion(actualVersion, VersionHelper.MINIMAL_HELM_VERSION);
    }
    VersionHelper.checkMinimalHelmVersion = checkMinimalHelmVersion;
    /**
     * Compare versions and return true if actual version is greater or equal to minimal.
     * The comparison will be done by major and minor versions.
     */
    function checkMinimalVersion(actual, minimal) {
        actual = removeVPrefix(actual);
        let vers = actual.split('.');
        const actualMajor = parseInt(vers[0], 10);
        const actualMinor = parseInt(vers[1], 10);
        minimal = removeVPrefix(minimal);
        vers = minimal.split('.');
        const minimalMajor = parseInt(vers[0], 10);
        const minimalMinor = parseInt(vers[1], 10);
        return (actualMajor > minimalMajor || (actualMajor === minimalMajor && actualMinor >= minimalMinor));
    }
    VersionHelper.checkMinimalVersion = checkMinimalVersion;
    function getMinimalVersionError(actualVersion, minimalVersion, component) {
        return new Error(`The minimal supported version of ${component} is '${minimalVersion} but '${actualVersion}' was found. To bypass version check use '--skip-version-check' flag.`);
    }
    VersionHelper.getMinimalVersionError = getMinimalVersionError;
    function getVersionWithOC(versionPrefix) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const command = 'oc';
            const args = ['version'];
            const { stdout } = yield execa(command, args, { timeout: 60000 });
            return stdout.split('\n').filter(value => value.startsWith(versionPrefix)).map(value => value.substring(versionPrefix.length))[0];
        });
    }
    function getVersionWithKubectl(versionPrefix) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const command = 'kubectl';
            const args = ['version', '--short'];
            const { stdout } = yield execa(command, args, { timeout: 60000 });
            return stdout.split('\n').filter(value => value.startsWith(versionPrefix)).map(value => value.substring(versionPrefix.length))[0];
        });
    }
    /**
     * Returns CodeReady Workspaces version.
     */
    function getCheVersion(flags) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const kube = new kube_1.KubeHelper(flags);
            const cheTasks = new che_1.CheTasks(flags);
            const cheCluster = yield kube.getCheCluster(flags.chenamespace);
            if (cheCluster && cheCluster.spec.server.cheFlavor !== 'che') {
                return cheCluster.status.cheVersion;
            }
            const chePodList = yield kube.getPodListByLabel(flags.chenamespace, cheTasks.cheSelector);
            const [chePodName] = chePodList.map(pod => pod.metadata && pod.metadata.name);
            if (!chePodName) {
                return 'UNKNOWN';
            }
            const command = util_1.getClusterClientCommand();
            const args = ['exec', chePodName, '--namespace', flags.chenamespace, 'cat', VersionHelper.CHE_POD_MANIFEST_FILE];
            try {
                const { stdout } = yield execa(command, args, { timeout: 60000 });
                return stdout.split('\n').filter(value => value.startsWith(VersionHelper.CHE_PREFFIX_VERSION)).map(value => value.substring(VersionHelper.CHE_PREFFIX_VERSION.length))[0];
            }
            catch (_a) {
                return 'UNKNOWN';
            }
        });
    }
    VersionHelper.getCheVersion = getCheVersion;
    /**
     * Returns latest crwctl version for the given channel.
     */
    function getLatestChectlVersion(channel) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (util_1.getProjectName() !== constants_1.CHECTL_PROJECT_NAME) {
                return;
            }
            const axiosInstance = axios_1.default.create({
                httpsAgent: new https.Agent({})
            });
            try {
                const { data } = yield axiosInstance.get(`https://che-incubator.github.io/crwctl/channels/${channel}/linux-x64`);
                return data.version;
            }
            catch (_a) {
                return;
            }
        });
    }
    VersionHelper.getLatestChectlVersion = getLatestChectlVersion;
    /**
     * Checks whether there is an update available for current crwctl.
     */
    function isChectlUpdateAvailable(cacheDir, forceRecheck = false) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            // Do not use ctx inside this function as the function is used from hook where ctx is not yet defined.
            if (util_1.getProjectName() !== constants_1.CHECTL_PROJECT_NAME) {
                // Do nothing for crwctl flavors
                return false;
            }
            const currentVersion = util_1.getProjectVersion();
            if (currentVersion === exports.CHECTL_DEVELOPMENT_VERSION) {
                // Skip it, crwctl is built from source
                return false;
            }
            const channel = currentVersion.includes('next') ? 'next' : 'stable';
            const newVersionInfoFilePath = path.join(cacheDir, `${channel}-${UPDATE_INFO_FILENAME}`);
            let newVersionInfo = {
                latestVersion: '0.0.0',
                lastCheck: 0,
            };
            if (yield fs.pathExists(newVersionInfoFilePath)) {
                try {
                    newVersionInfo = (yield fs.readJson(newVersionInfoFilePath, { encoding: 'utf8' }));
                }
                catch (_a) {
                    // file is corrupted
                }
            }
            // Check cache, if it is already known that newer version available
            const isCachedNewerVersionAvailable = semver.gt(newVersionInfo.latestVersion, currentVersion);
            const now = Date.now();
            const isCacheExpired = now - newVersionInfo.lastCheck > A_DAY_IN_MS;
            if (forceRecheck || (!isCachedNewerVersionAvailable && isCacheExpired)) {
                // Cached info is expired. Fetch actual info about versions.
                // undefined cannot be returned from getLatestChectlVersion as 'is flavor' check was done before.
                const latestVersion = (yield getLatestChectlVersion(channel));
                newVersionInfo = { latestVersion, lastCheck: now };
                yield fs.writeJson(newVersionInfoFilePath, newVersionInfo, { encoding: 'utf8' });
                return semver.gt(newVersionInfo.latestVersion, currentVersion);
            }
            // Information whether a newer version available is already in cache
            return isCachedNewerVersionAvailable;
        });
    }
    VersionHelper.isChectlUpdateAvailable = isChectlUpdateAvailable;
    /**
     * Indicates if stable version of CodeReady Workspaces is specified or meant implicitly.
     */
    function isDeployingStableVersion(flags) {
        return !!flags.version || !context_1.ChectlContext.get().isNightly;
    }
    VersionHelper.isDeployingStableVersion = isDeployingStableVersion;
    /**
     * Removes 'v' prefix from version string.
     * @param version version to process
     * @param checkForNumber if true remove prefix only if a numeric version follow it (e.g. v7.x -> 7.x, vNext -> vNext)
     */
    function removeVPrefix(version, checkForNumber = false) {
        if (version.startsWith('v') && version.length > 1) {
            if (checkForNumber) {
                const char2 = version.charAt(1);
                if (char2 >= '0' && char2 <= '9') {
                    return version.substr(1);
                }
            }
            return version.substr(1);
        }
        return version;
    }
    VersionHelper.removeVPrefix = removeVPrefix;
})(VersionHelper = exports.VersionHelper || (exports.VersionHelper = {}));
//# sourceMappingURL=version.js.map