"use strict";
/*********************************************************************
 * Copyright (c) 2019 Red Hat, Inc.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 **********************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const command_1 = require("@oclif/command");
const flags_1 = require("@oclif/parser/lib/flags");
const cli_ux_1 = require("cli-ux");
const Listr = require("listr");
const lodash_1 = require("lodash");
const semver = require("semver");
const context_1 = require("../../api/context");
const kube_1 = require("../../api/kube");
const common_flags_1 = require("../../common-flags");
const constants_1 = require("../../constants");
const common_tasks_1 = require("../../tasks/installers/common-tasks");
const installer_1 = require("../../tasks/installers/installer");
const api_1 = require("../../tasks/platforms/api");
const util_1 = require("../../util");
class Update extends command_1.Command {
    run() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const { flags } = this.parse(Update);
            flags.chenamespace = yield util_1.findWorkingNamespace(flags);
            const ctx = yield context_1.ChectlContext.initAndGet(flags, this);
            if (!flags.batch && ctx.isChectl) {
                yield util_1.askForChectlUpdateIfNeeded();
            }
            yield this.setDomainFlag(flags);
            if (!flags.installer) {
                yield this.setDefaultInstaller(flags);
                cli_ux_1.cli.info(`› Installer type is set to: '${flags.installer}'`);
            }
            yield this.config.runHook(constants_1.DEFAULT_ANALYTIC_HOOK_NAME, { command: Update.id, flags });
            if (!flags.templates && !flags.version) {
                // Use build-in templates if no custom templates nor version to deploy specified.
                // All flavors should use embedded templates if not custom templates is given.
                flags.templates = util_1.getEmbeddedTemplatesDirectory();
            }
            if (flags.version) {
                if (!ctx.isChectl) {
                    // Flavors of crwctl should not use upstream repositories, so version flag is not appliable
                    this.error(`${util_1.getProjectName()} does not support '--version' flag.`);
                }
                if (flags.installer === 'olm') {
                    this.error(`'--${common_flags_1.DEPLOY_VERSION_KEY}' flag is not supported for OLM installer. 'server:update' command automatically updates to the next available version.`);
                }
                if (flags.installer === 'operator' && semver.gt(constants_1.MIN_CHE_OPERATOR_INSTALLER_VERSION, flags.version)) {
                    throw new Error(this.getWrongVersionMessage(flags.version, constants_1.MIN_CHE_OPERATOR_INSTALLER_VERSION));
                }
            }
            const installerTasks = new installer_1.InstallerTasks();
            // pre update tasks
            const apiTasks = new api_1.ApiTasks();
            const preUpdateTasks = new Listr([], ctx.listrOptions);
            preUpdateTasks.add(apiTasks.testApiTasks(flags, this));
            preUpdateTasks.add(common_tasks_1.checkChectlAndCheVersionCompatibility(flags));
            preUpdateTasks.add(common_tasks_1.downloadTemplates(flags));
            preUpdateTasks.add(installerTasks.preUpdateTasks(flags, this));
            // update tasks
            const updateTasks = new Listr([], ctx.listrOptions);
            updateTasks.add({
                title: '↺  Updating...',
                task: () => new Listr(installerTasks.updateTasks(flags, this))
            });
            // post update tasks
            const postUpdateTasks = new Listr([], ctx.listrOptions);
            postUpdateTasks.add(common_tasks_1.getPrintHighlightedMessagesTask());
            try {
                yield preUpdateTasks.run(ctx);
                if (flags.installer === 'operator') {
                    if (!(yield this.checkAbilityToUpdateCheOperatorAndAskUser(flags))) {
                        // Exit
                        return;
                    }
                }
                yield this.checkComponentImages(flags);
                yield updateTasks.run(ctx);
                yield postUpdateTasks.run(ctx);
                this.log(util_1.getCommandSuccessMessage());
            }
            catch (err) {
                this.error(util_1.getCommandErrorMessage(err));
            }
            util_1.notifyCommandCompletedSuccessfully();
        });
    }
    /**
     * Tests if existing Che installation uses custom docker images.
     * If so, asks user whether keep custom images or revert to default images and update them.
     */
    checkComponentImages(flags) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const kubeHelper = new kube_1.KubeHelper(flags);
            const cheCluster = yield kubeHelper.getCheCluster(flags.chenamespace);
            if (cheCluster.spec.server.cheImage
                || cheCluster.spec.server.cheImageTag
                || cheCluster.spec.server.devfileRegistryImage
                || cheCluster.spec.database.postgresImage
                || cheCluster.spec.server.pluginRegistryImage
                || cheCluster.spec.auth.identityProviderImage) {
                let imagesListMsg = '';
                const resetImagesCrPatch = {};
                if (cheCluster.spec.server.pluginRegistryImage) {
                    imagesListMsg += `\n - Plugin registry image: ${cheCluster.spec.server.pluginRegistryImage}`;
                    lodash_1.merge(resetImagesCrPatch, { spec: { server: { pluginRegistryImage: '' } } });
                }
                if (cheCluster.spec.server.devfileRegistryImage) {
                    imagesListMsg += `\n - Devfile registry image: ${cheCluster.spec.server.devfileRegistryImage}`;
                    lodash_1.merge(resetImagesCrPatch, { spec: { server: { devfileRegistryImage: '' } } });
                }
                if (cheCluster.spec.server.postgresImage) {
                    imagesListMsg += `\n - Postgres image: ${cheCluster.spec.database.postgresImage}`;
                    lodash_1.merge(resetImagesCrPatch, { spec: { database: { postgresImage: '' } } });
                }
                if (cheCluster.spec.server.identityProviderImage) {
                    imagesListMsg += `\n - Identity provider image: ${cheCluster.spec.auth.identityProviderImage}`;
                    lodash_1.merge(resetImagesCrPatch, { spec: { auth: { identityProviderImage: '' } } });
                }
                if (cheCluster.spec.server.cheImage) {
                    imagesListMsg += `\n - CodeReady Workspaces server image name: ${cheCluster.spec.server.cheImage}`;
                    lodash_1.merge(resetImagesCrPatch, { spec: { server: { cheImage: '' } } });
                }
                if (cheCluster.spec.server.cheImageTag) {
                    imagesListMsg += `\n - CodeReady Workspaces server image tag: ${cheCluster.spec.server.cheImageTag}`;
                    lodash_1.merge(resetImagesCrPatch, { spec: { server: { cheImageTag: '' } } });
                }
                if (imagesListMsg) {
                    cli_ux_1.cli.warn(`Custom images found in '${cheCluster.metadata.name}' Custom Resource in the '${flags.chenamespace}' namespace: ${imagesListMsg}`);
                    if (flags.batch || flags.yes || (yield cli_ux_1.cli.confirm('Do you want to preserve custom images [y/n]?'))) {
                        cli_ux_1.cli.info('Keeping current images.\nNote, Update might fail if functionality of the custom images different from the default ones.');
                    }
                    else {
                        cli_ux_1.cli.info('Resetting custom images to default ones.');
                        const ctx = context_1.ChectlContext.get();
                        const crPatch = ctx[context_1.ChectlContext.CR_PATCH] || {};
                        lodash_1.merge(crPatch, resetImagesCrPatch);
                        ctx[context_1.ChectlContext.CR_PATCH] = crPatch;
                    }
                }
            }
        });
    }
    /**
     * Check whether crwctl should proceed with update.
     * Asks user for confirmation (unless assume yes is provided).
     * Is applicable to operator installer only.
     * Returns true if crwctl can/should proceed with update, false otherwise.
     */
    checkAbilityToUpdateCheOperatorAndAskUser(flags) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const ctx = context_1.ChectlContext.get();
            cli_ux_1.cli.info(`Existing CodeReady Workspaces operator: ${ctx.deployedCheOperatorImage}`);
            cli_ux_1.cli.info(`New CodeReady Workspaces operator     : ${ctx.newCheOperatorImage}`);
            if (ctx.deployedCheOperatorImageName === constants_1.DEFAULT_CHE_OPERATOR_IMAGE_NAME && ctx.newCheOperatorImageName === constants_1.DEFAULT_CHE_OPERATOR_IMAGE_NAME) {
                // Official images
                if (ctx.deployedCheOperatorImage === ctx.newCheOperatorImage) {
                    if (ctx.newCheOperatorImageTag === 'nightly') {
                        cli_ux_1.cli.info('Updating current CodeReady Workspaces nightly version to a new one.');
                        return true;
                    }
                    if (flags[common_flags_1.CHE_OPERATOR_CR_PATCH_YAML_KEY]) {
                        // Despite the operator image is the same, CR patch might contain some changes.
                        cli_ux_1.cli.info('Patching existing CodeReady Workspaces installation.');
                        return true;
                    }
                    else {
                        cli_ux_1.cli.info('CodeReady Workspaces is already up to date.');
                        return false;
                    }
                }
                if (this.isUpgrade(ctx.deployedCheOperatorImageTag, ctx.newCheOperatorImageTag)) {
                    // Upgrade
                    const currentChectlVersion = util_1.getProjectVersion();
                    if (!ctx.isNightly && (ctx.newCheOperatorImageTag === 'nightly' || semver.lt(currentChectlVersion, ctx.newCheOperatorImageTag))) {
                        // Upgrade is not allowed
                        if (ctx.newCheOperatorImageTag === 'nightly') {
                            cli_ux_1.cli.warn(`Stable ${util_1.getProjectName()} cannot update stable CodeReady Workspaces to nightly version`);
                        }
                        else {
                            cli_ux_1.cli.warn(`It is not possible to update CodeReady Workspaces to a newer version using the current '${currentChectlVersion}' version of crwctl. Please, update '${util_1.getProjectName()}' to a newer version using command '${util_1.getProjectName()} update' and then try again.`);
                        }
                        return false;
                    }
                    // Upgrade allowed
                    if (ctx.newCheOperatorImageTag === 'nightly') {
                        cli_ux_1.cli.info(`You are going to update CodeReady Workspaces ${ctx.deployedCheOperatorImageTag} to nightly version.`);
                    }
                    else {
                        cli_ux_1.cli.info(`You are going to update CodeReady Workspaces ${ctx.deployedCheOperatorImageTag} to ${ctx.newCheOperatorImageTag}`);
                    }
                }
                else {
                    // Downgrade
                    if (semver.gt(constants_1.MIN_CHE_OPERATOR_INSTALLER_VERSION, flags.version)) {
                        cli_ux_1.cli.info(`Given CodeReady Workspaces version ${flags.version} is too old to be downgraded to`);
                        return false;
                    }
                    cli_ux_1.cli.info(`You are going to downgrade CodeReady Workspaces ${ctx.deployedCheOperatorImageTag} to ${ctx.newCheOperatorImageTag}`);
                    cli_ux_1.cli.warn('DOWNGRADE IS NOT OFFICIALLY SUPPORTED, PROCEED ON YOUR OWN RISK');
                }
            }
            else {
                // At least one of the images is custom
                // Print message
                if (ctx.deployedCheOperatorImage === ctx.newCheOperatorImage) {
                    // Despite the image is the same it could be updated image, replace anyway.
                    cli_ux_1.cli.info(`You are going to replace CodeReady Workspaces operator image ${ctx.newCheOperatorImage}.`);
                }
                else if (ctx.deployedCheOperatorImageName !== constants_1.DEFAULT_CHE_OPERATOR_IMAGE_NAME && ctx.newCheOperatorImageName !== constants_1.DEFAULT_CHE_OPERATOR_IMAGE_NAME) {
                    // Both images are custom
                    cli_ux_1.cli.info(`You are going to update ${ctx.deployedCheOperatorImage} to ${ctx.newCheOperatorImage}`);
                }
                else {
                    // One of the images is offical
                    if (ctx.deployedCheOperatorImageName === constants_1.DEFAULT_CHE_OPERATOR_IMAGE_NAME) {
                        // Update from offical to custom image
                        cli_ux_1.cli.info(`You are going to update official ${ctx.deployedCheOperatorImage} image with user provided one: ${ctx.newCheOperatorImage}`);
                    }
                    else { // ctx.newCheOperatorImageName === DEFAULT_CHE_OPERATOR_IMAGE_NAME
                        // Update from custom to official image
                        cli_ux_1.cli.info(`You are going to update user provided image ${ctx.deployedCheOperatorImage} with official one: ${ctx.newCheOperatorImage}`);
                    }
                }
            }
            if (!flags.batch && !flags.yes && !(yield cli_ux_1.cli.confirm('If you want to continue - press Y'))) {
                cli_ux_1.cli.info('Update cancelled by user.');
                return false;
            }
            return true;
        });
    }
    /**
     * Checks if official operator image is replaced with a newer one.
     * Tags are allowed in format x.y.z or nightly.
     * nightly is considered the most recent.
     * For example:
     *  (7.22.1, 7.23.0) -> true,
     *  (7.22.1, 7.20.2) -> false,
     *  (7.22.1, nightly) -> true,
     *  (nightly, 7.20.2) -> false
     * @param oldTag old official operator image tag, e.g. 7.20.1
     * @param newTag new official operator image tag e.g. 7.22.0
     * @returns true if upgrade, false if downgrade
     * @throws error if tags are equal
     */
    isUpgrade(oldTag, newTag) {
        if (oldTag === newTag) {
            throw new Error(`Tags are the same: ${newTag}`);
        }
        // if newTag is nightly it is upgrade
        // if oldTag is nightly it is downgrade
        // otherwise just compare new and old tags
        // Note, that semver lib doesn't handle text tags and throws an error in case nightly is provided for comparation.
        return newTag === 'nightly' || (oldTag !== 'nightly' && semver.gt(newTag, oldTag));
    }
    /**
     * Copies spec.k8s.ingressDomain. It is needed later for updates.
     */
    setDomainFlag(flags) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const kubeHelper = new kube_1.KubeHelper(flags);
            const cheCluster = yield kubeHelper.getCheCluster(flags.chenamespace);
            if (cheCluster && cheCluster.spec.k8s && cheCluster.spec.k8s.ingressDomain) {
                flags.domain = cheCluster.spec.k8s.ingressDomain;
            }
        });
    }
    /**
     * Sets installer type depending on the previous installation.
     */
    setDefaultInstaller(flags) {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const kubeHelper = new kube_1.KubeHelper(flags);
            try {
                yield kubeHelper.getOperatorSubscription(constants_1.SUBSCRIPTION_NAME, flags.chenamespace);
                flags.installer = 'olm';
            }
            catch (_a) {
                flags.installer = 'operator';
            }
        });
    }
    getWrongVersionMessage(current, minimal) {
        return `This crwctl version can deploy ${minimal} version and higher, but ${current} is provided. If you really need to deploy that old version, please download corresponding legacy crwctl version.`;
    }
}
exports.default = Update;
Update.description = 'Update CodeReady Workspaces server.';
Update.examples = [
    '# Update CodeReady Workspaces:\n' +
        'crwctl server:update',
    '\n# Update CodeReady Workspaces in \'eclipse-che\' namespace:\n' +
        'crwctl server:update -n eclipse-che',
    '\n# Update CodeReady Workspaces and update its configuration in the custom resource:\n' +
        `crwctl server:update --${common_flags_1.CHE_OPERATOR_CR_PATCH_YAML_KEY} patch.yaml`,
];
Update.flags = {
    installer: flags_1.string({
        char: 'a',
        description: 'Installer type. If not set, default is autodetected depending on previous installation.',
        options: ['operator', 'olm'],
        hidden: true,
    }),
    platform: flags_1.string({
        char: 'p',
        description: 'Type of OpenShift platform. Valid values are \"openshift\", \"crc (for CodeReady Containers)\".',
        options: ['openshift', 'crc'],
        default: 'openshift'
    }),
    chenamespace: common_flags_1.cheNamespace,
    batch: common_flags_1.batch,
    templates: flags_1.string({
        char: 't',
        description: 'Path to the templates folder',
        env: 'CHE_TEMPLATES_FOLDER',
        exclusive: [common_flags_1.DEPLOY_VERSION_KEY],
    }),
    'che-operator-image': flags_1.string({
        description: 'Container image of the operator. This parameter is used only when the installer is the operator',
        hidden: true,
    }),
    'skip-version-check': command_1.flags.boolean({
        description: 'Skip minimal versions check.',
        default: false,
        hidden: true,
    }),
    'deployment-name': common_flags_1.cheDeployment,
    'listr-renderer': common_flags_1.listrRenderer,
    'skip-kubernetes-health-check': common_flags_1.skipKubeHealthzCheck,
    yes: common_flags_1.assumeYes,
    help: command_1.flags.help({ char: 'h' }),
    [common_flags_1.CHE_OPERATOR_CR_PATCH_YAML_KEY]: common_flags_1.cheOperatorCRPatchYaml,
    telemetry: common_flags_1.CHE_TELEMETRY,
    [common_flags_1.DEPLOY_VERSION_KEY]: common_flags_1.cheDeployVersion,
};
//# sourceMappingURL=update.js.map