/*
 * Copyright 2023 Red Hat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.wildfly.test.integration.observability.setuptask;

import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.STATISTICS_ENABLED;

import org.jboss.arquillian.testcontainers.api.DockerRequired;
import org.jboss.as.arquillian.container.ManagementClient;
import org.jboss.as.controller.client.helpers.Operations;
import org.jboss.as.test.shared.ServerReload;
import org.jboss.dmr.ModelNode;
import org.junit.AssumptionViolatedException;
import org.wildfly.test.integration.observability.container.OpenTelemetryCollectorContainer;

/**
 * Sets up a functioning Micrometer subsystem configuration. Requires functioning Docker environment!
 */
@DockerRequired(AssumptionViolatedException.class)
public class MicrometerSetupTask extends AbstractSetupTask {

    private final ModelNode micrometerExtension = Operations.createAddress("extension", "org.wildfly.extension.micrometer");
    private final ModelNode micrometerSubsystem = Operations.createAddress("subsystem", "micrometer");
    private OpenTelemetryCollectorContainer otelCollector;

    @Override
    public void setup(final ManagementClient managementClient, String containerId) throws Exception {
        executeOp(managementClient, writeAttribute("undertow", STATISTICS_ENABLED, "true"));

        if (!Operations.isSuccessfulOutcome(executeRead(managementClient, micrometerExtension))) {
            executeOp(managementClient, Operations.createAddOperation(micrometerExtension));
        }

        if (!Operations.isSuccessfulOutcome(executeRead(managementClient, micrometerSubsystem))) {
            ModelNode addOp = Operations.createAddOperation(micrometerSubsystem);
            addOp.get("endpoint").set("http://localhost:4318/v1/metrics"); // Default endpoint
            executeOp(managementClient, addOp);
        }

        otelCollector = OpenTelemetryCollectorContainer.getInstance();
        executeOp(managementClient, writeAttribute("micrometer", "endpoint",
                otelCollector.getOtlpHttpEndpoint() + "/v1/metrics"));
        executeOp(managementClient, writeAttribute("micrometer", "step", "1"));

        ServerReload.executeReloadAndWaitForCompletion(managementClient);
    }

    @Override
    public void tearDown(final ManagementClient managementClient, String containerId) throws Exception {
        otelCollector.stop();

        executeOp(managementClient, clearAttribute("undertow", STATISTICS_ENABLED));
        executeOp(managementClient, Operations.createRemoveOperation(micrometerSubsystem));
        executeOp(managementClient, Operations.createRemoveOperation(micrometerExtension));

        ServerReload.reloadIfRequired(managementClient);
    }
}
