/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ejb3.test.persistenceunits.unit;

import java.util.List;

import junit.framework.Test;

import org.jboss.ejb3.test.persistenceunits.Book;
import org.jboss.ejb3.test.persistenceunits.BookManager;
import org.jboss.ejb3.test.persistenceunits.Entity1;
import org.jboss.ejb3.test.persistenceunits.Entity2;
import org.jboss.ejb3.test.persistenceunits.EntityTest;
import org.jboss.ejb3.test.persistenceunits.User;
import org.jboss.ejb3.test.persistenceunits.UserManager;
import org.jboss.logging.Logger;
import org.jboss.test.JBossTestCase;

/**
 * @author <a href="mailto:bdecoste@jboss.com">William DeCoste</a>
 */
public class MultipleEarTestCase
extends JBossTestCase
{
   private static final Logger log = Logger.getLogger(MultipleEarTestCase.class);

   public MultipleEarTestCase(String name)
   {
      super(name);
   }

   public void testGoodEar() throws Exception
   {
      EntityTest test = (EntityTest) getInitialContext().lookup("persistenceunitscope-test1/EntityTestBean/remote");
      
      Entity1 entity1 = new Entity1();
      entity1.setData("ONE");
      Long id1 = test.persistEntity1(entity1);
      
      Entity2 entity2 = new Entity2();
      entity2.setData("TWO");
      Long id2 = test.persistEntity2(entity2);
      
      entity1 = test.loadEntity1(id1);
      assertEquals("ONE", entity1.getData());
      
      entity2 = test.loadEntity2(id2);
      assertEquals("TWO", entity2.getData());
   }
   
   /**
    * Tests that if an ear doesn't package a persistence unit, and tries to inject a persistence
    * unit deployed outside the ear, then the persistence unit is accessible and usable.
    * This is a jboss specific feature.
    * See https://jira.jboss.org/jira/browse/JBPAPP-3064 
    * @throws Exception
    */
   public void testPersistenceUnitAccessibleFromDifferentEar() throws Exception
   {
         // persistenceunitscope-test2.ear does *not* package a persistence unit,
         // so the PU will be picked up from persistenceunitscope-test1.ear (which pacakges
         // the PU). This is a jboss specific feature https://jira.jboss.org/jira/browse/JBPAPP-3064
         EntityTest test = (EntityTest) getInitialContext().lookup("persistenceunitscope-test2/EntityTestBean/remote");
         Entity1 entity1 = new Entity1();
         entity1.setData("hello");
         Long id1 = test.persistEntity1(entity1);
         entity1 = test.loadEntity1(id1);
         assertEquals("hello", entity1.getData());
         
   }
   
   /**
    * Test that if multiple EARs deploy a persistence unit with the same unit name, then
    * the deployments within the ears pick up the correct persistence unit.
    * 
    * @throws Exception
    */
   public void testMultipleEarWithSamePUName() throws Exception
   {
      // Both bookmanager.ear and usermanager.ear package a persistence unit
      // with the same name. However, each persistence unit maps different 
      // set of (exclusive) entities. The test ensures that the correct 
      // PU is picked up by the deployments for entitymanager operations.
      
      // first do some book manager operations. If an incorrect PU was picked
      // up then the org.jboss.ejb3.test.persistenceunits.Book entity will *not* be mapped to this PU
      // and the test will fail
      BookManager bookManager = (BookManager) getInitialContext().lookup(BookManager.JNDI_NAME);
      String bookName = "new book";
      bookManager.addBook(bookName);
      List<Book> books = bookManager.getBooks();
      assertNotNull("Books list was null", books);
      assertEquals("Unexpected number of books returned", 1, books.size());
      Book book = books.get(0);
      assertEquals("Unexpected book returned", bookName, book.getName());
      
      // now do some usermanager operations. If an incorrect PU was picked
      // up then the org.jboss.ejb3.test.persistenceunits.User entity will *not* be mapped to this PU
      // and the test will fail
      UserManager userManager = (UserManager) getInitialContext().lookup(UserManager.JNDI_NAME);
      String userName = "new book";
      userManager.addUser(userName);
      List<User> users = userManager.getUsers();
      assertNotNull("Users list was null", users);
      assertEquals("Unexpected number of books returned", 1, users.size());
      User user = users.get(0);
      assertEquals("Unexpected user returned", userName, user.getName());
      
   }


   public static Test suite() throws Exception
   {
      return getDeploySetup(MultipleEarTestCase.class,
            "persistenceunitscope-test1.ear, persistenceunitscope-test2.ear, bookmanager.ear, usermanager.ear");
   }

}
