/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.web.tomcat.service.session.distributedcache.spi;

import java.security.AccessController;
import java.security.PrivilegedAction;
import java.util.HashMap;
import java.util.Map;

/**
 * Factory for obtaining a DistributedCacheManagerFactory.
 * 
 * @author Brian Stansberry
 */
public class DistributedCacheManagerFactoryFactory
{
   /** 
    * Default name of the class that {@link #getDistributedCacheManagerFactory()} 
    * will try to instantiate if no factory is injected. 
    */
   public static final String DEFAULT_CLASS_NAME = "org.jboss.web.tomcat.service.session.distributedcache.impl.DistributedCacheManagerFactoryImpl";
   
   /** Singleton */
   private static final DistributedCacheManagerFactoryFactory instance = new DistributedCacheManagerFactoryFactory();
   
   /** Gets the singleton instance. */
   public static final DistributedCacheManagerFactoryFactory getInstance()
   {
      return instance;
   }
   
   private DistributedCacheManagerFactory distributedCacheManagerFactory;
   private String factoryClassName = DEFAULT_CLASS_NAME;
   private Map<TomcatClusterConfig, TomcatClusterDistributedCacheManagerFactory> tomcatClusterFactories = 
           new HashMap<TomcatClusterConfig, TomcatClusterDistributedCacheManagerFactory>();

   /**
    * Returns the factory, creating one if necessary by loading
    * {@link #getFactoryClassName() the factory class} and invoking 
    * {@link Class#newInstance()}.
    * 
    * @return the factory. Will not return <code>null</code>.
    * @throws ClusteringNotSupportedException if there is a problem instantiating a factory
    */
   public synchronized DistributedCacheManagerFactory getDistributedCacheManagerFactory()
      throws  ClusteringNotSupportedException
   {
      if (distributedCacheManagerFactory == null)
      {
         distributedCacheManagerFactory = instantiateFactory();
         factoryClassName = distributedCacheManagerFactory.getClass().getName();
      }
      return distributedCacheManagerFactory;
   }

   /**
    * Allows injection of the DistributedCacheManagerFactory.
    * 
    * @param distributedCacheManagerFactory the factory.
    */
   public synchronized void setDistributedCacheManagerFactory(DistributedCacheManagerFactory distributedCacheManagerFactory)
   {
      this.distributedCacheManagerFactory = distributedCacheManagerFactory;
   }
   
   public synchronized TomcatClusterDistributedCacheManagerFactory getTomcatClusterDistributedCacheManagerFactory(TomcatClusterConfig config)
   throws  ClusteringNotSupportedException
   {
      TomcatClusterDistributedCacheManagerFactory factory = tomcatClusterFactories.get(config);
      if (factory == null)
      {
         factory = (TomcatClusterDistributedCacheManagerFactory) instantiateFactory();
         tomcatClusterFactories.put(config, factory);
      }
      factory.setTomcatClusterConfig(config);
      return factory;
   }
   
   public synchronized void setTomcatClusterDistributedCacheManagerFactory(TomcatClusterConfig config,
                                                 TomcatClusterDistributedCacheManagerFactory factory)
   {
      tomcatClusterFactories.put(config, factory);
   }
   
   /**
    * Gets the class name of the factory; either the name of any existing 
    * factory, a class name injected via this property's setter, or
    * {@link #DEFAULT_CLASS_NAME}.
    * 
    * @return the factory class name; will not return <code>null</code>
    */
   public synchronized String getFactoryClassName()
   {
      return factoryClassName;
   }

   /**
    * Allows injection of the factory class name.
    * 
    * @param factoryClassName Fully qualified name of a class that implements 
    *                         {@link DistributedCacheManagerFactory} and exposes
    *                         a public no-arg constructor.
    */
   public synchronized void setFactoryClassName(String factoryClassName)
   {
      this.factoryClassName = factoryClassName == null ? DEFAULT_CLASS_NAME : factoryClassName;
   }

   /**
    * Loads the factory class and instantiates it.
    * 
    * @throws ClusteringNotSupportedException
    */
   private DistributedCacheManagerFactory instantiateFactory() throws ClusteringNotSupportedException
   {
      String className = factoryClassName == null ? DEFAULT_CLASS_NAME : factoryClassName;
      ClassLoader cl = getTCCL();
      try
      {
         return (DistributedCacheManagerFactory) cl.loadClass(className).newInstance();
      }
      catch (Exception e)
      {
         throw new ClusteringNotSupportedException("Cannot create " + DistributedCacheManagerFactory.class.getSimpleName(), e);
      }
   }

   private ClassLoader getTCCL()
   {
      if (System.getSecurityManager() == null)
      {
         return Thread.currentThread().getContextClassLoader();
      }
      else
      {
         return AccessController.doPrivileged(new PrivilegedAction<ClassLoader>() {
           
            public ClassLoader run()
            {
               return Thread.currentThread().getContextClassLoader();
            }
         });
      }
   }

   private DistributedCacheManagerFactoryFactory() {}
}