/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2009, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.as.ejb3.deployers;

import java.util.ArrayList;
import java.util.List;

import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.helpers.AbstractComponentDeployer;
import org.jboss.deployers.spi.deployer.helpers.AbstractDeploymentVisitor;
import org.jboss.deployers.spi.deployer.helpers.DeploymentVisitor;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.ejb3.common.deployers.spi.AttachmentNames;
import org.jboss.logging.Logger;
import org.jboss.metadata.ejb.jboss.JBossEnterpriseBeanMetaData;
import org.jboss.metadata.ejb.jboss.JBossMetaData;

/**
 * Create components out of EJB 3 JBossMetaData.
 * 
 * @author <a href="mailto:cdewolf@redhat.com">Carlo de Wolf</a>
 * @version $Revision: $
 */
public class EJBsDeployer extends AbstractComponentDeployer<JBossMetaData, JBossEnterpriseBeanMetaData>
{
   private static final Logger log = Logger.getLogger(EJBsDeployer.class);
   
   private static final JBossDeploymentVisitor deploymentVisitor = new JBossDeploymentVisitor();
   
   /**
    * 
    */
   public EJBsDeployer()
   {
      // Since we don't set component visitor
      addInput(deploymentVisitor.getVisitorType());
      setOutput(deploymentVisitor.getComponentType());
      
      // for ordering
      addInput(AttachmentNames.PROCESSED_METADATA);
      
      setDeploymentVisitor(deploymentVisitor);
   }
   
   private static class JBossDeploymentVisitor extends AbstractDeploymentVisitor<JBossEnterpriseBeanMetaData, JBossMetaData>
   {
      @Override
      public Class<JBossEnterpriseBeanMetaData> getComponentType()
      {
         return JBossEnterpriseBeanMetaData.class;
      }

      @Override
      protected List<? extends JBossEnterpriseBeanMetaData> getComponents(JBossMetaData deployment)
      {
         return new ArrayList<JBossEnterpriseBeanMetaData>(deployment.getEnterpriseBeans());
      }

      @Override
      protected String getComponentName(JBossEnterpriseBeanMetaData attachment)
      {
         return JBossEnterpriseBeanMetaData.class.getName() + "." + attachment.getEjbName();
      }

      public Class<JBossMetaData> getVisitorType()
      {
         return JBossMetaData.class;
      }
   }
   
   @Override
   protected <U> void deploy(DeploymentUnit unit, DeploymentVisitor<U> visitor) throws DeploymentException
   {
      // I only want a single attachment deployed
      
      U deployment = unit.getAttachment(visitor.getVisitorType());
      try
      {
         visitor.deploy(unit, deployment);
      }
      catch(Throwable t)
      {
         throw DeploymentException.rethrowAsDeploymentException("Error deploying: " + unit.getName(), t);
      }
   }
   
   @Override
   public void internalDeploy(DeploymentUnit unit) throws DeploymentException
   {
      if(!isValid(unit))
         return;
      
      log.debug("Processing " + unit);
      
      super.internalDeploy(unit);
   }
   
   @Override
   public void internalUndeploy(DeploymentUnit unit)
   {
      // if we don't check then undeploy will be done for stuff that was never deployed.
      if(!isValid(unit))
         return;
      
      super.internalUndeploy(unit);
   }
   
   protected boolean isValid(DeploymentUnit unit)
   {
      // for good measure, we only do this for EJB 3 deployments
      JBossMetaData md = unit.getAttachment(JBossMetaData.class);
      if(md == null || !md.isEJB3x())
         return false;
      return true;
   }
}
