/*
 * JBoss, Home of Professional Open Source
 * Copyright 2007, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.managed.plugins;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jboss.managed.api.DeploymentState;
import org.jboss.managed.api.DeploymentTemplateInfo;
import org.jboss.managed.api.ManagedComponent;
import org.jboss.managed.api.ManagedDeployment;
import org.jboss.managed.api.ManagedObject;
import org.jboss.managed.api.ManagedProperty;

/**
 * A simple ManagedDeployment bean implementation
 * 
 * @author Scott.Stark@jboss.org
 * @version $Revision: 86987 $
 */
public class ManagedDeploymentImpl implements ManagedDeployment, Serializable
{
   private static final long serialVersionUID = 1;
   /** The full deployment name */
   private String name;
   /** The simple deployment name */
   private String simpleName;
   /** The set of types assigned to the deployment */
   private Set<String> types;
   /** The profile service phase for the deployment */
   private DeploymentPhase phase;
   /** The deployment state for the deployment */
   private DeploymentState state = DeploymentState.UNKNOWN;
   /** The parent deployment if any */
   private ManagedDeployment parent;
   /** The deployment ManagedObjects */
   private Map<String, ManagedObject> unitMOs;
   /** The flattened map of all ManagedObject properties */   
   private Map<String, ManagedProperty> properties;
   /** The ManagedComponent wrappers for ManagedObjects marked as components */
   private Map<String, ManagedComponent> components = new HashMap<String, ManagedComponent>();
   /** The child deployemnts */
   private List<ManagedDeployment> children = new ArrayList<ManagedDeployment>();
   /** The attachments map */
   private transient Map<String, Object> attachments;
   
   public ManagedDeploymentImpl(String name, String simpleName)
   {
      this.name = name;
      this.simpleName = simpleName;
      this.unitMOs = Collections.emptyMap();
      this.properties = Collections.emptyMap();; 
   }
   
   @Deprecated
   public ManagedDeploymentImpl(String name, String simpleName, DeploymentPhase phase,
         ManagedDeployment parent, Map<String, ManagedObject> unitMOs)
   {
      this(name, simpleName, parent, unitMOs);
      this.phase = phase;
   }
   
   public ManagedDeploymentImpl(String name, String simpleName, 
         ManagedDeployment parent, Map<String, ManagedObject> unitMOs)
   {
      this.name = name;
      this.simpleName = simpleName;
      this.parent = parent;
      this.unitMOs = unitMOs;
      properties = new HashMap<String, ManagedProperty>();
      for(ManagedObject mo : unitMOs.values())
      {
         properties.putAll(mo.getProperties());
      }
      if(parent != null)
      {
         parent.getChildren().add(this);
      }
   }

   public String getName()
   {
      return name;
   }
   public String getSimpleName()
   {
      return simpleName;
   }

   public boolean addType(String type)
   {
      return types.add(type);
   }
   public Set<String> getTypes()
   {
      return types;
   }
   public void setTypes(Set<String> types)
   {
      this.types = types;
   }

   public DeploymentPhase getDeploymentPhase()
   {
      return phase;
   }
   
   public DeploymentState getDeploymentState()
   {
      return state;
   }
   public void setDeploymentState(DeploymentState state)
   {
      this.state = state;
   }
   
   public ManagedDeployment getParent()
   {
      return parent;
   }

   public Set<String> getComponentTemplateNames()
   {
      // TODO Auto-generated method stub
      return null;
   }
   public void addComponent(String name, ManagedComponent comp)
   {
      components.put(name, comp);
   }
   public ManagedComponent getComponent(String name)
   {
      return components.get(name);
   }

   public Map<String, ManagedComponent> getComponents()
   {
      return components;
   }

   public boolean removeComponent(String name)
   {
      ManagedComponent mc = components.remove(name);
      return mc != null;
   }

   public Set<String> getDeploymentTemplateNames()
   {
      // TODO Auto-generated method stub
      return null;
   }
   public List<ManagedDeployment> getChildren()
   {
      return children;
   }

   public ManagedDeployment addModule(String deplymentBaseName, DeploymentTemplateInfo info)
   {
      // TODO Auto-generated method stub
      return null;
   }

   public Map<String, ManagedProperty> getProperties()
   {
      return properties;
   }

   public ManagedProperty getProperty(String name)
   {
      return properties.get(name);
   }

   public Set<String> getPropertyNames()
   {
      return properties.keySet();
   }

   public Set<String> getManagedObjectNames()
   {
      return unitMOs.keySet();
   }
   public Map<String, ManagedObject> getManagedObjects()
   {
      return unitMOs;
   }
   public ManagedObject getManagedObject(String name)
   {
      return unitMOs.get(name);
   }

   public <T> T getAttachment(Class<T> expectedType)
   {
      T tvalue = null;
      Object value = getAttachment(expectedType.getName());
      if(value != null)
         tvalue = expectedType.cast(value);
      return tvalue;
   }

   public Object getAttachment(String name)
   {
      Object value = null;
      if(attachments != null)
         value = attachments.get(name);
      return value;
   }

   public synchronized void setAttachment(String name, Object attachment)
   {
      if(attachments == null)
         attachments = new HashMap<String, Object>();
      attachments.put(name, attachment);
   }

   public String toString()
   {
      StringBuilder tmp = new StringBuilder(super.toString());
      tmp.append('{');
      tmp.append("name=");
      tmp.append(getName());
      tmp.append(", types=");
      tmp.append(types);
      tmp.append(", state=");
      tmp.append(state);
      tmp.append(", parent=");
      if( parent != null )
      {
         tmp.append("ManagedDeployment@");
         tmp.append(System.identityHashCode(parent));
      }
      else
      {
         tmp.append("null");
      }
      tmp.append(", components=");
      tmp.append(components);
      tmp.append(", children=");
      tmp.append(children);
      tmp.append('}');
      return tmp.toString();
   }
}
