/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.logging.filter;

import java.lang.reflect.Method;
import java.net.URL;
import java.net.URLClassLoader;

/**
 * An appender filter that accepts log events based on whether the thread
 * <p/>
 * context class loader has a classpath URL that has the DeployURL
 * <p/>
 * attribute as a substring. A sample usage would be:
 * <p/>
 * <p/>
 * <appender name="JMX-CONSOLE" class="org.jboss.logging.appender.FileAppender">
 * <errorHandler class="org.jboss.logging.util.OnlyOnceErrorHandler"/>
 * <param name="File" value="${jboss.server.home.dir}/log/jmx-console.log"/>
 * <layout class="org.apache.log4j.PatternLayout">
 * <!-- The default pattern: Date Priority [Category] Message\n -->
 * <param name="ConversionPattern" value="%d %-5p [%c] %m%n"/>
 * </layout>
 * <p/>
 * <filter class="org.jboss.logging.filter.WebTCLFilter">
 * <p/>
 * <param name="AcceptOnMatch" value="true"/>
 * <p/>
 * <param name="DeployURL" value="jmx-console.war"/>
 * <p/>
 * </filter>
 * </appender>
 *
 * @author Scott.Stark@jboss.org
 * @version $Revision: 3621 $
 */
public class TCLFilter extends AbstractTCLFilter
{
   protected boolean matchClassLoader(ClassLoader cl)
   {
      URL[] urls = getClassLoaderURLs(cl);
      for (URL u : urls)
      {
         String file = u.getFile();
         if (file.indexOf(getDeployURL()) > 0)
            return true;
      }
      return false;
   }

   /**
    * First check if the <code>cl</code> is of type URLClassloader. If yes, then use the getURLs API without the
    * need for reflection. If its not a URLClassloader type then go for reflection
    * to access a URL[] getURLs method so that non-URLClassLoader
    * class loaders that support this method can provide info.
    *
    * @param cl classloader
    * @return classloader's urls
    */
   private static URL[] getClassLoaderURLs(ClassLoader cl)
   {
      URL[] urls = {};
      try
      {
         if (cl instanceof URLClassLoader)
         {
            URLClassLoader urlClassloader = (URLClassLoader)cl;
            urls = urlClassloader.getURLs();
         }
         else
         {
            Class<?> returnType = urls.getClass();
            Class<?>[] parameterTypes = {};
            Method getURLs = cl.getClass().getMethod("getURLs", parameterTypes);
            if (returnType.isAssignableFrom(getURLs.getReturnType()))
            {
               Object[] args = {};
               urls = (URL[])getURLs.invoke(cl, args);
            }
            if (urls == null || urls.length == 0)
            {
               getURLs = cl.getClass().getMethod("getClasspath", parameterTypes);
               if (returnType.isAssignableFrom(getURLs.getReturnType()))
               {
                  Object[] args = {};
                  urls = (URL[])getURLs.invoke(cl, args);
               }
            }
         }
      }
      catch (Exception ignore)
      {
      }
      return urls;
   }
}