/*
* JBoss, Home of Professional Open Source
* Copyright 2005, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.reflect.plugins.javassist;

import java.lang.annotation.Annotation;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javassist.*;

import org.jboss.reflect.plugins.AnnotationAttributeImpl;
import org.jboss.reflect.plugins.AnnotationHelper;
import org.jboss.reflect.plugins.AnnotationValueFactory;
import org.jboss.reflect.plugins.AnnotationValueImpl;
import org.jboss.reflect.plugins.EnumConstantInfoImpl;
import org.jboss.reflect.spi.AnnotationInfo;
import org.jboss.reflect.spi.AnnotationValue;
import org.jboss.reflect.spi.NumberInfo;
import org.jboss.reflect.spi.PrimitiveInfo;
import org.jboss.reflect.spi.TypeInfo;
import org.jboss.reflect.spi.TypeInfoFactory;
import org.jboss.util.JBossStringBuilder;
import org.jboss.util.collection.WeakClassCache;

/**
 * A javassist type factory.
 *
 * @author <a href="mailto:adrian@jboss.org">Adrian Brock</a>
 * @author <a href="mailto:ales.justin@jboss.org">Ales Justin</a>
 */
public class JavassistTypeInfoFactoryImpl extends WeakClassCache implements TypeInfoFactory, AnnotationHelper
{
   static final ClassPool pool = ClassPool.getDefault();

   static final AnnotationValue[] NO_ANNOTATIONS = new AnnotationValue[0];

   /** Tmp invocation results */
   private static ThreadLocal<Map<CtClass, TypeInfo>> results = new ThreadLocal<Map<CtClass, TypeInfo>>();

   /**
    * Raise NoClassDefFoundError for javassist not found
    *
    * @param name the name
    * @param e the not found error
    * @return never
    * @throws NoClassDefFoundError always
    */
   public static NoClassDefFoundError raiseClassNotFound(String name, NotFoundException e) throws NoClassDefFoundError
   {
      NoClassDefFoundError ex = new NoClassDefFoundError("Unable to find class " + name);
      if (e.getCause() != null)
         ex.initCause(e.getCause()); // Hide the javassist error
      throw ex;
   }

   /**
    * Raise NoClassDefFoundError for javassist not found
    *
    * @param name the name
    * @param e the not found error
    * @return never
    * @throws NoClassDefFoundError always
    */
   public static NoClassDefFoundError raiseClassNotFound(String name, ClassNotFoundException e) throws NoClassDefFoundError
   {
      NoClassDefFoundError ex = new NoClassDefFoundError("Unable to find class " + name);
      ex.initCause(e);
      throw ex;
   }

   /**
    * Raise NoClassDefFoundError for javassist not found
    *
    * @param name the name
    * @param e the not found error
    * @return never
    * @throws NoClassDefFoundError always
    */
   public static NoClassDefFoundError raiseMethodNotFound(String name, NotFoundException e) throws NoClassDefFoundError
   {
      NoSuchMethodError ex = new NoSuchMethodError("Unable to find method " + name);
      if (e.getCause() != null)
         ex.initCause(e.getCause()); // Hide the javassist error
      throw ex;
   }

   /**
    * Raise NoClassDefFoundError for javassist not found
    *
    * @param name the name
    * @param e the not found error
    * @return never
    * @throws NoClassDefFoundError always
    */
   public static NoClassDefFoundError raiseFieldNotFound(String name, NotFoundException e) throws NoClassDefFoundError
   {
      NoSuchFieldError ex = new NoSuchFieldError("Unable to find field " + name);
      if (e.getCause() != null)
         ex.initCause(e.getCause()); // Hide the javassist error
      throw ex;
   }

   @SuppressWarnings("unchecked")
   protected Object instantiate(Class clazz)
   {
      CtClass ctClass = getCtClass(clazz.getName());

      boolean start = false;
      Map<CtClass, TypeInfo> tmp = results.get();
      if (tmp == null)
      {
         start = true;
         tmp = new HashMap<CtClass, TypeInfo>();
         results.set(tmp);
      }

      // if we return here, it means we're already in a loop,
      // hence no need to cleanup
      TypeInfo cached = tmp.get(ctClass);
      if (cached != null)
         return cached;

      try
      {
         if (ctClass.isArray())
         {
            TypeInfo componentType = getTypeInfo(ctClass.getComponentType());
            TypeInfo result = new JavassistArrayInfoImpl(this, ctClass, clazz, componentType);
            tmp.put(ctClass, result);
            return result;
         }
         else if (ctClass.isAnnotation())
         {
            JavassistAnnotationInfo result = new JavassistAnnotationInfo(this, ctClass, clazz);
            tmp.put(ctClass, result);

            CtMethod[] methods = ctClass.getDeclaredMethods();
            AnnotationAttributeImpl[] atttributes = new AnnotationAttributeImpl[methods.length];
            for (int i = 0 ; i < methods.length ; i++)
            {
               atttributes[i] = new AnnotationAttributeImpl(methods[i].getName(), getTypeInfo(methods[i].getReturnType()), null);
            }
            result.setAttributes(atttributes);

            return result;
         }
         else if (ctClass.isEnum())
         {
            JavassistEnumInfo enumInfo = new JavassistEnumInfo(this, ctClass, clazz);
            tmp.put(ctClass, enumInfo);

            CtField[] fields = ctClass.getFields();
            List<EnumConstantInfoImpl> constants = new ArrayList<EnumConstantInfoImpl>();
            for (CtField field : fields)
            {
               if (Modifier.isEnum(field.getModifiers()))
               {
                  AnnotationValue[] annotations = getAnnotations(field);
                  constants.add(new EnumConstantInfoImpl(field.getName(), enumInfo, annotations));
               }
            }
            enumInfo.setEnumConstants(constants.toArray(new EnumConstantInfoImpl[constants.size()]));

            return enumInfo;
         }
         else
         {
            TypeInfo result = new JavassistTypeInfo(this, ctClass, clazz);
            tmp.put(ctClass, result);
            return result;
         }
      }
      catch (NotFoundException e)
      {
         throw new RuntimeException(e);
      }
      finally
      {
         if (start)
            results.remove();
      }
   }

   /**
    * Get the type info
    *
    * @param ctClass the ctClass
    * @return the typeinfo
    */
   protected TypeInfo getTypeInfo(CtClass ctClass)
   {
      try
      {
         String name = convertName(ctClass);
         return getTypeInfo(name, null);
      }
      catch (ClassNotFoundException e)
      {
         throw raiseClassNotFound(ctClass.getName(), e);
      }
   }

   /**
    * Convert name
    *
    * @param clazz the class
    * @return the converted name
    */
   protected String convertName(CtClass clazz)
   {
      CtClass temp = clazz;
      if (temp.isArray())
      {
         JBossStringBuilder buffer = new JBossStringBuilder();
         try
         {
            while (temp.isArray())
            {
               buffer.append('[');
               temp = temp.getComponentType();
            }
            if (temp.isPrimitive())
            {
               CtPrimitiveType primitive = (CtPrimitiveType) temp;
               buffer.append(Character.toString(primitive.getDescriptor()));
            }
            else
            {
               buffer.append('L');
               buffer.append(temp.getName());
               buffer.append(';');
            }
            return buffer.toString();
         }
         catch (NotFoundException e)
         {
            throw raiseClassNotFound(clazz.getName(), e);
         }
      }
      return clazz.getName();
   }

   /**
    * Get the CtClass
    *
    * @param name the name
    * @return the CtClass
    */
   protected CtClass getCtClass(String name)
   {
      try
      {
         return pool.get(name);
      }
      catch (NotFoundException e)
      {
         throw raiseClassNotFound(name, e);
      }
   }

   @SuppressWarnings("unchecked")
   protected void generate(Class clazz, Object result)
   {
      // Everything is done lazily
   }

   public TypeInfo getTypeInfo(Class<?> clazz)
   {
      if (clazz == null)
         throw new IllegalArgumentException("Null class");

      TypeInfo primitive = PrimitiveInfo.valueOf(clazz.getName());
      if (primitive != null)
         return primitive;

      NumberInfo number = NumberInfo.valueOf(clazz.getName());
      if (number != null)
      {
         synchronized (number)
         {
            if (number.getPhase() != NumberInfo.Phase.INITIALIZING)
            {
               if (number.getPhase() != NumberInfo.Phase.COMPLETE)
               {
                  number.initializing();
                  number.setDelegate((TypeInfo)get(clazz));
               }
               return number;
            }
         }
      }

      return (TypeInfo) get(clazz);
   }

   public TypeInfo getTypeInfo(String name, ClassLoader cl) throws ClassNotFoundException
   {
      if (name == null)
         throw new IllegalArgumentException("Null class name");
      if (cl == null)
         cl = Thread.currentThread().getContextClassLoader();

      TypeInfo primitive = PrimitiveInfo.valueOf(name);
      if (primitive != null)
         return primitive;

      NumberInfo number = NumberInfo.valueOf(name);
      if (number != null)
      {
         synchronized (number)
         {
            if (number.getPhase() != NumberInfo.Phase.INITIALIZING)
            {
               if (number.getPhase() != NumberInfo.Phase.COMPLETE)
               {
                  number.initializing();
                  number.setDelegate((TypeInfo)get(Class.forName(name, false, cl)));
               }
               return number;
            }
         }
      }

      Class<?> clazz = Class.forName(name, false, cl);
      return getTypeInfo(clazz);
   }
   
   public TypeInfo getTypeInfo(Type type)
   {
      if (type instanceof Class)
         return getTypeInfo((Class<?>) type);

      // TODO JBMICROCONT-129 getTypeInfo + NumberInfo
      throw new org.jboss.util.NotImplementedException("getTypeInfo");
   }

   public AnnotationValue[] getAnnotations(Object obj)
   {
      try
      {
         Object[] annotations;
         if (obj instanceof CtMember)
         {
            annotations = ((CtMember)obj).getAvailableAnnotations();
         }
         else if (obj instanceof CtClass)
         {
            annotations = ((CtClass)obj).getAvailableAnnotations();
         }
         else
         {
            throw new RuntimeException("Attempt was made to read annotations from unsupported type " + obj.getClass().getName() + ": " + obj);
         }

         if (annotations.length == 0)
         {
            return NO_ANNOTATIONS;
         }

         AnnotationValue[] annotationValues = new AnnotationValueImpl[annotations.length];
         for (int i = 0 ; i < annotations.length ; i++)
         {
            Class<?> clazz = ((Annotation)annotations[i]).annotationType();
            
            AnnotationInfo info = (AnnotationInfo)getTypeInfo(clazz);
            annotationValues[i] = AnnotationValueFactory.createAnnotationValue(this, this, info, annotations[i]);
         }
         return annotationValues;
      }
//      catch (ClassNotFoundException e)
//      {
//         throw new RuntimeException(e);
//      }
      catch (Throwable t)
      {
         throw new RuntimeException(t);
      }
   }

   public AnnotationValue createAnnotationValue(AnnotationInfo info, Object ann)
   {
      return AnnotationValueFactory.createAnnotationValue(this, this, info, ann);
   }
   
}
