/*
 * Hibernate, Relational Persistence for Idiomatic Java
 *
 * Copyright (c) 2008, Red Hat Middleware LLC or third-party contributors as
 * indicated by the @author tags or express copyright attribution
 * statements applied by the authors.  All third-party contributions are
 * distributed under license by Red Hat Middleware LLC.
 *
 * This copyrighted material is made available to anyone wishing to use, modify,
 * copy, or redistribute it subject to the terms and conditions of the GNU
 * Lesser General Public License, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 * for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this distribution; if not, write to:
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301  USA
 *
 */
package org.hibernate.loader.collection;

import java.util.Map;

import org.hibernate.MappingException;
import org.hibernate.engine.SessionFactoryImplementor;
import org.hibernate.loader.JoinWalker;
import org.hibernate.persister.collection.QueryableCollection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Loads a collection of values or a many-to-many association.
 * <br>
 * The collection persister must implement <tt>QueryableCOllection<tt>. For
 * other collections, create a customized subclass of <tt>Loader</tt>.
 *
 * @see OneToManyLoader
 * @author Gavin King
 */
public class BasicCollectionLoader extends CollectionLoader {

	private static final Logger log = LoggerFactory.getLogger(BasicCollectionLoader.class);

	public BasicCollectionLoader(
			QueryableCollection collectionPersister, 
			SessionFactoryImplementor session, 
			Map enabledFilters)
	throws MappingException {
		this(collectionPersister, 1, session, enabledFilters);
	}

	public BasicCollectionLoader(
			QueryableCollection collectionPersister, 
			int batchSize, 
			SessionFactoryImplementor factory, 
			Map enabledFilters)
	throws MappingException {
		this(collectionPersister, batchSize, null, factory, enabledFilters);
	}
	
	protected BasicCollectionLoader(
			QueryableCollection collectionPersister, 
			int batchSize, 
			String subquery, 
			SessionFactoryImplementor factory, 
			Map enabledFilters)
	throws MappingException {
		
		super(collectionPersister, factory, enabledFilters);
		
		JoinWalker walker = new BasicCollectionJoinWalker(
				collectionPersister, 
				batchSize, 
				subquery, 
				factory, 
				enabledFilters
			);
		initFromWalker( walker );

		postInstantiate();

		log.debug( "Static select for collection " + collectionPersister.getRole() + ": " + getSQLString() );
	}
	
}
