/*
 * Copyright (c) 2022. Red Hat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.as.test.integration.security.jaas;

import java.net.URI;
import java.net.URL;
import javax.security.auth.AuthPermission;
import javax.servlet.http.HttpServletResponse;
import org.hamcrest.CoreMatchers;
import org.hamcrest.MatcherAssert;
import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.container.test.api.RunAsClient;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.arquillian.test.api.ArquillianResource;
import org.jboss.as.test.integration.security.common.Utils;
import org.jboss.as.test.shared.integration.ejb.security.PermissionUtils;
import org.jboss.as.test.shared.util.AssumeTestGroupUtil;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.spec.WebArchive;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.runner.RunWith;

/**
 * <p>Test for RealmDirect login module that uses default <em>other</em>
 * security domain to perform a programmatic login context login.</p>
 *
 * @author rmartinc
 */
@RunWith(Arquillian.class)
@RunAsClient
public class RealmDirectLoginContextTestCase {

    @ArquillianResource
    private URL deploymentURL;

    @BeforeClass
    public static void beforeClass() {
        // legacy, don't run with elytron enabled
        AssumeTestGroupUtil.assumeElytronProfileEnabled();
    }

    @Deployment
    public static WebArchive createDeployment() {
        return ShrinkWrap.create(WebArchive.class, RealmDirectLoginContextTestCase.class.getSimpleName() + ".war")
                .addClasses(LoginContextServlet.class)
                .addAsManifestResource(PermissionUtils.createPermissionsXmlAsset(
                        new AuthPermission("createLoginContext.other"),
                        new RuntimePermission("org.jboss.security.config.SecurityConfiguration.addApplicationPolicy")),
                        "permissions.xml");
    }

    @Test
    public void testLoginOK() throws Exception {
        URI uri = new URI(deploymentURL.toExternalForm() + LoginContextServlet.SERVLET_PATH + "?username=user1&password=password1");
        String response = Utils.makeCall(uri, HttpServletResponse.SC_OK);
        MatcherAssert.assertThat(response, CoreMatchers.containsString("Login success"));
    }

    @Test
    public void testLoginInvalidPassword() throws Exception {
        URI uri = new URI(deploymentURL.toExternalForm() + LoginContextServlet.SERVLET_PATH + "?username=user1&password=bad-password");
        String response = Utils.makeCall(uri, HttpServletResponse.SC_OK);
        MatcherAssert.assertThat(response, CoreMatchers.containsString("Login failed"));
        MatcherAssert.assertThat(response, CoreMatchers.containsString("PBOX00070: Password invalid/Password required"));
    }

    @Test
    public void testLoginNoPassword() throws Exception {
        URI uri = new URI(deploymentURL.toExternalForm() + LoginContextServlet.SERVLET_PATH + "?username=user1");
        String response = Utils.makeCall(uri, HttpServletResponse.SC_OK);
        MatcherAssert.assertThat(response, CoreMatchers.containsString("Login failed"));
        MatcherAssert.assertThat(response, CoreMatchers.containsString("PBOX00070: Password invalid/Password required"));
    }

    @Test
    public void testLoginNoUser() throws Exception {
        URI uri = new URI(deploymentURL.toExternalForm() + LoginContextServlet.SERVLET_PATH + "?password=password1");
        String response = Utils.makeCall(uri, HttpServletResponse.SC_OK);
        MatcherAssert.assertThat(response, CoreMatchers.containsString("Login failed"));
        MatcherAssert.assertThat(response, CoreMatchers.containsString("PBOX00070: Password invalid/Password required"));
    }

    @Test
    public void testLoginNothing() throws Exception {
        URI uri = new URI(deploymentURL.toExternalForm() + LoginContextServlet.SERVLET_PATH);
        String response = Utils.makeCall(uri, HttpServletResponse.SC_OK);
        MatcherAssert.assertThat(response, CoreMatchers.containsString("Login failed"));
        MatcherAssert.assertThat(response, CoreMatchers.containsString("PBOX00070: Password invalid/Password required"));
    }
}
