/*
 * Copyright 2022 Red Hat, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.as.test.integration.domain;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import org.jboss.as.controller.PathAddress;
import org.jboss.as.controller.client.helpers.ClientConstants;
import org.jboss.as.controller.client.helpers.domain.DomainClient;
import org.jboss.as.controller.descriptions.ModelDescriptionConstants;
import org.jboss.as.test.integration.domain.management.util.DomainLifecycleUtil;
import org.jboss.as.test.integration.domain.management.util.WildFlyManagedConfiguration;
import org.jboss.dmr.ModelNode;
import org.junit.Test;
import org.jboss.as.controller.operations.common.Util;
import org.jboss.as.host.controller.HostControllerEnvironment;
import org.jboss.as.test.integration.domain.management.util.DomainTestUtils;
import org.jboss.as.test.integration.management.cli.CliProcessWrapper;
import org.jboss.as.test.shared.util.AssumeTestGroupUtil;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.rules.TemporaryFolder;

/**
 *
 * @author rmartinc
 */
public class DefaultConfigSmokeSE17TestCase extends BuildConfigurationTestBase {

    @ClassRule
    public static final TemporaryFolder WORKING_DIR = new TemporaryFolder();

    public static final String slaveAddress = System.getProperty("jboss.test.host.slave.address", "127.0.0.1");

    private static final Path JBOSS_DIST = Paths.get(System.getProperty("jboss.dist"));
    private static final Path JBOSS_HOME = Paths.get(System.getProperty("jboss.home"));
    private static final Path CONFIG_DIR_DIST = JBOSS_DIST.resolve("domain").resolve("configuration");
    private static final Path SCRIPT_FILE = JBOSS_HOME.resolve("docs").resolve("examples").resolve("enable-elytron-se17-domain.cli");

    @BeforeClass
    public static void runSE17ScriptForDomain() throws Exception {
        // only execute the class if elytron enabled (SE17)
        Assume.assumeTrue(AssumeTestGroupUtil.isElytronProfileEnabled());
        // copy files from distribution configuration to the working directory
        Files.walk(CONFIG_DIR_DIST, 1)
                .filter(Files::isRegularFile)
                .forEach(DefaultConfigSmokeSE17TestCase::copyFile);
        // execute the CLI file to perform the SE17 changes using the working directory
        CliProcessWrapper cli = new CliProcessWrapper()
                .addJavaOption("-D" + HostControllerEnvironment.DOMAIN_CONFIG_DIR
                        + "=" + WORKING_DIR.getRoot().getAbsolutePath())
                .addCliArgument("--file=" + SCRIPT_FILE.toAbsolutePath());
        String output = cli.executeNonInteractive();
        Assert.assertEquals("Script " + SCRIPT_FILE + " executed with errors: " + output, 0, cli.getProcessExitValue());
    }

    @Test
    public void testStandardHost() throws Exception {
        final WildFlyManagedConfiguration config = createConfiguration(WORKING_DIR.getRoot(),
                "domain.xml", "host.xml", getClass().getSimpleName());
        final DomainLifecycleUtil utils = new DomainLifecycleUtil(config);
        try {
            utils.start();

            DomainClient client = utils.getDomainClient();
            checkNoBootErrors(client, "master", "server-one", "server-two");
            checkProfile(client, "default");
            checkProfile(client, "full");
            checkProfile(client, "ha");
            checkProfile(client, "full-ha");
            checkExtensions(client);
            checkHost(client, "master");
        } finally {
            utils.stop(); // Stop
        }
    }

    @Test
    public void testMasterAndSlave() throws Exception {
        final WildFlyManagedConfiguration masterConfig = createConfiguration(WORKING_DIR.getRoot(),
                "domain.xml", "host-master.xml", getClass().getSimpleName());
        final DomainLifecycleUtil masterUtils = new DomainLifecycleUtil(masterConfig);
        final WildFlyManagedConfiguration slaveConfig = createConfiguration(WORKING_DIR.getRoot(),
                "domain.xml", "host-slave.xml", getClass().getSimpleName(),
                "slave", slaveAddress, 19990);
        final DomainLifecycleUtil slaveUtils = new DomainLifecycleUtil(slaveConfig);
        try {
            masterUtils.start();
            slaveUtils.start();

            DomainClient client = masterUtils.getDomainClient();
            checkNoBootErrors(client, "master");
            checkNoBootErrors(client, "slave", "server-one", "server-two");
            checkProfile(client, "default");
            checkProfile(client, "full");
            checkProfile(client, "ha");
            checkProfile(client, "full-ha");
            checkExtensions(client);
            checkHost(client, "master");
            checkHost(client, "slave");
        } finally {
            try {
                slaveUtils.stop();
            } finally {
                masterUtils.stop();
            }
        }
    }

    private static void copyFile(Path file) {
        try {
            Files.copy(file, WORKING_DIR.newFile(file.getFileName().toString()).toPath(), StandardCopyOption.REPLACE_EXISTING);
        } catch (IOException e) {
            throw new IllegalStateException(e);
        }
    }

    private boolean subsystemExists(DomainClient client, String profile, String name) throws Exception {
        ModelNode op =  Util.createEmptyOperation(ModelDescriptionConstants.READ_CHILDREN_NAMES_OPERATION,
                PathAddress.pathAddress(ModelDescriptionConstants.PROFILE, profile));
        op.get(ClientConstants.CHILD_TYPE).set(ModelDescriptionConstants.SUBSYSTEM);
        ModelNode res = DomainTestUtils.executeForResult(op, client);
        for (ModelNode node : res.asList()) {
            if (name.equals(node.asString())) {
                return true;
            }
        }
        return false;
    }

    private boolean extensionExists(DomainClient client, String name) throws Exception {
        ModelNode op =  Util.createEmptyOperation(ModelDescriptionConstants.READ_CHILDREN_NAMES_OPERATION,
                PathAddress.pathAddress());
        op.get(ClientConstants.CHILD_TYPE).set(ModelDescriptionConstants.EXTENSION);
        ModelNode res = DomainTestUtils.executeForResult(op, client);
        for (ModelNode node : res.asList()) {
            if (name.equals(node.asString())) {
                return true;
            }
        }
        return false;
    }

    private void checkPathExists(DomainClient client, PathAddress addr) throws Exception {
        ModelNode op =  Util.createEmptyOperation(ModelDescriptionConstants.READ_RESOURCE_OPERATION, addr);
        DomainTestUtils.executeForResult(op, client);
    }

    private void checkPathNotExists(DomainClient client, PathAddress addr) throws Exception {
        ModelNode op =  Util.createEmptyOperation(ModelDescriptionConstants.READ_RESOURCE_OPERATION, addr);
        DomainTestUtils.executeForFailure(op, client);
    }

    private void checkAttributeDefined(DomainClient client, PathAddress addr, String name) throws Exception {
        ModelNode op =  Util.createOperation(ModelDescriptionConstants.READ_ATTRIBUTE_OPERATION, addr);
        op.get(ModelDescriptionConstants.NAME).set(name);
        ModelNode res = DomainTestUtils.executeForResult(op, client);
        Assert.assertTrue("Attribute " + name + " is not defined", res.isDefined());
    }

    private void checkAttributeUndefined(DomainClient client, PathAddress addr, String name) throws Exception {
        ModelNode op =  Util.createOperation(ModelDescriptionConstants.READ_ATTRIBUTE_OPERATION, addr);
        op.get(ModelDescriptionConstants.NAME).set(name);
        ModelNode res = DomainTestUtils.executeForResult(op, client);
        Assert.assertFalse("Attribute " + name + " is defined", res.isDefined());
    }

    private void checkNoChildren(DomainClient client, PathAddress addr, String type) throws Exception {
        ModelNode op =  Util.createOperation(ModelDescriptionConstants.READ_CHILDREN_NAMES_OPERATION, addr);
        op.get(ModelDescriptionConstants.CHILD_TYPE).set(type);
        ModelNode res = DomainTestUtils.executeForResult(op, client);
        Assert.assertTrue("Child " + type + " is not empty", res.asList().isEmpty());
    }

    private void checkNoBootErrors(DomainClient client, String hostName, String... servers) throws Exception {
        ModelNode op = Util.createEmptyOperation("read-boot-errors",
                PathAddress.pathAddress(ModelDescriptionConstants.HOST, hostName)
                        .append(ModelDescriptionConstants.CORE_SERVICE, ModelDescriptionConstants.MANAGEMENT));
        ModelNode res = DomainTestUtils.executeForResult(op, client);
        Assert.assertTrue("Errors at boot", res.asList().isEmpty());

        if (servers != null) {
            for (String server : servers) {
                op = Util.createEmptyOperation("read-boot-errors",
                        PathAddress.pathAddress(ModelDescriptionConstants.HOST, hostName)
                                .append(ModelDescriptionConstants.SERVER, server)
                                .append(ModelDescriptionConstants.CORE_SERVICE, ModelDescriptionConstants.MANAGEMENT));
                res = DomainTestUtils.executeForResult(op, client);
                Assert.assertTrue("Errors at boot in server " + server, res.asList().isEmpty());
            }
        }
    }

    private void checkProfile(DomainClient client, String profile) throws Exception {
        // check elytron configuration is in place
        // /subsystem=elytron/http-authentication-factory=application-http-authentication
        checkPathExists(client, PathAddress.pathAddress(ModelDescriptionConstants.PROFILE, profile)
                .append(ModelDescriptionConstants.SUBSYSTEM, "elytron")
                .append("http-authentication-factory", "application-http-authentication"));
        // /subsystem=undertow/application-security-domain=other
        checkPathExists(client, PathAddress.pathAddress(ModelDescriptionConstants.PROFILE, profile)
                .append(ModelDescriptionConstants.SUBSYSTEM, "undertow")
                .append("application-security-domain", "other"));
        // /subsystem=undertow/server=default-server/https-listener=https:read-attribute(name=ssl-context)
        checkAttributeDefined(client, PathAddress.pathAddress(ModelDescriptionConstants.PROFILE, profile)
                .append(ModelDescriptionConstants.SUBSYSTEM, "undertow")
                .append("server", "default-server").append("https-listener", "https"), "ssl-context");
        // /subsystem=undertow/server=default-server/host=default-host/setting=http-invoker:read-attribute(name=http-authentication-factory)
        checkAttributeDefined(client, PathAddress.pathAddress(ModelDescriptionConstants.PROFILE, profile)
                .append(ModelDescriptionConstants.SUBSYSTEM, "undertow")
                .append("server", "default-server").append("host", "default-host").append("setting", "http-invoker"),
                "http-authentication-factory");
        // /subsystem=ejb3/application-security-domain=other
        checkPathExists(client, PathAddress.pathAddress(ModelDescriptionConstants.PROFILE, profile)
                .append(ModelDescriptionConstants.SUBSYSTEM, "ejb3")
                .append("application-security-domain", "other"));
        // /subsystem=batch-jberet:read-attribute(name=security-domain)
        checkAttributeDefined(client, PathAddress.pathAddress(ModelDescriptionConstants.PROFILE, profile)
                .append(ModelDescriptionConstants.SUBSYSTEM, "batch-jberet"),
                "security-domain");
        // activemq subsystem is not present in all configurations
        if (subsystemExists(client, profile, "messaging-activemq")) {
            // /subsystem=messaging-activemq/server=default:read-attribute(name=elytron-domain)
            checkAttributeDefined(client, PathAddress.pathAddress(ModelDescriptionConstants.PROFILE, profile)
                    .append(ModelDescriptionConstants.SUBSYSTEM, "messaging-activemq")
                    .append("server", "default"), "elytron-domain");
        }
        // /subsystem=remoting/http-connector=http-remoting-connector:read-attribute(name=sasl-authentication-factory)
        checkAttributeDefined(client, PathAddress.pathAddress(ModelDescriptionConstants.PROFILE, profile)
                .append(ModelDescriptionConstants.SUBSYSTEM, "remoting")
                .append("http-connector", "http-remoting-connector"), "sasl-authentication-factory");

        // subsystems and extensions removed
        Assert.assertFalse("Subsystem security is not removed", subsystemExists(client, profile, "security"));
        Assert.assertFalse("Subsystem picketlink-federation is not removed", subsystemExists(client, profile, "picketlink-federation"));
        Assert.assertFalse("Subsystem picketlink-identity-management is not removed", subsystemExists(client, profile, "picketlink-identity-management"));
    }

    private void checkHost(DomainClient client, String host) throws Exception {
        // /core-service=management/management-interface=http-interface:write-attribute(name=http-authentication-factory,value=management-http-authentication)
        checkAttributeDefined(client, PathAddress.pathAddress(ModelDescriptionConstants.HOST, host)
                .append(ModelDescriptionConstants.CORE_SERVICE, "management")
                .append(ModelDescriptionConstants.MANAGEMENT_INTERFACE, "http-interface"), "http-authentication-factory");
        // /core-service=management/management-interface=http-interface:undefine-attribute(name=security-realm)
        checkAttributeUndefined(client, PathAddress.pathAddress(ModelDescriptionConstants.HOST, host)
                .append(ModelDescriptionConstants.CORE_SERVICE, "management")
                .append(ModelDescriptionConstants.MANAGEMENT_INTERFACE, "http-interface"), "security-realm");
        // /core-service=management:read-children-names(child-type=security-realm)
        checkNoChildren(client, PathAddress.pathAddress(ModelDescriptionConstants.HOST, host)
                .append(ModelDescriptionConstants.CORE_SERVICE, "management"), "security-realm");
        // /core-service=vault:read-resource
        checkPathNotExists(client, PathAddress.pathAddress(ModelDescriptionConstants.HOST, host)
                .append(ModelDescriptionConstants.CORE_SERVICE, "vault"));
    }

    private void checkExtensions(DomainClient client) throws Exception {
        Assert.assertFalse("Extension org.jboss.as.security is not removed", extensionExists(client, "org.jboss.as.security"));
        Assert.assertFalse("Extension org.wildfly.extension.picketlink is not removed", extensionExists(client, "org.wildfly.extension.picketlink"));
    }
}
