package org.jboss.installer;

import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import java.io.File;
import java.nio.file.Path;

import static org.junit.Assert.*;

@SuppressWarnings("OptionalGetWithoutIsPresent")
public class ArgumentsParserTest {

    private ArgumentsParser argumentsParser = new ArgumentsParser();

    @Rule
    public TemporaryFolder temp = new TemporaryFolder();

    @Test
    public void testNoArguments() throws Exception {
        final ArgumentsParser.Arguments arguments = argumentsParser.parse(new String[]{});

        assertFalse(arguments.isAutomated());
        assertFalse(arguments.hasVariables());
    }

    @Test
    public void testAutomatedMode() throws Exception {
        final ArgumentsParser.Arguments arguments = argumentsParser.parse(new String[]{"auto.xml"});

        assertTrue(arguments.isAutomated());
        assertFalse(arguments.hasVariables());
    }

    @Test
    public void testUnattendedMode() throws Exception {
        final ArgumentsParser.Arguments arguments = argumentsParser.parse(new String[]{"auto.xml", "-variablesfile", "auto.xml.properties"});

        assertTrue(arguments.isAutomated());
        assertTrue(arguments.hasVariables());
    }

    @Test
    public void testOfflineMode() throws Exception {
        final String offlineRepo = temp.newFolder("offline").toPath().toAbsolutePath().toString();
        final ArgumentsParser.Arguments arguments = argumentsParser.parse(new String[]{"auto.xml", "-offline",
                offlineRepo});

        assertTrue(arguments.isAutomated());
        assertTrue(arguments.isOffline());
        assertFalse(arguments.hasVariables());
        assertEquals(offlineRepo, arguments.getOfflinePath().get().toString());
    }

    @Test
    public void testUnattendedOfflineMode() throws Exception {
        final Path offlineRepo = temp.newFolder("offline").toPath();
        final ArgumentsParser.Arguments arguments = argumentsParser.parse(
                new String[]{"auto.xml", "-variablesfile", "auto.xml.properties",
                        "-offline", offlineRepo.toString()});

        assertTrue(arguments.isAutomated());
        assertTrue(arguments.isOffline());
        assertTrue(arguments.hasVariables());
        assertEquals(offlineRepo, arguments.getOfflinePath().get());
    }

    @Test(expected = ArgumentsParsingException.class)
    public void testUnattendedModeInvertedArgumentIsIllegal() throws Exception {
        argumentsParser.parse(new String[]{"-variablesfile", "auto.xml.properties", "auto.xml"});
    }


    @Test(expected = ArgumentsParsingException.class)
    public void testMissingVariableFile() throws Exception {
        argumentsParser.parse(new String[]{"auto.xml", "-variablesfile"});
    }

    @Test(expected = ArgumentsParsingException.class)
    public void testMissingOfflinePath() throws Exception {
        argumentsParser.parse(new String[]{"auto.xml", "-offline"});
    }

    @Test(expected = ArgumentsParsingException.class)
    public void testTooManyParameters() throws Exception {
        argumentsParser.parse(new String[]{"auto.xml", "-variablesfile", "auto.xml.properties", "-foo"});
    }

    @Test(expected = ArgumentsParsingException.class)
    public void unrecognizedParameter() throws Exception {
        argumentsParser.parse(new String[]{"-foo"});
    }

    @Test(expected = ArgumentsParsingException.class)
    public void unrecognizedParameterAfterAutoFile() throws Exception {
        argumentsParser.parse(new String[]{"auto.xml", "-foo"});
    }

    @Test
    public void offlineArgumentAllowedInGUIMode() throws Exception {
        final File file = temp.newFile("test.zip");
        final ArgumentsParser.Arguments arguments = argumentsParser.parse(new String[]{"-offline", file.toPath().toString()});

        assertFalse(arguments.isAutomated());
        assertTrue(arguments.isOffline());
        assertEquals(file.toPath(), arguments.getOfflinePath().get());
    }

    @Test
    public void offlineRepoFileDoesntExist() throws Exception {
        assertThrows(ArgumentsParsingException.class, ()-> {
            argumentsParser.parse(new String[]{"-offline", "idontexist.zip"});
        });

    }
}