/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2023 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.installer.dialogs;

import org.jboss.installer.core.LanguageUtils;
import org.jboss.installer.core.MnemonicUtils;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.plaf.ComponentUI;
import javax.swing.plaf.metal.MetalFileChooserUI;

public class FileChooserWithMnemonicsUI extends MetalFileChooserUI {
    private static LanguageUtils langUtils;
    private JButton approveButton = null;
    private JButton cancelButton = null;
    /**
     * Constructs a new instance of {@code MetalFileChooserUI}.
     *
     * @param filechooser a {@code JFileChooser}
     */
    public FileChooserWithMnemonicsUI(JFileChooser filechooser) {
        super(filechooser);
    }

    public static ComponentUI createUI(JComponent c) {
        return new FileChooserWithMnemonicsUI((JFileChooser) c);
    }

    public static void setLangUtils(LanguageUtils langUtils) {
        FileChooserWithMnemonicsUI.langUtils = langUtils;
    }


    @Override
    protected JButton getApproveButton(JFileChooser fc) {
        return approveButton;
    }

    @Override
    public void installComponents(JFileChooser fc) {
        super.installComponents(fc);
        createNewControlButtons(fc);
    }

    private void createNewControlButtons(JFileChooser fc) {
        removeControlButtons();
        getButtonPanel().removeAll();

        if (approveButton == null) {
            approveButton = new JButton(getApproveButtonText(fc));
            approveButton.addActionListener(getApproveSelectionAction());
        }
        if (cancelButton == null){
            cancelButton = new JButton(getCancelText());
            cancelButton.addActionListener(getCancelSelectionAction());
        }
        getButtonPanel().add(cancelButton);
        getButtonPanel().add(approveButton);
        final MnemonicUtils mnemonicUtils = new MnemonicUtils();
        mnemonicUtils.findMnemonic(approveButton.getText(), true).ifPresent(approveButton::setMnemonic);
        mnemonicUtils.findMnemonic(cancelButton.getText(), true).ifPresent(cancelButton::setMnemonic);
        addControlButtons();
    }

    public String getApproveButtonText(JFileChooser fc){
        String text;
        if (fc.getDialogType() == JFileChooser.OPEN_DIALOG) {
            text = getOpenText();
        } else if (fc.getDialogType() == JFileChooser.SAVE_DIALOG) {
            text = getSaveText();
        } else {
            text = null;
        }
        if (text == null) {
            return null;
        }
        if (approveButton != null) {
            // the type of the dialog might have changed - change text and re-assign mnemonics
            if (!text.equals(approveButton.getText())) {
                approveButton.setText(text);
                final MnemonicUtils mnemonicUtils = new MnemonicUtils();
                mnemonicUtils.findMnemonic(approveButton.getText(), true).ifPresent(approveButton::setMnemonic);
                mnemonicUtils.findMnemonic(cancelButton.getText(), true).ifPresent(cancelButton::setMnemonic);
            }
        }

        return text;
    }

    private String getCancelText() {
        if (langUtils != null) {
            return langUtils.getString("generic.cancel");
        } else {
            return cancelButtonText;
        }
    }

    private String getOpenText() {
        if (langUtils != null) {
            return langUtils.getString("generic.open");
        } else {
            return openButtonText;
        }
    }

    private String getSaveText() {
        if (langUtils != null) {
            return langUtils.getString("generic.save");
        } else {
            return saveButtonText;
        }
    }
}
