/*
 * Copyright 2021 Red Hat, Inc. and/or its affiliates
 * and other contributors as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.validators;

import org.jboss.installer.core.FileUtils;
import org.jboss.installer.core.LanguageUtils;
import org.jboss.installer.core.ValidationResult;

import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.jar.JarFile;

public class JDBCDriverValidator {

    public static final String DUPLICATE_JAR_KEY = "jdbc_driver.duplicate_jar";
    public static final String DRIVER_CLASS_MISSING_KEY = "jdbc_driver.driver_class_missing";
    public static final String INVALID_JAR_KEY = "jdbc_driver.invalid_jar";
    public static final String EMPTY_JAR_PATH_KEY = "jdbc_driver.empty_jar";
    private final LanguageUtils langUtils;
    public JDBCDriverValidator(LanguageUtils langUtils) {
        this.langUtils = langUtils;
    }

    public ValidationResult validate(List<String> jarList, List<String> resolvedJarList, String xaClassName) {
        if (jarList.size() != resolvedJarList.size()) {
            throw new IllegalArgumentException("The jar lists passed as arguments have different sizes");
        }
        if (!allPathsAreUnique(jarList)) {
            return ValidationResult.error(langUtils.getString(DUPLICATE_JAR_KEY));
        }
        for (String jarPath : jarList) {
            if (jarPath == null) {
                return ValidationResult.error(langUtils.getString(EMPTY_JAR_PATH_KEY));
            }
            if (!jarPath.endsWith(".jar")) {
                return ValidationResult.error(langUtils.getString(INVALID_JAR_KEY, jarPath));
            }
        }

        for (String pathToJar : resolvedJarList) {
            try {
                if (jarContainsDriverClass(xaClassName, pathToJar)) {
                    return ValidationResult.ok();
                }
            } catch (IOException e) {
                return ValidationResult.error(langUtils.getString(INVALID_JAR_KEY, pathToJar));
            }
        }
        return ValidationResult.warning(langUtils.getString(DRIVER_CLASS_MISSING_KEY, xaClassName));
    }

    private boolean jarContainsDriverClass(String driverClassName, String pathToJar) throws IOException {
        JarFile jarFile = FileUtils.getJarFromPath(pathToJar);
        return jarFile.getJarEntry(getPathOfDriverClass(driverClassName)) != null;
    }

    private boolean allPathsAreUnique(List<String> jarList) {
        return jarList.size() == new HashSet<>(jarList).size();
    }

    private String getPathOfDriverClass(String driver) {
        return driver.replace(".", "/") + ".class";
    }
}
