/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2022 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.validators;

import org.apache.commons.lang3.StringUtils;
import org.jboss.installer.core.LanguageUtils;
import org.jboss.installer.core.ValidationResult;
import org.jboss.installer.postinstall.task.secdom.PropertiesSecurity;

import java.util.List;
import java.util.Optional;

import static org.jboss.installer.screens.domain.PropertiesSecurityDomainPanel.GROUPS_FILE_LABEL;
import static org.jboss.installer.screens.domain.PropertiesSecurityDomainPanel.HASH_SET_DESCRIPTION;
import static org.jboss.installer.screens.domain.PropertiesSecurityDomainPanel.USERS_FILE_LABEL;
import static org.jboss.installer.validators.CertificateSecurityDomainValidator.GENERIC_VALIDATION_EMPTY_FIELD;

public class PropertiesSecurityDomainValidator {

    private final LanguageUtils langUtils;

    public PropertiesSecurityDomainValidator(LanguageUtils langUtils) {
        this.langUtils = langUtils;
    }

    public ValidationResult validate(PropertiesSecurity.PropertiesFileConfig config) {
        if (config.getUsersFilePath() == null) {
            return ValidationResult.error(langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(USERS_FILE_LABEL)));
        }
        if (config.getGroupsFilePath() == null) {
            return ValidationResult.error(langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(GROUPS_FILE_LABEL)));
        }
        if (StringUtils.isEmpty(config.getHashCharset())) {
            return ValidationResult.error(langUtils.getString(GENERIC_VALIDATION_EMPTY_FIELD, langUtils.getString(HASH_SET_DESCRIPTION)));
        }

        final FileValidator fileValidator = new FileValidator(langUtils, true);
        final PathValidator pathValidator = PathValidator.newInstance("properties-domain", langUtils);
        final List<ValidationResult> results = List.of(fileValidator.validate(config.getUsersFilePath()),
                fileValidator.validate(config.getGroupsFilePath()),
                pathValidator.validate(config.getUsersFilePath().toString()),
                pathValidator.validate(config.getGroupsFilePath().toString()));
        final Optional<ValidationResult> error = results.stream().filter((res) -> res.getResult() == ValidationResult.Result.ERROR).findFirst();
        final Optional<ValidationResult> warning = results.stream().filter((res) -> res.getResult() == ValidationResult.Result.ERROR).findFirst();
        return error.orElseGet(() -> warning.orElseGet(ValidationResult::ok));
    }
}
