/*
 * Copyright The WildFly Authors
 * SPDX-License-Identifier: Apache-2.0
 */
package org.jboss.as.test.integration.management.api;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;

import java.io.IOException;
import java.util.List;

import org.jboss.arquillian.container.test.api.RunAsClient;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.as.controller.PathAddress;
import org.jboss.as.controller.descriptions.ModelDescriptionConstants;
import org.jboss.as.controller.operations.common.Util;
import org.jboss.as.test.integration.management.base.ContainerResourceMgmtTestBase;
import org.jboss.as.test.integration.management.util.MgmtOperationException;
import org.jboss.dmr.ModelNode;
import org.jboss.dmr.Property;
import org.junit.AssumptionViolatedException;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.runner.RunWith;

/**
 * Testing the product-info operation and the root management resource product information
 * on a standalone instance.
 * <p>
 * <strong>Note:</strong> WildFly Core's {@code org.wildfly.core.test.standalone.mgmt.api.core.ProductInfoUnitTestCase}
 * also covers {@code product-info}. The focus here is solely on checking values that cannot be tested in WildFly Core
 * because they are not set there.
 */
@RunWith(Arquillian.class)
@RunAsClient
public class ProductInfoTestCase  extends ContainerResourceMgmtTestBase {

    private static final String PRODUCT_NAME = System.getProperty("jboss.test.product.name");
    private static final String PRODUCT_VERSION = System.getProperty("jboss.test.product.version");

    @BeforeClass
    public static void beforeClass() {
        // If we're testing against an externally configured channel we have
        // no idea how it has set up the product info (particularly version),
        // so skip this test.
        if (System.getProperty("external.wildfly.channels") != null) {
            throw new AssumptionViolatedException("Unsuitable environment");
        }
    }

    @Test
    public void testProductInfoOperation() throws IOException, MgmtOperationException {
        List<Property> result = getProductInfo("product-info").asPropertyList();
        assertEquals(result.toString(), 1, result.size());
        assertEquals(result.toString(), "summary", result.get(0).getName());
        assertProductInfo(result.get(0).getValue());
    }

    @Test
    public void testRootResourceProductInfo() throws IOException, MgmtOperationException {
        ModelNode result = getProductInfo("read-resource");
        assertProductInfo(result);
    }

    private ModelNode getProductInfo(String operationName) throws IOException, MgmtOperationException {
        // Assert we were configured correctly to make debugging easier if we were not
        assertNotNull("Property jboss.test.product.name was not defined", PRODUCT_NAME);
        assertNotNull("Property jboss.test.product.version was not defined", PRODUCT_VERSION);
        assertFalse("Property jboss.test.product.name is empty", PRODUCT_NAME.isEmpty());
        assertFalse("Property jboss.test.product.version is empty", PRODUCT_VERSION.isEmpty());

        ModelNode op = Util.createEmptyOperation(operationName, PathAddress.EMPTY_ADDRESS);
        return executeOperation(op, true);

    }

    private static void assertProductInfo(ModelNode modelNode) {
        assertEquals(modelNode.toString(), PRODUCT_NAME, modelNode.get(ModelDescriptionConstants.PRODUCT_NAME).asString());
        assertEquals(modelNode.toString(), PRODUCT_VERSION, modelNode.get(ModelDescriptionConstants.PRODUCT_VERSION).asString());
    }
}
