/*
 * Copyright The WildFly Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package org.jboss.as.controller.registry;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.jboss.as.controller.OperationDefinition;
import org.jboss.as.controller.OperationStepHandler;
import org.jboss.as.controller.access.management.AccessConstraintDefinition;
import org.jboss.as.controller.descriptions.DescriptionProvider;

/**
 * Information about a registered {@code OperationStepHandler}.
 *
 * @author Emanuel Muckenhuber
 */
public class OperationEntry {
    public enum EntryType {
        PUBLIC, PRIVATE
    }

    /** Flags to indicate special characteristics of an operation */
    public enum Flag {
        /** Operation only reads, does not modify */
        READ_ONLY,
        /** The operation modifies the configuration and can be applied to the runtime without requiring a restart */
        RESTART_NONE,
        /** The operation modifies the configuration but can only be applied to the runtime via a full jvm restart */
        RESTART_JVM,
        /** The operation modifies the configuration but can only be applied to the runtime via a restart of all services;
         *  however it does not require a full jvm restart */
        RESTART_ALL_SERVICES,
        /** The operation modifies the configuration but can only be applied to the runtime via a restart of services,
         *  associated with the affected resource, but does not require a restart of all services or a full jvm restart */
        RESTART_RESOURCE_SERVICES,
        /** A domain or host-level operation that should be pushed to the servers even if the default behavior
         *  would indicate otherwise */
        DOMAIN_PUSH_TO_SERVERS,
        /** A host-level operation that should only be executed on the HostController and not on the servers,
         * even if the default behavior would indicate otherwise */
        HOST_CONTROLLER_ONLY,
        /** A domain-level operation that should only be executed on the master HostController and not on the slaves,
         * even if the default behavior would indicate otherwise */
        MASTER_HOST_CONTROLLER_ONLY,
        /** Operations with this flag do not affect the mode or change the installed services. The main intention for
         * this is to only make RUNTIME_ONLY methods on domain mode servers visible to end users. */
        RUNTIME_ONLY,
        /** Operations with this flag do not appear in management API description output but still can be invoked
         *  by external callers.  This is meant for operations that were not meant to be part of the supported external
         *  management API but users may have learned of them. Such ops should be evaluated for inclusion as normally
         *  described ops, or perhaps should be marked with {@link EntryType#PRIVATE} and external use thus disabled.
         *  This can also be used for ops that are invoked internally on one domain process by another domain process
         *  but where it's not possible for the caller to suppress the caller-type=user header from the op, making
         *  use of {@link EntryType#PRIVATE} not workable. */
        HIDDEN;

        private static final Map<Set<Flag>, Set<Flag>> flagSets = new ConcurrentHashMap<>(16);
        public static Set<OperationEntry.Flag> immutableSetOf(Set<OperationEntry.Flag> flags) {
            if (flags == null || flags.isEmpty()) {
                return Collections.emptySet();
            }
            Set<Flag> result = flagSets.get(flags);
            if (result == null) {
                Set<Flag> immutable = Collections.unmodifiableSet(flags);
                Set<Flag> existing = flagSets.putIfAbsent(flags, immutable);
                result = existing == null ? immutable : existing;
            }

            return result;
        }

    }

    private final OperationDefinition operationDefinition;
    private final OperationStepHandler operationHandler;
    private final boolean inherited;

    OperationEntry(final OperationDefinition definition, final OperationStepHandler operationHandler, final boolean inherited) {
        this.operationDefinition = definition;
        this.operationHandler = operationHandler;
        this.inherited = inherited;
    }

    public OperationDefinition getOperationDefinition() {
        return operationDefinition;
    }

    public OperationStepHandler getOperationHandler() {
        return operationHandler;
    }

    public DescriptionProvider getDescriptionProvider() {
        return operationDefinition.getDescriptionProvider();
    }

    public boolean isInherited() {
        return inherited;
    }

    public EntryType getType() {
        return operationDefinition.getEntryType();
    }

    public Set<Flag> getFlags() {
        return operationDefinition.getFlags();
    }

    public List<AccessConstraintDefinition> getAccessConstraints() {
        List<AccessConstraintDefinition> accessConstraints = operationDefinition.getAccessConstraints();
        return accessConstraints == null ? Collections.emptyList() : accessConstraints;
    }

}
