/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jboss.installer.validators;

import org.jboss.installer.core.DatabaseDriver;
import org.jboss.installer.core.LanguageUtils;
import org.jboss.installer.core.ValidationResult;
import org.jboss.installer.test.utils.MockLanguageUtils;
import org.junit.Assert;
import org.junit.Test;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

import static org.jboss.installer.validators.JDBCDriverValidator.DRIVER_CLASS_MISSING_KEY;
import static org.jboss.installer.validators.JDBCDriverValidator.DUPLICATE_JAR_KEY;
import static org.jboss.installer.validators.JDBCDriverValidator.INVALID_JAR_KEY;

public class JDBCDriverValidatorTest {

    public static final String INVALID_PATH_1 = "/invalid/path1";
    public static final String INVALID_PATH_2 = "/invalid/path2";
    LanguageUtils languageUtils = new MockLanguageUtils();
    JDBCDriverValidator validator = new JDBCDriverValidator(languageUtils);
    Path pathToIBMDB2Jar = Path.of(
            JDBCDriverValidatorTest.class.getClassLoader().getResource("jars/ibm-db2.jar").toURI()).toAbsolutePath();

    public JDBCDriverValidatorTest() throws Exception {
    }

    @Test
    public void uniquePathsTest() {
        List<String> uniqueJarsList = new ArrayList<>();
        uniqueJarsList.add(INVALID_PATH_1);
        uniqueJarsList.add(INVALID_PATH_2);

        ValidationResult result = validator.validate(uniqueJarsList, uniqueJarsList, DatabaseDriver.IBM_DB2.getXaClassName());
        Assert.assertTrue(result.getMessage().startsWith(INVALID_JAR_KEY));
        Assert.assertEquals(result.getMessage(), ValidationResult.Result.ERROR, result.getResult());
    }

    @Test
    public void notUniquePathsTest() {
        List<String> notUniqueJarsList = new ArrayList<>();
        notUniqueJarsList.add(INVALID_PATH_1);
        notUniqueJarsList.add(INVALID_PATH_1);

        ValidationResult result = validator.validate(notUniqueJarsList, notUniqueJarsList, DatabaseDriver.IBM_DB2.getXaClassName());
        Assert.assertEquals(DUPLICATE_JAR_KEY, result.getMessage());
        Assert.assertEquals(result.getMessage(), ValidationResult.Result.ERROR, result.getResult());
    }

    @Test
    public void noneOfTheJarContainDriverClassTest() {
        List<String> jarWithoutDriverClass = new ArrayList<>();
        jarWithoutDriverClass.add(pathToIBMDB2Jar.toString());
        ValidationResult result = validator.validate(jarWithoutDriverClass, jarWithoutDriverClass, DatabaseDriver.ORACLE.getXaClassName());
        Assert.assertEquals(result.getMessage(), ValidationResult.Result.WARNING, result.getResult());
    }

    @Test
    public void jarContainsDriverClassTest() {
        List<String> jarWithDriverClass = new ArrayList<>();
        jarWithDriverClass.add(pathToIBMDB2Jar.toString());
        ValidationResult result = validator.validate(jarWithDriverClass, jarWithDriverClass, DatabaseDriver.IBM_DB2.getXaClassName());
        Assert.assertEquals(result.getMessage(), ValidationResult.Result.OK, result.getResult());
    }

    @Test
    public void jarDoesNotContainDriverClassTest() {
        List<String> jarWithoutDriverClass = new ArrayList<>();
        String invalidJar = JDBCDriverValidatorTest.class.getClassLoader().getResource("jars/mojarra-impl.jar").getPath();
        jarWithoutDriverClass.add(invalidJar);
        ValidationResult result = validator.validate(jarWithoutDriverClass, jarWithoutDriverClass, DatabaseDriver.IBM_DB2.getXaClassName());
        Assert.assertEquals(result.getMessage(), ValidationResult.Result.WARNING, result.getResult());
        Assert.assertEquals(languageUtils.getString(DRIVER_CLASS_MISSING_KEY, DatabaseDriver.IBM_DB2.getXaClassName()), result.getMessage());
    }

    @Test
    public void oneOfTheJarPathsIsInvalidTest() {
        List<String> jarList = new ArrayList<>();
        jarList.add(pathToIBMDB2Jar.toString());
        jarList.add(INVALID_PATH_1);
        ValidationResult result = validator.validate(jarList, jarList, DatabaseDriver.IBM_DB2.getXaClassName());
        Assert.assertEquals(result.getMessage(), ValidationResult.Result.ERROR, result.getResult());
    }
}
