/*
 * Copyright The WildFly Authors
 * SPDX-License-Identifier: Apache-2.0
 */
package org.jboss.as.cli.parsing.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import java.util.List;

import org.jboss.as.cli.CommandFormatException;
import org.jboss.as.cli.operation.impl.DefaultCallbackHandler;
import org.junit.Test;

/**
 *
 * @author Alexey Loubyansky
 */
public class CommandTestCase {

    @Test
    public void testCommandOnly() throws Exception {

        DefaultCallbackHandler cmd = parse("some-command");
        assertEquals("some-command", cmd.getOperationName());
        assertFalse(cmd.hasProperties());
        assertFalse(cmd.hasOperator());
        assertFalse(cmd.endsOnSeparator());
        assertFalse(cmd.endsOnPropertyListStart());
        assertFalse(cmd.endsOnPropertySeparator());
        assertEquals(0, cmd.getLastChunkIndex());
    }

    @Test
    public void testPropertyListStart() throws Exception {

        DefaultCallbackHandler cmd = parse("some-command ");
        assertEquals("some-command", cmd.getOperationName());
        assertFalse(cmd.hasProperties());
        assertFalse(cmd.hasOperator());
        assertTrue(cmd.endsOnSeparator());
        assertTrue(cmd.endsOnPropertyListStart());
        assertFalse(cmd.endsOnPropertySeparator());
        assertFalse(cmd.endsOnPropertyValueSeparator());
        assertEquals(0, cmd.getLastChunkIndex());
    }

    @Test
    public void testSingleArgValue() throws Exception {

        DefaultCallbackHandler cmd = parse("some-command arg-value");
        assertEquals("some-command", cmd.getOperationName());
        assertTrue(cmd.hasProperties());
        assertFalse(cmd.hasOperator());
        assertFalse(cmd.endsOnSeparator());
        assertFalse(cmd.endsOnPropertyListStart());
        assertFalse(cmd.endsOnPropertySeparator());
        assertFalse(cmd.endsOnPropertyValueSeparator());
        assertTrue(cmd.getPropertyNames().isEmpty());
        assertEquals(1, cmd.getOtherProperties().size());
        assertEquals("arg-value", cmd.getOtherProperties().get(0));
        assertEquals(13, cmd.getLastChunkIndex());
    }

    @Test
    public void testSingleArgValueAndSeparator() throws Exception {

        DefaultCallbackHandler cmd = parse("some-command arg-value ");
        assertEquals("some-command", cmd.getOperationName());
        assertTrue(cmd.hasProperties());
        assertFalse(cmd.hasOperator());
        assertTrue(cmd.endsOnSeparator());
        assertFalse(cmd.endsOnPropertyListStart());
        assertTrue(cmd.endsOnPropertySeparator());
        assertFalse(cmd.endsOnPropertyValueSeparator());
        assertTrue(cmd.getPropertyNames().isEmpty());
        assertEquals(1, cmd.getOtherProperties().size());
        assertEquals("arg-value", cmd.getOtherProperties().get(0));
        assertEquals(13, cmd.getLastChunkIndex());
    }

    @Test
    public void testSingleArgName() throws Exception {

        DefaultCallbackHandler cmd = parse("some-command --arg-name");
        assertEquals("some-command", cmd.getOperationName());
        assertTrue(cmd.hasProperties());
        assertFalse(cmd.hasOperator());
        assertFalse(cmd.endsOnSeparator());
        assertFalse(cmd.endsOnPropertyListStart());
        assertFalse(cmd.endsOnPropertySeparator());
        assertFalse(cmd.endsOnPropertyValueSeparator());
        assertEquals(1, cmd.getPropertyNames().size());
        assertTrue(cmd.getOtherProperties().isEmpty());
        assertTrue(cmd.hasProperty("--arg-name"));
        assertTrue(cmd.getPropertyValue("--arg-name").equals("true"));
        assertEquals(13, cmd.getLastChunkIndex());
    }

    @Test
    public void testSingleArgNameAndSeparator() throws Exception {

        DefaultCallbackHandler cmd = parse("some-command --arg-name ");
        assertEquals("some-command", cmd.getOperationName());
        assertTrue(cmd.hasProperties());
        assertFalse(cmd.hasOperator());
        assertTrue(cmd.endsOnSeparator());
        assertFalse(cmd.endsOnPropertyListStart());
        assertTrue(cmd.endsOnPropertySeparator());
        assertFalse(cmd.endsOnPropertyValueSeparator());
        assertEquals(1, cmd.getPropertyNames().size());
        assertTrue(cmd.getOtherProperties().isEmpty());
        assertTrue(cmd.hasProperty("--arg-name"));
        assertTrue(cmd.getPropertyValue("--arg-name").equals("true"));
        assertEquals(13, cmd.getLastChunkIndex());
    }

    @Test
    public void testSingleArgNameAndValueSeparator() throws Exception {

        DefaultCallbackHandler cmd = parse("some-command --arg-name=");
        assertEquals("some-command", cmd.getOperationName());
        assertTrue(cmd.hasProperties());
        assertFalse(cmd.hasOperator());
        assertTrue(cmd.endsOnSeparator());
        assertFalse(cmd.endsOnPropertyListStart());
        assertFalse(cmd.endsOnPropertySeparator());
        assertTrue(cmd.endsOnPropertyValueSeparator());
        assertEquals(1, cmd.getPropertyNames().size());
        assertTrue(cmd.getOtherProperties().isEmpty());
        assertTrue(cmd.hasProperty("--arg-name"));
        assertNull(cmd.getPropertyValue("--arg-name"));
        assertEquals(23, cmd.getLastChunkIndex());
    }

    @Test
    public void testSingleArgNameWithValue() throws Exception {

        DefaultCallbackHandler cmd = parse("some-command --arg-name=value");
        assertEquals("some-command", cmd.getOperationName());
        assertTrue(cmd.hasProperties());
        assertFalse(cmd.hasOperator());
        assertFalse(cmd.endsOnSeparator());
        assertFalse(cmd.endsOnPropertyListStart());
        assertFalse(cmd.endsOnPropertySeparator());
        assertFalse(cmd.endsOnPropertyValueSeparator());
        assertEquals(1, cmd.getPropertyNames().size());
        assertTrue(cmd.getOtherProperties().isEmpty());
        assertTrue(cmd.hasProperty("--arg-name"));
        assertEquals("value", cmd.getPropertyValue("--arg-name"));
        assertEquals(24, cmd.getLastChunkIndex());
    }

    @Test
    public void testSingleArgNameWithValueAndSeparator() throws Exception {

        DefaultCallbackHandler cmd = parse("some-command --arg-name=value ");
        assertEquals("some-command", cmd.getOperationName());
        assertTrue(cmd.hasProperties());
        assertFalse(cmd.hasOperator());
        assertTrue(cmd.endsOnSeparator());
        assertFalse(cmd.endsOnPropertyListStart());
        assertTrue(cmd.endsOnPropertySeparator());
        assertFalse(cmd.endsOnPropertyValueSeparator());
        assertEquals(1, cmd.getPropertyNames().size());
        assertTrue(cmd.getOtherProperties().isEmpty());
        assertTrue(cmd.hasProperty("--arg-name"));
        assertEquals("value", cmd.getPropertyValue("--arg-name"));
        assertEquals(24, cmd.getLastChunkIndex());
    }

    @Test
    public void testEmptyString() throws Exception {

        DefaultCallbackHandler cmd = parse("");
        assertFalse(cmd.hasOperationName());
        assertFalse(cmd.hasProperties());
        assertFalse(cmd.hasOperator());
        assertFalse(cmd.endsOnSeparator());
        assertFalse(cmd.endsOnAddressOperationNameSeparator());
        assertFalse(cmd.endsOnPropertyListStart());
        assertFalse(cmd.endsOnPropertySeparator());
        assertFalse(cmd.endsOnPropertyValueSeparator());
        assertTrue(cmd.getPropertyNames().isEmpty());
        assertTrue(cmd.getOtherProperties().isEmpty());
        assertEquals(0, cmd.getLastChunkIndex());
    }

    @Test
    public void testWhitespaces() throws Exception {

        DefaultCallbackHandler cmd = parse("   ");
        assertFalse(cmd.hasOperationName());
        assertFalse(cmd.hasProperties());
        assertFalse(cmd.hasOperator());
        assertTrue(cmd.endsOnSeparator());
        assertTrue(cmd.endsOnAddressOperationNameSeparator());
        assertFalse(cmd.endsOnPropertyListStart());
        assertFalse(cmd.endsOnPropertySeparator());
        assertFalse(cmd.endsOnPropertyValueSeparator());
        assertTrue(cmd.getPropertyNames().isEmpty());
        assertTrue(cmd.getOtherProperties().isEmpty());
        assertEquals(0, cmd.getLastChunkIndex());
    }

    @Test
    public void testCommandWithArgsAndOutputTarget() throws Exception {

        DefaultCallbackHandler cmd = parse(" some-command --name=value --name1 value1 > command.log");
        assertEquals("some-command", cmd.getOperationName());
        assertTrue(cmd.hasProperties());
        assertTrue(cmd.hasProperty("--name"));
        assertEquals("value", cmd.getPropertyValue("--name"));
        assertTrue(cmd.hasProperty("--name1"));
        assertTrue(cmd.getPropertyValue("--name1").equals("true"));

        List<String> otherArgs = cmd.getOtherProperties();
        assertEquals(1, otherArgs.size());
        assertEquals("value1", otherArgs.get(0));

        assertTrue("command.log", cmd.hasOperator());
    }

    @Test
    public void testCommandWithArgsAndAppendTarget() throws Exception {

        DefaultCallbackHandler cmd = parse(" some-command --name=value --name1 value1 >> command.log");
        assertEquals("some-command", cmd.getOperationName());
        assertTrue(cmd.hasProperties());
        assertTrue(cmd.hasProperty("--name"));
        assertEquals("value", cmd.getPropertyValue("--name"));
        assertTrue(cmd.hasProperty("--name1"));
        assertTrue(cmd.getPropertyValue("--name1").equals("true"));

        List<String> otherArgs = cmd.getOtherProperties();
        assertEquals(1, otherArgs.size());
        assertEquals("value1", otherArgs.get(0));

        assertTrue("No operator", cmd.hasOperator());
    }

    @Test
    public void testCommandWithArgsAndPipe() throws Exception {

        DefaultCallbackHandler cmd = parse(" some-command --name=value --name1 value1 | grep ");
        assertEquals("some-command", cmd.getOperationName());
        assertTrue(cmd.hasProperties());
        assertTrue(cmd.hasProperty("--name"));
        assertEquals("value", cmd.getPropertyValue("--name"));
        assertTrue(cmd.hasProperty("--name1"));
        assertTrue(cmd.getPropertyValue("--name1").equals("true"));

        List<String> otherArgs = cmd.getOtherProperties();
        assertEquals(1, otherArgs.size());
        assertEquals("value1", otherArgs.get(0));

        assertTrue("No operator", cmd.hasOperator());
    }

    @Test
    public void testQuotedArgument() throws Exception {

        DefaultCallbackHandler cmd = parse(" some-command \"a b\"");
        assertEquals("some-command", cmd.getOperationName());
        assertTrue(cmd.hasProperties());
        assertTrue(cmd.getPropertyNames().isEmpty());
        final List<String> props = cmd.getOtherProperties();
        assertEquals(1, props.size());
        assertEquals("\"a b\"", props.get(0));
    }

    @Test
    public void testQuotesInQuotes() throws Exception {

        DefaultCallbackHandler cmd = parse("cmd --arg=\"a \\\"b\\\"\"");
        assertEquals("cmd", cmd.getOperationName());
        assertTrue(cmd.hasProperties());
        assertEquals(1, cmd.getPropertyNames().size());
        assertEquals(0, cmd.getOtherProperties().size());
        assertEquals("\"a \\\"b\\\"\"", cmd.getPropertyValue("--arg"));
    }

    @Test
    public void testCurlyBrackets() throws Exception {
        DefaultCallbackHandler cmd = parse("cmd --arg={a ");
        assertEquals("cmd", cmd.getOperationName());
        assertEquals("--arg", cmd.getLastParsedPropertyName());
        assertEquals("{a ", cmd.getPropertyValue("--arg"));
    }

    @Test
    public void testSpaceAfterValue() throws Exception {
        DefaultCallbackHandler cmd = parse("cmd --arg=a ");
        assertEquals("cmd", cmd.getOperationName());
        assertEquals("a", cmd.getPropertyValue("--arg"));
    }

    @Test
    public void testLeadingSpacesInArgumentValueAreTrimmed() throws Exception {
        DefaultCallbackHandler cmd = parse("cmd --arg= a");
        assertEquals("cmd", cmd.getOperationName());
        assertEquals("a", cmd.getPropertyValue("--arg"));
    }

    @Test
    public void testCurlyBracketsClosed() throws Exception {
        DefaultCallbackHandler cmd = parse("cmd --arg={a} b");
        assertEquals("cmd", cmd.getOperationName());
        assertNull(cmd.getLastParsedPropertyName());
        assertEquals("{a}", cmd.getPropertyValue("--arg"));
    }

    protected DefaultCallbackHandler parse(String line) {
        DefaultCallbackHandler args = new DefaultCallbackHandler();
        try {
            args.parse(null, line, null);
        } catch (CommandFormatException e) {
            e.printStackTrace();
            org.junit.Assert.fail(e.getLocalizedMessage());
        }
        return args;
    }
}
