/*
 * Copyright The WildFly Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package org.jboss.as.test.integration.domain.rbac;

import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.ADD;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.DEFAULT;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.HOST;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.REMOVE;
import static org.jboss.as.controller.descriptions.ModelDescriptionConstants.SHUTDOWN;
import static org.jboss.as.test.integration.management.rbac.RbacUtil.ADMINISTRATOR_USER;
import static org.jboss.as.test.integration.management.rbac.RbacUtil.AUDITOR_USER;
import static org.jboss.as.test.integration.management.rbac.RbacUtil.DEPLOYER_USER;
import static org.jboss.as.test.integration.management.rbac.RbacUtil.MAINTAINER_USER;
import static org.jboss.as.test.integration.management.rbac.RbacUtil.MONITOR_USER;
import static org.jboss.as.test.integration.management.rbac.RbacUtil.OPERATOR_USER;
import static org.jboss.as.test.integration.management.rbac.RbacUtil.SUPERUSER_USER;
import static org.jboss.as.test.integration.management.util.ModelUtil.createOpNode;

import java.io.IOException;

import org.jboss.as.controller.PathAddress;
import org.jboss.as.controller.client.ModelControllerClient;
import org.jboss.as.controller.operations.common.Util;
import org.jboss.as.test.integration.management.rbac.Outcome;
import org.jboss.as.test.integration.management.rbac.RbacUtil;
import org.jboss.dmr.ModelNode;
import org.junit.After;
import org.junit.Test;

/**
 * Abstract superclass of access control provider test cases covering the standard roles.
 *
 * @author Brian Stansberry (c) 2013 Red Hat Inc.
 */
public abstract class AbstractStandardRolesTestCase extends AbstractRbacTestCase implements RbacDomainRolesTests {


    @After
    public void tearDown() throws IOException {
        AssertionError assertionError = null;
        try {
            removeResource(DEPLOYMENT_2);
        } catch (AssertionError e) {
            assertionError = e;
        } finally {
            removeResource(TEST_PATH);
        }


        if (assertionError != null) {
            throw assertionError;
        }
    }

    protected abstract boolean isAllowLocalAuth();

    @Test
    public void testMonitor() throws Exception {
        ModelControllerClient client = getClientForUser(MONITOR_USER, isAllowLocalAuth(), primaryClientConfig);
        readWholeConfig(client, Outcome.UNAUTHORIZED, MONITOR_USER);
        checkStandardReads(client, null, null, MONITOR_USER);
        checkRootRead(client, null, null, Outcome.SUCCESS, MONITOR_USER);
        checkRootRead(client, PRIMARY, null, Outcome.SUCCESS, MONITOR_USER);
        checkRootRead(client, PRIMARY, PRIMARY_A, Outcome.SUCCESS, MONITOR_USER);
        checkSecurityDomainRead(client, null, null, Outcome.HIDDEN, MONITOR_USER);
        checkSecurityDomainRead(client, PRIMARY, PRIMARY_A, Outcome.HIDDEN, MONITOR_USER);
        checkSensitiveAttribute(client, null, null, false, MONITOR_USER);
        checkSensitiveAttribute(client, PRIMARY, PRIMARY_A, false, MONITOR_USER);

        if (readOnly) return;

        runGC(client, PRIMARY, null, Outcome.UNAUTHORIZED, MONITOR_USER);
        runGC(client, PRIMARY, PRIMARY_A, Outcome.UNAUTHORIZED, MONITOR_USER);
        addDeployment2(client, Outcome.UNAUTHORIZED, MONITOR_USER);
        addPath(client, Outcome.UNAUTHORIZED, MONITOR_USER);
        removeSecurityDomain(client, Outcome.HIDDEN, MONITOR_USER);
        restartServer(client, PRIMARY, PRIMARY_A, Outcome.UNAUTHORIZED, MONITOR_USER);

        // Monitor can't shutdown
        testWCORE1067(client, MONITOR_USER);

        stopServer(client, PRIMARY, PRIMARY_A, Outcome.UNAUTHORIZED, MONITOR_USER);
        killServer(client, PRIMARY, PRIMARY_A, Outcome.UNAUTHORIZED, MONITOR_USER);
        destroyServer(client, PRIMARY, PRIMARY_A, Outcome.UNAUTHORIZED, MONITOR_USER);
        killServersInGroup(client, Outcome.UNAUTHORIZED, MONITOR_USER);
        destroyServersInGroup(client, Outcome.UNAUTHORIZED, MONITOR_USER);
    }

    @Test
    public void testOperator() throws Exception {
        ModelControllerClient client = getClientForUser(OPERATOR_USER, isAllowLocalAuth(), primaryClientConfig);
        readWholeConfig(client, Outcome.UNAUTHORIZED, OPERATOR_USER);
        checkStandardReads(client, null, null, OPERATOR_USER);
        checkRootRead(client, null, null, Outcome.SUCCESS, OPERATOR_USER);
        checkRootRead(client, PRIMARY, null, Outcome.SUCCESS, OPERATOR_USER);
        checkRootRead(client, PRIMARY, PRIMARY_A, Outcome.SUCCESS, OPERATOR_USER);
        checkSecurityDomainRead(client, null, null, Outcome.HIDDEN, OPERATOR_USER);
        checkSecurityDomainRead(client, PRIMARY, PRIMARY_A, Outcome.HIDDEN, OPERATOR_USER);
        checkSensitiveAttribute(client, null, null, false, OPERATOR_USER);
        checkSensitiveAttribute(client, PRIMARY, PRIMARY_A, false, OPERATOR_USER);

        if (readOnly) return;

        runGC(client, PRIMARY, null, Outcome.SUCCESS, OPERATOR_USER);
        runGC(client, PRIMARY, PRIMARY_A, Outcome.SUCCESS, OPERATOR_USER);
        addDeployment2(client, Outcome.UNAUTHORIZED, OPERATOR_USER);
        addPath(client, Outcome.UNAUTHORIZED, OPERATOR_USER);
        removeSecurityDomain(client, Outcome.HIDDEN, OPERATOR_USER);
        restartServer(client, PRIMARY, PRIMARY_A, Outcome.SUCCESS, OPERATOR_USER);
    }

    @Test
    public void testMaintainer() throws Exception {
        ModelControllerClient client = getClientForUser(MAINTAINER_USER, isAllowLocalAuth(), primaryClientConfig);
        readWholeConfig(client, Outcome.UNAUTHORIZED, MAINTAINER_USER);
        checkStandardReads(client, null, null, MAINTAINER_USER);
        checkRootRead(client, null, null, Outcome.SUCCESS, MAINTAINER_USER);
        checkRootRead(client, PRIMARY, null, Outcome.SUCCESS, MAINTAINER_USER);
        checkRootRead(client, PRIMARY, PRIMARY_A, Outcome.SUCCESS, MAINTAINER_USER);
        checkSecurityDomainRead(client, null, null, Outcome.HIDDEN, MAINTAINER_USER);
        checkSecurityDomainRead(client, PRIMARY, PRIMARY_A, Outcome.HIDDEN, MAINTAINER_USER);
        checkSensitiveAttribute(client, null, null, false, MAINTAINER_USER);
        checkSensitiveAttribute(client, PRIMARY, PRIMARY_A, false, MAINTAINER_USER);

        if (readOnly) return;

        runGC(client, PRIMARY, null, Outcome.SUCCESS, MAINTAINER_USER);
        runGC(client, PRIMARY, PRIMARY_A, Outcome.SUCCESS, MAINTAINER_USER);
        addDeployment2(client, Outcome.SUCCESS, MAINTAINER_USER);
        addPath(client, Outcome.SUCCESS, MAINTAINER_USER);
        removeSecurityDomain(client, Outcome.HIDDEN, MAINTAINER_USER);
    }

    @Test
    public void testDeployer() throws Exception {
        ModelControllerClient client = getClientForUser(DEPLOYER_USER, isAllowLocalAuth(), primaryClientConfig);
        readWholeConfig(client, Outcome.UNAUTHORIZED, DEPLOYER_USER);
        checkStandardReads(client, null, null, DEPLOYER_USER);
        checkRootRead(client, null, null, Outcome.SUCCESS, DEPLOYER_USER);
        checkRootRead(client, PRIMARY, null, Outcome.SUCCESS, DEPLOYER_USER);
        checkRootRead(client, PRIMARY, PRIMARY_A, Outcome.SUCCESS, DEPLOYER_USER);
        checkSecurityDomainRead(client, null, null, Outcome.HIDDEN, DEPLOYER_USER);
        checkSecurityDomainRead(client, PRIMARY, PRIMARY_A, Outcome.HIDDEN, DEPLOYER_USER);
        checkSensitiveAttribute(client, null, null, false, DEPLOYER_USER);
        checkSensitiveAttribute(client, PRIMARY, PRIMARY_A, false, DEPLOYER_USER);

        if (readOnly) return;

        runGC(client, PRIMARY, null, Outcome.UNAUTHORIZED, DEPLOYER_USER);
        runGC(client, PRIMARY, PRIMARY_A, Outcome.UNAUTHORIZED, DEPLOYER_USER);
        addDeployment2(client, Outcome.SUCCESS, DEPLOYER_USER);
        addPath(client, Outcome.UNAUTHORIZED, DEPLOYER_USER);
        removeSecurityDomain(client, Outcome.HIDDEN, DEPLOYER_USER);
        restartServer(client, PRIMARY, PRIMARY_A, Outcome.UNAUTHORIZED, DEPLOYER_USER);

        // Deployer can't shutdown
        testWCORE1067(client, DEPLOYER_USER);

        stopServer(client, PRIMARY, PRIMARY_A, Outcome.UNAUTHORIZED, DEPLOYER_USER);
        killServer(client, PRIMARY, PRIMARY_A, Outcome.UNAUTHORIZED, DEPLOYER_USER);
        destroyServer(client, PRIMARY, PRIMARY_A, Outcome.UNAUTHORIZED, DEPLOYER_USER);
        killServersInGroup(client, Outcome.UNAUTHORIZED, DEPLOYER_USER);
        destroyServersInGroup(client, Outcome.UNAUTHORIZED, DEPLOYER_USER);
    }

    @Test
    public void testAdministrator() throws Exception {
        ModelControllerClient client = getClientForUser(ADMINISTRATOR_USER, isAllowLocalAuth(), primaryClientConfig);
        readWholeConfig(client, Outcome.SUCCESS, ADMINISTRATOR_USER);
        checkStandardReads(client, null, null, ADMINISTRATOR_USER);
        checkRootRead(client, null, null, Outcome.SUCCESS, ADMINISTRATOR_USER);
        checkRootRead(client, PRIMARY, null, Outcome.SUCCESS, ADMINISTRATOR_USER);
        checkRootRead(client, PRIMARY, PRIMARY_A, Outcome.SUCCESS, ADMINISTRATOR_USER);
        checkSecurityDomainRead(client, null, null, Outcome.SUCCESS, ADMINISTRATOR_USER);
        checkSecurityDomainRead(client, PRIMARY, PRIMARY_A, Outcome.SUCCESS, ADMINISTRATOR_USER);
        checkSensitiveAttribute(client, null, null, true, ADMINISTRATOR_USER);
        checkSensitiveAttribute(client, PRIMARY, PRIMARY_A, true, ADMINISTRATOR_USER);

        if (readOnly) return;

        runGC(client, PRIMARY, null, Outcome.SUCCESS, ADMINISTRATOR_USER);
        runGC(client, PRIMARY, PRIMARY_A, Outcome.SUCCESS, ADMINISTRATOR_USER);
        addDeployment2(client, Outcome.SUCCESS, ADMINISTRATOR_USER);
        addPath(client, Outcome.SUCCESS, ADMINISTRATOR_USER);
        addSecurityDomain(client, "test1", Outcome.SUCCESS, ADMINISTRATOR_USER);
        removeSecurityDomain(client, "test1", Outcome.SUCCESS, ADMINISTRATOR_USER);
    }

    @Test
    public void testAuditor() throws Exception {
        ModelControllerClient client = getClientForUser(AUDITOR_USER, isAllowLocalAuth(), primaryClientConfig);
        readWholeConfig(client, Outcome.SUCCESS, AUDITOR_USER);
        checkStandardReads(client, null, null, AUDITOR_USER);
        checkRootRead(client, null, null, Outcome.SUCCESS, AUDITOR_USER);
        checkRootRead(client, PRIMARY, null, Outcome.SUCCESS, AUDITOR_USER);
        checkRootRead(client, PRIMARY, PRIMARY_A, Outcome.SUCCESS, AUDITOR_USER);
        checkSecurityDomainRead(client, null, null, Outcome.SUCCESS, AUDITOR_USER);
        checkSecurityDomainRead(client, PRIMARY, PRIMARY_A, Outcome.SUCCESS, AUDITOR_USER);
        checkSensitiveAttribute(client, null, null, true, AUDITOR_USER);
        checkSensitiveAttribute(client, PRIMARY, PRIMARY_A, true, AUDITOR_USER);

        if (readOnly) return;

        runGC(client, PRIMARY, null, Outcome.UNAUTHORIZED, AUDITOR_USER);
        runGC(client, PRIMARY, PRIMARY_A, Outcome.UNAUTHORIZED, AUDITOR_USER);
        addDeployment2(client, Outcome.UNAUTHORIZED, AUDITOR_USER);
        addPath(client, Outcome.UNAUTHORIZED, AUDITOR_USER);
        removeSecurityDomain(client, Outcome.UNAUTHORIZED, AUDITOR_USER);
        restartServer(client, PRIMARY, PRIMARY_A, Outcome.UNAUTHORIZED, AUDITOR_USER);

        // Auditor can't shutdown
        testWCORE1067(client, AUDITOR_USER);

        stopServer(client, PRIMARY, PRIMARY_A, Outcome.UNAUTHORIZED, AUDITOR_USER);
        killServer(client, PRIMARY, PRIMARY_A, Outcome.UNAUTHORIZED, AUDITOR_USER);
        destroyServer(client, PRIMARY, PRIMARY_A, Outcome.UNAUTHORIZED, AUDITOR_USER);
        killServersInGroup(client, Outcome.UNAUTHORIZED, AUDITOR_USER);
        destroyServersInGroup(client, Outcome.UNAUTHORIZED, AUDITOR_USER);
    }

    @Test
    public void testSuperUser() throws Exception {
        ModelControllerClient client = getClientForUser(SUPERUSER_USER, isAllowLocalAuth(), primaryClientConfig);
        readWholeConfig(client, Outcome.SUCCESS, SUPERUSER_USER);
        checkStandardReads(client, null, null, SUPERUSER_USER);
        checkRootRead(client, null, null, Outcome.SUCCESS, SUPERUSER_USER);
        checkRootRead(client, PRIMARY, null, Outcome.SUCCESS, SUPERUSER_USER);
        checkRootRead(client, PRIMARY, PRIMARY_A, Outcome.SUCCESS, SUPERUSER_USER);
        checkSecurityDomainRead(client, null, null, Outcome.SUCCESS, SUPERUSER_USER);
        checkSecurityDomainRead(client, PRIMARY, PRIMARY_A, Outcome.SUCCESS, SUPERUSER_USER);
        checkSensitiveAttribute(client, null, null, true, SUPERUSER_USER);
        checkSensitiveAttribute(client, PRIMARY, PRIMARY_A, true, SUPERUSER_USER);

        if (readOnly) return;

        runGC(client, PRIMARY, null, Outcome.SUCCESS, SUPERUSER_USER);
        runGC(client, PRIMARY, PRIMARY_A, Outcome.SUCCESS, SUPERUSER_USER);
        addDeployment2(client, Outcome.SUCCESS, SUPERUSER_USER);
        addPath(client, Outcome.SUCCESS, SUPERUSER_USER);
        addSecurityDomain(client, "test2", Outcome.SUCCESS, SUPERUSER_USER);
        removeSecurityDomain(client, "test2", Outcome.SUCCESS, SUPERUSER_USER);
    }

    private void addSecurityDomain(ModelControllerClient client, String name, Outcome expected, String... roles) throws IOException {
        ModelNode op = createOpNode("profile=profile-a/subsystem=1/rbac-sensitive=" + name, ADD);
        op.get("cache-type").set(DEFAULT);
        configureRoles(op, roles);
        RbacUtil.executeOperation(client, op, expected);
    }

    private void removeSecurityDomain(ModelControllerClient client, String name, Outcome expected, String... roles) throws IOException {
        ModelNode op = createOpNode("profile=profile-a/subsystem=1/rbac-sensitive=" + name, REMOVE);
        configureRoles(op, roles);
        RbacUtil.executeOperation(client, op, expected);
    }

    private void removeSecurityDomain(ModelControllerClient client, Outcome expected, String... roles) throws IOException {
        removeSecurityDomain(client, "other", expected, roles);
    }

    private void testWCORE1067(ModelControllerClient client, String... roles) throws IOException {
        ModelNode op = Util.createEmptyOperation(SHUTDOWN, PathAddress.pathAddress(HOST, "primary"));
        configureRoles(op, roles);
        RbacUtil.executeOperation(client, op, Outcome.UNAUTHORIZED);
    }
}
