/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2023 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.jboss.installer.validators;

import org.jboss.installer.core.InstallationData;
import org.jboss.installer.core.ValidationResult;
import org.jboss.installer.postinstall.task.AbstractHttpsEnableConfig;
import org.jboss.installer.postinstall.task.AdminHttpsConfig;
import org.jboss.installer.test.utils.MockLanguageUtils;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import java.io.File;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.Set;

import static org.assertj.core.api.Assertions.assertThat;
import static org.jboss.installer.screens.AbstractSSLSecurityScreen.CIPHER_NAMES_LABEL_KEY;
import static org.jboss.installer.screens.AbstractSSLSecurityScreen.CIPHER_SUITES_LABEL_KEY;
import static org.jboss.installer.screens.AbstractSSLSecurityScreen.PROTOCOLS_LABEL_KEY;
import static org.jboss.installer.validators.FileValidator.NOT_ABSOLUTE_ERROR;
import static org.jboss.installer.validators.FileValidator.NOT_EXISTS_ERROR;
import static org.jboss.installer.validators.IntRangeValidator.INVALID_INT_VALUE;
import static org.jboss.installer.validators.IntRangeValidator.INVALID_MIN_VALUE;
import static org.jboss.installer.validators.KeystoreCredentialsValidator.KEYSTORE_VALIDATOR_AUTHENTICATION_FAILURE;
import static org.jboss.installer.validators.KeystoreCredentialsValidator.KEYSTORE_VALIDATOR_FILE_DOES_NOT_EXIST;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

public class SSLSecurityValidatorTest {

    private static final String KS_PREFIX = "key-store: keystore.";
    private static final String TS_PREFIX = "trust-store: keystore.";

    @Rule
    public TemporaryFolder tempFolder = new TemporaryFolder();

    private static final String KEYSTORE_PASSWORD = "keystore-password";
    private static final String TRUSTSTORE_PASSWORD = "truststore-password";

    private final SSLSecurityValidator validator = new SSLSecurityValidator(new MockLanguageUtils(), "key-store", "trust-store");
    private InstallationData idata;
    private File testKeyStore;
    private File testTrustStore;

    @Before
    public void setup() throws Exception {
        idata = new InstallationData();
        idata.setTargetFolder(Paths.get("target"));

        testKeyStore = tempFolder.newFile("test-ssl-keystore");
        KeystoreCredentialsValidatorTest.createTestKeystore(testKeyStore, KEYSTORE_PASSWORD);
        testTrustStore = tempFolder.newFile("test-ssl-truststore");
        KeystoreCredentialsValidatorTest.createTestKeystore(testTrustStore, TRUSTSTORE_PASSWORD);
    }

    @Test
    public void testValidity() {
        final AdminHttpsConfig number = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv12), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, null, KEYSTORE_PASSWORD,
                false, null, null,
                true, "CN=test", "60", null, false);
        assertEquals(ValidationResult.Result.OK, validator.validate(number, idata).getResult());
    }

    @Test
    public void testValidityInvalidNumber() {
        final AdminHttpsConfig invalidNumber = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv12), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, null, KEYSTORE_PASSWORD,
                false, null, null,
                true, "CN=test", "number", null, false);
        final ValidationResult result = validator.validate(invalidNumber, idata);
        assertEquals(ValidationResult.Result.ERROR, result.getResult());
        assertTrue(result.getMessage().contains(INVALID_INT_VALUE));
    }

    @Test
    public void testValidityNegative() {
        final AdminHttpsConfig negative = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv12), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, null, KEYSTORE_PASSWORD,
                false, null, null,
                true, "CN=test", "-1", null, false);
        final ValidationResult result = validator.validate(negative, idata);
        assertEquals(ValidationResult.Result.ERROR, result.getResult());
        assertTrue(result.getMessage().contains(INVALID_MIN_VALUE));
    }

    @Test
    public void testValidityZero() {
        final AdminHttpsConfig zero = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv12), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, null, KEYSTORE_PASSWORD,
                false, null, null,
                true, "CN=test", "0", null, false);
        final ValidationResult result = validator.validate(zero, idata);
        assertEquals(ValidationResult.Result.ERROR, result.getResult());
        assertTrue(result.getMessage().contains(INVALID_MIN_VALUE));
    }

    @Test
    public void testInvalidClientCert() {
        final String invalidPath = Paths.get("idont/exist").toAbsolutePath().toString();
        final AdminHttpsConfig invalid = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv12), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, null, KEYSTORE_PASSWORD,
                true, null, TRUSTSTORE_PASSWORD,
                true, "CN=test", "60", invalidPath, false);
        final ValidationResult result0 = validator.validate(invalid, idata);
        assertEquals(ValidationResult.Result.ERROR, result0.getResult());
        assertThat(result0.getMessage())
                .contains(NOT_EXISTS_ERROR);

        final String emptyPath = "";
        final AdminHttpsConfig empty = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv12), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, null, KEYSTORE_PASSWORD,
                true, null, TRUSTSTORE_PASSWORD,
                true, "CN=test", "60", emptyPath, false);
        final ValidationResult result1 = validator.validate(empty, idata);
        assertEquals(ValidationResult.Result.ERROR, result1.getResult());
        assertThat(result1.getMessage())
                .contains(NOT_ABSOLUTE_ERROR);
    }

    @Test
    public void testInvalidKeystorePath() {
        final String invalidPath = "/blah/blah";
        final AdminHttpsConfig invalid = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv12), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, invalidPath, KEYSTORE_PASSWORD,
                false, null, null,
                false, null, null, null, false);
        final ValidationResult result0 = validator.validate(invalid, idata);
        assertEquals(ValidationResult.Result.ERROR, result0.getResult());
        assertEquals(KS_PREFIX + KEYSTORE_VALIDATOR_FILE_DOES_NOT_EXIST, result0.getMessage());

        final String emptyPath = "";
        final AdminHttpsConfig empty = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv12), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, emptyPath, KEYSTORE_PASSWORD,
                false, null, null,
                false, null, null, null, false);
        final ValidationResult result1 = validator.validate(empty, idata);
        assertEquals(ValidationResult.Result.ERROR, result1.getResult());
        assertEquals(KS_PREFIX + KEYSTORE_VALIDATOR_FILE_DOES_NOT_EXIST, result1.getMessage());
    }

    @Test
    public void testInvalidKeystorePassword() {
        final AdminHttpsConfig invalid = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv12), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, testKeyStore.toString(), "wrong password",
                false, null, null,
                false, null, null, null, false);
        final ValidationResult result = validator.validate(invalid, idata);
        assertEquals(ValidationResult.Result.ERROR, result.getResult());
        assertEquals(KS_PREFIX + KEYSTORE_VALIDATOR_AUTHENTICATION_FAILURE, result.getMessage());
    }

    @Test
    public void testInvalidTruststorePath() {
        final String invalidPath = "/blah/blah";
        final AdminHttpsConfig invalid = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv12), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, testKeyStore.toString(), KEYSTORE_PASSWORD,
                true, invalidPath, TRUSTSTORE_PASSWORD,
                false, null, null, null, false);
        final ValidationResult result0 = validator.validate(invalid, idata);
        assertEquals(ValidationResult.Result.ERROR, result0.getResult());
        assertEquals(TS_PREFIX + KEYSTORE_VALIDATOR_FILE_DOES_NOT_EXIST, result0.getMessage());

        final String emptyPath = "";
        final AdminHttpsConfig empty = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv12), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, testKeyStore.toString(), KEYSTORE_PASSWORD,
                true, emptyPath, TRUSTSTORE_PASSWORD,
                false, null, null, null, false);
        final ValidationResult result1 = validator.validate(empty, idata);
        assertEquals(ValidationResult.Result.ERROR, result1.getResult());
        assertEquals(TS_PREFIX + KEYSTORE_VALIDATOR_FILE_DOES_NOT_EXIST, result1.getMessage());
    }

    @Test
    public void testMutualPasswords() {
        final AdminHttpsConfig config = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv12), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, testKeyStore.toString(), KEYSTORE_PASSWORD,
                true, testTrustStore.toString(), TRUSTSTORE_PASSWORD,
                false, null, null, null, false);
        assertEquals(ValidationResult.Result.OK, validator.validate(config, idata).getResult());
    }

    @Test
    public void testInvalidTrustStorePassword() {
        final AdminHttpsConfig mismatched = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv12), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, testKeyStore.toString(), KEYSTORE_PASSWORD,
                true, testTrustStore.toString(), "wrong password",
                false, null, null, null, false);
        final ValidationResult result = validator.validate(mismatched, idata);
        assertEquals(ValidationResult.Result.ERROR, result.getResult());
        assertEquals(TS_PREFIX + KEYSTORE_VALIDATOR_AUTHENTICATION_FAILURE, result.getMessage());

    }

    @Test
    public void testCipherSuiteNamesTLSv12() {
        final AdminHttpsConfig okSuites = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv12), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, null, KEYSTORE_PASSWORD,
                false, null, null,
                true, "CN=test", "60", null, false);
        assertEquals(ValidationResult.Result.OK, validator.validate(okSuites, idata).getResult());

        final AdminHttpsConfig invalidSuites = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv12), "myownsuite", AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, null, KEYSTORE_PASSWORD,
                false, null, null,
                true, "CN=test", "60", null, false);
        final ValidationResult result = validator.validate(invalidSuites, idata);
        assertEquals(ValidationResult.Result.ERROR, result.getResult());
        assertTrue(result.getMessage().contains("ELY05016"));
    }

    @Test
    public void testCipherSuiteNamesTLSv13() {
        final AdminHttpsConfig okSuites = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv13), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, null, KEYSTORE_PASSWORD,
                false, null, null,
                true, "CN=test", "60", null, false);
        assertEquals(ValidationResult.Result.OK, validator.validate(okSuites, idata).getResult());

        final AdminHttpsConfig invalidSuites = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv13), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, "myownsuite", null, KEYSTORE_PASSWORD,
                false, null, null,
                true, "CN=test", "60", null, false);
        final ValidationResult result = validator.validate(invalidSuites, idata);
        assertEquals(ValidationResult.Result.ERROR, result.getResult());
        assertTrue(result.getMessage().contains("ELY15000"));
    }

    @Test
    public void testEmptyCipherNamesTLSv13() {
        final AdminHttpsConfig invalidSuites = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv13), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, "", null, KEYSTORE_PASSWORD,
                false, null, null,
                true, "CN=test", "60", null, false);
        final ValidationResult result = validator.validate(invalidSuites, idata);
        assertEquals(ValidationResult.Result.ERROR, result.getResult());
        assertThat(result.getMessage())
                .contains("generic.validation.empty_field")
                .contains(CIPHER_NAMES_LABEL_KEY);
    }

    @Test
    public void testEmptyCipherNamesTLSv12() {
        final AdminHttpsConfig invalidSuites = new AdminHttpsConfig(
                Set.of(AbstractHttpsEnableConfig.PROTOCOL_TLSv12), "", AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, null, KEYSTORE_PASSWORD,
                false, null, null,
                true, "CN=test", "60", null, false);
        final ValidationResult result = validator.validate(invalidSuites, idata);
        assertEquals(ValidationResult.Result.ERROR, result.getResult());
        assertThat(result.getMessage())
                .contains("generic.validation.empty_field")
                .contains(CIPHER_SUITES_LABEL_KEY);
    }

    @Test
    public void testEmptyProtocols() {
        final AdminHttpsConfig invalidSuites = new AdminHttpsConfig(
                Collections.emptySet(), AbstractHttpsEnableConfig.DEFAULT_CIPHER_SUITES_TLS12, AbstractHttpsEnableConfig.DEFAULT_CIPHER_NAMES_TLS13, null, KEYSTORE_PASSWORD,
                false, null, null,
                true, "CN=test", "60", null, false);
        final ValidationResult result = validator.validate(invalidSuites, idata);
        assertEquals(ValidationResult.Result.ERROR, result.getResult());
        assertThat(result.getMessage())
                .contains("generic.validation.empty_field")
                .contains(PROTOCOLS_LABEL_KEY);
    }
}
