/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.web.tomcat.service.session.distributedcache.impl.jbc;


import org.jboss.cache.Cache;
import org.jboss.cache.CacheManager;
import org.jboss.cache.Fqn;
import org.jboss.cache.buddyreplication.BuddyManager;
import org.jboss.cache.pojo.PojoCache;
import org.jboss.ha.framework.server.CacheManagerLocator;
import org.jboss.ha.framework.server.PojoCacheManager;
import org.jboss.ha.framework.server.PojoCacheManagerLocator;
import org.jboss.metadata.web.jboss.ReplicationConfig;
import org.jboss.metadata.web.jboss.ReplicationGranularity;
import org.jboss.web.tomcat.service.session.distributedcache.spi.ClusteringNotSupportedException;
import org.jboss.web.tomcat.service.session.distributedcache.spi.LocalDistributableSessionManager;

/**
 * Utility methods related to JBoss distributed sessions.
 * 
 * @author Brian Stansberry
 * @version $Revision: 56542 $
 */
public class Util
{
   private static final int BUDDY_BACKUP_ROOT_OWNER_INDEX = BuddyManager.BUDDY_BACKUP_SUBTREE_FQN.size();
   private static final String DEAD_BUDDY_SUFFIX = ":DEAD";
   
   public static ReplicationGranularity getReplicationGranularity(LocalDistributableSessionManager localManager)
   {
      ReplicationConfig config = localManager.getReplicationConfig();
      return config.getReplicationGranularity();
   }  
   
   public static String getCacheConfigName(LocalDistributableSessionManager localManager)
   {
      ReplicationConfig config = localManager.getReplicationConfig();
      return config.getCacheName();
   }
   
   public static PojoCache findPojoCache(String cacheConfigName) throws ClusteringNotSupportedException
   {
      PojoCacheManager pcm = getManagerForPojoCache(cacheConfigName);
      
      try
      {
         return pcm.getPojoCache(cacheConfigName, true);
      }
      catch (RuntimeException re)
      {
         throw re;
      }
      catch (Exception e)
      {
         throw new RuntimeException("Problem accessing cache " + cacheConfigName, e);
      }
   }
   
   public static Cache<Object, Object> findPlainCache(String cacheConfigName) throws ClusteringNotSupportedException
   {
      CacheManager pcm = getManagerForCache(cacheConfigName);
      
      try
      {
         return pcm.getCache(cacheConfigName, true);
      }
      catch (RuntimeException re)
      {
         throw re;
      }
      catch (Exception e)
      {
         throw new RuntimeException("Problem accessing cache " + cacheConfigName, e);
      }
   }
   
   private static PojoCacheManager getManagerForPojoCache(String cacheConfigName) 
      throws ClusteringNotSupportedException
   {
      PojoCacheManager pcm = null;
      try
      {
         pcm = PojoCacheManagerLocator.getCacheManagerLocator().getCacheManager(null);
      }
      catch (Throwable t)
      {
         throw new ClusteringNotSupportedException("Could not access PojoCacheManager for JBossWeb clustering", t);
      }
      
      if (!pcm.getConfigurationNames().contains(cacheConfigName))
         throw new IllegalStateException("PojoCacheManager does not recognize config " + cacheConfigName);
      
      return pcm;
   }
   
   private static CacheManager getManagerForCache(String cacheConfigName) 
      throws ClusteringNotSupportedException
   {
      CacheManager cm = null;
      try
      {
         cm = CacheManagerLocator.getCacheManagerLocator().getCacheManager(null);
      }
      catch (Throwable t)
      {
         throw new ClusteringNotSupportedException("Could not access CacheManager for JBossWeb clustering", t);
      }
      
      if (!cm.getConfigurationNames().contains(cacheConfigName))
         throw new IllegalStateException("CacheManager does not recognize config " + cacheConfigName);
      
      return cm;
   }
   
   public static boolean isBuddyOwnerDead(Fqn<String> fqn)
   {
      String base = (String) fqn.get(BUDDY_BACKUP_ROOT_OWNER_INDEX);
      return base.endsWith(DEAD_BUDDY_SUFFIX);
   }
   
   /**
    * Extracts the owner portion of an buddy subtree Fqn.
    * 
    * @param fqn An Fqn that is a child of the buddy backup root node.
    */
   public static String getBuddyOwner(Fqn<String> fqn)
   {
      String base = (String) fqn.get(BUDDY_BACKUP_ROOT_OWNER_INDEX);
      return base.endsWith(DEAD_BUDDY_SUFFIX) ? base.substring(0, base.length() - DEAD_BUDDY_SUFFIX.length()): base;     
   }
   
   /**
    * Prevent instantiation.
    */
   private Util() {}

}
