/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.ejb.plugins.cmp.jdbc2;


import org.jboss.ejb.plugins.cmp.jdbc2.bridge.JDBCEntityBridge2;
import org.jboss.ejb.plugins.cmp.jdbc.metadata.JDBCQueryMetaData;
import org.jboss.ejb.plugins.cmp.jdbc.metadata.JDBCJBossQLQueryMetaData;
import org.jboss.ejb.plugins.cmp.jdbc.metadata.JDBCQlQueryMetaData;
import org.jboss.ejb.plugins.cmp.jdbc.metadata.JDBCDeclaredQueryMetaData;
import org.jboss.ejb.plugins.cmp.jdbc.metadata.JDBCDynamicQLQueryMetaData;
import org.jboss.deployment.DeploymentException;

import javax.ejb.FinderException;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;

/**
 * @author <a href="mailto:alex@jboss.org">Alexey Loubyansky</a>
 * @version <tt>$Revision: 81030 $</tt>
 */
public class QueryFactory
{
   private final Map queriesByMethod = new HashMap();
   private final JDBCEntityBridge2 entity;

   public QueryFactory(JDBCEntityBridge2 entity)
   {
      this.entity = entity;
   }

   public QueryCommand getQueryCommand(Method queryMethod) throws FinderException
   {
      QueryCommand queryCommand = (QueryCommand)queriesByMethod.get(queryMethod);
      if(queryCommand == null)
      {
         throw new FinderException("Unknown query method: " + queryMethod);
      }
      return queryCommand;
   }

   public void init() throws DeploymentException
   {
      Method findByPkMethod;
      Class home = entity.getHomeClass();
      if(home != null)
      {
         try
         {
            findByPkMethod = home.getMethod("findByPrimaryKey", new Class[]{entity.getPrimaryKeyClass()});
         }
         catch(NoSuchMethodException e)
         {
            throw new DeploymentException("Home interface " + home.getClass().getName() +
               " does not contain findByPrimaryKey(" + entity.getPrimaryKeyClass().getName() + ")");
         }

         FindByPrimaryKeyCommand findByPk = new FindByPrimaryKeyCommand(entity);
         queriesByMethod.put(findByPkMethod, findByPk);
      }

      Class local = entity.getLocalHomeClass();
      if(local != null)
      {
         try
         {
            findByPkMethod = local.getMethod("findByPrimaryKey", new Class[]{entity.getPrimaryKeyClass()});
         }
         catch(NoSuchMethodException e)
         {
            throw new DeploymentException("Local home interface " + local.getClass().getName() +
               " does not contain findByPrimaryKey(" + entity.getPrimaryKeyClass().getName() + ")");
         }

         FindByPrimaryKeyCommand findByPk = new FindByPrimaryKeyCommand(entity);
         queriesByMethod.put(findByPkMethod, findByPk);
      }

      //
      // Defined finders - Overrides automatic finders.
      //
      Iterator definedFinders = entity.getMetaData().getQueries().iterator();
      while(definedFinders.hasNext())
      {
         JDBCQueryMetaData q = (JDBCQueryMetaData) definedFinders.next();

         if(!queriesByMethod.containsKey(q.getMethod()))
         {
            if(q instanceof JDBCJBossQLQueryMetaData)
            {
               QueryCommand queryCommand = new JBossQLQueryCommand(entity, (JDBCJBossQLQueryMetaData)q);
               queriesByMethod.put(q.getMethod(), queryCommand);
            }
            else if(q instanceof JDBCQlQueryMetaData)
            {
               QueryCommand queryCommand = new EJBQLQueryCommand(entity, (JDBCQlQueryMetaData)q);
               queriesByMethod.put(q.getMethod(), queryCommand);
            }
            else if(q instanceof JDBCDeclaredQueryMetaData)
            {
               QueryCommand queryCommand = new DeclaredSQLQueryCommand(entity, (JDBCDeclaredQueryMetaData)q);
               queriesByMethod.put(q.getMethod(), queryCommand);
            }
            else if(q instanceof JDBCDynamicQLQueryMetaData)
            {
               QueryCommand queryCommand = new DynamicQueryCommand(entity, (JDBCDynamicQLQueryMetaData)q);
               queriesByMethod.put(q.getMethod(), queryCommand);
            }
            else
            {
               throw new DeploymentException("Unsupported query metadata: method=" + q.getMethod().getName() +
                  ", metadata=" + q);
            }
         }
      }
   }
}
