package org.hibernate.plugins.util;

import org.apache.maven.plugin.MojoExecutionException;

import java.io.*;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * @author vdedik@redhat.com
 */
public class Util {
    public static final Set<String> DIALECTS = new HashSet<String>() {{
        add("DB2Dialect");
        add("H2Dialect");
        add("MySQL5InnoDBDialect");
        add("Oracle10gDialect");
        add("PostgreSQLDialect");
        add("SQLServer2008Dialect");
    }};

    public static final String SQL_DELIMITER = ";";

    public static final String SQL_SEPARATOR = "\n\n    ";

    public static void writeFile(File outputFile, String content) throws MojoExecutionException {
        try {
            File outputDir = outputFile.getParentFile();
            if (!outputDir.exists()) {
                outputDir.mkdirs();
            }
            if (!outputFile.exists()) {
                outputFile.createNewFile();
            }

            PrintWriter writer = new PrintWriter(outputFile, "UTF-8");
            writer.write(content);
            writer.close();
        } catch (FileNotFoundException e) {
            throw new MojoExecutionException("File not found " + outputFile);
        } catch (UnsupportedEncodingException e) {
            throw new MojoExecutionException("Unsupported encoding UTF-8");
        } catch (IOException e) {
            throw new MojoExecutionException("An error occured while creating file " + outputFile);
        }
    }

    public static List<String> loadSqlStatementsFromFile(File sqlFile) throws MojoExecutionException {
        BufferedReader br = null;
        List<String> sqlStatements = new ArrayList<String>();
        try {
            br = new BufferedReader(new InputStreamReader(new FileInputStream(sqlFile)));
            StringBuilder sb = new StringBuilder();
            String line = br.readLine();

            while (line != null) {
                sb.append(line);
                sb.append('\n');

                Pattern pattern = Pattern.compile(".*;\\s*$");
                Matcher matcher = pattern.matcher(line);
                if (matcher.matches()) {
                    sqlStatements.add(sb.toString().trim());
                    sb = new StringBuilder();
                }

                line = br.readLine();
            }
        } catch(FileNotFoundException e) {
            throw new MojoExecutionException("File " + sqlFile.getName() + " not found.");
        } catch(IOException e) {
            throw new MojoExecutionException("An error occurred while reading file " + sqlFile.getName());
        } finally {
            try {
                if (br != null) {
                    br.close();
                }
            } catch (IOException e) {
                throw new MojoExecutionException("An error occurred while trying to close file " + sqlFile.getName());
            }
        }

        return sqlStatements;
    }

    public static String getObjectName(String sql, String token) {
        Pattern pattern = Pattern.compile("(?is).*" + token + ".*");
        Matcher matcher = pattern.matcher(sql);
        if (matcher.matches()) {
            return matcher.group(1);
        }
        return null;
    }

    public static Map<String, Set<File>> groupSqlFilesByDialects(Set<String> dialects, Set<File> sqlFiles) {
        Map<String, Set<File>> groupedSqlFiles = new HashMap<String, Set<File>>();

        for (String dialect : dialects) {
            if (!groupedSqlFiles.containsKey(dialect)) {
                groupedSqlFiles.put(dialect, new HashSet<File>());
            }

            for (File sqlFile : sqlFiles) {
                if (getSqlFileDialect(sqlFile).equals(dialect)) {
                    groupedSqlFiles.get(dialect).add(sqlFile);
                }
            }
        }

        return groupedSqlFiles;
    }

    public static Set<File> getSqlFiles(String dirPath) {
        Set<File> sqlFiles = new HashSet<File>();
        File[] files = new File(dirPath).listFiles();

        if (files != null) {
            for (File f : files) {
                if (f.isFile() && getFileExtension(f).equals("sql")) {
                    sqlFiles.add(f);
                }
            }
	}

        return sqlFiles;
    }

    public static String getFileExtension(File file) {
        String extension = "";
        String fileName = file.getName();

        int i = fileName.lastIndexOf('.');
        if (i > 0) {
            extension = fileName.substring(i+1);
        }

        return extension;
    }

    public static String getSqlFileDialect(File sqlFile) {
        String dialect = "";
        String fileName = sqlFile.getName();

        String[] split = fileName.split("\\.");
        if (split.length > 1) {
            dialect = split[split.length - 2];
        }

        return dialect;
    }

    public static boolean statementContains(String sql, String text) {
        return isWordNotNestedIn(sql, text, "'") && // String
                isWordNotNestedIn(sql, text, "\"") && // e.g. PostgreSQL escape char
                isWordNotNestedIn(sql, text, "`") && // e.g. MySQL escape char
                isWordNotNestedIn(sql, text, "[\\[\\]]"); // MSSQL escape char
    }

    public static boolean isWordNotNestedIn(String sentence, String word, String token) {
        String removedEscapes = sentence.replaceAll("\\\\" + token, "");
        String[] chunks = removedEscapes.split(token);

        for (int i = 0; i < chunks.length; i = i + 2) {
            if (chunks[i].toLowerCase().contains(word.toLowerCase())) {
                return true;
            }
        }
        return false;
    }

    public static String join(List<String> list) {
        return join(list, null);
    }

    public static String join(List<String> list, String conjunction) {
        StringBuilder sb = new StringBuilder();
        boolean first = true;
        for (String item : list) {
            if (first) {
                first = false;
            } else if (conjunction != null){
                sb.append(conjunction);
            }
            sb.append(item);
        }
        return sb.toString();
    }

    public static String removeSemicolon(String sql) {
        return sql.replaceAll(";\\s*$", "");
    }
}
