package org.jboss.ip.dbtool;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.*;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import java.io.*;
import java.util.List;

/**
 * Author: Alex Creasy &lt;acreasy@redhat.com$gt;
 * Date: 17/10/13
 * Time: 18:00
 */
public final class DriverInstaller {

    private static final Logger logger = LoggerFactory.getLogger(Main.LOGGER_NAME);


    public static File install(final Database db, final File dirToSearch, final File jbossHomeDir)
            throws IOException {

        logger.trace("DriverInstaller.install( db='{}', dirToSearch='{}', jbossHomeDir='{}' )", db, dirToSearch,
                jbossHomeDir);

        if (!dirToSearch.isDirectory())
            throw new IllegalArgumentException("Directory to search " + dirToSearch.getAbsolutePath() + " does not" +
                    "exist or is not a directory");

        if (!jbossHomeDir.isDirectory())
            throw new IllegalArgumentException("jbossHomeDir does not exist or is not a directory");

        if (db == null)
            throw new IllegalArgumentException("Null param db");

        final File installedDrv = findAndCopyJar(db, dirToSearch, jbossHomeDir);
        createModuleXml(db.getDriverModuleName(), installedDrv);

        return installedDrv;
    }

    private static File findAndCopyJar(final Database db, final File dirToSearch, final File jbossHomeDir)
            throws IOException {
        // Try to find the driver in the tool drivers dir.
        final List<File> drivers = Utils.recursiveDirSearch(dirToSearch,
                db.getDriverJarFileName().replaceAll("\\*", "\\.\\*\\?"));

        if (drivers.isEmpty())
            throw new FileNotFoundException(String.format("Unable to find driver: %s in %s", db.getDriverJarFileName(),
                    dirToSearch));

        logger.info("Found driver: {} in {}", drivers.get(0).getName(), dirToSearch);

        // Get the destination directory for the driver
        final File drvDest = new File(db.getDriverInstallLocation().replaceFirst("\\$\\{JBOSS_HOME\\}",
                jbossHomeDir.getAbsolutePath()), drivers.get(0).getName());

        logger.info("Destination location for driver: {}", drvDest);

        // Attempt to copy the driver to the destination directory
        if (!drvDest.exists()) {
            drvDest.getParentFile().mkdirs();
            drvDest.createNewFile();
        }

        Utils.copyFile(drivers.get(0), drvDest);

        if (!drvDest.exists())
            throw new IllegalStateException("Unable to create file " + drvDest.getAbsolutePath());

        return drvDest;
    }

    private static void createModuleXml(final String jdbcModuleName, final File jarFile)
            throws IOException {

        logger.trace("DriverInstaller.install( jdbcModuleName='{}', jarFile='{}' )", jdbcModuleName, jarFile);

        String location = jarFile.getParent();
        File module = new File(location, "module.xml");

        DocumentBuilderFactory docFactory = DocumentBuilderFactory.newInstance();
        DocumentBuilder docBuilder = null;
        try {
            docBuilder = docFactory.newDocumentBuilder();
        } catch (ParserConfigurationException e) {
            throw new IllegalStateException("Error creating module.xml for " + jdbcModuleName, e);
        }

        // module element and attributes
        Document doc = docBuilder.newDocument();
        Element root = doc.createElement("module");
        Attr xmlns = doc.createAttribute("xmlns");
        Attr moduleName = doc.createAttribute("name");
        xmlns.setValue("urn:jboss:module:1.0");
        moduleName.setValue(jdbcModuleName);
        root.setAttributeNode(xmlns);
        root.setAttributeNode(moduleName);
        doc.appendChild(root);

        // resources element and attributes (child of root)
        Element resources = doc.createElement("resources");
        root.appendChild(resources);

        // resource-root element and attributes (child of resources)
        Element resourceRoot = doc.createElement("resource-root");
        Attr resourcePath = doc.createAttribute("path");
        resourcePath.setValue(jarFile.getName());
        resourceRoot.setAttributeNode(resourcePath);
        resources.appendChild(resourceRoot);

        // dependencies element and attributes (child of root)
        Element dependencies = doc.createElement("dependencies");
        root.appendChild(dependencies);

        // module element and attributes (child of dependencies)
        Element module1 = doc.createElement("module");
        Attr mod1Name = doc.createAttribute("name");
        mod1Name.setValue("javax.api");
        module1.setAttributeNode(mod1Name);
        Element module2 = doc.createElement("module");
        Attr mod2Name = doc.createAttribute("name");
        mod2Name.setValue("javax.transaction.api");
        module2.setAttributeNode(mod2Name);
        dependencies.appendChild(module1);
        dependencies.appendChild(module2);

        TransformerFactory tFactory = TransformerFactory.newInstance();
        tFactory.setAttribute("indent-number", 4);
        Transformer trans = null;
        try {
            trans = tFactory.newTransformer();
        } catch (TransformerConfigurationException e) {
            throw new IllegalStateException("Error creating module.xml for " + jdbcModuleName, e);
        }
        trans.setOutputProperty(OutputKeys.INDENT, "yes");
        DOMSource source = new DOMSource(doc);

        // Write to file from a String to preserve indentation
        StreamResult result = new StreamResult(new StringWriter());
        try {
            trans.transform(source, result);
        } catch (TransformerException e) {
            throw new IllegalStateException("Error creating module.xml for " + jdbcModuleName, e);
        }

        String outputString = result.getWriter().toString();
        BufferedWriter writeOut = null;
        try {
            writeOut = new BufferedWriter(new FileWriter(module));
            writeOut.write(outputString);
        } finally {
            if (writeOut != null) {
                writeOut.close();
            }
        }
    }
}
