package org.jboss.ip.dbtool;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.util.logging.resources.logging;

import java.io.*;
import java.nio.channels.FileChannel;
import java.util.*;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * A collection of miscellaneous static utility methods used by DBTool.
 *
 * @author Alex Creasy <acreasy@redhat.com>
 */
public final class Utils {

    private static final Logger logger = LoggerFactory.getLogger(Main.LOGGER_NAME);

    /**
     * Scans a directory and returns a list of subdirectories or normal files.
     *
     * @param dirToScan   the directory to scan.
     * @param scanForDirs if true the directory is scanned for subdirectories. If false the
     *                    directory is scanned for normal files.
     *
     * @return File[] array of sub directories, the array will be empty if there are no matches.
     *
     * @throws IllegalArgumentException if dirToScan does not exist or is not a directory.
     * @throws SecurityException        If a security manager exists and its SecurityManager.checkRead(java.lang.String)
     *                                  method denies read access to the directory.
     */
    public static File[] scanDir(final File dirToScan, final boolean scanForDirs) {

        if (!dirToScan.isDirectory())
            throw new IllegalArgumentException(String.format("dirToScan: %s does not exist or is not a directory",
                    dirToScan));

        return dirToScan.listFiles(new FilenameFilter() {
            @Override
            public boolean accept(final File dir, final String name) {
                final File f = new File(dir, name);
                return scanForDirs ? f.isDirectory() : f.isFile();
            }
        });
    }

    /**
     * @param parentDir
     * @param regex
     *
     * @return
     *
     * @throws IllegalArgumentException
     * @throws SecurityException
     */
    public static List<File> recursiveDirSearch(final File parentDir, final String regex) {

        if (!parentDir.isDirectory())
            throw new IllegalArgumentException("parentDir does not exist or is not a directory");

        if (parentDir.listFiles().length == 0)
            return Collections.emptyList();

        final List<File> result = new ArrayList<File>();

        result.addAll(Arrays.asList(parentDir.listFiles(new FilenameFilter() {
            @Override
            public boolean accept(final File dir, final String name) {
                return name.matches(regex);
            }
        }
        )));

        for (final File subDir : scanDir(parentDir, true))
            result.addAll(recursiveDirSearch(subDir, regex));

        return result;
    }

    /**
     *
     */
    public static Collection<String> findResourcesInJar(final JarFile jarFile, final String regex) {
        Enumeration<JarEntry> entries = jarFile.entries();
        Set<String> result = new HashSet<String>(); //avoid duplicates in case it is a subdirectory

        while (entries.hasMoreElements()) {
            final String entry = entries.nextElement().getName();
            if (entry.matches(regex))
                result.add(entry);
        }
        return result;
    }

    /**
     * @param dirName
     *
     * @return
     */
    public static Map<String, Properties> getAllPropsFilesFromDirInJar(final JarFile jarFile, final String dirName) {
        final String regex = String.format("^%s/.+?\\.properties$", dirName);
        final Collection<String> propsFiles = findResourcesInJar(jarFile, regex);

        final Map<String, Properties> result = new HashMap<String, Properties>();

        for (final String p : propsFiles) {
            try {
                final Properties properties = loadPropertiesFileFromClasspath(p);
                result.put(p.replaceFirst(".properties", "").replaceFirst(dirName + "/", ""), properties);
            } catch (IOException ioe) {
                throw new RuntimeException("Unable to load .properties file: " + p, ioe);
            }
        }
        return result;
    }

    /**
     * Prints a Collection of objects to a given PrintStream adding a line break each time a
     * specified number of objects has been printed.
     * <p/>
     * Example usage:
     *
     * @param objsToPrint Collection of objects to print
     * @param objsPerLine the number of objects to print before adding a line break.
     * @param output      the PrintStream to output to
     *
     */
    public static void prettyPrinter(final Collection<?> objsToPrint, final int objsPerLine,
                                     final PrintStream output) {
        prettyPrinter(objsToPrint, objsPerLine, output, null);
    }

    /**
     * @param <T>
     */
    public interface Transformer<T> {
        /**
         * @param t
         *
         * @return
         */
        String prettyToString(T t);
    }

    /**
     * Prints a Collection of objects to a given PrintStream adding a line break each time a
     * specified number of objects has been printed. A Utils.Transformer instance can be
     * given that will be used to transform each object to a String before they are printed.
     * <p/>
     * Example usage:
     *
     * @param objsToPrint Collection of objects to print
     * @param objsPerLine the number of objects to print before adding a line break.
     * @param output      the PrintStream to output to.
     * @param transform   a Utils.Tranformer instance containing a function to be applied
     *                    to the objects before printing them. If null the objects toString()
     *                    method will be used instead.
     * @param <T>         The type of objects contained within objsToPrint and therefore to transform.
     */
    public static <T> void prettyPrinter(final Collection<T> objsToPrint, final int objsPerLine,
                                         final PrintStream output, final Transformer<T> transform) {

        if (objsPerLine < 1)
            return;

        int i = 0;

        for (T obj : objsToPrint) {
            if (i > 0) {
                if (i % objsPerLine == 0)
                    output.print("\n  ");
                else
                    output.print(",  ");
            } else
                output.print("  ");

            output.print(transform != null ? transform.prettyToString(obj) : obj.toString());

            i++;
        }
    }

    /**
     * Uses java.nio lib to copy a file to a given destination. This will
     * be unnecessary in Java 7.
     *
     * @param source      the File to copy.
     * @param destination where to create the file, note: the File object should
     *                    point to the destination including the filename, a File
     *                    pointing to the parent folder will result in an IOException
     *                    being thrown.
     *
     * @throws IOException if there is an error while copying the file.
     */
    public static void copyFile(final File source, final File destination) throws IOException {
        FileChannel inputChannel = null;
        FileChannel outputChannel = null;

        try {
            inputChannel = new FileInputStream(source).getChannel();
            outputChannel = new FileOutputStream(destination).getChannel();
            outputChannel.transferFrom(inputChannel, 0, inputChannel.size());
        } finally {
            if (inputChannel != null)
                inputChannel.close();
            if (outputChannel != null)
                outputChannel.close();
        }
    }

    public static Properties loadPropertiesFile(final File props) throws IOException {
        final Properties result = new Properties();
        result.load(new FileInputStream(props));
        return result;
    }

    public static Properties loadPropertiesFileFromClasspath(final String fileName) throws IOException {
        final Properties result = new Properties();
        result.load(Thread.currentThread().getContextClassLoader().getResourceAsStream(fileName));
        return result;
    }

    public static List<String> loadFile(File file) {
        List<String> lines = new ArrayList<String>();

        BufferedReader br = null;

        try {
            br = new BufferedReader(new FileReader(file));
            String line;
            while ((line = br.readLine()) != null) {
                lines.add(line);
            }
        } catch (IOException e) {
            logger.error("File " + file.getName() + " could not be read.", e);
        } finally {
            try {
                if (br != null) {
                    br.close();
                }
            } catch (IOException e) {
                // do nothing, there is nothing to be done
            }
        }
        return lines;
    }


    // Suppress default constructor to prevent instantiation.
    private Utils() {
    }

}


