/*
 * Copyright 2016 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version
 * 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied.  See the License for the specific language governing
 * permissions and limitations under the License.
 */
package org.jbosson.plugins.fuse;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.rhq.core.domain.configuration.Configuration;
import org.rhq.core.domain.configuration.PropertySimple;
import org.rhq.core.pluginapi.inventory.DiscoveredResourceDetails;
import org.rhq.core.pluginapi.inventory.InvalidPluginConfigurationException;
import org.rhq.core.pluginapi.inventory.ManualAddFacet;
import org.rhq.core.pluginapi.inventory.ResourceDiscoveryContext;
import org.rhq.core.util.Base64;
import org.rhq.modules.plugins.jbossas7.BaseComponent;
import org.rhq.modules.plugins.jbossas7.BaseServerComponent;
import org.rhq.modules.plugins.jbossas7.JBossProductType;
import org.rhq.modules.plugins.jbossas7.ManagedASComponent;
import org.rhq.modules.plugins.jbossas7.StandaloneASComponent;
import org.rhq.modules.plugins.jbossas7.helper.ServerPluginConfiguration;
import org.rhq.modules.plugins.jbossas7.jmx.ApplicationMBeansDiscoveryComponent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Discovery component for Fuse services in JBoss AS7, EAP6 and Wildfly.
 * @author dbokde
 */
public class FuseJBossAS7MBeansDiscoveryComponent extends ApplicationMBeansDiscoveryComponent implements ManualAddFacet<BaseComponent<?>> {

    private static final Logger LOG = LoggerFactory.getLogger(FuseJBossAS7MBeansDiscoveryComponent.class);

    private static final String HOSTNAME = "hostname";
    private static final String PORT = "port";
    private static final String USERNAME = "username";
    private static final String PASSWORD = "password";
    private static final String CLIENT_JAR_LOCATION = "clientJarLocation";
    // "remoting-jmx" for AS7 and EAP6, "http-remoting-jmx" for Wildfly8;
    private static final String PROTOCOL = "protocol";

    private static final int STANDALONE_REMOTING_PORT_OFFSET = 9;
    private static final int DOMAIN_REMOTING_PORT_DEFAULT = 4447;
    private static final String MANAGED_SERVER_PORT_OFFSET_PROPERTY_NAME = "socket-binding-port-offset";
    private static final int HTTP_PORT_DEFAULT = 8080;

    private ThreadLocal<ResourceDiscoveryContext<BaseComponent<?>>> context =
        new ThreadLocal<ResourceDiscoveryContext<BaseComponent<?>>>();

    @Override
    public Set<DiscoveredResourceDetails> discoverResources(ResourceDiscoveryContext<BaseComponent<?>> context)
        throws Exception {

        // store discovery context in TLS for later
        this.context.set(context);
        return super.discoverResources(context);
    }

    /**
     * Tries to read an ApplicationRealm username and password from rhq-user.properties.
     * @return username, password for ApplicationRealm user, or default rhqadmin/rhqadmin, which is likely to fail.
     */
    @Override
    protected String[] getCredentialsForManagedAS() {

        final ResourceDiscoveryContext<BaseComponent<?>> discoveryContext = context.get();
        BaseServerComponent serverComponent = discoveryContext.getParentResourceComponent().getServerComponent();
        final ServerPluginConfiguration serverConfig = serverComponent.getServerPluginConfiguration();
        final File homeDir = serverConfig.getHomeDir();

        // look up a user in rhq-user.properties
        final File rhqUserFile = new File(homeDir, "domain/configuration/rhq-user.properties");
        if (rhqUserFile.isFile()) {

            FileInputStream inputStream = null;
            try {

                inputStream = new FileInputStream(rhqUserFile);
                Properties userProperties = new Properties();
                userProperties.load(inputStream);

                if (userProperties.size() != 1) {

                    LOG.warn("File {} MUST contain a single entry of the form " +
                            "username=<BASE64 encoded utf-8-password>",
                        rhqUserFile);

                } else {

                    final Map.Entry<Object, Object> entry = userProperties.entrySet().iterator().next();
                    return new String[] {
                        entry.getKey().toString(),
                        new String(Base64.decode(entry.getValue().toString()), "UTF-8")
                    };
                }

            } catch (IOException e) {
                LOG.warn("Error reading rhq-user.properties file, resource "
                        + discoveryContext.getResourceType().getName()
                        + " must be manually imported for parent resource "
                        + discoveryContext.getParentResourceContext().getResourceDetails(),
                    e);
            } finally {
                if (inputStream != null) {
                    try {
                        inputStream.close();
                    } catch (IOException ignore) {
                    }
                }
            }
        }

        LOG.debug("Management user not configured in {}, trying default user rhqadmin", rhqUserFile);
        return super.getCredentialsForManagedAS();
    }

    // manual add
    public DiscoveredResourceDetails discoverResource(Configuration configuration,
                                                      ResourceDiscoveryContext<BaseComponent<?>> resourceDiscoveryContext)
        throws InvalidPluginConfigurationException {

        final BaseComponent<?> parentComponent = resourceDiscoveryContext.getParentResourceComponent();
        final BaseServerComponent serverComponent = parentComponent.getServerComponent();
        final ServerPluginConfiguration serverPluginConfiguration = serverComponent.getServerPluginConfiguration();
        final JBossProductType productType = serverPluginConfiguration.getProductType();

        // set properties for ApplicationMBeansDiscoveryComponent.loadEmsConnection()
        // auto-discover hostname if needed
        if (configuration.getSimpleValue(HOSTNAME) == null) {
            configuration.setSimpleValue(HOSTNAME, serverPluginConfiguration.getHostname());
        }

        // auto-discover port if needed
        if (configuration.getSimpleValue(PORT) == null) {

            int port;
            if (parentComponent instanceof ManagedASComponent) {

                ManagedASComponent managedASComponent = (ManagedASComponent) parentComponent;
                Configuration managedASConfig;
                try {
                    managedASConfig = managedASComponent.loadResourceConfiguration();
                } catch (Exception e) {
                    throw new InvalidPluginConfigurationException("Error adding resource: " + e.getMessage(), e);
                }

                PropertySimple offsetProp = managedASConfig.getSimple(MANAGED_SERVER_PORT_OFFSET_PROPERTY_NAME);
                if (offsetProp == null) {
                    throw new InvalidPluginConfigurationException(
                        "Could not find Managed Server socket binding offset, skipping discovery");
                }
                port = DOMAIN_REMOTING_PORT_DEFAULT;

                port += offsetProp.getIntegerValue();

            } else if (parentComponent instanceof StandaloneASComponent) {

                port = serverPluginConfiguration.getPort();
                port += STANDALONE_REMOTING_PORT_OFFSET;

                // overrides user provided username, password!!!
                configuration.setSimpleValue(USERNAME, serverPluginConfiguration.getUser());
                configuration.setSimpleValue(PASSWORD, serverPluginConfiguration.getPassword());

            } else {
                throw new InvalidPluginConfigurationException(
                    parentComponent + " is not a supported parent component");
            }

            configuration.setSimpleValue(PORT, String.valueOf(port));
        }

        // find client jar
        String clientJarPath = "bin" + File.separator + "client" + File.separator + "jboss-client.jar";
        File clientJarFile = new File(serverPluginConfiguration.getHomeDir(), clientJarPath);
        if (!clientJarFile.isFile()) {
            throw new InvalidPluginConfigurationException(clientJarFile + " does not exist.");
        }

        configuration.setSimpleValue(CLIENT_JAR_LOCATION, clientJarFile.getAbsolutePath());
        configuration.setSimpleValue(PROTOCOL, "remoting-jmx");

        // validate configuration
        if (ApplicationMBeansDiscoveryComponent.loadEmsConnection(configuration,
            resourceDiscoveryContext.getParentResourceContext().getTemporaryDirectory()) == null) {
            throw new InvalidPluginConfigurationException(
                "Unable to connect to management service, enable agent debug logging for more details");
        }

        return new DiscoveredResourceDetails(resourceDiscoveryContext.getResourceType(),
            getNewResourceKey(configuration), getNewResourceName(configuration), getNewResourceVersion(configuration),
            getNewResourceDescription(configuration), configuration, null);
    }
}
