/*
 * #%L
 * Fuse EAP :: Config
 * %%
 * Copyright (C) 2015 RedHat
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */
package org.jboss.fuse.eap.config;

import static org.wildfly.extras.config.LayerConfig.Type.INSTALLING;
import static org.wildfly.extras.config.LayerConfig.Type.OPTIONAL;
import static org.wildfly.extras.config.LayerConfig.Type.REQUIRED;

import static org.wildfly.extras.config.NamespaceConstants.NS_DOMAIN;
import static org.wildfly.extras.config.NamespaceConstants.NS_SECURITY;

import java.net.URL;
import java.util.Arrays;
import java.util.List;

import org.jdom.Element;
import org.jdom.Namespace;
import org.jdom.Text;
import org.wildfly.extras.config.ConfigContext;
import org.wildfly.extras.config.ConfigPlugin;
import org.wildfly.extras.config.ConfigSupport;
import org.wildfly.extras.config.LayerConfig;
import org.wildfly.extras.config.NamespaceRegistry;

public abstract class RTGovConfigPlugin implements ConfigPlugin {

    protected static final String NS_OVERLORD_CONFIG = "urn:jboss:domain:overlord-configuration";
    protected static final String NS_OVERLORD_DEPLOY = "urn:jboss:domain:overlord-deployment";

    private NamespaceRegistry registry;

    public RTGovConfigPlugin()
    {
        registerNamespaceVersions(new NamespaceRegistry());
    }

    public RTGovConfigPlugin(NamespaceRegistry registry)
    {
        registerNamespaceVersions(registry);
    }


    @Override
    public abstract String getConfigName();

    @Override
    public List<LayerConfig> getLayerConfigs() {
        return Arrays.asList(
            new LayerConfig("bpms", REQUIRED, 0),
            new LayerConfig("rtgov", INSTALLING, 1),
            new LayerConfig("soa", OPTIONAL, 2),
            new LayerConfig("fuse", REQUIRED, 3));
    }

    @Override
    public void applyStandaloneConfigChange(ConfigContext context, boolean enable) {
        updateExtensions (context, enable);
        updateVault (context, enable);
        updateSecurityDomain (context, enable);
        updateOverlordConfigSubsystem (context, enable);
        updateOverlordDeploySubsystem (context, enable);
    }

    @Override
    public void applyDomainConfigChange(ConfigContext context, boolean enable) {
         applyStandaloneConfigChange(context, enable);
    }

    // client & server
    private void updateExtensions(ConfigContext context, boolean enable) {
        String extPrefix = "org.overlord.commons.eap.extensions.";
        String[] extensionNames = new String[2];
        extensionNames[0] = extPrefix + "config";
        extensionNames[1] = extPrefix + "deploy";

        Namespace[] domainNamespaces = registry.getNamespaces(NS_DOMAIN);
        Element extensions = ConfigSupport.findChildElement(context.getDocument().getRootElement(), "extensions", domainNamespaces);
        ConfigSupport.assertExists(extensions, "Did not find the <extensions> element");
        Namespace domainNamespace = context.getDocument().getRootElement().getNamespace();
        for (String extName : extensionNames) {
            Element element = ConfigSupport.findElementWithAttributeValue(extensions, "extension", "module", extName, domainNamespaces);
            if (enable && element == null) {
                extensions.addContent(new Text("    "));
                extensions.addContent(new Element("extension", domainNamespace).setAttribute("module", extName));
                extensions.addContent(new Text("\n    "));
            }
            if (!enable && element != null) {
                element.getParentElement().removeContent(element);
            }
        }
    }

    // client & server
    private void updateVault (ConfigContext context, boolean enable) {
        Namespace[] domainNamespaces = registry.getNamespaces(NS_DOMAIN);
        Element root = context.getDocument().getRootElement();
        Element vault = ConfigSupport.findChildElement(root, "vault", domainNamespaces);

        Namespace domainNamespace = root.getNamespace();
        if (enable && vault == null) {
            // Insert the vault section before the management section.
            Element management = ConfigSupport.findChildElement(root, "management", domainNamespaces);

            ConfigSupport.assertExists (management, "Did not find the <management> element");
            int managementIndex = root.indexOf (management);
            URL resource = RTGovConfigPlugin.class.getResource ("/common/vault.xml");
            Element vaultElement = ConfigSupport.loadElementFrom(resource).setNamespace(domainNamespace);
            List<Element> children = vaultElement.getChildren();
            for (Element child : children)
            {
                child.setNamespace(domainNamespace);
            }

            root.addContent (managementIndex - 1, vaultElement);
            root.addContent (managementIndex - 1, new Text ("\n"));
        }

        if (!enable && vault != null) {
            root.removeContent (vault);
        }
    }

    // client & server
    private void updateSecurityDomain (ConfigContext context, boolean enable) {
        Namespace[] domainNamespaces = registry.getNamespaces(NS_DOMAIN);
        Namespace[] securityNamespaces = registry.getNamespaces(NS_SECURITY);

        List<Element> profiles = ConfigSupport.findProfileElements (context.getDocument (), domainNamespaces);
        for (Element profile : profiles) {
            Element security = ConfigSupport.findChildElement(profile, "subsystem", securityNamespaces);
            ConfigSupport.assertExists (security, "Did not find the security subsystem");
            Element domains = ConfigSupport.findChildElement(security, "security-domains", securityNamespaces);
            ConfigSupport.assertExists(domains, "Did not find the <security-domains> element");

            String[] domainNames = { "overlord-idp", "overlord-sp", "overlord-jaxrs" };
            for (String domainName : domainNames) {
                Element domain = ConfigSupport.findElementWithAttributeValue (domains, "security-domain", "name", domainName, securityNamespaces);
                if (enable && domain == null) {
                    URL resource = RTGovConfigPlugin.class.getResource ("/common/security-" + domainName + ".xml");
                    domains.addContent (new Text("    "));
                    domains.addContent (ConfigSupport.loadElementFrom (resource));
                    domains.addContent (new Text ("\n            "));
                }

                if (!enable && domain != null) {
                    domain.getParentElement().removeContent (domain);
                }
            }
        }
    }

    protected abstract void updateOverlordConfigSubsystem (ConfigContext context, boolean enable);

    protected abstract void updateOverlordDeploySubsystem (ConfigContext context, boolean enable);

    protected void updateSubsystem (ConfigContext context, boolean enable, String ns, String resourceName) {
        Namespace[] domainNamespaces = registry.getNamespaces(NS_DOMAIN);
        Namespace[] subsystemNamespaces = registry.getNamespaces(ns);
        List<Element> profiles = ConfigSupport.findProfileElements (context.getDocument(), domainNamespaces);
        for (Element profile : profiles) {
            Element element = ConfigSupport.findChildElement(profile, "subsystem", subsystemNamespaces);
            if (enable && element == null) {
                URL resource = RTGovConfigPlugin.class.getResource (resourceName);
                profile.addContent (new Text ("    "));
                profile.addContent (ConfigSupport.loadElementFrom (resource));
                profile.addContent (new Text ("\n    "));
            }
            if (!enable && element != null) {
                element.getParentElement().removeContent (element);
            }
        }
    }

    private void registerNamespaceVersions (NamespaceRegistry registry)
    {
        this.registry = registry;
        registry.registerNamespace(NS_DOMAIN, "1.8");
        registry.registerNamespace(NS_DOMAIN, "1.7");
        registry.registerNamespace(NS_SECURITY, "1.2");
        registry.registerNamespace(NS_OVERLORD_CONFIG, "1.0");
        registry.registerNamespace(NS_OVERLORD_DEPLOY, "1.0");
    }
}
