/*
 * #%L
 * Wildfly Camel :: Testsuite
 * %%
 * Copyright (C) 2013 - 2014 RedHat
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

package org.wildfly.camel.test.aws;

import static com.amazonaws.services.ec2.model.InstanceStateName.ShuttingDown;
import static com.amazonaws.services.ec2.model.InstanceStateName.Terminated;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.inject.Inject;

import org.apache.camel.ProducerTemplate;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.component.aws.ec2.EC2Constants;
import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.asset.EmptyAsset;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.wildfly.camel.test.aws.subA.EC2ClientProducer;
import org.wildfly.camel.test.aws.subA.EC2ClientProducer.EC2ClientProvider;
import org.wildfly.camel.test.common.aws.AWSUtils;
import org.wildfly.camel.test.common.aws.BasicCredentialsProvider;
import org.wildfly.camel.test.common.aws.EC2Utils;
import org.wildfly.extension.camel.CamelAware;
import org.wildfly.extension.camel.WildFlyCamelContext;

import com.amazonaws.services.ec2.AmazonEC2Client;
import com.amazonaws.services.ec2.model.Instance;
import com.amazonaws.services.ec2.model.InstanceStateName;
import com.amazonaws.services.ec2.model.InstanceType;
import com.amazonaws.services.ec2.model.RunInstancesResult;
import com.amazonaws.services.ec2.model.Tag;
import com.amazonaws.services.ec2.model.TerminateInstancesResult;

@CamelAware
@RunWith(Arquillian.class)
public class EC2IntegrationTest {

    @Inject
    private EC2ClientProvider provider;

    @Deployment
    public static JavaArchive deployment() {
        JavaArchive archive = ShrinkWrap.create(JavaArchive.class, "aws-ec2-tests.jar");
        archive.addClasses(EC2ClientProducer.class, EC2Utils.class, BasicCredentialsProvider.class);
        archive.addAsManifestResource(EmptyAsset.INSTANCE, "beans.xml");
        return archive;
    }

    public static void assertNoStaleInstances(AmazonEC2Client client, String when) {
        List<Instance> staleInstances = client.describeInstances().getReservations().stream()
        .flatMap(r -> r.getInstances().stream())
        .filter(i -> {
            String state = i.getState().getName();
            List<Tag> tags = i.getTags().stream().filter(t -> "Name".equals(t.getKey()) && "wildfly-camel".equals(t.getValue())).collect(Collectors.toList());
            boolean stillActive = !ShuttingDown.toString().equals(state) && !Terminated.toString().equals(state);
            boolean startedRecently = AWSUtils.HOUR < System.currentTimeMillis() - i.getLaunchTime().getTime();
            return tags.size() > 0 && stillActive && startedRecently;
        })
        .collect(Collectors.toList());
        Assert.assertEquals(String.format("Found stale EC2 instances %s running the test: %s", when, staleInstances), 0, staleInstances.size());
    }

    @Test
    public void testCreateInstance() throws Exception {

        AmazonEC2Client ec2Client = provider.getClient();
        Assume.assumeNotNull("AWS client not null", ec2Client);

        assertNoStaleInstances(ec2Client, "before");

        try {
            WildFlyCamelContext camelctx = new WildFlyCamelContext();
            camelctx.getNamingContext().bind("ec2Client", ec2Client);
            camelctx.addRoutes(new RouteBuilder() {
                @Override
                public void configure() throws Exception {
                    from("direct:createAndRun").to("aws-ec2://TestDomain?amazonEc2Client=#ec2Client&operation=createAndRunInstances");
                    from("direct:terminate").to("aws-ec2://TestDomain?amazonEc2Client=#ec2Client&operation=terminateInstances");
                }
            });

            camelctx.start();
            try {

                // Create and run an instance
                Map<String, Object> headers = new HashMap<>();
                headers.put(EC2Constants.IMAGE_ID, "ami-02ace471");
                headers.put(EC2Constants.INSTANCE_TYPE, InstanceType.T2Micro);
                headers.put(EC2Constants.SUBNET_ID, EC2Utils.getSubnetId(ec2Client));
                headers.put(EC2Constants.INSTANCE_MIN_COUNT, 1);
                headers.put(EC2Constants.INSTANCE_MAX_COUNT, 1);
                headers.put(EC2Constants.INSTANCES_TAGS, Arrays.asList(new Tag("Name", "wildfly-camel")));

                ProducerTemplate template = camelctx.createProducerTemplate();
                RunInstancesResult result1 = template.requestBodyAndHeaders("direct:createAndRun", null, headers, RunInstancesResult.class);
                String instanceId = result1.getReservation().getInstances().get(0).getInstanceId();
                System.out.println("InstanceId: " + instanceId);

                // Terminate the instance
                headers = new HashMap<>();
                headers.put(EC2Constants.INSTANCES_IDS, Collections.singleton(instanceId));

                TerminateInstancesResult result2 = template.requestBodyAndHeaders("direct:terminate", null, headers, TerminateInstancesResult.class);
                Assert.assertEquals(instanceId, result2.getTerminatingInstances().get(0).getInstanceId());
            } finally {
                camelctx.stop();
            }
        } finally {
            assertNoStaleInstances(ec2Client, "after");
        }
    }
}
