/*
 * Copyright Debezium Authors.
 *
 * Licensed under the Apache Software License version 2.0, available at http://www.apache.org/licenses/LICENSE-2.0
 */
package io.debezium.connector.mongodb;

import static org.fest.assertions.Assertions.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import java.util.List;

import org.junit.Before;
import org.junit.Test;

import com.mongodb.MongoException;
import com.mongodb.MongoSocketOpenException;
import com.mongodb.ServerAddress;
import com.mongodb.client.MongoClient;
import com.mongodb.connection.ClusterConnectionMode;
import com.mongodb.connection.ClusterDescription;
import com.mongodb.connection.ClusterType;
import com.mongodb.connection.ServerConnectionState;
import com.mongodb.connection.ServerDescription;

import io.debezium.util.Collect;

public class ReplicaSetDiscoveryTest {

    private ReplicaSetDiscovery replicaSetDiscovery;
    private MongoDbTaskContext context;
    private ConnectionContext connectionContext;
    private MongoClient mongoClient;
    private String seedHosts = "host1:27017,host2:27017";

    @Before
    public void setup() {
        context = mock(MongoDbTaskContext.class);
        connectionContext = mock(ConnectionContext.class);
        mongoClient = mock(MongoClient.class);
        when(context.getConnectionContext()).thenReturn(connectionContext);
        when(connectionContext.hosts()).thenReturn(seedHosts);
        when(connectionContext.clientFor(seedHosts)).thenReturn(mongoClient);
        replicaSetDiscovery = new ReplicaSetDiscovery(context);
    }

    /**
     * Testing the case where a member of a replicaset may be down.
     */
    @Test
    public void shouldGetFirstValidReplicaSetName() {
        // Skip the config server section of ReplicaSetDiscovery.getReplicaSets
        // by throwing an error here.
        when(mongoClient.listDatabaseNames()).thenThrow(new MongoException("dummy"));

        ServerAddress host1Address = new ServerAddress("host1");
        ServerAddress host2Address = new ServerAddress("host2");

        List<ServerDescription> serverDescriptions = Collect.arrayListOf(
                ServerDescription.builder()
                        .address(host1Address)
                        .state(ServerConnectionState.CONNECTING)
                        .exception(new MongoSocketOpenException("can't connect", host1Address))
                        .build(),
                ServerDescription.builder()
                        .address(host2Address)
                        .state(ServerConnectionState.CONNECTED)
                        .setName("my_rs")
                        .build());

        ClusterDescription clusterDescription = new ClusterDescription(
                ClusterConnectionMode.MULTIPLE,
                ClusterType.REPLICA_SET,
                serverDescriptions);

        when(mongoClient.getClusterDescription()).thenReturn(clusterDescription);

        ReplicaSets replicaSets = replicaSetDiscovery.getReplicaSets();
        assertThat(replicaSets.validReplicaSets().size()).isEqualTo(1);
        assertThat(replicaSets.validReplicaSets().get(0).replicaSetName()).isEqualTo("my_rs");
        assertThat(replicaSets.validReplicaSets().get(0).addresses()).isEqualTo(Collect.arrayListOf(host2Address));
    }
}
