/*******************************************************************************
 * Copyright (c) 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.websphere.logging.hpel.reader;

import java.util.Date;
import java.util.Properties;
import java.util.logging.Level;


/**
 * An interface for reading HPEL repositories.
 * <p>
 * The getLogLists() methods return lists of log record lists in which the log 
 * record (inner) list represents log records generated by one process.
 * <p>
 * For example, If your repository contains log records from two separate 
 * process lifespans, you'll have the following results:
 * <code><pre>
 * <b>for</b> ({@link ServerInstanceLogRecordList} oneProcessList: {@link #getLogLists()}) {
 * 	// this loop will be executed twice - once for each process lifespan
 *  
 * 	// The header contains information about the server
 * 	{@link Properties} header = oneProcessList.getHeader(); 
 * 	
 * 	<b>for</b> ({@link RepositoryLogRecord} record: oneProcessList) {
 * 		// This loop iterates over the list of RepositoryLogRecords from one process lifespan.
 * 	}
 * }
 * </pre></code>
 * 
 * All getLogList* queries have the ability to start from a particular
 * RepositoryLogRecord.  This is done by passing to the query method the 
 * RepositoryPointer information from that record. Here is an example:
 * <code><pre>
 * RepositoryPointer pointer;
 * <b>for</b> ({@link RepositoryLogRecord} record: {@link #getLogListForCurrentServerInstance()}) {
 * 	// process the record
 * 	<b>if</b> (&lt;this is the record we want to start from next time&gt;) {
 * 		pointer = record.getRepositoryPointer();
 * 	}
 * }
 * // after the break
 * <b>for</b> ({@link RepositoryLogRecord} record: getLogListForServerInstance(last)) {
 * 	// finish what was started.
 * }
 * </pre></code>
 * 
 * A simple sample of using the reader to report on all logs currently in a server repository:
 * <code><pre>
 *	public static void main(String[] args) {
 *				// Create a repository reader (requires base directory of repository 
 *		RepositoryReader logRepository = new RepositoryReaderImpl(args[0]) ;		
 *				// Get iterator of server instances (start/stop of the server) extracting all log messages with
 *				// severity between INFO and SEVERE.  Lots of different filtering options, this is just one sample
 *		Iterable&lt;ServerInstanceLogRecordList&gt; repResults = logRepository.getLogLists(Level.INFO, Level.SEVERE) ;
 *				// Go through each server instance
 *		for (ServerInstanceLogRecordList pidRecords: repResults) {		// For each list (server lifeCycle)
 *				// For each server instance, go through the records
 *			for (RepositoryLogRecord repositoryLogRecord : pidRecords) {
 *					// Just printing some key information here.  Note that the repositoryRecord exposes all fields
 *					// with simple get methods
 *					System.out.println("  "+repositoryLogRecord.getFormattedMessage());
 *			}
 *		}
 *	}
 * </pre></code>
 * 
 * Note that any records added to the repository after the repository reader
 * instance has been created will not be included in the record list. 
 * 
 * @ibm-api
 */
public interface RepositoryReader {	

	/**
	 * returns all log records from the repository.  
	 * 
	 * @return           the iterable instance of a list of log records within
	 *                   a process
	 * @throws LogRepositoryException indicating that an error occurred while reading list of instances from the server.
	 */	
	public Iterable<ServerInstanceLogRecordList> getLogLists() throws LogRepositoryException;
	
	/**
	 * returns all log records from the repository beyond a given repository
	 * location as specified.  Callers need to invoke
	 * {@link RepositoryLogRecord#getRepositoryPointer()} to obtain the
	 * RepositoryPointer for a particular log record.
	 * 
	 * @param after      pointer to a record the list will start after 
	 * @return           the iterable instance of a list of log records within 
	 *                   a process.  If no records meet the criteria, an Iterable is returned with no entries
	 * @throws LogRepositoryException indicating that an error occurred while reading list of instances from the server.
	 */	
	public Iterable<ServerInstanceLogRecordList> getLogLists(RepositoryPointer after) throws LogRepositoryException;

	/**
	 * returns log records from the repository that are within the level range as
	 * specified.
	 * 
	 * @param  minLevel  minimum {@link Level} that will be included in the 
	 *                   returned list
	 * @param  maxLevel  maximum {@link Level} that will be included in the
	 *                   returned list
	 * @return           the iterable instance of a list of records within 
	 *                   a process that are within the parameter range 
	 *                   If no records meet the criteria, an Iterable is returned with no entries
	 * @throws LogRepositoryException indicating that an error occurred while reading list of instances from the server.
	 */
	public Iterable<ServerInstanceLogRecordList> getLogLists(Level minLevel, Level maxLevel) throws LogRepositoryException;

	/**
	 * returns log records from the repository that are beyond a given 
	 * repository location and within the level range as specified.
	 * Callers need to invoke 
	 * {@link RepositoryLogRecord#getRepositoryPointer()} to obtain the 
	 * RepositoryPointer for a particular log record.
	 * 
	 * @param  after     pointer to a record the list will start after
	 * @param  minLevel  minimum {@link Level} that will be included in the 
	 *                   returned list
	 * @param  maxLevel  maximum {@link Level} that will be included in the
	 *                   returned list
	 * @return           the iterable instance of a list of log records within
	 *                   a process that are within the parameter range 
	 *                   If no records meet the criteria, an Iterable is returned with no entries
	 * @throws LogRepositoryException indicating that an error occurred while reading list of instances from the server.
	 */	
	public Iterable<ServerInstanceLogRecordList> getLogLists(RepositoryPointer after, Level minLevel, Level maxLevel) throws LogRepositoryException;

	/**
	 * returns log records from the repository that are within the date range 
	 * as specified.  The parameters can be null to indicate an open ended 
	 * range.  For example, a caller may pass <code>null</code>
	 * as the maxTime to indicate that the returned log records do not have
	 * to meet a maxTime criteria.
	 * 
	 * @param  minTime   minimum {@link Date} value that will be included in
	 *                   the returned list 
	 * @param  maxTime   maximum {@link Date} value that will be included in
	 *                   the returned list
	 * @return           the iterable instance of a list of log records within 
	 *                   a process that are within the parameter range 
	 *                   If no records meet the criteria, an Iterable is returned with no entries
	 * @throws LogRepositoryException indicating that an error occurred while reading list of instances from the server.
	 */	
	public Iterable<ServerInstanceLogRecordList> getLogLists(Date minTime, Date maxTime) throws LogRepositoryException;
	
	/**
	 * returns log records from the repository that are beyond a given 
	 * repository location and occurred before a given time as specified.
	 * Callers need to invoke 
	 * {@link RepositoryLogRecord#getRepositoryPointer()} to obtain the 
	 * RepositoryPointer for a particular log record.
	 * 
	 * @param  after     pointer to a record the list will start after
	 * @param  maxTime   maximum {@link Date} value that will be included in
	 *                   the returned list
	 * @return           the iterable instance of a list of log records within
	 *                   a process that are within the parameter range 
	 *                   If no records meet the criteria, an Iterable is returned with no entries
	 * @throws LogRepositoryException indicating that an error occurred while reading list of instances from the server.
	 */	
	public Iterable<ServerInstanceLogRecordList> getLogLists(RepositoryPointer after, Date maxTime) throws LogRepositoryException;
	
	/**
	 * returns log records from the repository according to the criteria specified
	 * by the log query bean.
	 * 
	 * @param  query     {@link LogQueryBean} instance representing set of criteria
	 *                   each of which need to be met by the return records.
	 * @return           the iterable instance of a list of log records within
	 *                   a process that are within the parameter range 
	 *                   If no records meet the criteria, an Iterable is returned with no entries
	 * @throws LogRepositoryException indicating that an error occurred while reading list of instances from the server.
	 */	
	public Iterable<ServerInstanceLogRecordList> getLogLists(LogQueryBean query) throws LogRepositoryException;
	
	/**
	 * returns log records from the repository that are beyond a given
	 * repository location, according to the criteria specified by the log query
	 * bean.
	 * 
	 * @param  after     pointer to a record the list will start after
	 * @param  query     {@link LogQueryBean} instance representing set of criteria
	 *                   each of which need to be met by the return records.
	 * @return           the iterable instance of a list of log records within
	 *                   a process that are within the parameter range 
	 *                   If no records meet the criteria, an Iterable is returned with no entries
	 * @throws LogRepositoryException indicating that an error occurred while reading list of instances from the server.
	 */	
	public Iterable<ServerInstanceLogRecordList> getLogLists(RepositoryPointer after, LogQueryBean query) throws LogRepositoryException;
	
	/**
	 * returns log records from the repository that were created since the last 
	 * server startup.
	 * 
	 * @return the iterable list of log records
	 * If no records meet the criteria, an Iterable is returned with no entries
	 * @throws LogRepositoryException indicating that an error occurred while reading records from the server.
	 */
	public ServerInstanceLogRecordList getLogListForCurrentServerInstance() throws LogRepositoryException;
	
	/**
	 * returns log records from the repository of a server instance running
	 * at the time specified.
	 * 
	 * @param  time     {@link Date} value used to determine the
	 *                  server instance where the server start time occurs
	 *                  before this value and the server stop time occurs
	 *                  after this value
	 * @return          the iterable list of log records
	 * 					If no records meet the criteria, a list is returned with a null properties object and a null record list
	 * @throws LogRepositoryException indicating that an error occurred while reading records from the server.
	 */
	public ServerInstanceLogRecordList getLogListForServerInstance(Date time) throws LogRepositoryException;
	
	/**
	 * returns log records from the repository that are beyond a given 
	 * repository location as specified.  Callers need to invoke
	 * {@link RepositoryLogRecord#getRepositoryPointer()} to obtain the
	 * RepositoryPointer for a particular log record.  The returned logs will
	 * be from one server instance.
	 * 
	 * @param  after    pointer to a record the list will start after
	 * @return          the iterable list of log records
	 * 					If no records meet the criteria, the list is empty.  
	 * @throws LogRepositoryException indicating that an error occurred while reading records from the server.
	 */
	public ServerInstanceLogRecordList getLogListForServerInstance(RepositoryPointer after) throws LogRepositoryException;

	/**
 	 * returns log records from the repository of a server instance running
	 * at a specified time, within the level range as specified.
	 * 
	 * @param  time     {@link Date} value used to determine the
	 *                  server instance where the server start time occurs
	 *                  before this value and the server stop time occurs
	 *                  after this value
	 * @param  minLevel  minimum {@link Level} that will be included in the 
	 *                   returned list
	 * @param  maxLevel  maximum {@link Level} that will be included in the
	 *                   returned list
	 * @return           the iterable list of log records
	 * 					 If no records meet the criteria, the list is empty.  
	 * @throws LogRepositoryException indicating that an error occurred while reading records from the server.
	 */
	public ServerInstanceLogRecordList getLogListForServerInstance(Date time, Level minLevel, Level maxLevel) throws LogRepositoryException;
	
	/**
	 * returns log records from the repository that are beyond a given 
	 * repository location and within the level range as specified.
	 * Callers need to invoke 
	 * {@link RepositoryLogRecord#getRepositoryPointer()} to obtain the
	 * RepositoryPointer for a particular log record.  The returned logs will
	 * be from one server instance.
	 * 
	 * @param  after     pointer to a record the list will start after
	 * @param  minLevel  minimum {@link Level} that will be included in the 
	 *                   returned list
	 * @param  maxLevel  maximum {@link Level} that will be included in the
	 *                   returned list
	 * @return           the iterable list of log records
	 * 					 If no records meet the criteria, the list is empty.  
	 * @throws LogRepositoryException indicating that an error occurred while reading records from the server.
	 */
	public ServerInstanceLogRecordList getLogListForServerInstance(RepositoryPointer after, Level minLevel, Level maxLevel) throws LogRepositoryException;

	/**
	 * returns log records from the repository of a server instance running
	 * at a specified time created by the given thread.
	 * 
	 * @param  time     {@link Date} value used to determine the
	 *                  server instance where the server start time occurs
	 *                  before this value and the server stop time occurs
	 *                  after this value
	 * @param  threadID  ID of the thread that the returned records will match
	 * @return           the iterable list of log records
	 * 					 If no records meet the criteria, the list is empty.  
	 * @throws LogRepositoryException indicating that an error occurred while reading records from the server.
	 */
	public ServerInstanceLogRecordList getLogListForServerInstance(Date time, int threadID) throws LogRepositoryException;
	
	/**
	 * returns log records from the repository that are beyond a given
	 * repository location and created by a given thread as specified.
	 * Callers need to invoke 
	 * {@link RepositoryLogRecord#getRepositoryPointer()} to obtain the
	 * RepositoryPointer for a particular log record.  The returned logs will
	 * be from one server instance.
	 * 
	 * @param  after     pointer to a record the list will start after
	 * @param  threadID  ID of the thread that the returned records will match
	 * @return           the iterable list of log records
	 * 					 If no records meet the criteria, the list is empty.  
	 * @throws LogRepositoryException indicating that an error occurred while reading records from the server.
	 */
	public ServerInstanceLogRecordList getLogListForServerInstance(RepositoryPointer after, int threadID) throws LogRepositoryException;

	/**
 	 * returns log records from the repository of a server instance running
	 * at a specified time, according to the criteria specified by the log query
	 * bean.
	 * 
	 * @param  time      {@link Date} value used to determine the
	 *                   server instance where the server start time occurs
	 *                   before this value and the server stop time occurs
	 *                   after this value
	 * @param query      {@link LogQueryBean} instance representing set of criteria
	 *                   each of which need to be met by the return records.
	 * @return           the iterable list of log records
	 * 					 If no records meet the criteria, the list is empty.
	 * @throws LogRepositoryException indicating that an error occurred while reading records from the server.
	 * @see LogQueryBean
	 */
	public ServerInstanceLogRecordList getLogListForServerInstance(Date time, LogQueryBean query) throws LogRepositoryException;
	
	/**
 	 * returns log records from the repository for one server instance that are
	 * beyond a given repository location, according to the criteria specified by the log query
	 * bean.
	 * 
	 * @param  after     pointer to a record the list will start after
	 * @param query      {@link LogQueryBean} instance representing set of criteria
	 *                   each of which need to be met by the return records.
	 * @return           the iterable list of log records
	 * 					 If no records meet the criteria, the list is empty.
	 * @throws LogRepositoryException indicating that an error occurred while reading records from the server.
	 * @see LogQueryBean
	 */
	public ServerInstanceLogRecordList getLogListForServerInstance(RepositoryPointer after, LogQueryBean query) throws LogRepositoryException;

}
