/*******************************************************************************
 * Copyright (c) 2012, 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package com.ibm.ws.sib.api.jms.impl;

import java.util.Hashtable;
import java.util.concurrent.ConcurrentHashMap;

import javax.jms.ConnectionFactory;
import javax.jms.ConnectionMetaData;
import javax.jms.InvalidDestinationException;
import javax.jms.JMSException;
import javax.jms.QueueConnectionFactory;
import javax.jms.TopicConnectionFactory;
import javax.naming.Context;
import javax.naming.Name;
import javax.naming.Reference;
import javax.naming.spi.ObjectFactory;
import javax.resource.ResourceException;

import com.ibm.websphere.ras.TraceComponent;
//Sanjay Liberty Changes
//import javax.resource.ResourceException;
import com.ibm.websphere.sib.api.jms.ApiJmsConstants;
import com.ibm.websphere.sib.api.jms.JmsConnectionFactory;
import com.ibm.websphere.sib.api.jms.JmsFactoryFactory;
import com.ibm.websphere.sib.api.jms.JmsQueue;
import com.ibm.websphere.sib.api.jms.JmsQueueConnectionFactory;
import com.ibm.websphere.sib.api.jms.JmsTopic;
import com.ibm.websphere.sib.api.jms.JmsTopicConnectionFactory;
import com.ibm.ws.sib.api.jms.JmsRAFactoryFactory;
import com.ibm.ws.sib.api.jmsra.JmsJcaConnectionFactory;
import com.ibm.ws.sib.api.jmsra.JmsJcaFactory;
import com.ibm.ws.sib.api.jmsra.JmsJcaManagedConnectionFactory;
import com.ibm.ws.sib.api.jmsra.JmsJcaManagedQueueConnectionFactory;
import com.ibm.ws.sib.api.jmsra.JmsJcaManagedTopicConnectionFactory;
import com.ibm.ws.sib.utils.ras.SibTr;

public class JmsFactoryFactoryImpl extends JmsFactoryFactory implements ApiJmsConstants, ObjectFactory, JmsRAFactoryFactory
{

    // ************************** TRACE INITIALISATION ***************************

    private static TraceComponent tc = SibTr.register(JmsFactoryFactoryImpl.class, ApiJmsConstants.MSG_GROUP_INT, ApiJmsConstants.MSG_BUNDLE_INT);

    // **************************** STATE VARIABLES ******************************

    /**
     * Utility class which performs URI decoding
     */
    private static URIDestinationCreator destCreator = new URIDestinationCreator();

    // *************************** INTERFACE METHODS *****************************

    /**
     * Only one instance of the clientIdTable exists always per JVM.
     * Its place holder for the tuple <ClientID, count>.
     * Where the "clientID" is a non-default client ID(defult = "clientID") which is either specified explicitly through Server.xml when defining ConnectionFactory
     * or calling setClientID() method on JMSContext/Connection object.
     * Where the "count" is number of connections sets the same client ID. Remember, it is possible to end up in a situation where more than one connection sets the same client ID.
     * For instance: Whenever client ID is set through javax.jms.JmsConnectionImpl's constructor(Constructor will be called whenever you define Connection Factory in Server.xml),
     * then system will not check for whether this clientID is already used by another connection.
     * */
    private static ConcurrentHashMap<String, Integer> clientIdTable = new ConcurrentHashMap<String, Integer>();

    /**
     * This method is called by the application to retrieve its first
     * ConnectionFactory object.
     */
    @Override
    public JmsConnectionFactory createConnectionFactory() throws JMSException {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(this, tc, "createConnectionFactory");
        JmsConnectionFactory jmscf = null;

        // get a jca managed connection factory, which is used to construct
        // the jms connection factory.
        JmsJcaFactory jcaFact = JmsJcaFactory.getInstance();

        if (jcaFact == null) {
            // This indicates that the jmsra.impl class could not be loaded by their
            // static initializer. Nothing we can do about that...

            // d238447 FFDC review. FFDC has already been generated by JmsJcaFactory, so
            //   don't generate one here.
            throw (JMSException) JmsErrorUtils.newThrowable(
                                                            JMSException.class,
                                                            "UNABLE_TO_CREATE_FACTORY_CWSIA0008",
                                                            new Object[] { "JmsJcaFactoryImpl", "sib.api.jmsraOutboundImpl.jar" },
                                                            tc
                            );
        }

        JmsJcaManagedConnectionFactory jcamcf = jcaFact.createManagedConnectionFactory();

        // invoke the construction of a jms connection factory - the jca managed
        // connection factory will call back into the jms api code by getting a
        // JmsRAFactoryFactory, and calling a create jms connection factory
        // method on it with jca connection factory and jca managed connection
        // factory objects as parameters (the jms cf will keep references to the
        // jca cf and jca mcf).
        try {
            jmscf = (JmsConnectionFactory) jcamcf.createConnectionFactory();
        } catch (ResourceException re) {
            // No FFDC code needed
            // d222942 review - no documentation for when a RE will be thrown, so
            // default exception ok. (A quick look at the RA suggests the exception isn't
            // ever thrown).
            // d238447 FFDC review. From the above comment, generating an FFDC seems ok.
            throw (JMSException) JmsErrorUtils.newThrowable(
                                                            JMSException.class,
                                                            "EXCEPTION_RECEIVED_CWSIA0007",
                                                            new Object[] { re, "JmsFactoryFactoryImpl.createConnectionFactory (#2)" },
                                                            re,
                                                            "JmsFactoryFactoryImpl.createConnectionFactory#2",
                                                            this,
                                                            tc
                            );
        }

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(this, tc, "createConnectionFactory", jmscf);
        return jmscf;
    }

    /**
     * This method is called by the application to retrieve its first
     * QueueConnectionFactory object.
     */
    @Override
    public JmsQueueConnectionFactory createQueueConnectionFactory() throws JMSException {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(this, tc, "createQueueConnectionFactory");
        JmsQueueConnectionFactory jmsqcf = null;

        // get a jca managed queue connection factory, which is used to construct
        // the jms queue connection factory.
        JmsJcaManagedQueueConnectionFactory jcamqcf = JmsJcaFactory.getInstance().createManagedQueueConnectionFactory();

        // invoke the construction of a jms queue connection factory - the jca
        // managed queue connection factory will call back into the jms api code
        // by getting a JmsRAFactoryFactory, and calling a create jms queue
        // connection factory method on it with jca connection factory and jca
        // managed queue connection factory objects as parameters (the jms qcf
        // will keep references to the jca cf and jca mqcf).
        try {
            jmsqcf = (JmsQueueConnectionFactory) jcamqcf.createConnectionFactory();
        } catch (ResourceException re) {
            // No FFDC code needed
            // d222942 review - no documentation for when a RE will be thrown, so
            // default exception ok. (A quick look at the RA suggests the exception isn't
            // ever thrown).
            // d238447 FFDC review. Comment above suggests an FFDC ok for this case.
            throw (JMSException) JmsErrorUtils.newThrowable(
                                                            JMSException.class,
                                                            "EXCEPTION_RECEIVED_CWSIA0007",
                                                            new Object[] { re, "JmsFactoryFactoryImpl.createQueueConnectionFactory (#1)" },
                                                            re,
                                                            "JmsFactoryFactoryImpl.createQueueConnectionFactory#1",
                                                            this,
                                                            tc
                            );
        }

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(this, tc, "createQueueConnectionFactory", jmsqcf);
        return jmsqcf;
    }

    /**
     * This method is called by the application to retrieve its first
     * TopicConnectionFactory object.
     */
    @Override
    public JmsTopicConnectionFactory createTopicConnectionFactory() throws JMSException {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(this, tc, "createTopicConnectionFactory");
        JmsTopicConnectionFactory jmstcf = null;

        // get a jca managed topic connection factory, which is used to construct
        // the jms topic connection factory.
        JmsJcaManagedTopicConnectionFactory jcamtcf = JmsJcaFactory.getInstance().createManagedTopicConnectionFactory();

        // invoke the construction of a jms topic connection factory - the jca
        // managed topic connection factory will call back into the jms api code
        // by getting a JmsRAFactoryFactory, and calling a create jms topic
        // connection factory method on it with jca connection factory and jca
        // managed topic connection factory objects as parameters (and the jms
        // tcf will keep references to the jca cf and jca mtcf).
        try {
            jmstcf = (JmsTopicConnectionFactory) jcamtcf.createConnectionFactory();
        } catch (ResourceException re) {
            // No FFDC code needed
            // d222942 review - no documentation for when a RE will be thrown, so
            // default exception ok. (A quick look at the RA suggests the exception isn't
            // ever thrown).
            // d238447 FFDC review. Comment above suggests FFDC ok.
            throw (JMSException) JmsErrorUtils.newThrowable(
                                                            JMSException.class,
                                                            "EXCEPTION_RECEIVED_CWSIA0007",
                                                            new Object[] { re, "JmsFactoryFactoryImpl.createTopicConnectionFactory (#1)" },
                                                            re,
                                                            "JmsFactoryFactoryImpl.createTopicConnectionFactory#1",
                                                            this,
                                                            tc
                            );
        }

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(this, tc, "createTopicConnectionFactory", jmstcf);
        return jmstcf;
    }

    /**
     * For use by the JCA resource adaptor.
     */
    @Override
    public ConnectionFactory createConnectionFactory(JmsJcaConnectionFactory jcaConnectionFactory) {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(this, tc, "createConnectionFactory", jcaConnectionFactory);
        ConnectionFactory cf = new JmsManagedConnectionFactoryImpl(jcaConnectionFactory);
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(this, tc, "createConnectionFactory", cf);
        return cf;
    }

    /**
     * For use by the JCA resource adaptor.
     */
    @Override
    public ConnectionFactory createConnectionFactory(JmsJcaConnectionFactory jcaConnectionFactory, JmsJcaManagedConnectionFactory jcaManagedConnectionFactory) {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(this, tc, "createConnectionFactory", new Object[] { jcaConnectionFactory, jcaManagedConnectionFactory });
        ConnectionFactory cf = new JmsConnectionFactoryImpl(jcaConnectionFactory, jcaManagedConnectionFactory);
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(this, tc, "createConnectionFactory", cf);
        return cf;
    }

    /**
     * For use by the JCA resource adaptor.
     */
    @Override
    public QueueConnectionFactory createQueueConnectionFactory(JmsJcaConnectionFactory jcaConnectionFactory) {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(this, tc, "createQueueConnectionFactory", jcaConnectionFactory);
        QueueConnectionFactory qcf = new JmsManagedQueueConnectionFactoryImpl(jcaConnectionFactory);
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(this, tc, "createQueueConnectionFactory", qcf);
        return qcf;
    }

    /**
     * For use by the JCA resource adaptor.
     */
    @Override
    public QueueConnectionFactory createQueueConnectionFactory(JmsJcaConnectionFactory jcaConnectionFactory, JmsJcaManagedQueueConnectionFactory jcaManagedQueueConnectionFactory) {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(this, tc, "createQueueConnectionFactory", new Object[] { jcaConnectionFactory, jcaManagedQueueConnectionFactory });
        QueueConnectionFactory qcf = new JmsQueueConnectionFactoryImpl(jcaConnectionFactory, jcaManagedQueueConnectionFactory);
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(this, tc, "createQueueConnectionFactory", qcf);
        return qcf;
    }

    /**
     * For use by the JCA resource adaptor.
     */
    @Override
    public TopicConnectionFactory createTopicConnectionFactory(JmsJcaConnectionFactory jcaConnectionFactory) {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(this, tc, "createTopicConnectionFactory", jcaConnectionFactory);
        TopicConnectionFactory tcf = new JmsManagedTopicConnectionFactoryImpl(jcaConnectionFactory);
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(this, tc, "createTopicConnectionFactory", tcf);
        return tcf;
    }

    /**
     * For use by the JCA resource adaptor.
     */
    @Override
    public TopicConnectionFactory createTopicConnectionFactory(JmsJcaConnectionFactory jcaConnectionFactory, JmsJcaManagedTopicConnectionFactory jcaManagedTopicConnectionFactory) {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(this, tc, "createTopicConnectionFactory", new Object[] { jcaConnectionFactory, jcaManagedTopicConnectionFactory });
        TopicConnectionFactory tcf = new JmsTopicConnectionFactoryImpl(jcaConnectionFactory, jcaManagedTopicConnectionFactory);
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(this, tc, "createTopicConnectionFactory", tcf);
        return tcf;
    }

    /**
     * This method is called by the application to create a jms administered
     * queue object.
     */
    @Override
    public JmsQueue createQueue(String name) throws JMSException {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(this, tc, "createQueue", name);
        JmsQueue queue = null;

        // if name string is null, empty or just "queue://", throw exception
        if ((name == null) || ("".equals(name)) || (JmsQueueImpl.QUEUE_PREFIX.equals(name))) {
            throw (InvalidDestinationException) JmsErrorUtils.newThrowable(
                                                                           InvalidDestinationException.class,
                                                                           "INVALID_VALUE_CWSIA0003",
                                                                           new Object[] { "Queue name", name },
                                                                           tc
                            );
        }

        // if name is "topic://" throw exception
        if (name.startsWith(JmsTopicImpl.TOPIC_PREFIX)) {
            throw (InvalidDestinationException) JmsErrorUtils.newThrowable(
                                                                           InvalidDestinationException.class,
                                                                           "MALFORMED_DESCRIPTOR_CWSIA0047",
                                                                           new Object[] { "Queue", name },
                                                                           tc
                            );
        }

        name = name.trim();
        queue = (JmsQueue) destCreator.createDestinationFromString(name, URIDestinationCreator.DestType.QUEUE);

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(this, tc, "createQueue", queue);
        return queue;
    }

    /**
     * This method is called by the application to create a jms administered
     * topic object.
     */
    @Override
    public JmsTopic createTopic(String name) throws JMSException {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(this, tc, "createTopic", name);
        JmsTopic topic = null;

        // if name string is null throw exception
        if (name == null) {
            throw (InvalidDestinationException) JmsErrorUtils.newThrowable(
                                                                           InvalidDestinationException.class,
                                                                           "INVALID_VALUE_CWSIA0003",
                                                                           new Object[] { "Topic name", null },
                                                                           tc
                            );
        }

        // if name is "queue://" throw exception
        if (name != null && name.startsWith(JmsQueueImpl.QUEUE_PREFIX)) {
            throw (InvalidDestinationException) JmsErrorUtils.newThrowable(
                                                                           InvalidDestinationException.class,
                                                                           "MALFORMED_DESCRIPTOR_CWSIA0047",
                                                                           new Object[] { "Topic", name },
                                                                           tc
                            );
        }

        name = name.trim();
        topic = (JmsTopic) destCreator.createDestinationFromString(name, URIDestinationCreator.DestType.TOPIC);

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(this, tc, "createTopic", topic);
        return topic;
    }

    /**
     * getObjectInstance(Object, Name, Context, Hashtable)
     */
    @Override
    public Object getObjectInstance(Object obj, Name name, Context nameCtx, Hashtable environment) throws Exception {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(this, tc, "getObjectInstance", new Object[] { obj, name, nameCtx, environment });
        Object ret = null;

        if (obj instanceof Reference) {
            Reference ref = (Reference) obj;
            String clsName = ref.getClassName();

            if (clsName.equals(ConnectionFactory.class.getName())) {
                ret = this.createConnectionFactory();
                ((JmsConnectionFactory) ret).setReference(ref);
            }

            else if (clsName.equals(QueueConnectionFactory.class.getName())) {
                ret = this.createQueueConnectionFactory();
                ((JmsQueueConnectionFactory) ret).setReference(ref);
            }

            else if (clsName.equals(TopicConnectionFactory.class.getName())) {
                ret = this.createTopicConnectionFactory();
                ((JmsTopicConnectionFactory) ret).setReference(ref);
            }

            else if (clsName.equals(JmsQueueImpl.class.getName())) {
                ret = new JmsQueueImpl(ref);
            }

            else if (clsName.equals(JmsTopicImpl.class.getName())) {
                ret = new JmsTopicImpl(ref);
            }

            else {
                if (TraceComponent.isAnyTracingEnabled() && tc.isDebugEnabled())
                    SibTr.debug(tc, "class name is " + clsName);
            }
        }

        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(this, tc, "getObjectInstance", ret);
        return ret;
    }

    /**
     * getMetaData()
     */
    @Override
    public ConnectionMetaData getMetaData() throws JMSException {
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.entry(this, tc, "getMetaData");
        ConnectionMetaData connectionMetaData = new JmsMetaDataImpl();
        if (TraceComponent.isAnyTracingEnabled() && tc.isEntryEnabled())
            SibTr.exit(this, tc, "getMetaData", connectionMetaData);
        return connectionMetaData;
    }

    /**
     * Method suppose to used internally by javax.jms.JmsConnectionImpl class only.
     * 
     * @return Client ID map.
     */
    protected static ConcurrentHashMap<String, Integer> getClientIdTable() {
        return clientIdTable;
    }
}
