/*******************************************************************************
 * Copyright (c) 2017,2021 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package web;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotSame;
import static org.junit.Assert.assertTrue;

import java.lang.management.ManagementFactory;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import javax.ejb.EJBContext;
import javax.management.MBeanServer;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.resource.spi.BootstrapContext;
import javax.resource.spi.XATerminator;
import javax.resource.spi.work.ExecutionContext;
import javax.resource.spi.work.TransactionContext;
import javax.resource.spi.work.WorkManager;
import javax.servlet.ServletException;
import javax.sql.DataSource;
import javax.transaction.UserTransaction;
import javax.transaction.xa.XAResource;
import javax.transaction.xa.Xid;

import componenttest.app.FATServlet;

public class DerbyRAServlet extends FATServlet {
    private static final long serialVersionUID = 7709282314904580334L;

    public static MBeanServer mbeanServer = ManagementFactory.getPlatformMBeanServer();
    private static final AtomicReference<Connection> nonDissociatableSharableHandleRef = new AtomicReference<Connection>();

    /**
     * Maximum number of nanoseconds a test should wait for something to happen
     */
    private static final long TIMEOUT_NS = TimeUnit.MINUTES.toNanos(2);

    public void initDatabaseTables() throws ServletException {
        try {
            DataSource ds = (DataSource) InitialContext.doLookup("java:module/env/eis/ds5ref-unshareable");
            try (Connection con = ds.getConnection()) {
                con.createStatement().execute("CREATE TABLE TESTTBL(NAME VARCHAR(80) NOT NULL PRIMARY KEY, VAL INT NOT NULL)");
            }
        } catch (NamingException | SQLException x) {
            throw new ServletException(x);
        }
    }

    /**
     * Verify that an admin object can be looked up directly as java.util.Map.
     */
    public void testAdminObjectDirectLookup() throws Throwable {
        @SuppressWarnings("unchecked")
        Map<String, String> map1 = (Map<String, String>) new InitialContext().lookup("eis/map1");
        try {
            map1.put("test", "testAdminObjectDirectLookup");
            String value = map1.get("test");
            if (!"testAdminObjectDirectLookup".equals(value))
                throw new Exception("Unexpected value: " + value);
        } finally {
            map1.clear();
        }
    }

    /**
     * Verify that an admin object, injected into DerbyRRAnnoServlet under java:app, can be looked up as java.util.Map.
     */
    public void testAdminObjectLookUpJavaApp() throws Throwable {
        @SuppressWarnings("unchecked")
        Map<String, String> map1 = (Map<String, String>) new InitialContext().lookup("java:app/env/eis/map1ref"); // from DerbyRAAnnoServlet
        try {
            map1.put("testName", "testAdminObjectLookUpJavaApp");
            String value = map1.remove("testName");
            if (!"testAdminObjectLookUpJavaApp".equals(value))
                throw new Exception("Unexpected value: " + value);
        } finally {
            map1.clear();
        }
    }

    /**
     * Verify that an admin object can be looked up via resource-env-ref as java.util.Map.
     */
    public void testAdminObjectResourceEnvRef() throws Throwable {

        @SuppressWarnings("unchecked")
        final Map<String, String> map1 = (Map<String, String>) new InitialContext().lookup("java:comp/env/eis/map1ref");
        try {
            map1.clear();
            if (!map1.isEmpty())
                throw new Exception("Not empty after clear");

            map1.put("k1", "v1");
            String previous = map1.put("k2", "v2");
            if (previous != null)
                throw new Exception("Should not have previous value: " + previous);

            int size = map1.size();
            if (size != 2)
                throw new Exception("Expect 2 entries in map, not " + size);

            String value = map1.get("k1");
            if (!"v1".equals(value))
                throw new Exception("Got wrong value: " + value);

            previous = map1.put("k1", "value1");
            if (!"v1".equals(previous))
                throw new Exception("Wrong previous value: " + previous);

            if (!map1.containsKey("k2"))
                throw new Exception("Doesn't contain key k2");

            if (map1.containsKey("k3"))
                throw new Exception("Erroneously says it contains key k3");

            if (map1.containsValue("v1"))
                throw new Exception("Erroneously says it contains value v1");

            if (!map1.containsValue("value1"))
                throw new Exception("Doesn't contain value value1");

            Set<Entry<String, String>> entries = map1.entrySet();
            if (entries.size() != 2)
                throw new Exception("wrong number of entries " + entries);

            String entriesToString = entries.toString();
            if (!entriesToString.contains("k1")
                || !entriesToString.contains("k2")
                || !entriesToString.contains("value1")
                || !entriesToString.contains("v2"))
                throw new Exception("Wrong entries: " + entries);

            value = map1.remove("k1");
            if (!"value1".equals(value))
                throw new Exception("Remove saw wrong value: " + value);

            Set<String> keys = map1.keySet();
            if (keys.size() != 1 || !keys.contains("k2"))
                throw new Exception("Wrong set of keys: " + keys);

            // Run without thread context to test that the resource adapter can lookup a java:global name
            // via a contextual proxy that it captured during start.
            ExecutorService unmanagedExecutor = Executors.newSingleThreadExecutor();
            Collection<String> values;
            try {
                values = unmanagedExecutor.submit(new Callable<Collection<String>>() {
                    @Override
                    public Collection<String> call() throws Exception {
                        return map1.values();
                    }
                }).get(TIMEOUT_NS, TimeUnit.NANOSECONDS);
            } finally {
                unmanagedExecutor.shutdown();
            }
            if (values.size() != 1 || !values.contains("v2"))
                throw new Exception("Wrong set of values: " + values);
        } finally {
            map1.clear();
        }
    }

    /**
     * Test for execution context where we roll back two units of work that share the same ExecutionContext (and thus the same xid).
     */
    public void testExecutionContext() throws Throwable {
        InitialContext initialContext = new InitialContext();
        BootstrapContext bootstrapContext = (BootstrapContext) initialContext.lookup("java:global/env/eis/bootstrapContext");
        DataSource ds1 = (DataSource) initialContext.lookup("java:comp/env/eis/ds1ref");
        Connection con = ds1.getConnection();
        try {
            // create the table
            Statement stmt = con.createStatement();
            try {
                stmt.executeUpdate("create table TestExecutionContextTBL (col1 int not null primary key, col2 varchar(50))");
                stmt.executeUpdate("insert into TestExecutionContextTBL values (1, 'one')");
                stmt.executeUpdate("insert into TestExecutionContextTBL values (2, 'two')");
            } finally {
                stmt.close();
            }

            // roll back inflown transaction
            Xid xid = new FATXID();
            ExecutionContext executionContext = new ExecutionContext();
            executionContext.setXid(xid);
            FATWork work = new FATWork("java:comp/env/eis/ds1ref", "update TestExecutionContextTBL set col2='uno' where col1=1");
            bootstrapContext.getWorkManager().doWork(work, WorkManager.INDEFINITE, executionContext, null);
            try {
                // reuse the execution context
                work = new FATWork("java:comp/env/eis/ds1ref", "update TestExecutionContextTBL set col2='dos' where col1=2");
                bootstrapContext.getWorkManager().doWork(work, WorkManager.INDEFINITE, executionContext, null);
            } finally {
                XATerminator xaTerminator = bootstrapContext.getXATerminator();
                int vote = xaTerminator.prepare(xid);
                if (vote == XAResource.XA_OK)
                    xaTerminator.rollback(xid);
                else
                    throw new Exception("Unexpected vote: " + vote);
            }

            // validate results
            stmt = con.createStatement();
            try {
                ResultSet results = stmt.executeQuery("select col1, col2 from TestExecutionContextTBL");
                Map<Integer, String> resultMap = new TreeMap<Integer, String>();
                while (results.next())
                    resultMap.put(results.getInt(1), results.getString(2));
                results.close();

                if (resultMap.size() != 2
                    || !"one".equals(resultMap.get(1))
                    || !"two".equals(resultMap.get(2)))
                    throw new Exception((new StringBuilder()).append("Unexpected results: ").append(resultMap).toString());
            } finally {
                stmt.close();
            }
        } finally {
            con.close();
        }
    }

    /**
     * Verify that a JCA data source can be looked up directly.
     */
    public void testJCADataSourceDirectLookup() throws Throwable {

        DataSource ds1 = (DataSource) new InitialContext().lookup("eis/ds1");
        int loginTimeout = ds1.getLoginTimeout();
        if (loginTimeout != 120)
            throw new Exception("Override of default loginTimeout in wlp-ra.xml not honored. Instead: " + loginTimeout);

        Connection con = ds1.getConnection("dbuser1", "dbpwd1");
        try {
            String userName = con.getMetaData().getUserName();
            if (!"dbuser1".equals(userName))
                throw new Exception("User name doesn't match. Instead: " + userName);

            Statement stmt = con.createStatement();
            try {
                ResultSet result = con.createStatement().executeQuery("values length('abcdefghijklmnopqrstuvwxyz')");

                if (!result.next())
                    throw new Exception("Missing result of query");

                int value = result.getInt(1);
                if (value != 26)
                    throw new Exception("Unexpected value: " + value);
            } finally {
                stmt.close();
            }
        } finally {
            con.close();
        }
    }

    /**
     * Verify that a JCA data source, injected into DerbyRRAnnoServlet under java:module, can be looked up as java.util.Map.
     */
    public void testJCADataSourceLookUpJavaModule() throws Throwable {

        DataSource ds1 = (DataSource) new InitialContext().lookup("java:module/env/eis/ds1ref");
        Connection con = ds1.getConnection();
        try {
            String userName = con.getMetaData().getUserName();
            if (!"DS1USER".equals(userName))
                throw new Exception("User name doesn't match configured value on containerAuthData. Instead: " + userName);

            Statement stmt = con.createStatement();
            try {
                ResultSet result = con.createStatement().executeQuery("values mod(87, 16)");

                if (!result.next())
                    throw new Exception("Missing result of query");

                int value = result.getInt(1);
                if (value != 7)
                    throw new Exception("Unexpected value: " + value);
            } finally {
                stmt.close();
            }
        } finally {
            con.close();
        }
    }

    /**
     * Verify that a JCA data source can be looked up via resource ref.
     */
    public void testJCADataSourceResourceRef() throws Throwable {

        DataSource ds1 = (DataSource) new InitialContext().lookup("java:comp/env/eis/ds1ref");
        Connection con = ds1.getConnection();
        try {
            String userName = con.getMetaData().getUserName();
            if (!"DS1USER".equals(userName))
                throw new Exception("User name doesn't match configured value on containerAuthData. Instead: " + userName);

            Statement stmt = con.createStatement();
            try {
                ResultSet result = con.createStatement().executeQuery("values absval(-40)");

                if (!result.next())
                    throw new Exception("Missing result of query");

                int value = result.getInt(1);
                if (value != 40)
                    throw new Exception("Unexpected value: " + value);
            } finally {
                stmt.close();
            }
        } finally {
            con.close();
        }
    }

    /**
     * Verify that a principal is added to the RA subject.
     */
    public void testJCADataSourceResourceRefSecurity() throws Throwable {

        DataSource ds1 = (DataSource) new InitialContext().lookup("java:comp/env/eis/ds1ref");
        Connection con = ds1.getConnection();

        try {
            String userName = con.getMetaData().getUserName();
            if (!"DS1USER".equals(userName))
                throw new Exception("User name doesn't match configured value on containerAuthData. Instead: " + userName);

            if (con.toString().contains("WSPrincipal:") == false) {
                throw new Exception("The subject does not contain a principal. " + con.toString());
            }
        } finally {
            con.close();
        }
    }

    /**
     * When HandleList is enabled, it automatically closes shareable parked connection handles that are
     * leaked across transaction scopes.
     */
    public void testNonDissociatableHandlesCannotBeParkedAcrossTransactionScopes() throws Exception {
        DerbyRABean bean = InitialContext.doLookup("java:global/derbyRAApp/fvtweb/DerbyRABean!web.DerbyRABean");

        Connection con = bean.runInNewGlobalTran(() -> {
            DataSource ds = (DataSource) InitialContext.doLookup("eis/ds5"); // shareable
            Connection c = ds.getConnection();
            Statement st = c.createStatement();
            st.executeUpdate("INSERT INTO TESTTBL VALUES('park-handle-across-transaction', 3000)");
            st.close();
            return c;
        });

        assertTrue(con.isClosed());
    }

    /**
     * When HandleList is enabled, shareable connection handles are parked across EJB methods within a transaction
     * if the resource adapter does not support DissociatableManagedConnection. The connection handle continues to be
     * usable in subsequent EJB methods within the transaction, remaining open until the EJB is destroyed, at which
     * point the HandleList closes connection handles that remain open and would otherwise be leaked.
     */
    public void testNonDissociatableHandlesParkedAcrossEJBMethods() throws Exception {
        DerbyConnectionCachingBean bean = InitialContext.doLookup("java:global/derbyRAApp/fvtweb/DerbyConnectionCachingBean!web.DerbyConnectionCachingBean");
        UserTransaction tx = InitialContext.doLookup("java:comp/UserTransaction");
        tx.begin();
        try {
            bean.connect();

            bean.insert("park-handle-across-EJB-methods-1", 1000);
            bean.insert("park-handle-across-EJB-methods-2", 2000);
            assertEquals(Integer.valueOf(1000), bean.find("park-handle-across-EJB-methods-1"));
            assertEquals(Integer.valueOf(2000), bean.find("park-handle-across-EJB-methods-2"));

            Connection cachedConnection = bean.getCachedConnection();
            assertFalse(cachedConnection.isClosed());

            bean.removeEJB();
            assertTrue(cachedConnection.isClosed());
        } finally {
            tx.commit();
        }

        // access the same data from another transaction:

        bean = InitialContext.doLookup("java:global/derbyRAApp/fvtweb/DerbyConnectionCachingBean!web.DerbyConnectionCachingBean");
        tx.begin();
        try {
            bean.connect();
            assertEquals(Integer.valueOf(1000), bean.find("park-handle-across-EJB-methods-1"));
            assertEquals(Integer.valueOf(2000), bean.find("park-handle-across-EJB-methods-2"));

            Connection cachedConnection = bean.getCachedConnection();
            assertFalse(cachedConnection.isClosed());

            bean.removeEJB();
            assertTrue(cachedConnection.isClosed());
        } finally {
            tx.commit();
        }
    }

    /**
     * Verifies that a non-dissociatable sharable connection handle that was left open by another servlet request
     * is closed now.
     *
     * Prerequisite - must run testNonDissociatableSharableHandleLeftOpenAfterServletMethod prior to this test
     */
    public void testNonDissociatableSharableHandleIsClosed() throws Exception {
        Connection con = nonDissociatableSharableHandleRef.getAndSet(null);
        assertTrue(con.isClosed());
    }

    /**
     * Intentionally leave a non-dissociatable, sharable handle open across the end of a servlet method.
     * The invoker must run testNonDissociatableSharableHandleIsClosed after this to complete the test.
     */
    public void testNonDissociatableSharableHandleLeftOpenAfterServletMethod() throws Exception {
        DataSource ds = (DataSource) InitialContext.doLookup("eis/ds5");
        Connection con = ds.getConnection();
        assertFalse(con.isClosed());
        assertTrue(nonDissociatableSharableHandleRef.compareAndSet(null, con));
        // intentionally leave the connection open across the end of the servlet request
    }

    /**
     * Rely on parking a sharable connection in the absence of support for DissociatableManagedConnection
     * to avoid exhausting the connection pool.
     */
    public void testParkNonDissociatableSharableHandle() throws Exception {
        DataSource ds = (DataSource) InitialContext.doLookup("eis/ds5"); // creates sharable connections

        // On another thread, use a sharable connection in a transaction, commit the transaction and wait
        CountDownLatch transactionCommitted = new CountDownLatch(1);
        CountDownLatch servletThreadDoneWithConnection = new CountDownLatch(1);
        ExecutorService executor = InitialContext.doLookup("java:comp/DefaultManagedExecutorService");
        Future<?> future = executor.submit(() -> {
            UserTransaction tx = InitialContext.doLookup("java:comp/UserTransaction");
            tx.begin();
            Connection con1 = ds.getConnection();
            con1.createStatement().executeQuery("VALUES (51)").getStatement().close();
            tx.commit();
            transactionCommitted.countDown();

            // Connection handle remains open, but should now be associated to the parking ManagedConnection.
            // Remain in this state until allowed to continue.

            assertTrue(servletThreadDoneWithConnection.await(TIMEOUT_NS, TimeUnit.NANOSECONDS));

            // It would be possible to use the connection handle again here, if it weren't closed during the cleanup.

            return null;
        });

        assertTrue(transactionCommitted.await(TIMEOUT_NS, TimeUnit.NANOSECONDS));

        // At this point, the pool's single connection has been used on another thread, and hopefully freed up by
        // parking the handle to the parking ManagedConnection. During this time, it should be possible to obtain
        // the connection for use by the current thread:
        UserTransaction tx = InitialContext.doLookup("java:comp/UserTransaction");
        tx.begin();
        try {
            Connection con2 = ds.getConnection();
            con2.createStatement().executeQuery("VALUES (52)").getStatement().close();
            con2.close();
        } finally {
            tx.commit();
        }

        servletThreadDoneWithConnection.countDown();

        // Surface any errors that occurred on the other thread
        future.get(TIMEOUT_NS, TimeUnit.NANOSECONDS);
    }

    /**
     * Test for transaction context where we do two-phase commit of two resources with the same xid but different timeout.
     */
    public void testTransactionContext() throws Throwable {
        InitialContext initialContext = new InitialContext();
        BootstrapContext bootstrapContext = (BootstrapContext) initialContext.lookup("java:global/env/eis/bootstrapContext");
        DataSource ds1 = (DataSource) initialContext.lookup("java:comp/env/eis/ds1ref");
        Connection con = ds1.getConnection();
        try {
            // create the table
            Statement stmt = con.createStatement();
            try {
                stmt.executeUpdate("create table TestTransactionContextTBL (col1 int not null primary key, col2 varchar(50))");
                stmt.executeUpdate("insert into TestTransactionContextTBL values (3, 'three')");
                stmt.executeUpdate("insert into TestTransactionContextTBL values (4, 'four')");
            } finally {
                stmt.close();
            }

            // commit inflown transaction
            Xid xid = new FATXID();
            TransactionContext transactionContext = new TransactionContext();
            transactionContext.setXid(xid);
            FATWork work = new FATWorkAndContext("java:comp/env/eis/ds1ref", "update TestTransactionContextTBL set col2='III' where col1=3", transactionContext);
            bootstrapContext.getWorkManager().doWork(work, WorkManager.INDEFINITE, null, null);
            try {
                // create new transaction context for same xid, but different transaction timeout
                transactionContext = new TransactionContext();
                transactionContext.setXid(xid);
                transactionContext.setTransactionTimeout(5000);
                work = new FATWorkAndContext("java:comp/env/eis/ds1ref", "update TestTransactionContextTBL set col2='IV' where col1=4", transactionContext);
                bootstrapContext.getWorkManager().doWork(work, WorkManager.INDEFINITE, null, null);
            } finally {
                XATerminator xaTerminator = bootstrapContext.getXATerminator();
                int vote = xaTerminator.prepare(xid);
                if (vote == XAResource.XA_OK)
                    xaTerminator.commit(xid, false);
                else
                    throw new Exception("Unexpected vote: " + vote);
            }

            // validate results
            stmt = con.createStatement();
            try {
                ResultSet results = stmt.executeQuery("select col1, col2 from TestTransactionContextTBL");
                Map<Integer, String> resultMap = new TreeMap<Integer, String>();
                while (results.next())
                    resultMap.put(results.getInt(1), results.getString(2));
                results.close();

                if (resultMap.size() != 2
                    || !"III".equals(resultMap.get(3))
                    || !"IV".equals(resultMap.get(4)))
                    throw new Exception((new StringBuilder()).append("Unexpected results: ").append(resultMap).toString());
            } finally {
                stmt.close();
            }
        } finally {
            con.close();
        }
    }

    /**
     * Use an unshared connection within two EJB methods that run under different transactions, one of which rolls back
     * and the other of which commits.
     */
    public void testUnsharableConnectionAcrossEJBGlobalTran() throws Exception {
        DerbyRABean bean = InitialContext.doLookup("java:global/derbyRAApp/fvtweb/DerbyRABean!web.DerbyRABean");
        final DataSource ds = (DataSource) InitialContext.doLookup("java:module/env/eis/ds5ref-unshareable");
        final Connection[] c = new Connection[1];
        c[0] = ds.getConnection();
        try {
            c[0].createStatement().execute("create table testUnsharableConEJBTable (name varchar(40) not null primary key, val int not null)");
            c[0].close();

            bean.runInNewGlobalTran(new Callable<Void>() {
                @Override
                public Void call() throws Exception {
                    c[0] = ds.getConnection();
                    c[0].createStatement().executeUpdate("insert into testUnsharableConEJBTable values ('first', 1)");

                    DerbyRABean bean = InitialContext.doLookup("java:global/derbyRAApp/fvtweb/DerbyRABean!web.DerbyRABean");
                    bean.runInNewGlobalTran(new Callable<Integer>() {
                        @Override
                        public Integer call() throws Exception {
                            return c[0].createStatement().executeUpdate("insert into testUnsharableConEJBTable values ('second', 2)");
                        }
                    });

                    EJBContext ejbContext = InitialContext.doLookup("java:comp/EJBContext");
                    ejbContext.setRollbackOnly();
                    return null;
                }
            });

            int updateCount;
            updateCount = c[0].createStatement().executeUpdate("delete from testUnsharableConEJBTable where name='second'");
            // TODO connection is not enlisting in the transaction of the second EJB method.
            //if (updateCount != 1)
            //    throw new Exception("Did not find the entry that should have been committed under the EJB transaction.");

            updateCount = c[0].createStatement().executeUpdate("delete from testUnsharableConEJBTable where name='first'");
            if (updateCount != 0)
                throw new Exception("Found an entry that should have been rolled back under the Servlet's global transaction.");

        } finally {
            c[0].close();
        }
    }

    public void testConnPoolStatsExceptionInDestroy() throws Exception {
        UserTransaction tran = (UserTransaction) new InitialContext().lookup("java:comp/UserTransaction");
        tran.begin();
        DataSource ds2 = (DataSource) new InitialContext().lookup("eis/ds2");
        Connection conn = null;
        try {
            conn = ds2.getConnection();
            ObjectName objName = getConnPoolStatsMBeanObjName("ds2");
            int initialDestroyCount = getMonitorData(objName, "DestroyCount");
            mbeanServer.invoke(getMBeanObjectInstance("eis/ds2").getObjectName(), "purgePoolContents", new Object[] { "Normal" }, null);
            tran.commit();

            int finalDestroyCount = getMonitorData(objName, "DestroyCount");
            int numManagedConns = getMonitorData(objName, "ManagedConnectionCount");
            int numFreeConns = getMonitorData(objName, "FreeConnectionCount");
            if (finalDestroyCount != initialDestroyCount + 1) { //Should increase by 1 even though there is an exception destroying one connection
                throw new Exception("Expected destroyCount to increase by one. Initial: " + initialDestroyCount + " Final: " + finalDestroyCount);
            }
            if (numManagedConns != 0) { //Should be 0
                throw new Exception("Expected 0 managed connections, but found: " + numManagedConns);
            }
            if (numFreeConns != 0) { //Should be 0
                throw new Exception("Expected 0 free connections, but found: " + numFreeConns);
            }

        } finally {
            if (conn != null)
                conn.close();
        }
    }

    /**
     * Make sure that when a connection error event notification is sent to the
     * connection event listener and the connection is "STATE_ACTIVE_FREE"
     * That the only that connection is marked as unusable, and nothing is purged.
     */
    public void testErrorInFreeConn() throws Exception {
        DataSource ds = (DataSource) new InitialContext().lookup("eis/ds4");
        SQLException sqe = new SQLException("APP_SPECIFIED_CONN_ERROR");

        //Get and close connection
        Object managedConn1 = null;
        Connection con1 = null;
        Class<?> derbyConnClass1 = null;
        try {
            con1 = ds.getConnection();
            derbyConnClass1 = con1.getClass();
            Field f = derbyConnClass1.getDeclaredField("mc");
            managedConn1 = f.get(con1);
            Statement stmt = con1.createStatement();
            stmt.close();
        } finally {
            con1.close();
        }

        //Connection in free pool: Report an error
        Class<?> c = managedConn1.getClass();
        Method m = c.getMethod("notify", int.class, derbyConnClass1, Exception.class);
        m.invoke(managedConn1, 5, con1, sqe); //5 indicates connection error

        //Get pool size
        long poolSizeAfterError = (long) mbeanServer.invoke(getMBeanObjectInstance("eis/ds4").getObjectName(), "getSize", null, null);

        //After the error, there should be 1 connections in the pool.  Delaying the remove of a free managed connection.
        assertEquals("Unexpected number of connections found, connection should not have been purged.", 1, poolSizeAfterError);

        //Report a duplicate error
        m.invoke(managedConn1, 5, con1, sqe); //5 indicates connection error
        poolSizeAfterError = (long) mbeanServer.invoke(getMBeanObjectInstance("eis/ds4").getObjectName(), "getSize", null, null);

        //After the error, there should be 1 connections in the pool.  Its not safe to remove the managed connection if its free.
        //The free connection is only marked to not be reused.  Next use will remove this managed connection.
        assertEquals("Unexpected number of connections found, connection should not have been purged.", 1, poolSizeAfterError);

        //Get and close another connection
        Object managedConn2 = null;
        Connection con2 = null;
        Class<?> derbyConnClass2 = null;
        try {
            con2 = ds.getConnection();
            derbyConnClass2 = con2.getClass();
            Field f = derbyConnClass2.getDeclaredField("mc");
            managedConn2 = f.get(con2);
            Statement stmt = con2.createStatement();
            stmt.close();
        } finally {
            con2.close();
        }

        //Ensure these two connections are not the same connection
        assertNotSame("We must have a new managed connection, review trace log", managedConn1, managedConn2);

        //After the failing connection is removed and a new one is created, there should be 1 connections in the pool.
        poolSizeAfterError = (long) mbeanServer.invoke(getMBeanObjectInstance("eis/ds4").getObjectName(), "getSize", null, null);
        assertEquals("Unexpected number of connections found, failing connection should have been replaced.", 1, poolSizeAfterError);
    }

    /**
     * Make sure that when a connection error event notification is sent to the
     * connection event listener and the connection is "STATE_ACTIVE_INUSE"
     * That the only that connection is marked as unusable, and nothing is purged.
     */
    public void testErrorInUsedConn() throws Exception {
        DataSource ds = (DataSource) new InitialContext().lookup("eis/ds4");
        SQLException sqe = new SQLException("APP_SPECIFIED_CONN_ERROR");

        //Get first connection
        Object managedConn1 = null;
        Class<?> derbyConnClass1 = null;
        try (Connection con1 = ds.getConnection(); Statement stmt1 = con1.createStatement();) {
            derbyConnClass1 = con1.getClass();
            Field f1 = derbyConnClass1.getDeclaredField("mc");
            managedConn1 = f1.get(con1);

            //get second connection
            Object managedConn2 = null;
            Class<?> derbyConnClass2 = null;
            try (Connection con2 = ds.getConnection(); Statement stmt2 = con2.createStatement();) {
                derbyConnClass2 = con2.getClass();
                Field f2 = derbyConnClass2.getDeclaredField("mc");
                managedConn2 = f2.get(con2);

                //Assert that connections are not the same.
                assertNotSame("Connections should not be the same", managedConn1, managedConn2);
                long poolSizeBeforeError = (long) mbeanServer.invoke(getMBeanObjectInstance("eis/ds4").getObjectName(), "getSize", null, null);

                //Close first connection so that it gets returned to free pool
                stmt1.close();
                con1.close();

                //Now cause an error on the in-use connection
                Class<?> c = managedConn2.getClass();
                Method m = c.getMethod("notify", int.class, derbyConnClass2, Exception.class);
                m.invoke(managedConn2, 5, con2, sqe); //5 indicates connection error
                long poolSizeAfterError = (long) mbeanServer.invoke(getMBeanObjectInstance("eis/ds4").getObjectName(), "getSize", null, null);

                //Assert pool sizes before and after error
                assertEquals("Incorrect pool size before error.", 2, poolSizeBeforeError);
                assertEquals("Incorrect pool size after error. All connections should have been purged.", 0, poolSizeAfterError);
            }
        }

    }

    private int getMonitorData(ObjectName name, String attribute) throws Exception {
        return Integer.parseInt((mbeanServer.getAttribute(name, attribute)).toString());
    }

    private ObjectName getConnPoolStatsMBeanObjName(String dsName) throws Exception {
        Set<ObjectInstance> mxBeanSet;
        ObjectInstance oi;
        mxBeanSet = mbeanServer.queryMBeans(new ObjectName("WebSphere:type=ConnectionPoolStats,name=*" + dsName + "*"), null);
        if (mxBeanSet != null && mxBeanSet.size() > 0) {
            Iterator<ObjectInstance> it = mxBeanSet.iterator();
            oi = it.next();
            return oi.getObjectName();
        } else
            throw new Exception("ConnectionPoolStatsMBean:NotFound");
    }

    private ObjectInstance getMBeanObjectInstance(String jndiName) throws Exception {
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        ObjectName obn = new ObjectName("WebSphere:type=com.ibm.ws.jca.cm.mbean.ConnectionManagerMBean,jndiName=" + jndiName + ",*");
        Set<ObjectInstance> s = mbs.queryMBeans(obn, null);
        if (s.size() != 1) {
            System.out.println("ERROR: Found incorrect number of MBeans (" + s.size() + ")");
            for (ObjectInstance i : s)
                System.out.println("  Found MBean: " + i.getObjectName());
            throw new Exception("Expected to find exactly 1 MBean, instead found " + s.size());
        }
        return s.iterator().next();
    }

}
