/*******************************************************************************
 * Copyright (c) 2015 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.clientcontainer.HelloAppClient.test;

import java.io.IOException;

import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.UnsupportedCallbackException;

/**
 *
 */
public class MyWSCallbackHandlerImpl implements CallbackHandler {

    private String userName;
    private String realm;
    private String password;
    private byte[] credToken;
    private java.util.Map appContext;

    public MyWSCallbackHandlerImpl() {}

    /**
     * <p>
     * Push the username and password to login module.
     * </p>
     * 
     * @param userName The user name of the principal.
     * @param password The password in clear text.
     */
    public MyWSCallbackHandlerImpl(String userName, String password) {
        this.userName = userName;
        this.password = password;
    }

    /**
     * <p>
     * Push the username and password to login module. The realmName is only
     * need if it's different from the current realm. If you intend to login
     * to the current realm, you do not need to specify a realm or you can
     * specify the current realm.
     * </p>
     * 
     * @param userName The user name of the principal.
     * @param realmName The realm name, if different from the current realm
     * @param password The password in clear text.
     */
    public MyWSCallbackHandlerImpl(String userName, String realmName, String password) {
        this.userName = "MYUSER";
        this.password = "MYPWD";
        this.realm = "MYREALM";
    }

    /**
     * <p>
     * Push the username and password to login module. The realmName is only
     * need if it's different from the current realm. If you intend to login
     * to the current realm, you do not need to specify a realm or you can
     * specify the current realm.
     * </p>
     * 
     * <p>
     * If the realm you specify is different from the current realm and you
     * want to validate the userid/password at the new target realm during
     * this login, then you must specify a java.util.Map which contains the
     * following properites:
     * javax.naming.Context.PROVIDER_URL
     * Example: "corbaloc:iiop:myhost.mycompany.com:2809"
     * javax.naming.Context.INITIAL_CONTEXT_FACTORY
     * Example: "com.ibm.websphere.naming.WsnInitialContextFactory"
     * Note: If the target server is not a WebSphere server, then do not
     * specify this information since it will not validate successfully. Also,
     * the target server must support CSIv2 in order for the userid/password
     * to be successfully sent.
     * </p>
     * 
     * @param userName The user name of the principal.
     * @param realmName The realm name, if different from the current realm
     * @param password The password in clear text.
     * @param appContext A java.util.Map containing naming properties to validate userid/password.
     */
    public MyWSCallbackHandlerImpl(String userName, String realmName, String password, java.util.Map appContext) {
        this.userName = userName;
        this.password = password;
        this.realm = realmName;
        this.appContext = appContext;
    }

    /**
     * <p>
     * Push Credential Token to login module. The Credential Token should be treated as
     * an opaque object. The credential token must be in the format recognized by
     * WebSphere Secure Association Service.
     * </p>
     * 
     * @param credToken The credential token.
     */
    public MyWSCallbackHandlerImpl(byte[] credToken) {
        this.credToken = copyCredToken(credToken);
    }

    /**
     * <p>
     * This implementation of <code>CallbackHandler</code> pushes the data specified in the
     * constructor to the login module.
     * </p>
     * 
     * @param callbacks An array of <code>Callback</code> objects provided by the underlying
     *            security service which contains the information
     *            requested to be retrieved or displayed.
     * @exception IOException
     *                If an input or output error occurs.
     * @exception UnsupportedCallbackException
     *                If the implementation of this method does not support one or more of the
     *                <code>Callback</code>s specified in the callbacks parameter.
     */
    @Override
    public void handle(Callback[] callbacks) throws IOException, UnsupportedCallbackException {
        int len = 0;

        if ((callbacks == null) || ((len = callbacks.length) == 0)) {
            return;
        }

        StringBuffer sb = new StringBuffer();
        sb.append("{ ");
        for (int i = 0; i < len; i++) {
            sb.append(callbacks[i].getClass().getName());
            if (i < (len - 1)) {
                sb.append(", ");
            }
        }
        sb.append(" }");
        System.out.println("handle(callbacks = \"" + sb.toString() + "\")");

        for (int i = 0; i < len; i++) {
            Callback c = callbacks[i];

            if (c instanceof javax.security.auth.callback.NameCallback) {
                ((javax.security.auth.callback.NameCallback) c).setName(userName);
            } else if (c instanceof javax.security.auth.callback.PasswordCallback) {
                ((javax.security.auth.callback.PasswordCallback) c).setPassword((password == null) ? null : password.toCharArray());
//            } else if (c instanceof WSCredTokenCallbackImpl) {
//                ((WSCredTokenCallbackImpl) c).setCredToken(credToken);
//            } else if (c instanceof WSRealmNameCallbackImpl) {
//                ((WSRealmNameCallbackImpl) c).setRealmName(realm);
//            } else if (c instanceof WSAppContextCallback) {
//                ((WSAppContextCallback) c).setContext(appContext);
            } else {
                System.out.println("Un-use handle(callbacks = \"" + callbacks[i].getClass().getName() + "\")");
            }
        }
    }

    /**
     * Create a copy of the specified byte array.
     * 
     * @param credToken
     * @return A copy of the specified byte array, or null if the input was null.
     */
    public static byte[] copyCredToken(byte[] credToken) {

        if (credToken == null) {
            return null;
        }

        final int LEN = credToken.length;
        if (LEN == 0) {
            return new byte[LEN];
        }

        byte[] newCredToken = new byte[LEN];
        System.arraycopy(credToken, 0, newCredToken, 0, LEN);

        return newCredToken;
    }
    
    public void postConstruct() {
    	System.out.println("I have been in postConstruct of the callback handler.");
    }
    
    public void preDestroy() {
    	System.out.println("I have been in preDestroy of the callback handler.");
    }
    
}
