/*******************************************************************************
 * Copyright (c) 2020 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package com.ibm.ws.query.testlogic;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Arrays;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;

import javax.persistence.EntityManager;
import javax.persistence.Query;

import org.junit.Assert;

import com.ibm.ws.query.entities.xml.XMLAddressBean;
import com.ibm.ws.query.entities.xml.XMLAddressPK;
import com.ibm.ws.query.entities.xml.XMLDeptBean;
import com.ibm.ws.query.entities.xml.XMLEmbAddress;
import com.ibm.ws.query.entities.xml.XMLEmpBean;
import com.ibm.ws.query.entities.xml.XMLProjectBean;
import com.ibm.ws.query.entities.xml.XMLTaskBean;
import com.ibm.ws.query.utils.DeptEmpListView;
import com.ibm.ws.query.utils.DeptEmpView;
import com.ibm.ws.query.utils.SimpleDeptEmpView;
import com.ibm.ws.testtooling.database.DatabaseVendor;
import com.ibm.ws.testtooling.jpaprovider.JPAPersistenceProvider;
import com.ibm.ws.testtooling.testinfo.TestExecutionContext;
import com.ibm.ws.testtooling.testlogic.AbstractTestLogic;
import com.ibm.ws.testtooling.tranjacket.TransactionJacket;
import com.ibm.ws.testtooling.vehicle.resources.JPAResource;
import com.ibm.ws.testtooling.vehicle.resources.TestExecutionResources;

public class JULoopQueryXMLTest extends AbstractTestLogic {
    final EntityValue dep100 = new EntityValue(XMLDeptBean.class, "no", 100);
    final EntityValue dep200 = new EntityValue(XMLDeptBean.class, "no", 200);
    final EntityValue dep210 = new EntityValue(XMLDeptBean.class, "no", 210);
    final EntityValue dep220 = new EntityValue(XMLDeptBean.class, "no", 220);
    final EntityValue dep300 = new EntityValue(XMLDeptBean.class, "no", 300);

    final EntityValue emp1 = new EntityValue(XMLEmpBean.class, "empid", 1);
    final EntityValue emp2 = new EntityValue(XMLEmpBean.class, "empid", 2);
    final EntityValue emp3 = new EntityValue(XMLEmpBean.class, "empid", 3);
    final EntityValue emp4 = new EntityValue(XMLEmpBean.class, "empid", 4);
    final EntityValue emp5 = new EntityValue(XMLEmpBean.class, "empid", 5);
    final EntityValue emp6 = new EntityValue(XMLEmpBean.class, "empid", 6);
    final EntityValue emp7 = new EntityValue(XMLEmpBean.class, "empid", 7);
    final EntityValue emp8 = new EntityValue(XMLEmpBean.class, "empid", 8);
    final EntityValue emp9 = new EntityValue(XMLEmpBean.class, "empid", 9);
    final EntityValue emp10 = new EntityValue(XMLEmpBean.class, "empid", 10);

    final EntityValue proj1000 = new EntityValue(XMLProjectBean.class, "projid", 1000);
    final EntityValue proj2000 = new EntityValue(XMLProjectBean.class, "projid", 2000);
    final EntityValue proj3000 = new EntityValue(XMLProjectBean.class, "projid", 3000);

    final EntityValue task1010 = new EntityValue(XMLTaskBean.class, "taskid", 1010);
    final EntityValue task1020 = new EntityValue(XMLTaskBean.class, "taskid", 1020);
    final EntityValue task1030 = new EntityValue(XMLTaskBean.class, "taskid", 1030);
    final EntityValue task2010 = new EntityValue(XMLTaskBean.class, "taskid", 2010);
    final EntityValue task2020 = new EntityValue(XMLTaskBean.class, "taskid", 2020);

//    TEST1; select d.no, d.name, d.mgr.empid, d.mgr.name, d.reportsTo.no from XMLDeptBean d order by d.no
    public void testLoop001(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.no, d.name, d.mgr.empid, d.mgr.name, d.reportsTo.no from XMLDeptBean d order by d.no";
            Query q = em.createQuery(qStr);

//          TEST1; select d.no, d.name, d.mgr.empid, d.mgr.name, d.reportsTo.no from XMLDeptBean d order by d.no
//          d.no   d.name    d.mgr.empid  d.mgr.name  d.reportsTo.no
//          ~~~~ ~~~~~~~~~~~ ~~~~~~~~~~~ ~~~~~~~~~~~~ ~~~~~~~~~~~~~~
//          100      CEO         10      Catalina Wei      100
//          200     Admin         8      Tom Rayburn       100
//          210  Development      3         minmei         200
//          220    Service        4         george         200
//          300     Sales         6         ahmad          100
//           TEST1; 5 tuples

            List<Object[]> rList = q.getResultList();

            Object[] targets[] = {
                                   //             d.no   d.name    d.mgr.empid  d.mgr.name  d.reportsTo.no
                                   new Object[] { 100, "CEO", 10, "Catalina Wei", 100 },
                                   new Object[] { 200, "Admin", 8, "Tom Rayburn", 100 },
                                   new Object[] { 210, "Development", 3, "minmei", 200 },
                                   new Object[] { 220, "Service", 4, "george", 200 },
                                   new Object[] { 300, "Sales", 6, "ahmad", 100 } };

            validateQueryResult(testName, qStr, rList, targets, true);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST2; select e.empid, e.name, e.dept.no from XMLEmpBean e order by e.empid
    public void testLoop002(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        JPAPersistenceProvider provider = JPAPersistenceProvider.resolveJPAPersistenceProvider(jpaResource);

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, e.name, e.dept.no from XMLEmpBean e order by e.empid";
            Query q = em.createQuery(qStr);

//            TEST2; select e.empid, e.name, e.dept.no from XMLEmpBean e order by e.empid
//            e.empid   e.name    e.dept.no
//            ~~~~~~~ ~~~~~~~~~~~ ~~~~~~~~~
//               1       david       210
//               2      andrew       210
//               3      minmei       200
//               4      george       200
//               5      ritika       220
//               6       ahmad       100
//               7     charlene      210
//               8    Tom Rayburn    100
//               9       harry       210
//             TEST2; 9 tuples

            List<Object[]> rList = q.getResultList();

            /*
             * TODO: investigate what the correct behavior should be.
             * EclipseLink/OpenJPA join against the XMLDeptBean table, eliminating null values.
             * Hibernate only queries the XMLEmpBean table, which should be sufficient.
             *
             * https://bugs.eclipse.org/bugs/show_bug.cgi?id=431371
             */
            Object[] targets = new Object[] {
                                              //             e.empid   e.name    e.dept.no
                                              new Object[] { 1, "david", 210 },
                                              new Object[] { 2, "andrew", 210 },
                                              new Object[] { 3, "minmei", 200 },
                                              new Object[] { 4, "george", 200 },
                                              new Object[] { 5, "ritika", 220 },
                                              new Object[] { 6, "ahmad", 100 },
                                              new Object[] { 7, "charlene", 210 },
                                              new Object[] { 8, "Tom Rayburn", 100 },
                                              new Object[] { 9, "harry", 210 }
            };

            if (JPAPersistenceProvider.HIBERNATE.equals(provider)) {
                targets = new Object[] {
                                         //             e.empid   e.name    e.dept.no
                                         new Object[] { 1, "david", 210 },
                                         new Object[] { 2, "andrew", 210 },
                                         new Object[] { 3, "minmei", 200 },
                                         new Object[] { 4, "george", 200 },
                                         new Object[] { 5, "ritika", 220 },
                                         new Object[] { 6, "ahmad", 100 },
                                         new Object[] { 7, "charlene", 210 },
                                         new Object[] { 8, "Tom Rayburn", 100 },
                                         new Object[] { 9, "harry", 210 },
                                         new Object[] { 10, "Catalina Wei", null }
                };
            }

            validateQueryResult(testName, qStr, rList, targets, true);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST3; select e.empid, e.name, d. no from XMLDeptBean d left join d.emps e order by e.empid
    public void testLoop003(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        final String dbProductName = (testProps == null) ? "UNKNOWN" : ((testProps.get("dbProductName") == null) ? "UNKNOWN" : (String) testProps.get("dbProductName"));

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, e.name, d.no from XMLDeptBean d left join d.emps e order by e.empid";
            Query q = em.createQuery(qStr);

//            TEST3; select e.empid, e.name, d. no from XMLDeptBean d left join d.emps e order by e.empid
//            e.empid   e.name    d. no
//            ~~~~~~~ ~~~~~~~~~~~ ~~~~~
//               1       david     210
//               2      andrew     210
//               3      minmei     200
//               4      george     200
//               5      ritika     220
//               6       ahmad     100
//               7     charlene    210
//               8    Tom Rayburn  100
//               9       harry     210
//             null      null      300
//             TEST3; 10 tuples

            List<Object[]> rList = q.getResultList();

            // SQLServer treats null as lower value
            Object[] targets[] = (!DatabaseVendor.checkDBProductName(dbProductName, DatabaseVendor.SQLSERVER)) ? new Object[][] {
                                                                                                                                  // e.empid   e.name    d. no
                                                                                                                                  new Object[] { 1, "david", 210 },
                                                                                                                                  new Object[] { 2, "andrew", 210 },
                                                                                                                                  new Object[] { 3, "minmei", 200 },
                                                                                                                                  new Object[] { 4, "george", 200 },
                                                                                                                                  new Object[] { 5, "ritika", 220 },
                                                                                                                                  new Object[] { 6, "ahmad", 100 },
                                                                                                                                  new Object[] { 7, "charlene", 210 },
                                                                                                                                  new Object[] { 8, "Tom Rayburn", 100 },
                                                                                                                                  new Object[] { 9, "harry", 210 },
                                                                                                                                  new Object[] { null, null, 300 }
            } : new Object[][] {
                                 // e.empid   e.name    d. no
                                 new Object[] { null, null, 300 },
                                 new Object[] { 1, "david", 210 },
                                 new Object[] { 2, "andrew", 210 },
                                 new Object[] { 3, "minmei", 200 },
                                 new Object[] { 4, "george", 200 },
                                 new Object[] { 5, "ritika", 220 },
                                 new Object[] { 6, "ahmad", 100 },
                                 new Object[] { 7, "charlene", 210 },
                                 new Object[] { 8, "Tom Rayburn", 100 },
                                 new Object[] { 9, "harry", 210 }

            };

            validateQueryResult(testName, qStr, rList, targets, true);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST4; select e.empid, e.name, e.work, e.home from XMLEmpBean e order by e.empid
    public void testLoop004(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        JPAPersistenceProvider provider = JPAPersistenceProvider.resolveJPAPersistenceProvider(jpaResource);

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, e.name, e.work, e.home from XMLEmpBean e order by e.empid";
            Query q = em.createQuery(qStr);

//            TEST4; select e.empid, e.name, e.work, e.home from XMLEmpBean e order by e.empid
//            e.empid    e.name           AddressBean               AddressBean
//            ~~~~~~~ ~~~~~~~~~~~~ ~~~~~~~~~~~~~~~~~~~~~~~~~~ ~~~~~~~~~~~~~~~~~~~~~~~~
//               1       david     [555 Silicon Valley Drive]    [1780 Mercury Way]
//               2       andrew    [555 Silicon Valley Drive]    [1780 Mercury Way]
//               3       minmei    [555 Silicon Valley Drive]    [1780 Mercury Way]
//               4       george    [555 Silicon Valley Drive]    [512 Venus Drive]
//               5       ritika    [555 Silicon Valley Drive]  [12440 Vulcan Avenue]
//               6       ahmad      [4983 Plutonium Avenue]   [4983 Plutonium Avenue]
//               7      charlene   [555 Silicon Valley Drive]   [182 Martian Street]
//               8    Tom Rayburn  [555 Silicon Valley Drive]    [6200 Vegas Drive]
//               9       harry        [8900 Jupiter Park]     [150 North First Apt E1]
//              10    Catalina Wei [555 Silicon Valley Drive]           null
//             TEST4; 10 tuples

            List<Object[]> rList = q.getResultList();

            XMLAddressPK _555SiliconValleyDrive = new XMLAddressPK("555 Silicon Valley Drive");
            XMLAddressPK _1780MercuryWay = new XMLAddressPK("1780 Mercury Way");
            XMLAddressPK _512VenusDrive = new XMLAddressPK("512 Venus Drive");
            XMLAddressPK _12440VulcaneAve = new XMLAddressPK("12440 Vulcan Avenue");
            XMLAddressPK _4983PlutoniumAve = new XMLAddressPK("4983 Plutonium Avenue");
            XMLAddressPK _182MartianSt = new XMLAddressPK("182 Martian Street");
            XMLAddressPK _6200VegasDr = new XMLAddressPK("6200 Vegas Drive");
            XMLAddressPK _8900JupiterPark = new XMLAddressPK("8900 Jupiter Park");
            XMLAddressPK _150NFAptE1 = new XMLAddressPK("150 North First Apt E1");

            /*
             * TODO: investigate what the correct behavior should be.
             * EclipseLink/OpenJPA queries include NULL values
             */
            Object[] targets = new Object[] {
                                              // e.empid    e.name           AddressBean               AddressBean
                                              new Object[] { 1, "david", new EntityValue(XMLAddressBean.class, "street", _555SiliconValleyDrive),
                                                             new EntityValue(XMLAddressBean.class, "street", _1780MercuryWay) },
                                              new Object[] { 2, "andrew", new EntityValue(XMLAddressBean.class, "street", _555SiliconValleyDrive),
                                                             new EntityValue(XMLAddressBean.class, "street", _1780MercuryWay) },
                                              new Object[] { 3, "minmei", new EntityValue(XMLAddressBean.class, "street", _555SiliconValleyDrive),
                                                             new EntityValue(XMLAddressBean.class, "street", _1780MercuryWay) },
                                              new Object[] { 4, "george", new EntityValue(XMLAddressBean.class, "street", _555SiliconValleyDrive),
                                                             new EntityValue(XMLAddressBean.class, "street", _512VenusDrive) },
                                              new Object[] { 5, "ritika", new EntityValue(XMLAddressBean.class, "street", _555SiliconValleyDrive),
                                                             new EntityValue(XMLAddressBean.class, "street", _12440VulcaneAve) },
                                              new Object[] { 6, "ahmad", new EntityValue(XMLAddressBean.class, "street", _4983PlutoniumAve),
                                                             new EntityValue(XMLAddressBean.class, "street", _4983PlutoniumAve) },
                                              new Object[] { 7, "charlene", new EntityValue(XMLAddressBean.class, "street", _555SiliconValleyDrive),
                                                             new EntityValue(XMLAddressBean.class, "street", _182MartianSt) },
                                              new Object[] { 8, "Tom Rayburn", new EntityValue(XMLAddressBean.class, "street", _555SiliconValleyDrive),
                                                             new EntityValue(XMLAddressBean.class, "street", _6200VegasDr) },
                                              new Object[] { 9, "harry", new EntityValue(XMLAddressBean.class, "street", _8900JupiterPark),
                                                             new EntityValue(XMLAddressBean.class, "street", _150NFAptE1) },
                                              new Object[] { 10, "Catalina Wei", new EntityValue(XMLAddressBean.class, "street", _555SiliconValleyDrive), null }
            };

            if (JPAPersistenceProvider.HIBERNATE.equals(provider)) {
                targets = new Object[] {
                                         // e.empid    e.name           AddressBean               AddressBean
                                         new Object[] { 1, "david", new EntityValue(XMLAddressBean.class, "street", _555SiliconValleyDrive),
                                                        new EntityValue(XMLAddressBean.class, "street", _1780MercuryWay) },
                                         new Object[] { 2, "andrew", new EntityValue(XMLAddressBean.class, "street", _555SiliconValleyDrive),
                                                        new EntityValue(XMLAddressBean.class, "street", _1780MercuryWay) },
                                         new Object[] { 3, "minmei", new EntityValue(XMLAddressBean.class, "street", _555SiliconValleyDrive),
                                                        new EntityValue(XMLAddressBean.class, "street", _1780MercuryWay) },
                                         new Object[] { 4, "george", new EntityValue(XMLAddressBean.class, "street", _555SiliconValleyDrive),
                                                        new EntityValue(XMLAddressBean.class, "street", _512VenusDrive) },
                                         new Object[] { 5, "ritika", new EntityValue(XMLAddressBean.class, "street", _555SiliconValleyDrive),
                                                        new EntityValue(XMLAddressBean.class, "street", _12440VulcaneAve) },
                                         new Object[] { 6, "ahmad", new EntityValue(XMLAddressBean.class, "street", _4983PlutoniumAve),
                                                        new EntityValue(XMLAddressBean.class, "street", _4983PlutoniumAve) },
                                         new Object[] { 7, "charlene", new EntityValue(XMLAddressBean.class, "street", _555SiliconValleyDrive),
                                                        new EntityValue(XMLAddressBean.class, "street", _182MartianSt) },
                                         new Object[] { 8, "Tom Rayburn", new EntityValue(XMLAddressBean.class, "street", _555SiliconValleyDrive),
                                                        new EntityValue(XMLAddressBean.class, "street", _6200VegasDr) },
                                         new Object[] { 9, "harry", new EntityValue(XMLAddressBean.class, "street", _8900JupiterPark),
                                                        new EntityValue(XMLAddressBean.class, "street", _150NFAptE1) }
                };
            }

            validateQueryResult(testName, qStr, rList, targets, true);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST5; select p.projid, p.name, p.dept.no from ProjectBean p order by p.projid
    public void testLoop005(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        JPAPersistenceProvider provider = JPAPersistenceProvider.resolveJPAPersistenceProvider(jpaResource);

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select p.projid, p.name, p.dept.no from XMLProjectBean p order by p.projid";
            Query q = em.createQuery(qStr);

//            TEST5; select p.projid, p.name, p.dept.no from ProjectBean p order by p.projid
//            p.projid    p.name    p.dept.no
//            ~~~~~~~~ ~~~~~~~~~~~~ ~~~~~~~~~
//              1000   Project:1000    210
//              2000   Project:2000    220
//             TEST5; 2 tuples

            List<Object[]> rList = q.getResultList();

            /*
             * TODO: investigate what the correct behavior should be.
             * EclipseLink/OpenJPA join against the XMLDeptBean table, eliminating null values.
             * Hibernate only queries the XMLProjectBean table, which should be sufficient.
             *
             * https://bugs.eclipse.org/bugs/show_bug.cgi?id=431371
             */
            Object[] targets = new Object[] {
                                              //  p.projid    p.name    p.dept.no
                                              new Object[] { 1000, "Project:1000", 210 },
                                              new Object[] { 2000, "Project:2000", 220 },
            };

            if (JPAPersistenceProvider.HIBERNATE.equals(provider)) {
                /*
                 * select projectbea0_.projid as col_0_0_, projectbea0_.name as col_1_0_, projectbea0_.dept_deptno as col_2_0_
                 * from JPAProjectBean projectbea0_
                 * order by projectbea0_.projid
                 */
                targets = new Object[] {
                                         //  p.projid    p.name    p.dept.no
                                         new Object[] { 1000, "Project:1000", 210 },
                                         new Object[] { 2000, "Project:2000", 220 },
                                         new Object[] { 3000, "Project:3000", null }
                };
            }

            validateQueryResult(testName, qStr, rList, targets, true);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST6; select p.projid, p.name, t9.taskid, t9.name, e.empid, e.name from ProjectBean p left join p.tasks t9 left join t9.emps e
    public void testLoop006(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select p.projid, p.name, t9.taskid, t9.name, e.empid, e.name from XMLProjectBean p left join p.tasks t9 left join t9.emps e";
            Query q = em.createQuery(qStr);

//            TEST6; select p.projid, p.name, t9.taskid, t9.name, e.empid, e.name from ProjectBean p left join p.tasks t9 left join t9.emps e
//            p.projid    p.name    t9.taskid  t9.name   e.empid e.name
//            ~~~~~~~~ ~~~~~~~~~~~~ ~~~~~~~~~ ~~~~~~~~~~ ~~~~~~~ ~~~~~~
//              1000   Project:1000   1010     Design       1    david
//              1000   Project:1000   1020       Code       1    david
//              1000   Project:1000   1020       Code       2    andrew
//              1000   Project:1000   1020       Code       9    harry
//              1000   Project:1000   1030       Test       5    ritika
//              1000   Project:1000   1030       Test       9    harry
//              2000   Project:2000   2010      Design      1    david
//              2000   Project:2000   2020    Code, Test  null    null
//              3000   Project:3000   null       null     null    null
//             TEST6; 9 tuples

            List<Object[]> rList = q.getResultList();

            Object[] targets[] = {
                                   // p.projid    p.name    t9.taskid  t9.name   e.empid e.name
                                   new Object[] { 1000, "Project:1000", 1010, "Design", 1, "david" },
                                   new Object[] { 1000, "Project:1000", 1020, "Code", 1, "david" },
                                   new Object[] { 1000, "Project:1000", 1020, "Code", 2, "andrew" },
                                   new Object[] { 1000, "Project:1000", 1020, "Code", 9, "harry" },
                                   new Object[] { 1000, "Project:1000", 1030, "Test", 5, "ritika" },
                                   new Object[] { 1000, "Project:1000", 1030, "Test", 9, "harry" },
                                   new Object[] { 2000, "Project:2000", 2010, "Design", 1, "david" },
                                   new Object[] { 2000, "Project:2000", 2020, "Code, Test", null, null },
                                   new Object[] { 3000, "Project:3000", null, null, null, null },
            };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST7; select count(d) from XMLDeptBean d
    public void testLoop007(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select count(d) from XMLDeptBean d";
            Query q = em.createQuery(qStr);

//            TEST7; select count(d) from XMLDeptBean d
//            count(d)
//            ~~~~~~~~
//               5
//             TEST7; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { 5l };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST8; select count(e) from XMLEmpBean e
    public void testLoop008(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select count(e) from XMLEmpBean e";
            Query q = em.createQuery(qStr);

//            TEST8; select count(e) from XMLEmpBean e
//            count(e)
//            ~~~~~~~~
//               10
//             TEST8; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { 10l };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST9; select count(a) from AddressBean a
    public void testLoop009(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select count(a) from XMLAddressBean a";
            Query q = em.createQuery(qStr);

//            TEST9; select count(a) from AddressBean a
//            count(a)
//            ~~~~~~~~
//               9
//             TEST9; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { 9l };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST10; select count(p) from ProjectBean p
    public void testLoop010(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select count(p) from XMLProjectBean p";
            Query q = em.createQuery(qStr);

//            TEST10; select count(p) from ProjectBean p
//            count(p)
//            ~~~~~~~~
//               3
//             TEST10; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { 3l };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST11; select count(t) from TaskBean t
    public void testLoop011(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select count(t) from XMLTaskBean t";
            Query q = em.createQuery(qStr);

//            TEST11; select count(t) from TaskBean t
//            count(t)
//            ~~~~~~~~
//               5
//             TEST11; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { 5l };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST12; select d from XMLDeptBean d
    public void testLoop012(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d";
            Query q = em.createQuery(qStr);

//            TEST12; select d from XMLDeptBean d
//            DeptBean
//            ~~~~~~~~
//             [100]
//             [200]
//             [210]
//             [220]
//             [300]
//             TEST12; 5 tuples

            List rList = q.getResultList();
            Object[] targets = { dep100, dep200, dep210, dep220, dep300 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST13; select d from XMLDeptBean d left join d.mgr m left join m.dept md left join md.mgr dm left join dm.dept x order by x.name
    public void testLoop013(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d left join d.mgr m left join m.dept md left join md.mgr dm left join dm.dept x order by x.name";
            Query q = em.createQuery(qStr);

//            TEST13; select d from XMLDeptBean d left join d.mgr m left join m.dept md left join md.mgr dm left join dm.dept x order by x.name
//            DeptBean
//            ~~~~~~~~
//             [100]
//             [200]
//             [210]
//             [220]
//             [300]
//             TEST13; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { dep100, dep200, dep210, dep220, dep300 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST14; select d from XMLDeptBean d left join d.mgr m left join m.dept md left join md.mgr dm order by dm.name asc
    public void testLoop014(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d left join d.mgr m left join m.dept md left join md.mgr dm order by dm.name asc";
            Query q = em.createQuery(qStr);

//            TEST14; select d from XMLDeptBean d left join d.mgr m left join m.dept md left join md.mgr dm order by dm.name asc
//            DeptBean
//            ~~~~~~~~
//             [100]
//             [200]
//             [210]
//             [220]
//             [300]
//             TEST14; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { dep100, dep200, dep210, dep220, dep300 };

            // TODO: Validator needs to have smarts for alternative correct orders.
//            validateQueryResult(testName, qStr, rList, targets, true);

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST15; select d from XMLDeptBean d left join d.mgr m left join m.dept md order by md.name asc
    public void testLoop015(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d left join d.mgr m left join m.dept md order by md.name asc";
            Query q = em.createQuery(qStr);

//            TEST15; select d from XMLDeptBean d left join d.mgr m left join m.dept md order by md.name asc
//            DeptBean
//            ~~~~~~~~
//             [100]
//             [200]
//             [210]
//             [220]
//             [300]
//             TEST15; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { dep100, dep200, dep210, dep220, dep300 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST16; select d from XMLDeptBean d order by d.mgr.dept.mgr.dept.name
    public void testLoop016(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d order by d.mgr.dept.mgr.dept.name ";
            Query q = em.createQuery(qStr);

//            TEST16; select d from XMLDeptBean d order by d.mgr.dept.mgr.dept.name
//            DeptBean
//            ~~~~~~~~
//             [210] [220]
//             TEST16; 2 tuples

            List rList = q.getResultList();

            Object[] targets = { dep210, dep220 };

            // TODO: Validator needs to have smarts for alternative correct orders.
//            validateQueryResult(testName, qStr, rList, targets, true);
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST17; select d from XMLDeptBean d order by d.mgr.dept.mgr.name asc
    public void testLoop017(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d order by d.mgr.dept.mgr.name asc ";
            Query q = em.createQuery(qStr);

//            TEST17; select d from XMLDeptBean d order by d.mgr.dept.mgr.name asc
//            DeptBean
//            ~~~~~~~~
//             [200]
//             [210]
//             [220]
//             [300]
//             TEST17; 4 tuples

            List rList = q.getResultList();

            Object[] targets = { dep200, dep210, dep220, dep300 };

            // TODO: Validator needs to have smarts for alternative correct orders.
//          validateQueryResult(testName, qStr, rList, targets, true);
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST18; select d from XMLDeptBean d order by d.mgr.dept.name asc
    public void testLoop018(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d order by d.mgr.dept.name asc ";
            Query q = em.createQuery(qStr);

//            TEST18; select d from XMLDeptBean d order by d.mgr.dept.name asc
//            DeptBean
//            ~~~~~~~~
//             [200]
//             [210]
//             [220]
//             [300]
//             TEST18; 4 tuples

            List rList = q.getResultList();

            Object[] targets = { dep200, dep210, dep220, dep300 };

            // TODO: Validator needs to have smarts for alternative correct orders.
//          validateQueryResult(testName, qStr, rList, targets, true);
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST19; select d from XMLDeptBean d where (d.mgr = (select d.mgr from XMLDeptBean d where d.no = 100))
    public void testLoop019(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where (d.mgr = (select d.mgr from XMLDeptBean d where d.no = 100)) ";
            Query q = em.createQuery(qStr);

//            TEST19; select d from XMLDeptBean d where (d.mgr = (select d.mgr from XMLDeptBean d where d.no = 100))
//            DeptBean
//            ~~~~~~~~
//             [100]
//             TEST19; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { dep100 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST20; select d from XMLDeptBean d where (d.name = 'Dave''s Dept')
    public void testLoop020(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where (d.name = 'Dave''s Dept') ";
            Query q = em.createQuery(qStr);

//            TEST20; select d from XMLDeptBean d where (d.name = 'Dave''s Dept')
//            TEST20; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST21; select d from XMLDeptBean d where (d.name = 'WebSphere')
    public void testLoop021(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where (d.name = 'WebSphere')";
            Query q = em.createQuery(qStr);

//            TEST21; select d from XMLDeptBean d where (d.name = 'WebSphere')
//            TEST21; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST22; select d from XMLDeptBean d where (d.no < 200 and d.no = 200 or d.no >= 300)
    public void testLoop022(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where (d.no < 200 and d.no = 200 or d.no >= 300) ";
            Query q = em.createQuery(qStr);

//            TEST22; select d from XMLDeptBean d where (d.no < 200 and d.no = 200 or d.no >= 300)
//            DeptBean
//            ~~~~~~~~
//             [300]
//             TEST22; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { dep300 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST23; select d from XMLDeptBean d where (d.no <= 200 or d.no > 301)
    public void testLoop023(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where (d.no <= 200 or d.no > 301) ";
            Query q = em.createQuery(qStr);

//            TEST23; select d from XMLDeptBean d where (d.no <= 200 or d.no > 301)
//            DeptBean
//            ~~~~~~~~
//             [100]
//             [200]
//             TEST23; 2 tuples

            List rList = q.getResultList();

            Object[] targets = { dep100, dep200 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } catch (Throwable t) {
            // Catch any Exceptions thrown by the test case for proper error logging.
            t.printStackTrace();
            Assert.fail("Caught an unexpected Exception during test execution." + t);
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST24; select d from XMLDeptBean d where (not d.no > 300 or d.no =4)
    public void testLoop024(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where (not d.no > 300 or d.no =4) ";
            Query q = em.createQuery(qStr);

//            TEST24; select d from XMLDeptBean d where (not d.no > 300 or d.no =4)
//            DeptBean
//            ~~~~~~~~
//             [100]
//             [200]
//             [210]
//             [220]
//             [300]
//             TEST24; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { dep100, dep200, dep210, dep220, dep300 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST25; select d from XMLDeptBean d where d = d.mgr.dept
    public void testLoop025(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where d = d.mgr.dept ";
            Query q = em.createQuery(qStr);

//            TEST25; select d from XMLDeptBean d where d = d.mgr.dept
//            TEST25; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST26; select d from XMLDeptBean d where d.mgr.salary=10
    public void testLoop026(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where d.mgr.salary=10  ";
            Query q = em.createQuery(qStr);

//            TEST26; select d from XMLDeptBean d where d.mgr.salary=10
//            TEST26; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST27; select d from XMLDeptBean d where d.mgr.salary>0.0 and d.name = 'Sales'
    public void testLoop027(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where d.mgr.salary>0.0 and d.name = 'Sales' ";
            Query q = em.createQuery(qStr);

//            TEST27; select d from XMLDeptBean d where d.mgr.salary>0.0 and d.name = 'Sales'
//            TEST27; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST28; select d from XMLDeptBean d where d.name = 'Sales' or d.name = 'Service'
    public void testLoop028(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where d.name = 'Sales' or d.name = 'Service' ";
            Query q = em.createQuery(qStr);

//            TEST28; select d from XMLDeptBean d where d.name = 'Sales' or d.name = 'Service'
//                            DeptBean
//                            ~~~~~~~~
//                             [220]
//                             [300]
//                             TEST28; 2 tuples

            List rList = q.getResultList();

            Object[] targets = { dep220, dep300 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST29; select d from XMLDeptBean d where d.name = 'nonexisting'
    public void testLoop029(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where d.name = 'nonexisting' ";
            Query q = em.createQuery(qStr);

//          TEST29; select d from XMLDeptBean d where d.name = 'nonexisting'
//          TEST29; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST30; select d from XMLDeptBean d, in(d.emps) as e where e.salary > 13
    public void testLoop030(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d, in(d.emps) as e where e.salary > 13 ";
            Query q = em.createQuery(qStr);

//            TEST30; select d from XMLDeptBean d, in(d.emps) as e where e.salary > 13
//            DeptBean
//            ~~~~~~~~
//             [200]
//             [210]
//             TEST30; 2 tuples

            List rList = q.getResultList();

            Object[] targets = { dep200, dep210 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST31; select d from XMLDeptBean d, in(d.emps) e where (e.isManager = FALSE)
    public void testLoop031(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d, in(d.emps) e where (e.isManager = FALSE) ";
            Query q = em.createQuery(qStr);

//            TEST31; select d from XMLDeptBean d, in(d.emps) e where (e.isManager = FALSE)
//            DeptBean
//            ~~~~~~~~
//             [210]
//             [210]
//             TEST31; 2 tuples

            List rList = q.getResultList();

            Object[] targets = { dep210, dep210 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST32; select d from XMLDeptBean d, in(d.projects) as p
    public void testLoop032(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d, in(d.projects) as p ";
            Query q = em.createQuery(qStr);

//            TEST32; select d from XMLDeptBean d, in(d.projects) as p
//            DeptBean
//            ~~~~~~~~
//             [210]
//             [220]
//             TEST32; 2 tuples

            List rList = q.getResultList();

            Object[] targets = { dep210, dep220 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST33; select d from XMLEmpBean e join e.dept d
    public void testLoop033(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e join e.dept d";
            Query q = em.createQuery(qStr);

//            TEST33; select d from XMLEmpBean e join e.dept d
//            DeptBean
//            ~~~~~~~~
//             [100]
//             [100]
//             [200]
//             [200]
//             [210]
//             [210]
//             [210]
//             [210]
//             [220]
//             TEST33; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { dep100, dep100, dep200, dep200, dep210, dep210, dep210, dep210, dep220 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST34; select d from XMLEmpBean e join e.dept d where d.name = 'dept1'
    public void testLoop034(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e join e.dept d where d.name = 'dept1'";
            Query q = em.createQuery(qStr);

//          TEST34; select d from XMLEmpBean e join e.dept d where d.name = 'dept1'
//          TEST34; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST35; select d from XMLEmpBean e join e.dept d where d.no > 0
    public void testLoop035(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e join e.dept d where d.no > 0";
            Query q = em.createQuery(qStr);

//            TEST35; select d from XMLEmpBean e join e.dept d where d.no > 0
//            DeptBean
//            ~~~~~~~~
//             [100]
//             [100]
//             [200]
//             [200]
//             [210]
//             [210]
//             [210]
//             [210]
//             [220]
//             TEST35; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { dep100, dep100, dep200, dep200, dep210, dep210, dep210, dep210, dep220 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST36; select d from XMLEmpBean e left join e.dept d
    public void testLoop036(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e left join e.dept d";
            Query q = em.createQuery(qStr);

//            TEST36; select d from XMLEmpBean e left join e.dept d
//            DeptBean
//            ~~~~~~~~
//              null
//             [100]
//             [100]
//             [200]
//             [200]
//             [210]
//             [210]
//             [210]
//             [210]
//             [220]
//             TEST36; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { null, dep100, dep100, dep200, dep200, dep210, dep210, dep210, dep210, dep220 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST37; select d from XMLEmpBean e left join e.dept d where (e.name = 'john' and e.name = 'ahmad')
    public void testLoop037(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e left join e.dept d where (e.name = 'john' and e.name = 'ahmad')";
            Query q = em.createQuery(qStr);

//            TEST37; select d from XMLEmpBean e left join e.dept d where (e.name = 'john' and e.name = 'ahmad')
//            TEST37; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST38; select d from XMLEmpBean e left join e.dept d where (e.name = 'john' or e.name = 'ahmad') and d.name is null
    public void testLoop038(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e left join e.dept d where (e.name = 'john' or e.name = 'ahmad') and d.name is null";
            Query q = em.createQuery(qStr);

//            TEST38; select d from XMLEmpBean e left join e.dept d where (e.name = 'john' or e.name = 'ahmad') and d.name is null
//            TEST38; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST39; select d from XMLEmpBean e left join e.dept d where d.mgr.name = 'Dave' or e.empid > 0
    public void testLoop039(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e left join e.dept d where d.mgr.name = 'Dave' or e.empid > 0";
            Query q = em.createQuery(qStr);

//            TEST39; select d from XMLEmpBean e left join e.dept d where d.mgr.name = 'Dave' or e.empid > 0
//                            DeptBean
//                            ~~~~~~~~
//                              null
//                             [100]
//                             [100]
//                             [200]
//                             [200]
//                             [210]
//                             [210]
//                             [210]
//                             [210]
//                             [220]
//                             TEST39; 10 tuples

            List<XMLDeptBean> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(10, rList.size());

            boolean found[] = { false, false, false, false, false, false, false, false, false, false };
            Integer targets[] = { null, 100, 100, 200, 200, 210, 210, 210, 210, 220 };

            for (XMLDeptBean result : rList) {
                System.out.println("Result = " + result);
                for (int idx = 0; idx < targets.length; idx++) {
                    if (found[idx] == true) {
                        continue;
                    }

                    if (targets[idx] == null) {
                        if (result == null) {
                            found[idx] = true;
                            break;
                        }
                    } else {
                        if (result != null && targets[idx].equals(result.getNo())) {
                            found[idx] = true;
                            break;
                        }
                    }
                }
            }

            boolean allFound = true;
            for (boolean b : found) {
                allFound = allFound && b;
            }

            System.out.print("Results: ");
            for (boolean f : found) {
                System.out.print(f + " ");
            }
            System.out.println();
            Assert.assertTrue(allFound);

            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST40; select d from XMLEmpBean e left join e.dept d where d.name = 'dept1'
    public void testLoop040(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e left join e.dept d where d.name = 'dept1'";
            Query q = em.createQuery(qStr);

//          TEST40; select d from XMLEmpBean e left join e.dept d where d.name = 'dept1'
//          TEST40; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);

            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST41; select d from XMLEmpBean e left join e.dept d where d.name is null
    public void testLoop041(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e left join e.dept d where d.name is null";
            Query q = em.createQuery(qStr);

//          TEST41; select d from XMLEmpBean e left join e.dept d where d.name is null
//            d
//           ~~~~
//           null
//            TEST41; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { null };
            validateQueryResult(testName, qStr, rList, targets);

            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST42; select d from XMLEmpBean e left join e.dept d where d.no > 0
    public void testLoop042(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e left join e.dept d where d.no > 0";
            Query q = em.createQuery(qStr);

//            TEST42; select d from XMLEmpBean e left join e.dept d where d.no > 0
//            DeptBean
//            ~~~~~~~~
//             [100]
//             [100]
//             [200]
//             [200]
//             [210]
//             [210]
//             [210]
//             [210]
//             [220]
//             TEST42; 9 tuples

            List<XMLDeptBean> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(9, rList.size());

            boolean found[] = { false, false, false, false, false, false, false, false, false };
            Integer targets[] = { 100, 100, 200, 200, 210, 210, 210, 210, 220 };

            for (XMLDeptBean result : rList) {
                System.out.println("Result = " + result);
                for (int idx = 0; idx < targets.length; idx++) {
                    if (found[idx] == true) {
                        continue;
                    }

                    if (targets[idx] == null) {
                        if (result == null) {
                            found[idx] = true;
                            break;
                        }
                    } else {
                        if (result != null && targets[idx].equals(result.getNo())) {
                            found[idx] = true;
                            break;
                        }
                    }
                }
            }

            boolean allFound = true;
            for (boolean b : found) {
                allFound = allFound && b;
            }

            System.out.print("Results: ");
            for (boolean f : found) {
                System.out.print(f + " ");
            }
            System.out.println();
            Assert.assertTrue(allFound);

            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST43; select d from XMLEmpBean e left join e.dept d where e.name = 'john' and e.name = 'ahmad'
    public void testLoop043(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e left join e.dept d where e.name = 'john' and e.name = 'ahmad'";
            Query q = em.createQuery(qStr);

//          TEST43; select d from XMLEmpBean e left join e.dept d where e.name = 'john' and e.name = 'ahmad'
//          TEST43; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST44; select d from XMLEmpBean e left join e.dept d where e.name is null
    public void testLoop044(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e left join e.dept d where e.name is null";
            Query q = em.createQuery(qStr);

//            TEST44; select d from XMLEmpBean e left join e.dept d where e.name is null
//            TEST44; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST45; select d from XMLEmpBean e left join e.dept d where e.name='name1'
    public void testLoop045(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e left join e.dept d where e.name='name1' ";
            Query q = em.createQuery(qStr);

//          TEST45; select d from XMLEmpBean e left join e.dept d where e.name='name1'
//          TEST45; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST46; select d from XMLEmpBean e left join e.dept d where e.name='name3' and e.bonus > 100
    public void testLoop046(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e left join e.dept d where e.name='name3' and e.bonus > 100 ";
            Query q = em.createQuery(qStr);

//          TEST46; select d from XMLEmpBean e left join e.dept d where e.name='name3' and e.bonus > 100
//          TEST46; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST47; select d from XMLEmpBean e left join e.dept d where e.salary = 1000 and e.name = 'Bijan'
    public void testLoop047(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e left join e.dept d where e.salary = 1000 and e.name = 'Bijan'";
            Query q = em.createQuery(qStr);

//          TEST47; select d from XMLEmpBean e left join e.dept d where e.salary = 1000 and e.name = 'Bijan'
//          TEST47; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST48; select d from XMLEmpBean e left join e.dept d where e.salary = 1000 or e.name = 'Ahmad'
    public void testLoop048(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e left join e.dept d where e.salary = 1000 or e.name = 'Ahmad'";
            Query q = em.createQuery(qStr);

//          TEST48; select d from XMLEmpBean e left join e.dept d where e.salary = 1000 or e.name = 'Ahmad'
//          TEST48; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST49; select d from XMLEmpBean e left join e.dept d, ProjectBean p where e.name = 'john' and e.salary = p.budget
    public void testLoop049(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e left join e.dept d, ProjectBean p where e.name = 'john' and e.salary = p.budget";
            Query q = em.createQuery(qStr);

//          TEST49; select d from XMLEmpBean e left join e.dept d, ProjectBean p where e.name = 'john' and e.salary = p.budget
//          TEST49; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST50; select d from XMLEmpBean e left join e.dept d, ProjectBean p where e.name = 'john' and e.salary = p.budget and NOT(d.name is null)
    public void testLoop050(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e left join e.dept d, ProjectBean p where e.name = 'john' and e.salary = p.budget and NOT(d.name is null) ";
            Query q = em.createQuery(qStr);

//           TEST50; select d from XMLEmpBean e left join e.dept d, ProjectBean p where e.name = 'john' and e.salary = p.budget and NOT(d.name is null)
//           TEST50; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST51; select d from XMLEmpBean e left join e.dept d, ProjectBean p where e.name = 'john' and e.salary = p.budget and d.name is null
    public void testLoop051(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLEmpBean e left join e.dept d, ProjectBean p where e.name = 'john' and e.salary = p.budget and d.name is null";
            Query q = em.createQuery(qStr);

//          TEST51; select d from XMLEmpBean e left join e.dept d, ProjectBean p where e.name = 'john' and e.salary = p.budget and d.name is null
//          TEST51; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST52; select d, e.name, p.name from XMLDeptBean as d left outer join d.emps as e left outer join d.projects as p
    public void testLoop052(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d, e.name, p.name from XMLDeptBean as d left outer join d.emps as e left outer join d.projects as p ";
            Query q = em.createQuery(qStr);

//            TEST52; select d, e.name, p.name from XMLDeptBean as d left outer join d.emps as e left outer join d.projects as p
//            DeptBean   e.name       p.name
//            ~~~~~~~~ ~~~~~~~~~~~ ~~~~~~~~~~~~
//             [100]      ahmad        null
//             [100]   Tom Rayburn     null
//             [200]     george        null
//             [200]     minmei        null
//             [210]      david    Project:1000
//             [210]      harry    Project:1000
//             [210]     andrew    Project:1000
//             [210]    charlene   Project:1000
//             [220]     ritika    Project:2000
//             [300]      null         null
//             TEST52; 10 tuples

            List<Object[]> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(10, rList.size());

            Object[] targets[] = {
                                   // DeptBean.   e.name,       p.name
                                   new Object[] { dep100, "ahmad", null },
                                   new Object[] { dep100, "Tom Rayburn", null },
                                   new Object[] { dep200, "george", null },
                                   new Object[] { dep200, "minmei", null },
                                   new Object[] { dep210, "david", "Project:1000" },
                                   new Object[] { dep210, "harry", "Project:1000" },
                                   new Object[] { dep210, "andrew", "Project:1000" },
                                   new Object[] { dep210, "charlene", "Project:1000" },
                                   new Object[] { dep220, "ritika", "Project:2000" },
                                   new Object[] { dep300, null, null }
            };

            validateQueryResult("testLoop052", qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST53; select d,e from XMLEmpBean e join e.dept d where e.bonus<100.02 or e.name='name6'
    public void testLoop053(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d,e from XMLEmpBean e join e.dept d where e.bonus<100.02 or e.name='name6'";
            Query q = em.createQuery(qStr);

//            TEST53; select d,e from XMLEmpBean e join e.dept d where e.bonus<100.02 or e.name='name6'
//            DeptBean EmpBean
//            ~~~~~~~~ ~~~~~~~
//             [100]     [6]
//             [100]     [8]
//             [200]     [3]
//             [200]     [4]
//             [210]     [1]
//             [210]     [2]
//             [210]     [7]
//             [210]     [9]
//             [220]     [5]
//             TEST53; 9 tuples

            List<Object[]> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(9, rList.size());

            Object[] targets[] = {
                                   // DeptBean,  EmpBean
                                   new Object[] { dep100, emp6 },
                                   new Object[] { dep100, emp8 },
                                   new Object[] { dep200, emp3 },
                                   new Object[] { dep200, emp4 },
                                   new Object[] { dep210, emp1 },
                                   new Object[] { dep210, emp2 },
                                   new Object[] { dep210, emp7 },
                                   new Object[] { dep210, emp9 },
                                   new Object[] { dep220, emp5 }
            };

            validateQueryResult("testLoop053", qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST54; select d,e from XMLEmpBean e left join e.dept d where d.name='dept1'
    public void testLoop054(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d,e from XMLEmpBean e left join e.dept d where d.name='dept1' ";
            Query q = em.createQuery(qStr);

//          TEST54; select d,e from XMLEmpBean e left join e.dept d where d.name='dept1'
//          TEST54; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST55; select d,e from XMLEmpBean e left join e.dept d where e.bonus<100.02 or e.name='name2'
    public void testLoop055(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d,e from XMLEmpBean e left join e.dept d where e.bonus<100.02 or e.name='name2' ";
            Query q = em.createQuery(qStr);

//            TEST55; select d,e from XMLEmpBean e left join e.dept d where e.bonus<100.02 or e.name='name2'
//                            DeptBean EmpBean
//                            ~~~~~~~~ ~~~~~~~
//                              null    [10]
//                             [100]     [6]
//                             [100]     [8]
//                             [200]     [3]
//                             [200]     [4]
//                             [210]     [1]
//                             [210]     [2]
//                             [210]     [7]
//                             [210]     [9]
//                             [220]     [5]
//                             TEST55; 10 tuples

            List<Object[]> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(10, rList.size());

            Object[] targets[] = {
                                   // DeptBean,  EmpBean
                                   new Object[] { null, emp10 },
                                   new Object[] { dep100, emp6 },
                                   new Object[] { dep100, emp8 },
                                   new Object[] { dep200, emp3 },
                                   new Object[] { dep200, emp4 },
                                   new Object[] { dep210, emp1 },
                                   new Object[] { dep210, emp2 },
                                   new Object[] { dep210, emp7 },
                                   new Object[] { dep210, emp9 },
                                   new Object[] { dep220, emp5 },

            };

            validateQueryResult("testLoop056", qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST56; select d,e from XMLEmpBean e left join e.dept d where e.name='name1'
    public void testLoop056(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d,e from XMLEmpBean e left join e.dept d where e.name='name1' ";
            Query q = em.createQuery(qStr);

//          TEST56; select d,e from XMLEmpBean e left join e.dept d where e.name='name1'
//          TEST56; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST57; select d,m,md,dm from XMLDeptBean d left join d.mgr m left join m.dept md left join md.mgr dm left join dm.dept x order by x.name
    public void testLoop057(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d,m,md,dm from XMLDeptBean d left join d.mgr m left join m.dept md left join md.mgr dm left join dm.dept x order by x.name ";
            Query q = em.createQuery(qStr);

//           TEST57; select d,m,md,dm from XMLDeptBean d left join d.mgr m left join m.dept md left join md.mgr dm left join dm.dept x order by x.name
//            DeptBean EmpBean DeptBean EmpBean
//            ~~~~~~~~ ~~~~~~~ ~~~~~~~~ ~~~~~~~
//             [100]    [10]     null    null
//             [200]     [8]    [100]    [10]
//             [210]     [3]    [200]     [8]
//             [220]     [4]    [200]     [8]
//             [300]     [6]    [100]    [10]
//             TEST57; 5 tuples

            List<Object[]> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(5, rList.size());

            Object[] targets[] = {
                                   // DeptBean EmpBean DeptBean EmpBean
                                   new Object[] { dep100, emp10, null, null },
                                   new Object[] { dep200, emp8,
                                                  dep100, emp10 },
                                   new Object[] { dep210, emp3,
                                                  dep200, emp8 },
                                   new Object[] { dep220, emp4,
                                                  dep200, emp8 },
                                   new Object[] { dep300, emp6,
                                                  dep100, emp10 },

            };
            validateQueryResult("testLoop057", qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST58; select d.mgr from XMLEmpBean e join e.dept d where d.mgr.name = 'Ahmad'
    public void testLoop058(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.mgr from XMLEmpBean e join e.dept d where d.mgr.name = 'Ahmad' ";
            Query q = em.createQuery(qStr);

//          TEST58; select d.mgr from XMLEmpBean e join e.dept d where d.mgr.name = 'Ahmad'
//          TEST58; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST59; select d.mgr from XMLEmpBean e join e.dept d where d.name = 'davedept' or e.empid > 0
    public void testLoop059(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.mgr from XMLEmpBean e join e.dept d where d.name = 'davedept' or e.empid > 0 ";
            Query q = em.createQuery(qStr);

//          TEST59; select d.mgr from XMLEmpBean e join e.dept d where d.name = 'davedept' or e.empid > 0
//            EmpBean
//            ~~~~~~~
//              [3]
//              [3]
//              [3]
//              [3]
//              [4]
//              [8]
//              [8]
//             [10]
//             [10]
//             TEST59; 9 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(9, rList.size());

            int targets[] = { 3, 3, 3, 3, 4, 8, 8, 10, 10 };
            boolean found[] = new boolean[targets.length];
            Arrays.fill(found, false);

            for (Object result : rList) {
                for (int idx = 0; idx < targets.length; idx++) {
                    if (found[idx] == true) {
                        continue;
                    }

                    int targetVal = targets[idx];

                    XMLEmpBean eb1 = (XMLEmpBean) result;
                    System.out.println("EmpBean = " + eb1);
                    if (eb1 == null) {
                        continue;
                    } else {
                        if (eb1.getEmpid() != targetVal)
                            continue;
                    }

                    found[idx] = true;
                    break;
                }
            }

            boolean allFound = true;
            for (boolean b : found) {
                allFound = allFound && b;
            }

            System.out.print("Results: ");
            for (boolean f : found) {
                System.out.print(f + " ");
            }
            System.out.println();
            Assert.assertTrue(allFound);

            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } catch (Throwable t) {
            // Catch any Exceptions thrown by the test case for proper error logging.
            t.printStackTrace();
            Assert.fail("Caught an unexpected Exception during test execution." + t);
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST60; select d.mgr from XMLEmpBean e join e.dept d where e.empid > 1000
    public void testLoop060(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.mgr from XMLEmpBean e join e.dept d where e.empid > 1000 ";
            Query q = em.createQuery(qStr);

//          TEST60; select d.mgr from XMLEmpBean e join e.dept d where e.empid > 1000
//            TEST60; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);

            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST61; select d.mgr from XMLEmpBean e left join e.dept d where d.mgr.name = 'Ahmad'
    public void testLoop061(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.mgr from XMLEmpBean e left join e.dept d where d.mgr.name = 'Ahmad' ";
            Query q = em.createQuery(qStr);

//          TEST61; select d.mgr from XMLEmpBean e left join e.dept d where d.mgr.name = 'Ahmad'
//            TEST61; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);

            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST62; select d.mgr from XMLEmpBean e left join e.dept d where d.name = 'davedept' or e.empid > 0
    public void testLoop062(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.mgr from XMLEmpBean e left join e.dept d where d.name = 'davedept' or e.empid > 0";
            Query q = em.createQuery(qStr);

//            TEST62; select d.mgr from XMLEmpBean e left join e.dept d where d.name = 'davedept' or e.empid > 0
//                            EmpBean
//                            ~~~~~~~
//                              [3]
//                              [3]
//                              [3]
//                              [3]
//                              [4]
//                              [8]
//                              [8]
//                             [10]
//                             [10]
//                             null
//                             TEST62; 10 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(10, rList.size());

            Integer targets[] = { 3, 3, 3, 3, 4, 8, 8, 10, 10, null };
            boolean found[] = new boolean[targets.length];
            Arrays.fill(found, false);

            for (Object result : rList) {
                System.out.println("Empbean = " + result);
                for (int idx = 0; idx < targets.length; idx++) {
                    if (found[idx] == true) {
                        continue;
                    }

                    Integer targetVal = targets[idx];
                    if (targetVal == null) {
                        if (result != null)
                            continue;
                    } else {
                        XMLEmpBean eb1 = (XMLEmpBean) result;
                        if (eb1 == null) {
                            continue;
                        } else {
                            if (!targetVal.equals(eb1.getEmpid()))
                                continue;
                        }
                    }

                    found[idx] = true;
                    break;
                }
            }

            boolean allFound = true;
            for (boolean b : found) {
                allFound = allFound && b;
            }

            System.out.print("Results: ");
            for (boolean f : found) {
                System.out.print(f + " ");
            }
            System.out.println();
            Assert.assertTrue(allFound);

            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } catch (Throwable t) {
            // Catch any Exceptions thrown by the test case for proper error logging.
            t.printStackTrace();
            Assert.fail("Caught an unexpected Exception during test execution." + t);
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST63; select d.mgr from XMLEmpBean e left join e.dept d where d.name = 'dmqa' and e.salary = 20
    public void testLoop063(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.mgr from XMLEmpBean e left join e.dept d where d.name = 'dmqa' and e.salary = 20";
            Query q = em.createQuery(qStr);

//            TEST63; select d.mgr from XMLEmpBean e left join e.dept d where d.name = 'dmqa' and e.salary = 20
//            TEST63; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST64; select d.mgr from XMLEmpBean e left join e.dept d where d.name is null
    public void testLoop064(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.mgr from XMLEmpBean e left join e.dept d where d.name is null";
            Query q = em.createQuery(qStr);

//            TEST64; select d.mgr from XMLEmpBean e left join e.dept d where d.name is null
//            d.mgr
//            ~~~~~
//            null
//             TEST64; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { null };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST65; select d.mgr from XMLEmpBean e left join e.dept d where e.empid > 1000
    public void testLoop065(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.mgr from XMLEmpBean e left join e.dept d where e.empid > 1000";
            Query q = em.createQuery(qStr);

//            TEST65; select d.mgr from XMLEmpBean e left join e.dept d where e.empid > 1000
//            TEST65; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST66; select d.mgr.name from XMLEmpBean e join e.dept d where e.name = 'Dave'
    public void testLoop066(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.mgr.name from XMLEmpBean e join e.dept d where e.name = 'Dave'";
            Query q = em.createQuery(qStr);

//            TEST66; select d.mgr.name from XMLEmpBean e join e.dept d where e.name = 'Dave'
//            TEST66; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST67; select d.name from XMLDeptBean as d left outer join d.emps as e  where e.dept = d
    public void testLoop067(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name from XMLDeptBean as d left outer join d.emps as e  where e.dept = d";
            Query q = em.createQuery(qStr);

//            TEST67; select d.name from XMLDeptBean as d left outer join d.emps as e  where e.dept = d
//                            d.name
//                          ~~~~~~~~~~~
//                              CEO
//                              CEO
//                             Admin
//                             Admin
//                            Service
//                          Development
//                          Development
//                          Development
//                          Development
//                           TEST67; 9 tuples

            List<String> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(9, rList.size());

            String[] expList = { "CEO", "CEO", "Admin", "Admin", "Service", "Development", "Development", "Development", "Development" };

            for (int idx = 0; idx < expList.length; idx++) {
                final String str = expList[idx];
                Assert.assertNotNull(str);
                Assert.assertEquals(expList[idx], str);
            }

            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } catch (Throwable t) {
            // Catch any Exceptions thrown by the test case for proper error logging.
            t.printStackTrace();
            Assert.fail("Caught an unexpected Exception during test execution." + t);
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST68; select d.name from XMLDeptBean as d
    public void testLoop068(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name from XMLDeptBean as d ";
            Query q = em.createQuery(qStr);

//            TEST68; select d.name from XMLDeptBean as d
//            d.name
//          ~~~~~~~~~~~
//              CEO
//             Admin
//             Sales
//            Service
//          Development
//           TEST68; 5 tuples

            List<String> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(5, rList.size());

            String[] expList = { "CEO", "Admin", "Sales", "Service", "Development" };

            for (int idx = 0; idx < expList.length; idx++) {
                final String str = expList[idx];
                Assert.assertNotNull(str);
                Assert.assertEquals(expList[idx], str);
            }

            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } catch (Throwable t) {
            // Catch any Exceptions thrown by the test case for proper error logging.
            t.printStackTrace();
            Assert.fail("Caught an unexpected Exception during test execution." + t);
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST69; select d.name from XMLDeptBean d left join d.emps e where d.name = 'DEV'
    public void testLoop069(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name from XMLDeptBean d left join d.emps e where d.name = 'DEV' ";
            Query q = em.createQuery(qStr);

//            TEST69; select d.name from XMLDeptBean d left join d.emps e where d.name = 'DEV'
//            TEST69; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST70; select d.name from XMLDeptBean d left join d.emps e where d.name = 'DEV' or e.name = 'harry'
    public void testLoop070(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name from XMLDeptBean d left join d.emps e where d.name = 'DEV' or e.name = 'harry'";
            Query q = em.createQuery(qStr);

//            TEST70; select d.name from XMLDeptBean d left join d.emps e where d.name = 'DEV' or e.name = 'harry'
//                            d.name
//                          ~~~~~~~~~~~
//                          Development
//                           TEST70; 1 tuple

            List<String> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(1, rList.size());

            String[] expList = { "Development" };

            for (int idx = 0; idx < expList.length; idx++) {
                final String str = expList[idx];
                Assert.assertNotNull(str);
                Assert.assertEquals(expList[idx], str);
            }

            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } catch (Throwable t) {
            // Catch any Exceptions thrown by the test case for proper error logging.
            t.printStackTrace();
            Assert.fail("Caught an unexpected Exception during test execution." + t);
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST71; select d.name from XMLDeptBean d where d.name = ''
    public void testLoop071(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name from XMLDeptBean d where d.name = ''";
            Query q = em.createQuery(qStr);

//            TEST71; select d.name from XMLDeptBean d where d.name = ''
//            TEST71; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST72; select d.name from XMLEmpBean e join e.dept d where e.name = 'Dave' or e.empid > 0
    public void testLoop072(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name from XMLEmpBean e join e.dept d where e.name = 'Dave' or e.empid > 0";
            Query q = em.createQuery(qStr);

//            TEST72; select d.name from XMLEmpBean e join e.dept d where e.name = 'Dave' or e.empid > 0
//                            d.name
//                          ~~~~~~~~~~~
//                              CEO
//                              CEO
//                             Admin
//                             Admin
//                            Service
//                          Development
//                          Development
//                          Development
//                          Development
//                           TEST72; 9 tuples

            List<String> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(9, rList.size());

            String[] expList = { "CEO", "CEO", "Admin", "Admin", "Service", "Development", "Development", "Development", "Development" };

            for (int idx = 0; idx < expList.length; idx++) {
                final String str = expList[idx];
                Assert.assertNotNull(str);
                Assert.assertEquals(expList[idx], str);
            }

            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } catch (Throwable t) {
            // Catch any Exceptions thrown by the test case for proper error logging.
            t.printStackTrace();
            Assert.fail("Caught an unexpected Exception during test execution." + t);
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST73; select d.name, e.name from XMLDeptBean d join d.emps e
    public void testLoop073(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name, e.name from XMLDeptBean d join d.emps e";
            Query q = em.createQuery(qStr);

//            TEST73; select d.name, e.name from XMLDeptBean d join d.emps e
//            d.name      e.name
//          ~~~~~~~~~~~ ~~~~~~~~~~~
//              CEO        ahmad
//              CEO     Tom Rayburn
//             Admin      george
//             Admin      minmei
//            Service     ritika
//          Development    david
//          Development    harry
//          Development   andrew
//          Development  charlene
//           TEST73; 9 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(9, rList.size());

            Object[] targets[] = {
                                   //             d.name      e.name
                                   new String[] { "CEO", "ahmad" },
                                   new String[] { "CEO", "Tom Rayburn" },
                                   new String[] { "Admin", "george" },
                                   new String[] { "Admin", "minmei" },
                                   new String[] { "Service", "ritika" },
                                   new String[] { "Development", "david" },
                                   new String[] { "Development", "harry" },
                                   new String[] { "Development", "andrew" },
                                   new String[] { "Development", "charlene" }
            };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST74; select d.name, e.name from XMLDeptBean d left join d.emps e
    public void testLoop074(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name, e.name from XMLDeptBean d left join d.emps e";
            Query q = em.createQuery(qStr);

//            TEST74; select d.name, e.name from XMLDeptBean d left join d.emps e
//            d.name      e.name
//          ~~~~~~~~~~~ ~~~~~~~~~~~
//              CEO        ahmad
//              CEO     Tom Rayburn
//             Admin      george
//             Admin      minmei
//             Sales       null
//            Service     ritika
//          Development    david
//          Development    harry
//          Development   andrew
//          Development  charlene
//           TEST74; 10 tuples

            List<Object[]> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(10, rList.size());

            boolean found[] = { false, false, false, false, false, false, false, false, false, false };
            String[] targets[] = {
                                   //             d.name      e.name
                                   new String[] { "CEO", "ahmad" },
                                   new String[] { "CEO", "Tom Rayburn" },
                                   new String[] { "Admin", "george" },
                                   new String[] { "Admin", "minmei" },
                                   new String[] { "Sales", null },
                                   new String[] { "Service", "ritika" },
                                   new String[] { "Development", "david" },
                                   new String[] { "Development", "harry" },
                                   new String[] { "Development", "andrew" },
                                   new String[] { "Development", "charlene" }
            };

            validateQueryResult("testLoop074", qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST75; select d.name, e.name, m.name from XMLDeptBean d left join d.emps e left join d.mgr m
    public void testLoop075(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name, e.name, m.name from XMLDeptBean d left join d.emps e left join d.mgr m";
            Query q = em.createQuery(qStr);

//            TEST75; select d.name, e.name, m.name from XMLDeptBean d left join d.emps e left join d.mgr m
//            d.name      e.name       m.name
//          ~~~~~~~~~~~ ~~~~~~~~~~~ ~~~~~~~~~~~~
//              CEO        ahmad    Catalina Wei
//              CEO     Tom Rayburn Catalina Wei
//             Admin      george    Tom Rayburn
//             Admin      minmei    Tom Rayburn
//             Sales       null        ahmad
//            Service     ritika       george
//          Development    david       minmei
//          Development    harry       minmei
//          Development   andrew       minmei
//          Development  charlene      minmei
//           TEST75; 10 tuples

            List<String[]> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(10, rList.size());

            boolean found[] = { false, false, false, false, false, false, false, false, false, false };
            String[] targets[] = {
                                   //             d.name      e.name     n.name
                                   new String[] { "CEO", "ahmad", "Catalina Wei" },
                                   new String[] { "CEO", "Tom Rayburn", "Catalina Wei" },
                                   new String[] { "Admin", "george", "Tom Rayburn" },
                                   new String[] { "Admin", "minmei", "Tom Rayburn" },
                                   new String[] { "Sales", null, "ahmad" },
                                   new String[] { "Service", "ritika", "george" },
                                   new String[] { "Development", "david", "minmei" },
                                   new String[] { "Development", "harry", "minmei" },
                                   new String[] { "Development", "andrew", "minmei" },
                                   new String[] { "Development", "charlene", "minmei" }
            };

            for (Object[] result : rList) {
                for (int idx = 0; idx < targets.length; idx++) {
                    if (found[idx] == true) {
                        continue;
                    }

                    String[] targetVals = targets[idx];
                    if (targetVals == null && result == null) {
                        found[idx] = true;
                        break;
                    }

                    String dName = (String) result[0];
                    if (dName == null) {
                        if (targetVals[0] != null)
                            continue;
                    } else {
                        if (!targetVals[0].equals(dName))
                            continue;
                    }
                    String eName = (String) result[1];
                    if (eName == null) {
                        if (targetVals[1] != null)
                            continue;
                    } else {
                        if (!targetVals[1].equals(eName))
                            continue;
                    }

                    String nName = (String) result[2];
                    if (nName == null) {
                        if (targetVals[2] != null)
                            continue;
                    } else {
                        if (!targetVals[2].equals(nName))
                            continue;
                    }

                    found[idx] = true;
                    break;
                }
            }

            boolean allFound = true;
            for (boolean b : found) {
                allFound = allFound && b;
            }

            Assert.assertTrue(allFound);

            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } catch (Throwable t) {
            // Catch any Exceptions thrown by the test case for proper error logging.
            t.printStackTrace();
            Assert.fail("Caught an unexpected Exception during test execution." + t);
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST76; select d.name, e.name, m.name, p.name from XMLDeptBean d left join d.emps e left join d.mgr m left join e.tasks p
    public void testLoop076(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name, e.name, m.name, p.name from XMLDeptBean d left join d.emps e left join d.mgr m left join e.tasks p";
            Query q = em.createQuery(qStr);

//            TEST76; select d.name, e.name, m.name, p.name from XMLDeptBean d left join d.emps e left join d.mgr m left join e.tasks p
//            d.name      e.name       m.name    p.name
//          ~~~~~~~~~~~ ~~~~~~~~~~~ ~~~~~~~~~~~~ ~~~~~~
//              CEO        ahmad    Catalina Wei  null
//              CEO     Tom Rayburn Catalina Wei  null
//             Admin      george    Tom Rayburn   null
//             Admin      minmei    Tom Rayburn   null
//             Sales       null        ahmad      null
//            Service     ritika       george     Test
//          Development    david       minmei     Code
//          Development    david       minmei    Design
//          Development    david       minmei    Design
//          Development    harry       minmei     Code
//          Development    harry       minmei     Test
//          Development   andrew       minmei     Code
//          Development  charlene      minmei     null
//           TEST76; 13 tuples

            List<Object[]> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(13, rList.size());

//            boolean found[] = { false, false, false, false, false, false, false, false, false, false, false, false, false };
            String[] targets[] = {
                                   //             d.name      e.name     n.name    p.name
                                   new String[] { "CEO", "ahmad", "Catalina Wei", null },
                                   new String[] { "CEO", "Tom Rayburn", "Catalina Wei", null },
                                   new String[] { "Admin", "george", "Tom Rayburn", null },
                                   new String[] { "Admin", "minmei", "Tom Rayburn", null },
                                   new String[] { "Sales", null, "ahmad", null },
                                   new String[] { "Service", "ritika", "george", "Test" },
                                   new String[] { "Development", "david", "minmei", "Code" },
                                   new String[] { "Development", "david", "minmei", "Design" },
                                   new String[] { "Development", "david", "minmei", "Design" },
                                   new String[] { "Development", "harry", "minmei", "Code" },
                                   new String[] { "Development", "harry", "minmei", "Test" },
                                   new String[] { "Development", "andrew", "minmei", "Code" },
                                   new String[] { "Development", "charlene", "minmei", null }
            };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST77; select d.name, e.name, p.name from XMLDeptBean d join d.emps e join e.tasks p
    public void testLoop077(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name, e.name, p.name from XMLDeptBean d join d.emps e join e.tasks p";
            Query q = em.createQuery(qStr);

//            TEST77; select d.name, e.name, p.name from XMLDeptBean d join d.emps e join e.tasks p
//            d.name    e.name p.name
//          ~~~~~~~~~~~ ~~~~~~ ~~~~~~
//            Service   ritika  Test
//          Development andrew  Code
//          Development david   Code
//          Development david  Design
//          Development david  Design
//          Development harry   Code
//          Development harry   Test
//           TEST77; 7 tuples

            List<Object[]> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(7, rList.size());

            String[] targets[] = {
                                   //             d.name    e.name p.name
                                   new String[] { "Service", "ritika", "Test" },
                                   new String[] { "Development", "andrew", "Code" },
                                   new String[] { "Development", "david", "Code" },
                                   new String[] { "Development", "david", "Design" },
                                   new String[] { "Development", "david", "Design" },
                                   new String[] { "Development", "harry", "Code" },
                                   new String[] { "Development", "harry", "Test" },
            };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST78; select d.name, e.name, p.name from XMLDeptBean d join d.emps e, in (e.tasks) p
    public void testLoop078(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name, e.name, p.name from XMLDeptBean d join d.emps e, in (e.tasks) p ";
            Query q = em.createQuery(qStr);

//            TEST78; select d.name, e.name, p.name from XMLDeptBean d join d.emps e, in (e.tasks) p
//            d.name    e.name p.name
//          ~~~~~~~~~~~ ~~~~~~ ~~~~~~
//            Service   ritika  Test
//          Development andrew  Code
//          Development david   Code
//          Development david  Design
//          Development david  Design
//          Development harry   Code
//          Development harry   Test
//           TEST78; 7 tuples

            List<Object[]> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(7, rList.size());

            String[] targets[] = {
                                   //             d.name    e.name p.name
                                   new String[] { "Service", "ritika", "Test" },
                                   new String[] { "Development", "andrew", "Code" },
                                   new String[] { "Development", "david", "Code" },
                                   new String[] { "Development", "david", "Design" },
                                   new String[] { "Development", "david", "Design" },
                                   new String[] { "Development", "harry", "Code" },
                                   new String[] { "Development", "harry", "Test" },
            };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST79; select d.name, e.name, p.name from XMLDeptBean d left join d.emps e left join e.tasks p
    public void testLoop079(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name, e.name, p.name from XMLDeptBean d left join d.emps e left join e.tasks p";
            Query q = em.createQuery(qStr);

//            TEST79; select d.name, e.name, p.name from XMLDeptBean d left join d.emps e left join e.tasks p
//            d.name      e.name    p.name
//          ~~~~~~~~~~~ ~~~~~~~~~~~ ~~~~~~
//              CEO        ahmad     null
//              CEO     Tom Rayburn  null
//             Admin      george     null
//             Admin      minmei     null
//             Sales       null      null
//            Service     ritika     Test
//          Development    david     Code
//          Development    david    Design
//          Development    david    Design
//          Development    harry     Code
//          Development    harry     Test
//          Development   andrew     Code
//          Development  charlene    null
//           TEST79; 13 tuples

            List<Object[]> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(13, rList.size());

            String[] targets[] = {
                                   //             d.name      e.name    p.name
                                   new String[] { "CEO", "ahmad", null },
                                   new String[] { "CEO", "Tom Rayburn", null },
                                   new String[] { "Admin", "george", null },
                                   new String[] { "Admin", "minmei", null },
                                   new String[] { "Sales", null, null },
                                   new String[] { "Service", "ritika", "Test" },
                                   new String[] { "Development", "david", "Code" },
                                   new String[] { "Development", "david", "Design" },
                                   new String[] { "Development", "david", "Design" },
                                   new String[] { "Development", "harry", "Code" },
                                   new String[] { "Development", "harry", "Test" },
                                   new String[] { "Development", "andrew", "Code" },
                                   new String[] { "Development", "charlene", null }
            };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST80; select d.name, m.name from XMLDeptBean d, in (d.mgr) m
    public void testLoop080(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name, m.name from XMLDeptBean d, in (d.mgr) m";
            Query q = em.createQuery(qStr);

//            TEST80; select d.name, m.name from XMLDeptBean d, in (d.mgr) m
//            d.name       m.name
//          ~~~~~~~~~~~ ~~~~~~~~~~~~
//              CEO     Catalina Wei
//             Admin    Tom Rayburn
//             Sales       ahmad
//            Service      george
//          Development    minmei
//           TEST80; 5 tuples

            List<Object[]> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(5, rList.size());

            String[] targets[] = {
                                   //             d.name       m.name
                                   new String[] { "CEO", "Catalina Wei" },
                                   new String[] { "Admin", "Tom Rayburn" },
                                   new String[] { "Sales", "ahmad" },
                                   new String[] { "Service", "george" },
                                   new String[] { "Development", "minmei" }
            };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST81; select d.name, p.name from XMLDeptBean d join d.mgr m join m.tasks p
    public void testLoop081(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name, p.name from XMLDeptBean d join d.mgr m join m.tasks p";
            Query q = em.createQuery(qStr);

//            TEST81; select d.name, p.name from XMLDeptBean d join d.mgr m join m.tasks p
//            TEST81; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST82; select d.no from  DeptBean d, in(d.emps) e
    public void testLoop082(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.no from  DeptBean d, in(d.emps) e";
            Query q = em.createQuery(qStr);

//            TEST82; select d.no from  DeptBean d, in(d.emps) e
//            d.no
//            ~~~~
//            100 100 200 200 210 210 210 210 220
//             TEST82; 9 tuples

            List rList = q.getResultList();
            Object[] targets = { 100, 100, 200, 200, 210, 210, 210, 210, 220 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST83; select d.no from XMLDeptBean d where d.name <> 'Department1'
    public void testLoop083(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.no from XMLDeptBean d where d.name <> 'Department1'   ";
            Query q = em.createQuery(qStr);

//            TEST83; select d.no from XMLDeptBean d where d.name <> 'Department1'
//            d.no
//            ~~~~
//            100
//            200
//            210
//            220
//            300
//             TEST83; 5 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(5, rList.size());

            Object[] targets = { 100, 200, 210, 220, 300 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST84; select d.no,d.name,d.budget,d.mgr from XMLDeptBean d
    public void testLoop084(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.no,d.name,d.budget,d.mgr from XMLDeptBean d ";
            Query q = em.createQuery(qStr);

//            TEST84; select d.no,d.name,d.budget,d.mgr from XMLDeptBean d
//            d.no   d.name    d.budget EmpBean
//            ~~~~ ~~~~~~~~~~~ ~~~~~~~~ ~~~~~~~
//            100      CEO       2.1     [10]
//            200     Admin      2.1      [8]
//            210  Development   2.1      [3]
//            220    Service     2.1      [4]
//            300     Sales      2.1      [6]
//             TEST84; 5 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(5, rList.size());

            Object[] targets[] = {
                                   new Object[] { 100, "CEO", 2.1f, emp10 },
                                   new Object[] { 200, "Admin", 2.1f, emp8 },
                                   new Object[] { 210, "Development", 2.1f, emp3 },
                                   new Object[] { 220, "Service", 2.1f, emp4 },
                                   new Object[] { 300, "Sales", 2.1f, emp6 },
            };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST85; select e from XMLDeptBean d inner join d.emps e
    public void testLoop085(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLDeptBean d inner join d.emps e";
            Query q = em.createQuery(qStr);

//            TEST85; select e from XMLDeptBean d inner join d.emps e
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [3]
//              [4]
//              [5]
//              [6]
//              [7]
//              [8]
//              [9]
//             TEST85; 9 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(9, rList.size());

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST86; select e from XMLDeptBean d join d.emps e
    public void testLoop086(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLDeptBean d join d.emps e ";
            Query q = em.createQuery(qStr);

//            TEST86; select e from XMLDeptBean d join d.emps e
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [3]
//              [4]
//              [5]
//              [6]
//              [7]
//              [8]
//              [9]
//             TEST86; 9 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(9, rList.size());

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST87; select e from XMLDeptBean d left join d.emps e
    public void testLoop087(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLDeptBean d left join d.emps e";
            Query q = em.createQuery(qStr);

//            TEST87; select e from XMLDeptBean d left join d.emps e
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [3]
//              [4]
//              [5]
//              [6]
//              [7]
//              [8]
//              [9]
//             null
//             TEST87; 10 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(10, rList.size());

            final Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, null };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST88; select e from XMLDeptBean d, EmpBean e where d.no = e.salary
    public void testLoop088(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLDeptBean d, EmpBean e where d.no = e.salary ";
            Query q = em.createQuery(qStr);

//          TEST88; select e from XMLDeptBean d, EmpBean e where d.no = e.salary
//          TEST88; 0 tuples

            List<XMLEmpBean> rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(0, rList.size());

            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST89; select e from XMLDeptBean d, EmpBean e where e.name = 'john'
    public void testLoop089(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLDeptBean d, EmpBean e where e.name = 'john' ";
            Query q = em.createQuery(qStr);

//            TEST89; select e from XMLDeptBean d, EmpBean e where e.name = 'john'
//                            TEST89; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);

            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST90; select e from XMLDeptBean d, EmpBean e where e.salary=d.no
    public void testLoop090(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLDeptBean d, EmpBean e where e.salary=d.no ";
            Query q = em.createQuery(qStr);

//            TEST90; select e from XMLDeptBean d, EmpBean e where e.salary=d.no
//                            TEST90; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST91; select e from XMLDeptBean d, in(d.emps) e where d.name='Sales'
    public void testLoop091(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLDeptBean d, in(d.emps) e where d.name='Sales' ";
            Query q = em.createQuery(qStr);

//            TEST91; select e from XMLDeptBean d, in(d.emps) e where d.name='Sales'
//                            TEST91; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST92; select e from XMLEmpBean e  order by e.name
    public void testLoop092(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e  order by e.name ";
            Query q = em.createQuery(qStr);

//            TEST92; select e from XMLEmpBean e  order by e.name
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [3]
//              [4]
//              [5]
//              [6]
//              [7]
//              [8]
//              [9]
//             [10]
//             TEST92; 10 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(10, rList.size());

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST93; select e from XMLEmpBean e join e.dept d join d.mgr m join m.tasks p , DeptBean d2 left join d2.emps q left join q.tasks p2 where p2.name = 'abc' and p.name is null order by d.name, q.salary, m.name
    public void testLoop093(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e join e.dept d join d.mgr m join m.tasks p , DeptBean d2 left join d2.emps q left join q.tasks p2 where p2.name = 'abc' and p.name is null order by d.name, q.salary, m.name";
            Query q = em.createQuery(qStr);

//            TEST93; select e from XMLEmpBean e join e.dept d join d.mgr m join m.tasks p , DeptBean d2 left join d2.emps q left join q.tasks p2 where p2.name = 'abc' and p.name is null order by d.name, q.salary, m.name
//            TEST93; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST94; select e from XMLEmpBean e join e.dept d join d.mgr m join m.tasks p , DeptBean d2 left join d2.emps q left join q.tasks p2 where p2.name = 'abc' and p.name is null order by e.dept.name, q.salary, d.mgr.name
    public void testLoop094(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e join e.dept d join d.mgr m join m.tasks p , DeptBean d2 left join d2.emps q left join q.tasks p2 where p2.name = 'abc' and p.name is null order by e.dept.name, q.salary, d.mgr.name";
            Query q = em.createQuery(qStr);

//            TEST94; select e from XMLEmpBean e join e.dept d join d.mgr m join m.tasks p , DeptBean d2 left join d2.emps q left join q.tasks p2 where p2.name = 'abc' and p.name is null order by e.dept.name, q.salary, d.mgr.name
//            TEST94; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST95; select e from XMLEmpBean e join e.dept d join d.mgr m join m.tasks p , DeptBean d2 left join d2.emps q left join q.tasks p2 where p2.name = 'abc' and p.name is null order by e.dept.name, q.salary, e.dept.mgr.name
    public void testLoop095(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e join e.dept d join d.mgr m join m.tasks p , DeptBean d2 left join d2.emps q left join q.tasks p2 where p2.name = 'abc' and p.name is null order by e.dept.name, q.salary, e.dept.mgr.name";
            Query q = em.createQuery(qStr);

//            TEST95; select e from XMLEmpBean e join e.dept d join d.mgr m join m.tasks p , DeptBean d2 left join d2.emps q left join q.tasks p2 where p2.name = 'abc' and p.name is null order by e.dept.name, q.salary, e.dept.mgr.name
//            TEST95; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST96; select e from XMLEmpBean e join e.dept d left join d.mgr m left join m.tasks p left join e.tasks p2 where p2.name = 'abc'
    public void testLoop096(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e join e.dept d left join d.mgr m left join m.tasks p left join e.tasks p2 where p2.name = 'abc'";
            Query q = em.createQuery(qStr);

//            TEST96; select e from XMLEmpBean e join e.dept d left join d.mgr m left join m.tasks p left join e.tasks p2 where p2.name = 'abc'
//            TEST96; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST97; select e from XMLEmpBean e left join e.dept d left join d.mgr m order by m.name asc
    public void testLoop097(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e left join e.dept d left join d.mgr m order by m.name asc";
            Query q = em.createQuery(qStr);

//            TEST97; select e from XMLEmpBean e left join e.dept d left join d.mgr m order by m.name asc
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [3]
//              [4]
//              [5]
//              [6]
//              [7]
//              [8]
//              [9]
//             [10]
//             TEST97; 10 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(10, rList.size());

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST98; select e from XMLEmpBean e left join e.dept ed, DeptBean d where e.name = 'john' order by ed.name
    public void testLoop098(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e left join e.dept ed, DeptBean d where e.name = 'john' order by ed.name ";
            Query q = em.createQuery(qStr);

//            TEST98; select e from XMLEmpBean e left join e.dept ed, DeptBean d where e.name = 'john' order by ed.name
//            TEST98; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST99; select e from XMLEmpBean e left join e.dept ed, DeptBean d where e.name = 'john'and ed.name = 'dept1'  order by ed.name
    public void testLoop099(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e left join e.dept ed, DeptBean d where e.name = 'john'and ed.name = 'dept1'  order by ed.name ";
            Query q = em.createQuery(qStr);

//            TEST99; select e from XMLEmpBean e left join e.dept ed, DeptBean d where e.name = 'john'and ed.name = 'dept1'  order by ed.name
//            TEST99; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST100; select e from XMLEmpBean e order  by e.name desc
    public void testLoop100(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e order  by e.name desc ";
            Query q = em.createQuery(qStr);

//            TEST100; select e from XMLEmpBean e order  by e.name desc
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [3]
//              [4]
//              [5]
//              [6]
//              [7]
//              [8]
//              [9]
//             [10]
//             TEST100; 10 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(10, rList.size());

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST101; select e from XMLEmpBean e order by e.dept.mgr.name asc
    public void testLoop101(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e order by e.dept.mgr.name asc ";
            Query q = em.createQuery(qStr);

//            TEST101; select e from XMLEmpBean e order by e.dept.mgr.name asc
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [3]
//              [4]
//              [5]
//              [6]
//              [7]
//              [8]
//              [9]
//             TEST101; 9 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(9, rList.size());

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9 };

            validateQueryResult(testName, qStr, rList, targets, true);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST102; select e from XMLEmpBean e order by e.name asc
    public void testLoop102(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e order by e.name asc ";
            Query q = em.createQuery(qStr);

//            TEST102; select e from XMLEmpBean e order by e.name asc
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [3]
//              [4]
//              [5]
//              [6]
//              [7]
//              [8]
//              [9]
//             [10]
//             TEST102; 10 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(10, rList.size());

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };

            validateQueryResult(testName, qStr, rList, targets, true);

            em.clear();

        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST103; select e from XMLEmpBean e order by e.name asc, e.dept.mgr.name desc , e.salary asc, e.dept.budget desc
    public void testLoop103(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e order by e.name asc, e.dept.mgr.name desc , e.salary asc, e.dept.budget desc ";
            Query q = em.createQuery(qStr);

//            TEST103; select e from XMLEmpBean e order by e.name asc, e.dept.mgr.name desc , e.salary asc, e.dept.budget desc
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [3]
//              [4]
//              [5]
//              [6]
//              [7]
//              [8]
//              [9]
//             TEST103; 9 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(10, rList.size());

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9 };

            validateQueryResult(testName, qStr, rList, targets, true);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST104; select e from XMLEmpBean e where (NOT (e.isManager = true))
    public void testLoop104(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (NOT (e.isManager = true))";
            Query q = em.createQuery(qStr);

//            TEST104; select e from XMLEmpBean e where (NOT (e.isManager = true))
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//             TEST104; 2 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(2, rList.size());

            Object[] targets = { emp1, emp2 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST105; select e from XMLEmpBean e where (e.dept.mgr.dept.mgr.name ='Tom')
    public void testLoop105(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.dept.mgr.dept.mgr.name ='Tom') ";
            Query q = em.createQuery(qStr);

//            TEST105; select e from XMLEmpBean e where (e.dept.mgr.dept.mgr.name ='Tom')
//            TEST105; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST106; select e from XMLEmpBean e where (e.dept.mgr.name = 'Tom')
    public void testLoop106(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.dept.mgr.name = 'Tom')";
            Query q = em.createQuery(qStr);

//            TEST106; select e from XMLEmpBean e where (e.dept.mgr.name = 'Tom')
//            TEST106; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST107; select e from XMLEmpBean e where (e.dept.no = 10)
    public void testLoop107(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.dept.no = 10)";
            Query q = em.createQuery(qStr);

//            TEST107; select e from XMLEmpBean e where (e.dept.no = 10)
//            TEST107; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST108; select e from XMLEmpBean e where (e.empid > 5) order by e.name
    public void testLoop108(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.empid > 5) order by e.name ";
            Query q = em.createQuery(qStr);

//            TEST108; select e from XMLEmpBean e where (e.empid > 5) order by e.name
//            EmpBean
//            ~~~~~~~
//              [6]
//              [7]
//              [8]
//              [9]
//             [10]
//             TEST108; 5 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(5, rList.size());

            Object[] targets = { emp6, emp7, emp8, emp9, emp10, };

            validateQueryResult(testName, qStr, rList, targets, true);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST109; select e from XMLEmpBean e where (e.isManager <> TRUE)
    public void testLoop109(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.isManager <> TRUE)";
            Query q = em.createQuery(qStr);

//            TEST109; select e from XMLEmpBean e where (e.isManager <> TRUE)
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//             TEST109; 2 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(2, rList.size());

            Object[] targets = { emp1, emp2 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST110; select e from XMLEmpBean e where (e.isManager = FALSE)
    public void testLoop110(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.isManager = FALSE) ";
            Query q = em.createQuery(qStr);

//            TEST110; select e from XMLEmpBean e where (e.isManager = FALSE)
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//             TEST110; 2 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(2, rList.size());

            Object[] targets = { emp1, emp2 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST111; select e from XMLEmpBean e where (e.isManager = TRUE)
    public void testLoop111(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.isManager = TRUE) ";
            Query q = em.createQuery(qStr);

//            TEST111; select e from XMLEmpBean e where (e.isManager = TRUE)
//            EmpBean
//            ~~~~~~~
//              [3]
//              [4]
//              [5]
//              [6]
//              [7]
//              [8]
//              [9]
//             [10]
//             TEST111; 8 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(8, rList.size());

            Object[] targets = { emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST112; select e from XMLEmpBean e where (e.salary = 65034.28)
    public void testLoop112(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary = 65034.28) ";
            Query q = em.createQuery(qStr);

//            TEST112; select e from XMLEmpBean e where (e.salary = 65034.28)
//            TEST112; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST113; select e from XMLEmpBean e where 12 <= e.salary and 15 > e.salary
    public void testLoop113(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where 12 <= e.salary and 15 > e.salary";
            Query q = em.createQuery(qStr);

//            TEST113; select e from XMLEmpBean e where 12 <= e.salary and 15 > e.salary
//                            EmpBean
//                            ~~~~~~~
//                              [1]
//                              [2]
//                             TEST113; 2 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(2, rList.size());

            Object[] targets = { emp1, emp2 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST114; select e from XMLEmpBean e where 12.1 <= e.salary and 15.5 > e.salary
    public void testLoop114(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where 12.1 <= e.salary and 15.5 > e.salary ";
            Query q = em.createQuery(qStr);

//            TEST114; select e from XMLEmpBean e where 12.1 <= e.salary and 15.5 > e.salary
//                            EmpBean
//                            ~~~~~~~
//                              [1]
//                              [2]
//                             TEST114; 2 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(2, rList.size());

            Object[] targets = { emp1, emp2 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST115; select e from XMLEmpBean e where 5 < e.salary
    public void testLoop115(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where 5 < e.salary";
            Query q = em.createQuery(qStr);

//            TEST115; select e from XMLEmpBean e where 5 < e.salary
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [3]
//             TEST115; 3 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(3, rList.size());

            Object[] targets = { emp1, emp2, emp3 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST116; select e from XMLEmpBean e where e = e.dept.mgr
    public void testLoop116(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e = e.dept.mgr ";
            Query q = em.createQuery(qStr);

//            TEST116; select e from XMLEmpBean e where e = e.dept.mgr
//                            TEST116; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST117; select e from XMLEmpBean e where e = e.dept.mgr and e.name = 'abc'
    public void testLoop117(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e = e.dept.mgr and e.name = 'abc' ";
            Query q = em.createQuery(qStr);

//            TEST117; select e from XMLEmpBean e where e = e.dept.mgr and e.name = 'abc'
//                            TEST117; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST118; select e from XMLEmpBean e where e.dept <> e.dept.mgr.dept
    public void testLoop118(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.dept <> e.dept.mgr.dept";
            Query q = em.createQuery(qStr);

//            TEST118; select e from XMLEmpBean e where e.dept <> e.dept.mgr.dept
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [3]
//              [4]
//              [5]
//              [7]
//              [9]
//             TEST118; 7 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(7, rList.size());

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp7, emp9 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST119; select e from XMLEmpBean e where e.dept.mgr.salary = 10
    public void testLoop119(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.dept.mgr.salary = 10 ";
            Query q = em.createQuery(qStr);

//            TEST119; select e from XMLEmpBean e where e.dept.mgr.salary = 10
//                            TEST119; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST120; select e from XMLEmpBean e where e.dept.name = 'Sales'
    public void testLoop120(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.dept.name = 'Sales'";
            Query q = em.createQuery(qStr);

//            TEST120; select e from XMLEmpBean e where e.dept.name = 'Sales'
//                            TEST120; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST121; select e from XMLEmpBean e where e.empid = 0
    public void testLoop121(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.empid = 0 ";
            Query q = em.createQuery(qStr);

//            TEST121; select e from XMLEmpBean e where e.empid = 0
//                            TEST121; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST122; select e from XMLEmpBean e where e.empid = 8 and e.dept.no = 2
    public void testLoop122(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.empid = 8 and e.dept.no = 2 ";
            Query q = em.createQuery(qStr);

//            TEST122; select e from XMLEmpBean e where e.empid = 8 and e.dept.no = 2
//                            TEST122; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST123; select e from XMLEmpBean e where e.empid<3 order by e.empid
    public void testLoop123(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.empid<3 order by e.empid";
            Query q = em.createQuery(qStr);

//            TEST123; select e from XMLEmpBean e where e.empid<3 order by e.empid
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//             TEST123; 2 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(2, rList.size());

            Object[] targets = { emp1, emp2 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST124; select e from XMLEmpBean e where e.name = 'david' or e.name = 'andrew'
    public void testLoop124(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.name = 'david' or e.name = 'andrew' ";
            Query q = em.createQuery(qStr);

//            TEST124; select e from XMLEmpBean e where e.name = 'david' or e.name = 'andrew'
//                            EmpBean
//                            ~~~~~~~
//                              [1]
//                              [2]
//                             TEST124; 2 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(2, rList.size());

            Object[] targets = { emp1, emp2 };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST125; select e from XMLEmpBean e where e.salary < (select avg(e.salary) from XMLEmpBean e where e.salary >0)
    public void testLoop125(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary < (select avg(e.salary) from XMLEmpBean e where e.salary >0) ";
            Query q = em.createQuery(qStr);

//            TEST125; select e from XMLEmpBean e where e.salary < (select avg(e.salary) from XMLEmpBean e where e.salary >0)
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [4]
//              [5]
//              [6]
//              [7]
//              [8]
//              [9]
//             [10]
//             TEST125; 9 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(9, rList.size());

            Object[] targets = { emp1, emp2, emp4, emp5, emp6, emp7, emp8, emp9, emp10, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST126; select e from XMLEmpBean e where e.salary > 10 and not (e.name='david' and e.salary = 13.1)
    public void testLoop126(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary > 10 and not (e.name='david' and e.salary = 13.1) ";
            Query q = em.createQuery(qStr);

//            TEST126; select e from XMLEmpBean e where e.salary > 10 and not (e.name='david' and e.salary = 13.1)
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [3]
//             TEST126; 3 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(3, rList.size());

            Object[] targets = { emp1, emp2, emp3, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST127; select e from XMLEmpBean e where e.salary >0 and e.salary <  100
    public void testLoop127(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary >0 and e.salary <  100 ";
            Query q = em.createQuery(qStr);

//            TEST127; select e from XMLEmpBean e where e.salary >0 and e.salary <  100
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [3]
//             TEST127; 3 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(3, rList.size());

            Object[] targets = { emp1, emp2, emp3, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST128; select e from XMLEmpBean e where e.salary >0.0 and e.salary <  100.0
    public void testLoop128(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary >0.0 and e.salary <  100.0 ";
            Query q = em.createQuery(qStr);

//            TEST128; select e from XMLEmpBean e where e.salary >0.0 and e.salary <  100.0
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [3]
//             TEST128; 3 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(3, rList.size());

            Object[] targets = { emp1, emp2, emp3, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST129; select e from XMLEmpBean e where e.salary between 0 and 15.4
    public void testLoop129(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary between 0 and 15.4";
            Query q = em.createQuery(qStr);

//            TEST129; select e from XMLEmpBean e where e.salary between 0 and 15.4
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [4]
//              [5]
//              [6]
//              [7]
//              [8]
//              [9]
//             [10]
//             TEST129; 9 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(9, rList.size());

            Object[] targets = { emp1, emp2, emp4, emp5, emp6, emp7, emp8, emp9, emp10, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST130; select e from XMLEmpBean e where not ( e.salary > 10)
    public void testLoop130(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where not ( e.salary > 10) ";
            Query q = em.createQuery(qStr);

//            TEST130; select e from XMLEmpBean e where not ( e.salary > 10)
//            EmpBean
//            ~~~~~~~
//              [4]
//              [5]
//              [6]
//              [7]
//              [8]
//              [9]
//             [10]
//             TEST130; 7 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(7, rList.size());

            Object[] targets = { emp4, emp5, emp6, emp7, emp8, emp9, emp10, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST131; select e from XMLEmpBean e where not e.salary > 10 and e.salary =10 or e.salary < 20
    public void testLoop131(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where not e.salary > 10 and e.salary =10 or e.salary < 20 ";
            Query q = em.createQuery(qStr);

//            TEST131; select e from XMLEmpBean e where not e.salary > 10 and e.salary =10 or e.salary < 20
//                            EmpBean
//                            ~~~~~~~
//                              [1]
//                              [2]
//                              [3]
//                              [4]
//                              [5]
//                              [6]
//                              [7]
//                              [8]
//                              [9]
//                             [10]
//                             TEST131; 10 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(10, rList.size());

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST132; select e from XMLEmpBean e, DeptBean d where d.no = e.salary
    public void testLoop132(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e, DeptBean d where d.no = e.salary ";
            Query q = em.createQuery(qStr);

//            TEST132; select e from XMLEmpBean e, DeptBean d where d.no = e.salary
//                            TEST132; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST133; select e from XMLEmpBean e, DeptBean d where d.no = e.salary+90
    public void testLoop133(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e, DeptBean d where d.no = e.salary+90 ";
            Query q = em.createQuery(qStr);

//            TEST133; select e from XMLEmpBean e, DeptBean d where d.no = e.salary+90
//                            TEST133; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST134; select count(e) from XMLEmpBean e, DeptBean d where e.name = 'john'
    public void testLoop134(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select count(e) from XMLEmpBean e, DeptBean d where e.name = 'john'";
            Query q = em.createQuery(qStr);

//            TEST134; select count(e) from XMLEmpBean e, DeptBean d where e.name = 'john'
//                            count(e)
//                            ~~~~~~~~
//                               0
//                             TEST134; 1 tuple

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(1, rList.size());

            Object[] targets = { 0l };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST135; select count(e) from XMLEmpBean e, DeptBean d where e.name = 'john'and e.dept.name = 'dept1'
    public void testLoop135(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select count(e) from XMLEmpBean e, DeptBean d where e.name = 'john'and e.dept.name = 'dept1' ";
            Query q = em.createQuery(qStr);

//            TEST135; select count(e) from XMLEmpBean e, DeptBean d where e.name = 'john'and e.dept.name = 'dept1'
//                            count(e)
//                            ~~~~~~~~
//                               0
//                             TEST135; 1 tuple

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(1, rList.size());

            Object[] targets = { 0l };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST136; select count(e) from XMLEmpBean e, DeptBean d where e.salary=d.no
    public void testLoop136(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select count(e) from XMLEmpBean e, DeptBean d where e.salary=d.no ";
            Query q = em.createQuery(qStr);

//            TEST136; select count(e) from XMLEmpBean e, DeptBean d where e.salary=d.no
//                            count(e)
//                            ~~~~~~~~
//                               0
//                             TEST136; 1 tuple

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(1, rList.size());

            Object[] targets = { 0l };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST137; select e from XMLEmpBean e, in( e.dept) as d where d.name = 'Sales'
    public void testLoop137(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e, in(e.dept) as d where d.name = 'Sales' ";
            Query q = em.createQuery(qStr);

//            TEST137; select e from XMLEmpBean e, in( e.dept) as d where d.name = 'Sales'
//                            TEST137; 0 tuples

            List rList = q.getResultList();
            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST138; select e from XMLEmpBean e, in( e.dept.mgr) as m where m.salary > 13
    public void testLoop138(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e, in( e.dept.mgr) as m where m.salary > 13 ";
            Query q = em.createQuery(qStr);

//            TEST138; select e from XMLEmpBean e, in( e.dept.mgr) as m where m.salary > 13
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [7]
//              [9]
//             TEST138; 4 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(4, rList.size());

            Object[] targets = { emp1, emp2, emp7, emp9, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST139; select e from XMLEmpBean e, in(e.dept.emps) as emps where emps.salary > 13
    public void testLoop139(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e, in(e.dept.emps) as emps where emps.salary > 13 ";
            Query q = em.createQuery(qStr);

//            TEST139; select e from XMLEmpBean e, in(e.dept.emps) as emps where emps.salary > 13
//            EmpBean
//            ~~~~~~~
//              [1]
//              [2]
//              [3]
//              [4]
//              [7]
//              [9]
//             TEST139; 6 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(6, rList.size());

            Object[] targets = { emp1, emp2, emp3, emp4, emp7, emp9, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST140; select e, d from XMLEmpBean e, DeptBean d
    public void testLoop140(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e, d from XMLEmpBean e, DeptBean d ";
            Query q = em.createQuery(qStr);

//            TEST140; select e, d from XMLEmpBean e, DeptBean d
//            EmpBean DeptBean
//            ~~~~~~~ ~~~~~~~~
//              [1]    [100]
//              [1]    [200]
//              [1]    [210]
//              [1]    [220]
//              [1]    [300]
//              [2]    [100]
//              [2]    [200]
//              [2]    [210]
//              [2]    [220]
//              [2]    [300]
//              [3]    [100]
//              [3]    [200]
//              [3]    [210]
//              [3]    [220]
//              [3]    [300]
//              [4]    [100]
//              [4]    [200]
//              [4]    [210]
//              [4]    [220]
//              [4]    [300]
//              [5]    [100]
//              [5]    [200]
//              [5]    [210]
//              [5]    [220]
//              [5]    [300]
//              [6]    [100]
//              [6]    [200]
//              [6]    [210]
//              [6]    [220]
//              [6]    [300]
//              [7]    [100]
//              [7]    [200]
//              [7]    [210]
//              [7]    [220]
//              [7]    [300]
//              [8]    [100]
//              [8]    [200]
//              [8]    [210]
//              [8]    [220]
//              [8]    [300]
//              [9]    [100]
//              [9]    [200]
//              [9]    [210]
//              [9]    [220]
//              [9]    [300]
//             [10]    [100]
//             [10]    [200]
//             [10]    [210]
//             [10]    [220]
//             [10]    [300]
//             TEST140; 50 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(50, rList.size());

            Object[] targets[] = { //  EmpBean DeptBean
                                   new Object[] { emp1, dep100, },
                                   new Object[] { emp1, dep200, },
                                   new Object[] { emp1, dep210, },
                                   new Object[] { emp1, dep220, },
                                   new Object[] { emp1, dep300, },

                                   new Object[] { emp2, dep100, },
                                   new Object[] { emp2, dep200, },
                                   new Object[] { emp2, dep210, },
                                   new Object[] { emp2, dep220, },
                                   new Object[] { emp2, dep300, },

                                   new Object[] { emp3, dep100, },
                                   new Object[] { emp3, dep200, },
                                   new Object[] { emp3, dep210, },
                                   new Object[] { emp3, dep220, },
                                   new Object[] { emp3, dep300, },

                                   new Object[] { emp4, dep100, },
                                   new Object[] { emp4, dep200, },
                                   new Object[] { emp4, dep210, },
                                   new Object[] { emp4, dep220, },
                                   new Object[] { emp4, dep300, },

                                   new Object[] { emp5, dep100, },
                                   new Object[] { emp5, dep200, },
                                   new Object[] { emp5, dep210, },
                                   new Object[] { emp5, dep220, },
                                   new Object[] { emp5, dep300, },

                                   new Object[] { emp6, dep100, },
                                   new Object[] { emp6, dep200, },
                                   new Object[] { emp6, dep210, },
                                   new Object[] { emp6, dep220, },
                                   new Object[] { emp6, dep300, },

                                   new Object[] { emp7, dep100, },
                                   new Object[] { emp7, dep200, },
                                   new Object[] { emp7, dep210, },
                                   new Object[] { emp7, dep220, },
                                   new Object[] { emp7, dep300, },

                                   new Object[] { emp8, dep100, },
                                   new Object[] { emp8, dep200, },
                                   new Object[] { emp8, dep210, },
                                   new Object[] { emp8, dep220, },
                                   new Object[] { emp8, dep300, },

                                   new Object[] { emp9, dep100, },
                                   new Object[] { emp9, dep200, },
                                   new Object[] { emp9, dep210, },
                                   new Object[] { emp9, dep220, },
                                   new Object[] { emp9, dep300, },

                                   new Object[] { emp10, dep100, },
                                   new Object[] { emp10, dep200, },
                                   new Object[] { emp10, dep210, },
                                   new Object[] { emp10, dep220, },
                                   new Object[] { emp10, dep300, },
            };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST141; select e,d from XMLEmpBean e join e.dept d
    public void testLoop141(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e,d from XMLEmpBean e join e.dept d ";
            Query q = em.createQuery(qStr);

//            TEST141; select e,d from XMLEmpBean e join e.dept d
//            EmpBean DeptBean
//            ~~~~~~~ ~~~~~~~~
//              [1]    [210]
//              [2]    [210]
//              [3]    [200]
//              [4]    [200]
//              [5]    [220]
//              [6]    [100]
//              [7]    [210]
//              [8]    [100]
//              [9]    [210]
//             TEST141; 9 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(9, rList.size());

            Object[] targets[] = { //  EmpBean DeptBean
                                   new Object[] { emp1, dep210, },
                                   new Object[] { emp2, dep210, },
                                   new Object[] { emp3, dep200, },
                                   new Object[] { emp4, dep200, },
                                   new Object[] { emp5, dep220, },

                                   new Object[] { emp6, dep100, },
                                   new Object[] { emp7, dep210, },
                                   new Object[] { emp8, dep100, },
                                   new Object[] { emp9, dep210, },
            };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST142; select e,d from XMLEmpBean e left join e.dept d
    public void testLoop142(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e,d from XMLEmpBean e left join e.dept d ";
            Query q = em.createQuery(qStr);

//            TEST142; select e,d from XMLEmpBean e left join e.dept d
//            EmpBean DeptBean
//            ~~~~~~~ ~~~~~~~~
//              [1]    [210]
//              [2]    [210]
//              [3]    [200]
//              [4]    [200]
//              [5]    [220]
//              [6]    [100]
//              [7]    [210]
//              [8]    [100]
//              [9]    [210]
//             [10]     null
//             TEST142; 10 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(10, rList.size());

            Object[] targets[] = { //  EmpBean DeptBean
                                   new Object[] { emp1, dep210, },
                                   new Object[] { emp2, dep210, },
                                   new Object[] { emp3, dep200, },
                                   new Object[] { emp4, dep200, },
                                   new Object[] { emp5, dep220, },

                                   new Object[] { emp6, dep100, },
                                   new Object[] { emp7, dep210, },
                                   new Object[] { emp8, dep100, },
                                   new Object[] { emp9, dep210, },
                                   new Object[] { emp10, null },
            };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST143; select e,d from XMLEmpBean e left join e.dept ed, DeptBean d where e.name='name1' and ed.name='dept1'
    public void testLoop143(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e,d from XMLEmpBean e left join e.dept ed, DeptBean d where e.name='name1' and ed.name='dept1' ";
            Query q = em.createQuery(qStr);

//            TEST143; select e,d from XMLEmpBean e left join e.dept ed, DeptBean d where e.name='name1' and ed.name='dept1'
//                            TEST143; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST144; select e.dept from  DeptBean d, in(d.emps) e
    public void testLoop144(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept from  DeptBean d, in(d.emps) e ";
            Query q = em.createQuery(qStr);

//            TEST144; select e.dept from  DeptBean d, in(d.emps) e
//            DeptBean
//            ~~~~~~~~
//             [100]
//             [100]
//             [200]
//             [200]
//             [210]
//             [210]
//             [210]
//             [210]
//             [220]
//             TEST144; 9 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(9, rList.size());

            Object[] targets = { dep100, dep100, dep200, dep200, dep210, dep210, dep210, dep210, dep220, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST145; select e.dept from XMLEmpBean e
    public void testLoop145(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept from XMLEmpBean e";
            Query q = em.createQuery(qStr);

//            TEST145; select e.dept from XMLEmpBean e
//            DeptBean
//            ~~~~~~~~
//              null
//             [100]
//             [100]
//             [200]
//             [200]
//             [210]
//             [210]
//             [210]
//             [210]
//             [220]
//             TEST145; 10 tuples

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(10, rList.size());

            Object[] targets = { null, dep100, dep100, dep200, dep200, dep210, dep210, dep210, dep210, dep220, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST146; select e.dept from XMLEmpBean e where (e.name = 'john' or e.name = 'ahmad')
    public void testLoop146(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept from XMLEmpBean e where (e.name = 'john' or e.name = 'ahmad')";
            Query q = em.createQuery(qStr);

//            TEST146; select e.dept from XMLEmpBean e where (e.name = 'john' or e.name = 'ahmad')
//            DeptBean
//            ~~~~~~~~
//             [100]
//             TEST146; 1 tuple

            List rList = q.getResultList();
            Assert.assertNotNull(rList);
            Assert.assertEquals(1, rList.size());

            Object[] targets = { dep100 };

            validateQueryResult("testLoop146", qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST147; select e.dept from XMLEmpBean e where (e.name = 'john' or e.name = 'ahmad') and e.dept.name is null
    public void testLoop147(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept from XMLEmpBean e where (e.name = 'john' or e.name = 'ahmad') and e.dept.name is null ";
            Query q = em.createQuery(qStr);

//            TEST147; select e.dept from XMLEmpBean e where (e.name = 'john' or e.name = 'ahmad') and e.dept.name is null
//            TEST147; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST148; select e.dept from XMLEmpBean e where e.dept.mgr.name = 'Ahmad'
    public void testLoop148(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept from XMLEmpBean e where e.dept.mgr.name = 'Ahmad'";
            Query q = em.createQuery(qStr);

//            TEST148; select e.dept from XMLEmpBean e where e.dept.mgr.name = 'Ahmad'
//                            TEST148; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST149; select e.dept from XMLEmpBean e where e.dept.mgr.name = 'Dave' or e.empid > 0
    public void testLoop149(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept from XMLEmpBean e where e.dept.mgr.name = 'Dave' or e.empid > 0";
            Query q = em.createQuery(qStr);

//            TEST149; select e.dept from XMLEmpBean e where e.dept.mgr.name = 'Dave' or e.empid > 0
//                            DeptBean
//                            ~~~~~~~~
//                              null
//                             [100]
//                             [100]
//                             [200]
//                             [200]
//                             [210]
//                             [210]
//                             [210]
//                             [210]
//                             [220]
//                             TEST149; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { null, dep100, dep100, dep200, dep200, dep210, dep210, dep210, dep210, dep220, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST150; select e.dept from XMLEmpBean e where e.dept.name = 'Sales'
    public void testLoop150(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept from XMLEmpBean e where e.dept.name = 'Sales' ";
            Query q = em.createQuery(qStr);

//            TEST150; select e.dept from XMLEmpBean e where e.dept.name = 'Sales'
//                            TEST150; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST151; select e.dept from XMLEmpBean e where e.dept.name is null
    public void testLoop151(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept from XMLEmpBean e where e.dept.name is null";
            Query q = em.createQuery(qStr);

//            TEST151; select e.dept from XMLEmpBean e where e.dept.name is null
//                            TEST151; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST152; select e.dept from XMLEmpBean e where e.name = 'john' and  e.name = 'ahmad'
    public void testLoop152(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept from XMLEmpBean e where e.name = 'john' and  e.name = 'ahmad' ";
            Query q = em.createQuery(qStr);

//            TEST152; select e.dept from XMLEmpBean e where e.name = 'john' and  e.name = 'ahmad'
//                            TEST152; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST153; select e.dept from XMLEmpBean e where e.name is null
    public void testLoop153(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept from XMLEmpBean e where e.name is null";
            Query q = em.createQuery(qStr);

//            TEST153; select e.dept from XMLEmpBean e where e.name is null
//            TEST153; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//   TEST154; select e.dept from XMLEmpBean e where e.salary = 1000 and e.name = 'Bijan'
    public void testLoop154(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept from XMLEmpBean e where e.salary = 1000 and e.name = 'Bijan'";
            Query q = em.createQuery(qStr);

//            TEST154; select e.dept from XMLEmpBean e where e.salary = 1000 and e.name = 'Bijan'
//                            TEST154; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST155; select e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and e.salary = p.budget
    public void testLoop155(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and e.salary = p.budget";
            Query q = em.createQuery(qStr);

//           TEST155; select e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and e.salary = p.budget
//                           TEST155; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST156; select e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and e.salary = p.budget and NOT(e.dept.name is null)
    public void testLoop156(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and e.salary = p.budget and NOT(e.dept.name is null)";
            Query q = em.createQuery(qStr);

//            TEST156; select e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and e.salary = p.budget and NOT(e.dept.name is null)
//                            TEST156; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST157; select e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and e.salary = p.budget and e.dept.name is not null
    public void testLoop157(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and e.salary = p.budget and e.dept.name is not null ";
            Query q = em.createQuery(qStr);

//            TEST157; select e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and e.salary = p.budget and e.dept.name is not null
//                            TEST157; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST158; select e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and e.salary = p.budget and e.dept.name is null
    public void testLoop158(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and e.salary = p.budget and e.dept.name is null";
            Query q = em.createQuery(qStr);

//            TEST158; select e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and e.salary = p.budget and e.dept.name is null
//                            TEST158; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST159; select e.dept from XMLEmpBean e, in(e.dept)d where d.no > 0
    public void testLoop159(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept from XMLEmpBean e, in(e.dept)d where d.no > 0 ";
            Query q = em.createQuery(qStr);

//            TEST159; select e.dept from XMLEmpBean e, in(e.dept)d where d.no > 0
//            DeptBean
//            ~~~~~~~~
//             [100]
//             [100]
//             [200]
//             [200]
//             [210]
//             [210]
//             [210]
//             [210]
//             [220]
//             TEST159; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { null, dep100, dep100, dep200, dep200, dep210, dep210, dep210, dep210, dep220, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST160; select e.dept, e.name from XMLEmpBean e
    public void testLoop160(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept, e.name from XMLEmpBean e ";
            Query q = em.createQuery(qStr);

//            TEST160; select e.dept, e.name from XMLEmpBean e
//            DeptBean    e.name
//            ~~~~~~~~ ~~~~~~~~~~~~
//              null   Catalina Wei
//             [100]      ahmad
//             [100]   Tom Rayburn
//             [200]      george
//             [200]      minmei
//             [210]      andrew
//             [210]      david
//             [210]      harry
//             [210]     charlene
//             [220]      ritika
//             TEST160; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { null, "Catalina Wei" },
                                 new Object[] { dep100, "ahmad" },
                                 new Object[] { dep100, "Tom Rayburn" },
                                 new Object[] { dep200, "george" },
                                 new Object[] { dep200, "minmei" },
                                 new Object[] { dep210, "andrew" },
                                 new Object[] { dep210, "david" },
                                 new Object[] { dep210, "harry" },
                                 new Object[] { dep210, "charlene" },
                                 new Object[] { dep220, "ritika" },
            };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST161; select e.dept.mgr from XMLEmpBean e
    public void testLoop161(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.mgr from XMLEmpBean e";
            Query q = em.createQuery(qStr);

//            TEST161; select e.dept.mgr from XMLEmpBean e
//            EmpBean
//            ~~~~~~~
//              [3]
//              [3]
//              [3]
//              [3]
//              [4]
//              [8]
//              [8]
//             [10]
//             [10]
//             TEST161; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { emp3, emp3, emp3, emp3, emp4, emp8, emp8, emp10, emp10, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST162; select e.dept.mgr from XMLEmpBean e where NOT(e.dept.name is not null)
    public void testLoop162(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.mgr from XMLEmpBean e where NOT(e.dept.name is not null)";
            Query q = em.createQuery(qStr);

//            TEST162; select e.dept.mgr from XMLEmpBean e where NOT(e.dept.name is not null)
//            TEST162; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST163; select e.dept.mgr from XMLEmpBean e where NOT(e.dept.name is null)
    public void testLoop163(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.mgr from XMLEmpBean e where NOT(e.dept.name is null)";
            Query q = em.createQuery(qStr);

//            TEST163; select e.dept.mgr from XMLEmpBean e where NOT(e.dept.name is null)
//            EmpBean
//            ~~~~~~~
//              [3]
//              [3]
//              [3]
//              [3]
//              [4]
//              [8]
//              [8]
//             [10]
//             [10]
//             TEST163; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { emp3, emp3, emp3, emp3, emp4, emp8, emp8, emp10, emp10, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST164; select e.dept.mgr from XMLEmpBean e where e.dept.mgr.name = 'Dave' or e.empid > 0
    public void testLoop164(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.mgr from XMLEmpBean e where e.dept.mgr.name = 'Dave' or e.empid > 0";
            Query q = em.createQuery(qStr);

//            TEST164; select e.dept.mgr from XMLEmpBean e where e.dept.mgr.name = 'Dave' or e.empid > 0
//                            EmpBean
//                            ~~~~~~~
//                              [3]
//                              [3]
//                              [3]
//                              [3]
//                              [4]
//                              [8]
//                              [8]
//                             [10]
//                             [10]
//                             TEST164; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { emp3, emp3, emp3, emp3, emp4, emp8, emp8, emp10, emp10, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST165; select e.dept.mgr from XMLEmpBean e where e.dept.mgr.name = 'John'
    public void testLoop165(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.mgr from XMLEmpBean e where e.dept.mgr.name = 'John'";
            Query q = em.createQuery(qStr);

//            TEST165; select e.dept.mgr from XMLEmpBean e where e.dept.mgr.name = 'John'
//                            TEST165; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST166; select e.dept.mgr from XMLEmpBean e where e.dept.mgr.name = 'John' and e.salary = 20
    public void testLoop166(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.mgr from XMLEmpBean e where e.dept.mgr.name = 'John' and e.salary = 20";
            Query q = em.createQuery(qStr);

//            TEST166; select e.dept.mgr from XMLEmpBean e where e.dept.mgr.name = 'John' and e.salary = 20
//                            TEST166; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST167; select e.dept.mgr from XMLEmpBean e where e.dept.mgr.name is null
    public void testLoop167(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.mgr from XMLEmpBean e where e.dept.mgr.name is null";
            Query q = em.createQuery(qStr);

//            TEST167; select e.dept.mgr from XMLEmpBean e where e.dept.mgr.name is null
//            TEST167; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST168; select e.dept.mgr from XMLEmpBean e where e.dept.name = 'Ahmaddept'
    public void testLoop168(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.mgr from XMLEmpBean e where e.dept.name = 'Ahmaddept'";
            Query q = em.createQuery(qStr);

//            TEST168; select e.dept.mgr from XMLEmpBean e where e.dept.name = 'Ahmaddept'
//                            TEST168; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST169; select e.dept.mgr from XMLEmpBean e where e.dept.name = 'davedept' or e.empid > 0
    public void testLoop169(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.mgr from XMLEmpBean e where e.dept.name = 'davedept' or e.empid > 0";
            Query q = em.createQuery(qStr);

//            TEST169; select e.dept.mgr from XMLEmpBean e where e.dept.name = 'davedept' or e.empid > 0
//                            EmpBean
//                            ~~~~~~~
//                              [3]
//                              [3]
//                              [3]
//                              [3]
//                              [4]
//                              [8]
//                              [8]
//                             [10]
//                             [10]
//                             TEST169; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { emp3, emp3, emp3, emp3, emp4, emp8, emp8, emp10, emp10, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST170; select e.dept.mgr from XMLEmpBean e where e.dept.name = 'dmqa' and e.salary = 20
    public void testLoop170(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.mgr from XMLEmpBean e where e.dept.name = 'dmqa' and e.salary = 20  ";
            Query q = em.createQuery(qStr);

//            TEST170; select e.dept.mgr from XMLEmpBean e where e.dept.name = 'dmqa' and e.salary = 20
//                            TEST170; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST171; select e.dept.mgr from XMLEmpBean e where e.dept.name is not null
    public void testLoop171(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.mgr from XMLEmpBean e where e.dept.name is not null";
            Query q = em.createQuery(qStr);

//            TEST171; select e.dept.mgr from XMLEmpBean e where e.dept.name is not null
//            EmpBean
//            ~~~~~~~
//              [3]
//              [3]
//              [3]
//              [3]
//              [4]
//              [8]
//              [8]
//             [10]
//             [10]
//             TEST171; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { emp3, emp3, emp3, emp3, emp4, emp8, emp8, emp10, emp10, };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST172; select e.dept.mgr from XMLEmpBean e where e.dept.name is null
    public void testLoop172(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.mgr from XMLEmpBean e where e.dept.name is null";
            Query q = em.createQuery(qStr);

//            TEST172; select e.dept.mgr from XMLEmpBean e where e.dept.name is null
//            TEST172; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST173; select e.dept.mgr from XMLEmpBean e where e.salary = 1000 and e.name = 'Steve'
    public void testLoop173(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.mgr from XMLEmpBean e where e.salary = 1000 and e.name = 'Steve'";
            Query q = em.createQuery(qStr);

//            TEST173; select e.dept.mgr from XMLEmpBean e where e.salary = 1000 and e.name = 'Steve'
//            TEST173; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST174; select e.dept.mgr from XMLEmpBean e where e.salary = 1000 or e.name = 'Ahmad'
    public void testLoop174(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        final String dbProductName = (testProps == null) ? "UNKNOWN" : ((testProps.get("dbProductName") == null) ? "UNKNOWN" : (String) testProps.get("dbProductName"));

        // TODO: This test doesn't pass on SQLServer on Eclipselink, its output does not match the answer file
//        Validate Output for test "JULoopQueryAnoTest.testLoop174":
//            SQL: select e.dept.mgr from XMLEmpBean e where e.salary = 1000 or e.name = 'Ahmad'
//            Expected Output:
//
//            Generated Results:
//              0  EmpBean [empid=10, name=Catalina Wei]
        if (DatabaseVendor.checkDBProductName(dbProductName, DatabaseVendor.SQLSERVER)) {
            System.out.println("Test has been excluded from SQLServer.");
            return;
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.mgr from XMLEmpBean e where e.salary = 1000 or e.name = 'Ahmad'";
            Query q = em.createQuery(qStr);

//            TEST174; select e.dept.mgr from XMLEmpBean e where e.salary = 1000 or e.name = 'Ahmad'
//            TEST174; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST175; select e.dept.mgr from XMLEmpBean e, ProjectBean p where e.name = 'john' and p.budget = e.salary
    public void testLoop175(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.mgr from XMLEmpBean e, ProjectBean p where e.name = 'john' and p.budget = e.salary ";
            Query q = em.createQuery(qStr);

//            TEST175; select e.dept.mgr from XMLEmpBean e, ProjectBean p where e.name = 'john' and p.budget = e.salary
//            TEST175; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST176; select e.dept.mgr from XMLEmpBean e, in(e.dept.mgr.tasks)p where p.taskid > 0
    public void testLoop176(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.mgr from XMLEmpBean e, in(e.dept.mgr.tasks)p where p.taskid > 0 ";
            Query q = em.createQuery(qStr);

//            TEST176; select e.dept.mgr from XMLEmpBean e, in(e.dept.mgr.tasks)p where p.taskid > 0
//            TEST176; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST177; select e.dept.mgr.name from XMLEmpBean e
    public void testLoop177(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.mgr.name from XMLEmpBean e ";
            Query q = em.createQuery(qStr);

//            TEST177; select e.dept.mgr.name from XMLEmpBean e
//            e.dept.mgr.name
//            ~~~~~~~~~~~~~~~
//                george
//                minmei
//                minmei
//                minmei
//                minmei
//              Tom Rayburn
//              Tom Rayburn
//             Catalina Wei
//             Catalina Wei
//             TEST177; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { "george", "minmei", "minmei", "minmei", "minmei", "Tom Rayburn", "Tom Rayburn", "Catalina Wei", "Catalina Wei" };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST178; select e.dept.no, e.salary, e.bonus from XMLEmpBean e order by e.dept.no, e.salary
    public void testLoop178(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.no, e.salary, e.bonus from XMLEmpBean e order by e.dept.no, e.salary";
            Query q = em.createQuery(qStr);

//            TEST178; select e.dept.no, e.salary, e.bonus from XMLEmpBean e order by e.dept.no, e.salary
//            e.dept.no e.salary e.bonus
//            ~~~~~~~~~ ~~~~~~~~ ~~~~~~~
//               100      0.0      0.0
//               100      0.0      0.0
//               200      0.0      0.0
//               200      15.5     0.0
//               210      0.0      0.0
//               210      0.0      0.0
//               210      12.1     0.0
//               210      13.1     0.0
//               220      0.0      0.0
//             TEST178; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 100, 0.0, 0.0 },
                                 new Object[] { 100, 0.0, 0.0 },
                                 new Object[] { 200, 0.0, 0.0 },
                                 new Object[] { 200, 15.5, 0.0 },
                                 new Object[] { 210, 0.0, 0.0 },
                                 new Object[] { 210, 0.0, 0.0 },
                                 new Object[] { 210, 12.1, 0.0 },
                                 new Object[] { 210, 13.1, 0.0 },
                                 new Object[] { 220, 0.0, 0.0 },
            };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST179; select e.empid from XMLDeptBean d join d.emps e
    public void testLoop179(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid from XMLDeptBean d join d.emps e ";
            Query q = em.createQuery(qStr);

//            TEST179; select e.empid from XMLDeptBean d join d.emps e
//            e.empid
//            ~~~~~~~
//               1
//               2
//               3
//               4
//               5
//               6
//               7
//               8
//               9
//             TEST179; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { 1, 2, 3, 4, 5, 6, 7, 8, 9 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST180; select e.empid from XMLEmpBean e join e.dept d
    public void testLoop180(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid from XMLEmpBean e join e.dept d ";
            Query q = em.createQuery(qStr);

//            TEST180; select e.empid from XMLEmpBean e join e.dept d
//            e.empid
//            ~~~~~~~
//               1
//               2
//               3
//               4
//               5
//               6
//               7
//               8
//               9
//             TEST180; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { 1, 2, 3, 4, 5, 6, 7, 8, 9 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST181; select e.empid from XMLEmpBean e where e.dept = e.dept.mgr.dept
    public void testLoop181(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid from XMLEmpBean e where e.dept = e.dept.mgr.dept ";
            Query q = em.createQuery(qStr);

//          TEST181; select e.empid from XMLEmpBean e where e.dept = e.dept.mgr.dept
//            TEST181; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST182; select e.empid, d.name from XMLEmpBean e, DeptBean d where e.empid=d.no
    public void testLoop182(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, d.name from XMLEmpBean e, DeptBean d where e.empid=d.no  ";
            Query q = em.createQuery(qStr);

//          TEST182; select e.empid, d.name from XMLEmpBean e, DeptBean d where e.empid=d.no
//            TEST182; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST183; select e.empid, e.bonus, e.salary, e.bonus, e.salary, e.bonus from XMLEmpBean e
    public void testLoop183(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, e.bonus, e.salary, e.bonus, e.salary, e.bonus from XMLEmpBean e ";
            Query q = em.createQuery(qStr);

//            TEST183; select e.empid, e.bonus, e.salary, e.bonus, e.salary, e.bonus from XMLEmpBean e
//            e.empid e.bonus e.salary e.bonus e.salary e.bonus
//            ~~~~~~~ ~~~~~~~ ~~~~~~~~ ~~~~~~~ ~~~~~~~~ ~~~~~~~
//               1      0.0     12.1     0.0     12.1     0.0
//               2      0.0     13.1     0.0     13.1     0.0
//               3      0.0     15.5     0.0     15.5     0.0
//               4      0.0     0.0      0.0     0.0      0.0
//               5      0.0     0.0      0.0     0.0      0.0
//               6      0.0     0.0      0.0     0.0      0.0
//               7      0.0     0.0      0.0     0.0      0.0
//               8      0.0     0.0      0.0     0.0      0.0
//               9      0.0     0.0      0.0     0.0      0.0
//              10      0.0     0.0      0.0     0.0      0.0
//             TEST183; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, 0.0, 12.1, 0.0, 12.1, 0.0 },
                                 new Object[] { 2, 0.0, 13.1, 0.0, 13.1, 0.0 },
                                 new Object[] { 3, 0.0, 15.5, 0.0, 15.5, 0.0 },
                                 new Object[] { 4, 0.0, 0.0, 0.0, 0.0, 0.0 },
                                 new Object[] { 5, 0.0, 0.0, 0.0, 0.0, 0.0 },
                                 new Object[] { 6, 0.0, 0.0, 0.0, 0.0, 0.0 },
                                 new Object[] { 7, 0.0, 0.0, 0.0, 0.0, 0.0 },
                                 new Object[] { 8, 0.0, 0.0, 0.0, 0.0, 0.0 },
                                 new Object[] { 9, 0.0, 0.0, 0.0, 0.0, 0.0 },
                                 new Object[] { 10, 0.0, 0.0, 0.0, 0.0, 0.0 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST184; select e.empid, e.bonus, e.salary, e.isManager from XMLEmpBean e
    public void testLoop184(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, e.bonus, e.salary, e.isManager from XMLEmpBean e";
            Query q = em.createQuery(qStr);

//            TEST184; select e.empid, e.bonus, e.salary, e.isManager from XMLEmpBean e
//            e.empid e.bonus e.salary e.isManager
//            ~~~~~~~ ~~~~~~~ ~~~~~~~~ ~~~~~~~~~~~
//               1      0.0     12.1      false
//               2      0.0     13.1      false
//               3      0.0     15.5      true
//               4      0.0     0.0       true
//               5      0.0     0.0       true
//               6      0.0     0.0       true
//               7      0.0     0.0       true
//               8      0.0     0.0       true
//               9      0.0     0.0       true
//              10      0.0     0.0       true
//             TEST184; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, 0.0, 12.1, false },
                                 new Object[] { 2, 0.0, 13.1, false },
                                 new Object[] { 3, 0.0, 15.5, true },
                                 new Object[] { 4, 0.0, 0.0, true },
                                 new Object[] { 5, 0.0, 0.0, true },
                                 new Object[] { 6, 0.0, 0.0, true },
                                 new Object[] { 7, 0.0, 0.0, true },
                                 new Object[] { 8, 0.0, 0.0, true },
                                 new Object[] { 9, 0.0, 0.0, true },
                                 new Object[] { 10, 0.0, 0.0, true },

            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST185; select e.empid, e.dept.no, e.dept.name from XMLEmpBean e where e.empid<9  order by e.empid
    public void testLoop185(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, e.dept.no, e.dept.name from XMLEmpBean e where e.empid<9  order by e.empid ";
            Query q = em.createQuery(qStr);

//            TEST185; select e.empid, e.dept.no, e.dept.name from XMLEmpBean e where e.empid<9  order by e.empid
//            e.empid e.dept.no e.dept.name
//            ~~~~~~~ ~~~~~~~~~ ~~~~~~~~~~~
//               1       210    Development
//               2       210    Development
//               3       200       Admin
//               4       200       Admin
//               5       220      Service
//               6       100        CEO
//               7       210    Development
//               8       100        CEO
//             TEST185; 8 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, 210, "Development" },
                                 new Object[] { 2, 210, "Development" },
                                 new Object[] { 3, 200, "Admin" },
                                 new Object[] { 4, 200, "Admin" },
                                 new Object[] { 5, 220, "Service" },
                                 new Object[] { 6, 100, "CEO" },
                                 new Object[] { 7, 210, "Development" },
                                 new Object[] { 8, 100, "CEO" },
            };
            validateQueryResult(testName, qStr, rList, targets, true);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST186; select e.empid, e.hireDate,  e.dept.no,  e.salary from XMLEmpBean e where e.empid<9  order by e.empid
    public void testLoop186(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, e.hireDate,  e.dept.no,  e.salary from XMLEmpBean e where e.empid<9  order by e.empid  ";
            Query q = em.createQuery(qStr);

//            TEST186; select e.empid, e.hireDate,  e.dept.no,  e.salary from XMLEmpBean e where e.empid<9  order by e.empid
//            e.empid e.hireDate e.dept.no e.salary
//            ~~~~~~~ ~~~~~~~~~~ ~~~~~~~~~ ~~~~~~~~
//               1    1969-12-31    210      12.1
//               2    1969-12-31    210      13.1
//               3    1969-12-31    200      15.5
//               4    1969-12-31    200      0.0
//               5    1969-12-31    220      0.0
//               6    1969-12-31    100      0.0
//               7    1969-12-31    210      0.0
//               8    1969-12-31    100      0.0
//             TEST186; 8 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, hireDate, 210, 12.1 },
                                 new Object[] { 2, hireDate, 210, 13.1 },
                                 new Object[] { 3, hireDate, 200, 15.5 },
                                 new Object[] { 4, hireDate, 200, 0.0 },
                                 new Object[] { 5, hireDate, 220, 0.0 },
                                 new Object[] { 6, hireDate, 100, 0.0 },
                                 new Object[] { 7, hireDate, 210, 0.0 },
                                 new Object[] { 8, hireDate, 100, 0.0 },
            };
            validateQueryResult(testName, qStr, rList, targets, true);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST187; select e.empid, e.name from XMLEmpBean e where e.dept.name='Sales' and e.salary > 1.0
    public void testLoop187(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, e.name from XMLEmpBean e where e.dept.name='Sales' and e.salary > 1.0 ";
            Query q = em.createQuery(qStr);

//          TEST187; select e.empid, e.name from XMLEmpBean e where e.dept.name='Sales' and e.salary > 1.0
//            TEST187; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST188; select e.empid, e.name, d.name from XMLDeptBean d, EmpBean e where e.empid=d.no
    public void testLoop188(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, e.name, d.name from XMLDeptBean d, EmpBean e where e.empid=d.no ";
            Query q = em.createQuery(qStr);

//          TEST188; select e.empid, e.name, d.name from XMLDeptBean d, EmpBean e where e.empid=d.no
//            TEST188; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST189; select e.empid, e.name, e.dept.no from XMLEmpBean e where e.empid<10 OR e.empid>=10
    public void testLoop189(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, e.name, e.dept.no from XMLEmpBean e where e.empid<10 OR e.empid>=10";
            Query q = em.createQuery(qStr);

//            TEST189; select e.empid, e.name, e.dept.no from XMLEmpBean e where e.empid<10 OR e.empid>=10
//                            e.empid   e.name    e.dept.no
//                            ~~~~~~~ ~~~~~~~~~~~ ~~~~~~~~~
//                               1       david       210
//                               2      andrew       210
//                               3      minmei       200
//                               4      george       200
//                               5      ritika       220
//                               6       ahmad       100
//                               7     charlene      210
//                               8    Tom Rayburn    100
//                               9       harry       210
//                             TEST189; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, "david", 210 },
                                 new Object[] { 2, "andrew", 210 },
                                 new Object[] { 3, "minmei", 200 },
                                 new Object[] { 4, "george", 200 },
                                 new Object[] { 5, "ritika", 220 },
                                 new Object[] { 6, "ahmad", 100 },
                                 new Object[] { 7, "charlene", 210 },
                                 new Object[] { 8, "Tom Rayburn", 100 },
                                 new Object[] { 9, "harry", 210 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST190; select e.empid, e.name, e.dept.no from XMLEmpBean e where e.empid<10 and e.empid>=10
    public void testLoop190(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, e.name, e.dept.no from XMLEmpBean e where e.empid<10 and e.empid>=10 ";
            Query q = em.createQuery(qStr);

//          TEST190; select e.empid, e.name, e.dept.no from XMLEmpBean e where e.empid<10 and e.empid>=10
//            TEST190; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST191; select e.empid, e.name, e.dept.no from XMLEmpBean e where e.empid<11
    public void testLoop191(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, e.name, e.dept.no from XMLEmpBean e where e.empid<11 ";
            Query q = em.createQuery(qStr);

//            TEST191; select e.empid, e.name, e.dept.no from XMLEmpBean e where e.empid<11
//            e.empid   e.name    e.dept.no
//            ~~~~~~~ ~~~~~~~~~~~ ~~~~~~~~~
//               1       david       210
//               2      andrew       210
//               3      minmei       200
//               4      george       200
//               5      ritika       220
//               6       ahmad       100
//               7     charlene      210
//               8    Tom Rayburn    100
//               9       harry       210
//             TEST191; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, "david", 210 },
                                 new Object[] { 2, "andrew", 210 },
                                 new Object[] { 3, "minmei", 200 },
                                 new Object[] { 4, "george", 200 },
                                 new Object[] { 5, "ritika", 220 },
                                 new Object[] { 6, "ahmad", 100 },
                                 new Object[] { 7, "charlene", 210 },
                                 new Object[] { 8, "Tom Rayburn", 100 },
                                 new Object[] { 9, "harry", 210 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST192; select e.empid, e.name, e.dept.no from XMLEmpBean e where e.empid<3
    public void testLoop192(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, e.name, e.dept.no from XMLEmpBean e where e.empid<3 ";
            Query q = em.createQuery(qStr);

//            TEST192; select e.empid, e.name, e.dept.no from XMLEmpBean e where e.empid<3
//            e.empid e.name e.dept.no
//            ~~~~~~~ ~~~~~~ ~~~~~~~~~
//               1    david     210
//               2    andrew    210
//             TEST192; 2 tuples

            List rList = q.getResultList();

            Object[] targets = {
                                 new Object[] { 1, "david", 210 },
                                 new Object[] { 2, "andrew", 210 },

            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST193; select e.empid, e.name, e.dept.no from XMLEmpBean e where e.empid>3
    public void testLoop193(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, e.name, e.dept.no from XMLEmpBean e where e.empid>3 ";
            Query q = em.createQuery(qStr);

//            TEST193; select e.empid, e.name, e.dept.no from XMLEmpBean e where e.empid>3
//            e.empid   e.name    e.dept.no
//            ~~~~~~~ ~~~~~~~~~~~ ~~~~~~~~~
//               4      george       200
//               5      ritika       220
//               6       ahmad       100
//               7     charlene      210
//               8    Tom Rayburn    100
//               9       harry       210
//             TEST193; 6 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 4, "george", 200 },
                                 new Object[] { 5, "ritika", 220 },
                                 new Object[] { 6, "ahmad", 100 },
                                 new Object[] { 7, "charlene", 210 },
                                 new Object[] { 8, "Tom Rayburn", 100 },
                                 new Object[] { 9, "harry", 210 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST194; select e.empid, e.name, e.salary  from XMLEmpBean e group by e.salary, e.name, e.empid  order by e.salary
    public void testLoop194(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, e.name, e.salary  from XMLEmpBean e group by e.salary, e.name, e.empid  order by e.salary ";
            Query q = em.createQuery(qStr);

//            TEST194; select e.empid, e.name, e.salary  from XMLEmpBean e group by e.salary, e.name, e.empid  order by e.salary
//            e.empid    e.name    e.salary
//            ~~~~~~~ ~~~~~~~~~~~~ ~~~~~~~~
//               1       david       12.1
//               2       andrew      13.1
//               3       minmei      15.5
//               4       george      0.0
//               5       ritika      0.0
//               6       ahmad       0.0
//               7      charlene     0.0
//               8    Tom Rayburn    0.0
//               9       harry       0.0
//              10    Catalina Wei   0.0
//             TEST194; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, "david", 12.1 },
                                 new Object[] { 2, "andrew", 13.1 },
                                 new Object[] { 3, "minmei", 15.5 },
                                 new Object[] { 4, "george", 0.0 },
                                 new Object[] { 5, "ritika", 0.0 },
                                 new Object[] { 6, "ahmad", 0.0 },
                                 new Object[] { 7, "charlene", 0.0 },
                                 new Object[] { 8, "Tom Rayburn", 0.0 },
                                 new Object[] { 9, "harry", 0.0 },
                                 new Object[] { 10, "Catalina Wei", 0.0 },
            };
            validateQueryResult(testName, qStr, rList, targets, true);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST195; select e.empid, e.name, e.salary from XMLEmpBean e group by e.name, e.empid, e.salary  having e.empid = 1 order by e.salary
    public void testLoop195(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, e.name, e.salary from XMLEmpBean e group by e.name, e.empid, e.salary  having e.empid = 1 order by e.salary ";
            Query q = em.createQuery(qStr);

//            TEST195; select e.empid, e.name, e.salary from XMLEmpBean e group by e.name, e.empid, e.salary  having e.empid = 1 order by e.salary
//                            e.empid e.name e.salary
//                            ~~~~~~~ ~~~~~~ ~~~~~~~~
//                               1    david    12.1
//                             TEST195; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, "david", 12.1 }, };
            validateQueryResult(testName, qStr, rList, targets, true);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST196; select e.empid,e.hireTime  from XMLEmpBean e where e.empid = 0
    public void testLoop196(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid,e.hireTime  from XMLEmpBean e where e.empid = 0";
            Query q = em.createQuery(qStr);

//          TEST196; select e.empid,e.hireTime  from XMLEmpBean e where e.empid = 0
//            TEST196; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST197; select e.empid,e.name,e.salary,e.bonus,e.hireDate,e.hireTime,e.hireTimestamp,e.isManager,e.dept from XMLEmpBean e
    public void testLoop197(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid,e.name,e.salary,e.bonus,e.hireDate,e.hireTime,e.hireTimestamp,e.isManager,e.dept from XMLEmpBean e ";
            Query q = em.createQuery(qStr);

//            TEST197; select e.empid,e.name,e.salary,e.bonus,e.hireDate,e.hireTime,e.hireTimestamp,e.isManager,e.dept from XMLEmpBean e
//            e.empid    e.name    e.salary e.bonus e.hireDate e.hireTime    e.hireTimestamp    e.isManager DeptBean
//            ~~~~~~~ ~~~~~~~~~~~~ ~~~~~~~~ ~~~~~~~ ~~~~~~~~~~ ~~~~~~~~~~ ~~~~~~~~~~~~~~~~~~~~~ ~~~~~~~~~~~ ~~~~~~~~
//               1       david       12.1     0.0   1969-12-31  16:00:00  1969-12-31 16:00:00.0    false     [210]
//               2       andrew      13.1     0.0   1969-12-31  16:00:00  1969-12-31 16:00:00.0    false     [210]
//               3       minmei      15.5     0.0   1969-12-31  16:00:00  1969-12-31 16:00:00.0    true      [200]
//               4       george      0.0      0.0   1969-12-31  16:00:00  1969-12-31 16:00:00.0    true      [200]
//               5       ritika      0.0      0.0   1969-12-31  16:00:00  1969-12-31 16:00:00.0    true      [220]
//               6       ahmad       0.0      0.0   1969-12-31  16:00:00  1969-12-31 16:00:00.0    true      [100]
//               7      charlene     0.0      0.0   1969-12-31  16:00:00  1969-12-31 16:00:00.0    true      [210]
//               8    Tom Rayburn    0.0      0.0   1969-12-31  16:00:00  1969-12-31 16:00:00.0    true      [100]
//               9       harry       0.0      0.0   1969-12-31  16:00:00  1969-12-31 16:00:00.0    true      [210]
//              10    Catalina Wei   0.0      0.0   1969-12-31  16:00:00  1969-12-31 16:00:00.0    true       null
//             TEST197; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, "david", 12.1, 0.0, hireDate, hireTime, hireTimestamp, false, dep210 },
                                 new Object[] { 2, "andrew", 13.1, 0.0, hireDate, hireTime, hireTimestamp, false, dep210 },
                                 new Object[] { 3, "minmei", 15.5, 0.0, hireDate, hireTime, hireTimestamp, true, dep200 },
                                 new Object[] { 4, "george", 0.0, 0.0, hireDate, hireTime, hireTimestamp, true, dep200 },
                                 new Object[] { 5, "ritika", 0.0, 0.0, hireDate, hireTime, hireTimestamp, true, dep220 },
                                 new Object[] { 6, "ahmad", 0.0, 0.0, hireDate, hireTime, hireTimestamp, true, dep100 },
                                 new Object[] { 7, "charlene", 0.0, 0.0, hireDate, hireTime, hireTimestamp, true, dep210 },
                                 new Object[] { 8, "Tom Rayburn", 0.0, 0.0, hireDate, hireTime, hireTimestamp, true, dep100 },
                                 new Object[] { 9, "harry", 0.0, 0.0, hireDate, hireTime, hireTimestamp, true, dep210 },
                                 new Object[] { 10, "Catalina Wei", 0.0, 0.0, hireDate, hireTime, hireTimestamp, true, null },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST198; select e.name from XMLDeptBean d join d.emps e where d.name = 'Sales' and e.empid >  5
    public void testLoop198(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLDeptBean d join d.emps e where d.name = 'Sales' and e.empid >  5 ";
            Query q = em.createQuery(qStr);

//         TEST198; select e.name from XMLDeptBean d join d.emps e where d.name = 'Sales' and e.empid >  5
//            TEST198; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST199; select e.name from XMLDeptBean d left join d.emps e where d.name = 'Sales' and e.empid >  0
    public void testLoop199(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLDeptBean d left join d.emps e where d.name = 'Sales' and e.empid >  0 ";
            Query q = em.createQuery(qStr);

//         TEST199; select e.name from XMLDeptBean d left join d.emps e where d.name = 'Sales' and e.empid >  0
//            TEST199; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST200; select e.name from XMLDeptBean d left join d.emps e where d.name = 'Sales' and e.salary >  0
    public void testLoop200(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLDeptBean d left join d.emps e where d.name = 'Sales' and e.salary >  0";
            Query q = em.createQuery(qStr);

//         TEST200; select e.name from XMLDeptBean d left join d.emps e where d.name = 'Sales' and e.salary >  0
//            TEST200; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST201; select e.name from XMLEmpBean e join e.dept d where e.name = 'david' and d.no >  200
    public void testLoop201(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e join e.dept d where e.name = 'david' and d.no >  200";
            Query q = em.createQuery(qStr);

//            TEST201; select e.name from XMLEmpBean e join e.dept d where e.name = 'david' and d.no >  200
//                            e.name
//                            ~~~~~~
//                            david
//                             TEST201; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { "david" };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST202; select e.name from XMLEmpBean e order by e.salary desc, e.name
    public void testLoop202(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e order by e.salary desc, e.name";
            Query q = em.createQuery(qStr);

//            TEST202; select e.name from XMLEmpBean e order by e.salary desc, e.name
//            e.name
//         ~~~~~~~~~~~~
//            ahmad
//            andrew
//            david
//            george
//            harry
//            minmei
//            ritika
//           charlene
//         Catalina Wei
//         Tom Rayburn
//          TEST202; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { "ahmad",
                                 "andrew",
                                 "david",
                                 "george",
                                 "harry",
                                 "minmei",
                                 "ritika",
                                 "charlene",
                                 "Catalina Wei",
                                 "Tom Rayburn",
            };
            validateQueryResult(testName, qStr, rList, targets, true);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST203; select e.name from XMLEmpBean e where e.hireDate > 0
    public void testLoop203(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        JPAPersistenceProvider provider = JPAPersistenceProvider.resolveJPAPersistenceProvider(jpaResource);

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.hireDate > 0";

//            e.name
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//Filter invalid. Cannot compare field hireDate of type java.sql.Date to value of type java.lang.Long. Numeric comparisons must be between numeric types only. To enable such comparisons for backwards-compatibility, add "QuotedNumbersInQueries=true" to the org.apache.openjpa.Compatibility setting in your configuration.
//TEST203; 1 tuple

            // Expect an Exception
            try {
                em.createQuery(qStr).getResultList();
                Assert.fail("Expected Exception was not thrown.");
            } catch (RuntimeException e) {
                String eStr = e.getMessage();
                if (JPAPersistenceProvider.ECLIPSELINK.equals(provider)) {
                    String lookFor = "The object [0], of class [class java.lang.Integer], from mapping [org.eclipse.persistence.mappings.DirectToFieldMapping[hireDate-->JPAEmpBean.HIREDATE]] with descriptor [RelationalDescriptor(com.ibm.ws.query.entities.xml.EmpBean --> [DatabaseTable(JPAEmpBean)])], could not be converted to [class java.sql.Date]";
                    System.out.println("eStr = " + eStr);
                    System.out.println("lookFor = " + lookFor);
                    boolean result = eStr.contains(lookFor);
                    Assert.assertTrue("Received unexpected exception cause: " + eStr, result);
                } else if (JPAPersistenceProvider.OPENJPA.equals(provider)) {
                    Assert.fail();
                } else if (JPAPersistenceProvider.HIBERNATE.equals(provider)) {
                    Assert.fail();
                } else {
                    Assert.fail();
                }
            }

            em.clear();

        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST204; select e.name from XMLEmpBean e where e.hireTime > 0
    public void testLoop204(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        JPAPersistenceProvider provider = JPAPersistenceProvider.resolveJPAPersistenceProvider(jpaResource);

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.hireTime > 0";

//            TEST204; select e.name from XMLEmpBean e where e.hireTime > 0
//            e.name
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//Filter invalid. Cannot compare field hireTime of type java.sql.Time to value of type java.lang.Long. Numeric comparisons must be between numeric types only. To enable such comparisons for backwards-compatibility, add "QuotedNumbersInQueries=true" to the org.apache.openjpa.Compatibility setting in your configuration.
//TEST204; 1 tuple

            // Expect an Exception
            try {
                em.createQuery(qStr).getResultList();
                Assert.fail("Expected Exception was not thrown.");
            } catch (RuntimeException e) {
                String eStr = e.getMessage();
                if (JPAPersistenceProvider.ECLIPSELINK.equals(provider)) {
                    String lookFor = "The object [0], of class [class java.lang.Integer], from mapping [org.eclipse.persistence.mappings.DirectToFieldMapping[hireTime-->JPAEmpBean.HIRETIME]] with descriptor [RelationalDescriptor(com.ibm.ws.query.entities.xml.EmpBean --> [DatabaseTable(JPAEmpBean)])], could not be converted to [class java.sql.Time].";
                    System.out.println("eStr = " + eStr);
                    System.out.println("lookFor = " + lookFor);
                    boolean result = eStr.contains(lookFor);
                    Assert.assertTrue("Received unexpected exception cause: " + eStr, result);
                } else if (JPAPersistenceProvider.OPENJPA.equals(provider)) {
                    Assert.fail();
                } else if (JPAPersistenceProvider.HIBERNATE.equals(provider)) {
                    Assert.fail();
                } else {
                    Assert.fail();
                }
            }

            em.clear();

        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST205; select e.name from XMLEmpBean e where e.hireTimestamp > 0
    public void testLoop205(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        JPAPersistenceProvider provider = JPAPersistenceProvider.resolveJPAPersistenceProvider(jpaResource);

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.hireTimestamp > 0";

//            TEST205; select e.name from XMLEmpBean e where e.hireTimestamp > 0
//            e.name
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//Filter invalid. Cannot compare field hireTimestamp of type java.sql.Timestamp to value of type java.lang.Long. Numeric comparisons must be between numeric types only. To enable such comparisons for backwards-compatibility, add "QuotedNumbersInQueries=true" to the org.apache.openjpa.Compatibility setting in your configuration.
//TEST205; 1 tuple

            // Expect an Exception
            try {
                em.createQuery(qStr).getResultList();
                Assert.fail("Expected Exception was not thrown.");
            } catch (RuntimeException e) {
                String eStr = e.getMessage();
                if (JPAPersistenceProvider.ECLIPSELINK.equals(provider)) {
                    String lookFor = "The object [0], of class [class java.lang.Integer], from mapping [org.eclipse.persistence.mappings.DirectToFieldMapping[hireTimestamp-->JPAEmpBean.HIRETIMESTAMP]] with descriptor [RelationalDescriptor(com.ibm.ws.query.entities.xml.XMLEmpBean --> [DatabaseTable(JPAEmpBean)])], could not be converted to [class java.sql.Timestamp].";
                    System.out.println("eStr = " + eStr);
                    System.out.println("lookFor = " + lookFor);
                    boolean result = eStr.contains(lookFor);
                    Assert.assertTrue("Received unexpected exception cause: " + eStr, result);
                } else if (JPAPersistenceProvider.OPENJPA.equals(provider)) {
                    Assert.fail();
                } else if (JPAPersistenceProvider.HIBERNATE.equals(provider)) {
                    Assert.fail();
                } else {
                    Assert.fail();
                }
            }

            em.clear();

        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST206; select e.name from XMLEmpBean e where e.isManager = TRUE
    public void testLoop206(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.isManager = TRUE ";
            Query q = em.createQuery(qStr);

//            TEST206; select e.name from XMLEmpBean e where e.isManager = TRUE
//                            e.name
//                         ~~~~~~~~~~~~
//                            ahmad
//                            george
//                            harry
//                            minmei
//                            ritika
//                           charlene
//                         Catalina Wei
//                         Tom Rayburn
//                          TEST206; 8 tuples

            List rList = q.getResultList();

            Object[] targets = { "ahmad",
                                 "george",
                                 "harry",
                                 "minmei",
                                 "ritika",
                                 "charlene",
                                 "Catalina Wei",
                                 "Tom Rayburn",
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST207; select e.name from XMLEmpBean e where e.name ='nonexisting'
    public void testLoop207(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.name ='nonexisting'";
            Query q = em.createQuery(qStr);

//            TEST207; select e.name from XMLEmpBean e where e.name ='nonexisting'
//                            TEST207; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST208; select e.name from XMLEmpBean e where e.name='Sales'
    public void testLoop208(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.name='Sales' ";
            Query q = em.createQuery(qStr);

//            TEST208; select e.name from XMLEmpBean e where e.name='Sales'
//                            TEST208; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST209; select e.name from XMLEmpBean e where e.salary =  0.0
    public void testLoop209(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.salary =  0.0 ";
            Query q = em.createQuery(qStr);

//            TEST209; select e.name from XMLEmpBean e where e.salary =  0.0
//                            e.name
//                         ~~~~~~~~~~~~
//                            ahmad
//                            george
//                            harry
//                            ritika
//                           charlene
//                         Catalina Wei
//                         Tom Rayburn
//                          TEST209; 7 tuples

            List rList = q.getResultList();

            Object[] targets = { "ahmad",
                                 "george",
                                 "harry",
                                 "ritika",
                                 "charlene",
                                 "Catalina Wei",
                                 "Tom Rayburn"
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

// TEST210; select e.name from XMLEmpBean e where e.salary >  0.0
    public void testLoop210(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.salary >  0.0 ";
            Query q = em.createQuery(qStr);

//            TEST210; select e.name from XMLEmpBean e where e.salary >  0.0
//            e.name
//            ~~~~~~
//            andrew
//            david
//            minmei
//             TEST210; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { "andrew",
                                 "david",
                                 "minmei"
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST211; select e.name from XMLEmpBean e where e.salary >  0.0 and e.name = 'david'
    public void testLoop211(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.salary >  0.0 and e.name = 'david' ";
            Query q = em.createQuery(qStr);

//            TEST211; select e.name from XMLEmpBean e where e.salary >  0.0 and e.name = 'david'
//                            e.name
//                            ~~~~~~
//                            david
//                             TEST211; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { "david" };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST212; select e.name from XMLEmpBean e where e.salary > 0 and e.dept.mgr.name = 'minmei'
    public void testLoop212(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.salary > 0 and e.dept.mgr.name = 'minmei'";
            Query q = em.createQuery(qStr);

//            TEST212; select e.name from XMLEmpBean e where e.salary > 0 and e.dept.mgr.name = 'minmei'
//                            e.name
//                            ~~~~~~
//                            andrew
//                            david
//                             TEST212; 2 tuples

            List rList = q.getResultList();

            Object[] targets = { "andrew", "david" };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST213; select e.name from XMLEmpBean e where e.salary > 0 and e.dept.no > 0
    public void testLoop213(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.salary > 0 and e.dept.no > 0 ";
            Query q = em.createQuery(qStr);

//            TEST213; select e.name from XMLEmpBean e where e.salary > 0 and e.dept.no > 0
//            e.name
//            ~~~~~~
//            andrew
//            david
//            minmei
//             TEST213; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { "andrew", "david", "minmei" };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST214; select e.name from XMLEmpBean e where e.salary > 0 and e.dept.no > 0.0
    public void testLoop214(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.salary > 0 and e.dept.no > 0.0";
            Query q = em.createQuery(qStr);

//            TEST214; select e.name from XMLEmpBean e where e.salary > 0 and e.dept.no > 0.0
//            e.name
//            ~~~~~~
//            andrew
//            david
//            minmei
//             TEST214; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { "andrew", "david", "minmei" };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST215; select e.name from XMLEmpBean e where e.salary > 0 and e.salary > e.empid
    public void testLoop215(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.salary > 0 and e.salary > e.empid ";
            Query q = em.createQuery(qStr);

//            TEST215; select e.name from XMLEmpBean e where e.salary > 0 and e.salary > e.empid
//            e.name
//            ~~~~~~
//            andrew
//            david
//            minmei
//             TEST215; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { "andrew", "david", "minmei" };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST216; select e.name, d.name from XMLEmpBean as e, DeptBean as d where e.dept = d and d.name = 'Sales' and e.salary >  0.0
    public void testLoop216(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name, d.name from XMLEmpBean as e, DeptBean as d where e.dept = d and d.name = 'Sales' and e.salary >  0.0";
            Query q = em.createQuery(qStr);

//            TEST216; select e.name, d.name from XMLEmpBean as e, DeptBean as d where e.dept = d and d.name = 'Sales' and e.salary >  0.0
//                            TEST216; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST217; select e.name, e.salary from XMLEmpBean e where (not e.isManager = true and not e.empid <= 7)
    public void testLoop217(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name, e.salary from XMLEmpBean e where (not e.isManager = true and not e.empid <= 7)";
            Query q = em.createQuery(qStr);

//            TEST217; select e.name, e.salary from XMLEmpBean e where (not e.isManager = true and not e.empid <= 7)
//            TEST217; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST218; select e.salary from XMLEmpBean e where e.name = 'david' or e.name = 'andrew' or e.name='minmei'
    public void testLoop218(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.salary from XMLEmpBean e where e.name = 'david' or e.name = 'andrew' or e.name='minmei' ";
            Query q = em.createQuery(qStr);

//            TEST218; select e.salary from XMLEmpBean e where e.name = 'david' or e.name = 'andrew' or e.name='minmei'
//                            e.salary
//                            ~~~~~~~~
//                              12.1
//                              13.1
//                              15.5
//                             TEST218; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { 12.1, 13.1, 15.5 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST219; select e.salary, e.bonus from XMLEmpBean e where (e.dept.no <> 1)
    public void testLoop219(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.salary, e.bonus from XMLEmpBean e where (e.dept.no <> 1) ";
            Query q = em.createQuery(qStr);

//            TEST219; select e.salary, e.bonus from XMLEmpBean e where (e.dept.no <> 1)
//            e.salary e.bonus
//            ~~~~~~~~ ~~~~~~~
//              0.0      0.0
//              0.0      0.0
//              0.0      0.0
//              0.0      0.0
//              0.0      0.0
//              0.0      0.0
//              12.1     0.0
//              13.1     0.0
//              15.5     0.0
//             TEST219; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 0.0, 0.0 },
                                 new Object[] { 0.0, 0.0 },
                                 new Object[] { 0.0, 0.0 },
                                 new Object[] { 0.0, 0.0 },
                                 new Object[] { 0.0, 0.0 },
                                 new Object[] { 0.0, 0.0 },
                                 new Object[] { 12.1, 0.0 },
                                 new Object[] { 13.1, 0.0 },
                                 new Object[] { 15.5, 0.0 }
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST220; select f from  DeptBean AS d, IN (d.emps)e, IN (e.dept.reportsTo.emps) f where f.empid = 1 order by f.empid
    public void testLoop220(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select f from  DeptBean AS d, IN (d.emps)e, IN (e.dept.reportsTo.emps) f where f.empid = 1 order by f.empid ";
            Query q = em.createQuery(qStr);

//            TEST220; select f from  DeptBean AS d, IN (d.emps)e, IN (e.dept.reportsTo.emps) f where f.empid = 1 order by f.empid
//                            TEST220; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST221; select f from  DeptBean AS d, IN (d.emps)e, IN (e.dept.reportsTo.emps) f where f.empid > 5  order by f.empid
    public void testLoop221(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select f from  DeptBean AS d, IN (d.emps)e, IN (e.dept.reportsTo.emps) f where f.empid > 5  order by f.empid ";
            Query q = em.createQuery(qStr);

//            TEST221; select f from  DeptBean AS d, IN (d.emps)e, IN (e.dept.reportsTo.emps) f where f.empid > 5  order by f.empid
//            EmpBean
//            ~~~~~~~
//              [6] [6] [6] [6] [8] [8] [8] [8]
//             TEST221; 8 tuples

            List rList = q.getResultList();

            Object[] targets = { emp6, emp6, emp6, emp6, emp8, emp8, emp8, emp8 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST222; select f from XMLDeptBean d, in (d.emps) as e, in (e.dept.emps) as f
    public void testLoop222(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select f from XMLDeptBean d, in (d.emps) as e, in (e.dept.emps) as f ";
            Query q = em.createQuery(qStr);

//            TEST222; select f from XMLDeptBean d, in (d.emps) as e, in (e.dept.emps) as f
//            EmpBean
//            ~~~~~~~
//              [1] [1] [1] [1] [2] [2] [2] [2] [3] [3] [4] [4] [5] [6] [6] [7] [7] [7] [7] [8] [8] [9] [9] [9] [9]
//             TEST222; 25 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp1, emp1, emp1,
                                 emp2, emp2, emp2, emp2,
                                 emp3, emp3,
                                 emp4, emp4,
                                 emp5,
                                 emp6, emp6,
                                 emp7, emp7, emp7, emp7,
                                 emp8, emp8,
                                 emp9, emp9, emp9, emp9
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST223; select m from XMLEmpBean e join e.dept d join d.mgr m
    public void testLoop223(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e join e.dept d join d.mgr m";
            Query q = em.createQuery(qStr);

//            TEST223; select m from XMLEmpBean e join e.dept d join d.mgr m
//            EmpBean
//            ~~~~~~~
//              [3] [3] [3] [3] [4] [8] [8] [10] [10]
//             TEST223; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { emp3, emp3, emp3, emp3, emp4, emp8, emp8, emp10, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST224; select m from XMLEmpBean e join e.dept d join d.mgr m where m.name = 'Dave' or e.empid > 0
    public void testLoop224(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e join e.dept d join d.mgr m where m.name = 'Dave' or e.empid > 0";
            Query q = em.createQuery(qStr);

//            TEST224; select m from XMLEmpBean e join e.dept d join d.mgr m where m.name = 'Dave' or e.empid > 0
//                            EmpBean
//                            ~~~~~~~
//                              [3] [3] [3] [3] [4] [8] [8] [10] [10]
//                             TEST224; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { emp3, emp3, emp3, emp3, emp4, emp8, emp8, emp10, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST225; select m from XMLEmpBean e join e.dept d left join d.mgr m
    public void testLoop225(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e join e.dept d left join d.mgr m";
            Query q = em.createQuery(qStr);

//            TEST225; select m from XMLEmpBean e join e.dept d left join d.mgr m
//            EmpBean
//            ~~~~~~~
//              [3] [3] [3] [3] [4] [8] [8] [10] [10]
//             TEST225; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { emp3, emp3, emp3, emp3, emp4, emp8, emp8, emp10, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST226; select m from XMLEmpBean e join e.dept d left join d.mgr m left join m.tasks p where p.taskid > 0
    public void testLoop226(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e join e.dept d left join d.mgr m left join m.tasks p where p.taskid > 0";
            Query q = em.createQuery(qStr);

//            TEST226; select m from XMLEmpBean e join e.dept d left join d.mgr m left join m.tasks p where p.taskid > 0
//            TEST226; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST227; select m from XMLEmpBean e left join e.dept d left join d.mgr m
    public void testLoop227(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e left join e.dept d left join d.mgr m";
            Query q = em.createQuery(qStr);

//            TEST227; select m from XMLEmpBean e left join e.dept d left join d.mgr m
//            EmpBean
//            ~~~~~~~
//              [3] [3] [3] [3] [4] [8] [8] [10] [10] null
//             TEST227; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp3, emp3, emp3, emp3, emp4, emp8, emp8, emp10, emp10, null };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST228; select m from XMLEmpBean e left join e.dept d left join d.mgr m where NOT(d.name is not null)
    public void testLoop228(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e left join e.dept d left join d.mgr m where NOT(d.name is not null)";
            Query q = em.createQuery(qStr);

//            TEST228; select m from XMLEmpBean e left join e.dept d left join d.mgr m where NOT(d.name is not null)
//            m
//           ~~~~
//           null
//            TEST228; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { null };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST229; select m from XMLEmpBean e left join e.dept d left join d.mgr m where NOT(d.name is null)
    public void testLoop229(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e left join e.dept d left join d.mgr m where NOT(d.name is null)";
            Query q = em.createQuery(qStr);

//            TEST229; select m from XMLEmpBean e left join e.dept d left join d.mgr m where NOT(d.name is null)
//            EmpBean
//            ~~~~~~~
//              [3] [3] [3] [3] [4] [8] [8] [10] [10]
//             TEST229; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { emp3, emp3, emp3, emp3, emp4, emp8, emp8, emp10, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST230; select m from XMLEmpBean e left join e.dept d left join d.mgr m where d.name = 'Ahmaddept'
    public void testLoop230(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e left join e.dept d left join d.mgr m where d.name = 'Ahmaddept'";
            Query q = em.createQuery(qStr);

//            TEST230; select m from XMLEmpBean e left join e.dept d left join d.mgr m where d.name = 'Ahmaddept'
//                            TEST230; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST231; select m from XMLEmpBean e left join e.dept d left join d.mgr m where d.name is not null
    public void testLoop231(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e left join e.dept d left join d.mgr m where d.name is not null";
            Query q = em.createQuery(qStr);

//            TEST231; select m from XMLEmpBean e left join e.dept d left join d.mgr m where d.name is not null
//            EmpBean
//            ~~~~~~~
//              [3] [3] [3] [3] [4] [8] [8] [10] [10]
//             TEST231; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { emp3, emp3, emp3, emp3, emp4, emp8, emp8, emp10, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST232; select m from XMLEmpBean e left join e.dept d left join d.mgr m where e.salary = 1000 and e.name = 'Steve'
    public void testLoop232(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e left join e.dept d left join d.mgr m where e.salary = 1000 and e.name = 'Steve'";
            Query q = em.createQuery(qStr);

//            TEST232; select m from XMLEmpBean e left join e.dept d left join d.mgr m where e.salary = 1000 and e.name = 'Steve'
//                            TEST232; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST233; select m from XMLEmpBean e left join e.dept d left join d.mgr m where e.salary = 1000 or e.name = 'Ahmad'
    public void testLoop233(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        final String dbProductName = (testProps == null) ? "UNKNOWN" : ((testProps.get("dbProductName") == null) ? "UNKNOWN" : (String) testProps.get("dbProductName"));

        // TODO: This test doesn't pass on SQLServer on Eclipselink, its output does not match the answer file
//        Validate Output for test "JULoopQueryAnoTest.testLoop233":
//            SQL: select m from XMLEmpBean e left join e.dept d left join d.mgr m where e.salary = 1000 or e.name = 'Ahmad'
//            Expected Output:
//
//            Generated Results:
//              0  EmpBean [empid=10, name=Catalina Wei]
        if (DatabaseVendor.checkDBProductName(dbProductName, DatabaseVendor.SQLSERVER)) {
            System.out.println("Test has been excluded from SQLServer.");
            return;
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e left join e.dept d left join d.mgr m where e.salary = 1000 or e.name = 'Ahmad'";
            Query q = em.createQuery(qStr);

//            TEST233; select m from XMLEmpBean e left join e.dept d left join d.mgr m where e.salary = 1000 or e.name = 'Ahmad'
//                            TEST233; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST234; select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name = 'Dave' or e.empid > 0
    public void testLoop234(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name = 'Dave' or e.empid > 0";
            Query q = em.createQuery(qStr);

//            TEST234; select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name = 'Dave' or e.empid > 0
//                            EmpBean
//                            ~~~~~~~
//                              [3] [3] [3] [3] [4] [8] [8] [10] [10] null
//                             TEST234; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp3, emp3, emp3, emp3, emp4, emp8, emp8, emp10, emp10, null };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST235; select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name = 'John'
    public void testLoop235(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name = 'John'";
            Query q = em.createQuery(qStr);

//            TEST235; select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name = 'John'
//                            TEST235; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST236; select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name = 'John' and e.salary = 20
    public void testLoop236(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name = 'John' and e.salary = 20";
            Query q = em.createQuery(qStr);

//            TEST236; select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name = 'John' and e.salary = 20
//                            TEST236; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {
            };
            validateQueryResult(testName, qStr, rList, targets);

            em.clear();

        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST237; select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name is not null
    public void testLoop237(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name is not null";
            Query q = em.createQuery(qStr);

//            TEST237; select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name is not null
//            EmpBean
//            ~~~~~~~
//              [3] [3] [3] [3] [4] [8] [8] [10] [10]
//             TEST237; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { emp3, emp3, emp3, emp3, emp4, emp8, emp8, emp10, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST238; select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name is null
    public void testLoop238(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name is null ";
            Query q = em.createQuery(qStr);

//            TEST238; select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name is null
//            m
//           ~~~~
//           null
//            TEST238; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { null };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST239; select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name='name1' and e.bonus<100.02
    public void testLoop239(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name='name1' and e.bonus<100.02 ";
            Query q = em.createQuery(qStr);

//            TEST239; select m from XMLEmpBean e left join e.dept d left join d.mgr m where m.name='name1' and e.bonus<100.02
//                            TEST239; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST240; select m from XMLEmpBean e left join e.dept d left join d.mgr m, ProjectBean p where e.name = 'john' and p.budget = e.salary
    public void testLoop240(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from XMLEmpBean e left join e.dept d left join d.mgr m, ProjectBean p where e.name = 'john' and p.budget = e.salary ";
            Query q = em.createQuery(qStr);

//            TEST240; select m from XMLEmpBean e left join e.dept d left join d.mgr m, ProjectBean p where e.name = 'john' and p.budget = e.salary
//                            TEST240; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST241; select m from ProjectBean p, EmpBean e left join e.dept d left join d.mgr m where e.name = 'john' and p.budget = e.salary
    public void testLoop241(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m from ProjectBean p, EmpBean e left join e.dept d left join d.mgr m where e.name = 'john' and p.budget = e.salary ";
            Query q = em.createQuery(qStr);

//            TEST241; select m from ProjectBean p, EmpBean e left join e.dept d left join d.mgr m where e.name = 'john' and p.budget = e.salary
//                            TEST241; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST242; select m,d,p from XMLEmpBean e left join e.dept d left join d.mgr m, ProjectBean p where e.name='name2' and p.cost > 9000
    public void testLoop242(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m,d,p from XMLEmpBean e left join e.dept d left join d.mgr m, ProjectBean p where e.name='name2' and p.cost > 9000 ";
            Query q = em.createQuery(qStr);

//            TEST242; select m,d,p from XMLEmpBean e left join e.dept d left join d.mgr m, ProjectBean p where e.name='name2' and p.cost > 9000
//                            TEST242; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST243; select m,e,d from XMLEmpBean e join e.dept d join d.mgr m where m.name='name4' or e.empid > 1002
    public void testLoop243(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m,e,d from XMLEmpBean e join e.dept d join d.mgr m where m.name='name4' or e.empid > 1002 ";
            Query q = em.createQuery(qStr);

//            TEST243; select m,e,d from XMLEmpBean e join e.dept d join d.mgr m where m.name='name4' or e.empid > 1002
//                            TEST243; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST244; select m,e,d from XMLEmpBean e join e.dept d left join d.mgr m
    public void testLoop244(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m,e,d from XMLEmpBean e join e.dept d left join d.mgr m ";
            Query q = em.createQuery(qStr);

//            TEST244; select m,e,d from XMLEmpBean e join e.dept d left join d.mgr m
//            EmpBean EmpBean DeptBean
//            ~~~~~~~ ~~~~~~~ ~~~~~~~~
//              [3]     [1]    [210]
//              [3]     [2]    [210]
//              [3]     [7]    [210]
//              [3]     [9]    [210]
//              [4]     [5]    [220]
//              [8]     [3]    [200]
//              [8]     [4]    [200]
//             [10]     [6]    [100]
//             [10]     [8]    [100]
//             TEST244; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { emp3, emp1, dep210 },
                                 new Object[] { emp3, emp2, dep210 },
                                 new Object[] { emp3, emp7, dep210 },
                                 new Object[] { emp3, emp9, dep210 },
                                 new Object[] { emp4, emp5, dep220 },
                                 new Object[] { emp8, emp3, dep200 },
                                 new Object[] { emp8, emp4, dep200 },
                                 new Object[] { emp10, emp6, dep100 },
                                 new Object[] { emp10, emp8, dep100 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST245; select m,e,d from XMLEmpBean e join e.dept d left join d.mgr m where e.bonus<100.02 or e.name='name6'
    public void testLoop245(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m,e,d from XMLEmpBean e join e.dept d left join d.mgr m where e.bonus<100.02 or e.name='name6' ";
            Query q = em.createQuery(qStr);

//            TEST245; select m,e,d from XMLEmpBean e join e.dept d left join d.mgr m where e.bonus<100.02 or e.name='name6'
//                            EmpBean EmpBean DeptBean
//                            ~~~~~~~ ~~~~~~~ ~~~~~~~~
//                              [3]     [1]    [210]
//                              [3]     [2]    [210]
//                              [3]     [7]    [210]
//                              [3]     [9]    [210]
//                              [4]     [5]    [220]
//                              [8]     [3]    [200]
//                              [8]     [4]    [200]
//                             [10]     [6]    [100]
//                             [10]     [8]    [100]
//                             TEST245; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { emp3, emp1, dep210 },
                                 new Object[] { emp3, emp2, dep210 },
                                 new Object[] { emp3, emp7, dep210 },
                                 new Object[] { emp3, emp9, dep210 },
                                 new Object[] { emp4, emp5, dep220 },
                                 new Object[] { emp8, emp3, dep200 },
                                 new Object[] { emp8, emp4, dep200 },
                                 new Object[] { emp10, emp6, dep100 },
                                 new Object[] { emp10, emp8, dep100 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST246; select m,e,d from XMLEmpBean e left join e.dept d join d.mgr m
    public void testLoop246(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m,e,d from XMLEmpBean e left join e.dept d join d.mgr m ";
            Query q = em.createQuery(qStr);

//            TEST246; select m,e,d from XMLEmpBean e left join e.dept d join d.mgr m
//            EmpBean EmpBean DeptBean
//            ~~~~~~~ ~~~~~~~ ~~~~~~~~
//              [3]     [1]    [210]
//              [3]     [2]    [210]
//              [3]     [7]    [210]
//              [3]     [9]    [210]
//              [4]     [5]    [220]
//              [8]     [3]    [200]
//              [8]     [4]    [200]
//             [10]     [6]    [100]
//             [10]     [8]    [100]
//             null    [10]     null
//             TEST246; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { emp3, emp1, dep210 },
                                 new Object[] { emp3, emp2, dep210 },
                                 new Object[] { emp3, emp7, dep210 },
                                 new Object[] { emp3, emp9, dep210 },
                                 new Object[] { emp4, emp5, dep220 },
                                 new Object[] { emp8, emp3, dep200 },
                                 new Object[] { emp8, emp4, dep200 },
                                 new Object[] { emp10, emp6, dep100 },
                                 new Object[] { emp10, emp8, dep100 },
                                 new Object[] { null, emp10, null },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST247; select m,e,d from XMLEmpBean e left join e.dept d left join d.mgr m
    public void testLoop247(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m,e,d from XMLEmpBean e left join e.dept d left join d.mgr m ";
            Query q = em.createQuery(qStr);

//            TEST247; select m,e,d from XMLEmpBean e left join e.dept d left join d.mgr m
//            EmpBean EmpBean DeptBean
//            ~~~~~~~ ~~~~~~~ ~~~~~~~~
//              [3]     [1]    [210]
//              [3]     [2]    [210]
//              [3]     [7]    [210]
//              [3]     [9]    [210]
//              [4]     [5]    [220]
//              [8]     [3]    [200]
//              [8]     [4]    [200]
//             [10]     [6]    [100]
//             [10]     [8]    [100]
//             null    [10]     null
//             TEST247; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { emp3, emp1, dep210 },
                                 new Object[] { emp3, emp2, dep210 },
                                 new Object[] { emp3, emp7, dep210 },
                                 new Object[] { emp3, emp9, dep210 },
                                 new Object[] { emp4, emp5, dep220 },
                                 new Object[] { emp8, emp3, dep200 },
                                 new Object[] { emp8, emp4, dep200 },
                                 new Object[] { emp10, emp6, dep100 },
                                 new Object[] { emp10, emp8, dep100 },
                                 new Object[] { null, emp10, null },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST248; select m,e,d from XMLEmpBean e left join e.dept d left join d.mgr m where e.bonus<100.02 or e.name='name6'
    public void testLoop248(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m,e,d from XMLEmpBean e left join e.dept d left join d.mgr m where e.bonus<100.02 or e.name='name6' ";
            Query q = em.createQuery(qStr);

//            TEST248; select m,e,d from XMLEmpBean e left join e.dept d left join d.mgr m where e.bonus<100.02 or e.name='name6'
//                            EmpBean EmpBean DeptBean
//                            ~~~~~~~ ~~~~~~~ ~~~~~~~~
//                              [3]     [1]    [210]
//                              [3]     [2]    [210]
//                              [3]     [7]    [210]
//                              [3]     [9]    [210]
//                              [4]     [5]    [220]
//                              [8]     [3]    [200]
//                              [8]     [4]    [200]
//                             [10]     [6]    [100]
//                             [10]     [8]    [100]
//                             null    [10]     null
//                             TEST248; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { emp3, emp1, dep210 },
                                 new Object[] { emp3, emp2, dep210 },
                                 new Object[] { emp3, emp7, dep210 },
                                 new Object[] { emp3, emp9, dep210 },
                                 new Object[] { emp4, emp5, dep220 },
                                 new Object[] { emp8, emp3, dep200 },
                                 new Object[] { emp8, emp4, dep200 },
                                 new Object[] { emp10, emp6, dep100 },
                                 new Object[] { emp10, emp8, dep100 },
                                 new Object[] { null, emp10, null },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST249; select m,m.name,d,d.name,e  from XMLEmpBean e join e.dept d join d.mgr m
    public void testLoop249(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m,m.name,d,d.name,e  from XMLEmpBean e join e.dept d join d.mgr m";
            Query q = em.createQuery(qStr);

//            TEST249; select m,m.name,d,d.name,e  from XMLEmpBean e join e.dept d join d.mgr m
//            EmpBean    m.name    DeptBean   d.name    EmpBean
//            ~~~~~~~ ~~~~~~~~~~~~ ~~~~~~~~ ~~~~~~~~~~~ ~~~~~~~
//              [3]      minmei     [210]   Development   [1]
//              [3]      minmei     [210]   Development   [2]
//              [3]      minmei     [210]   Development   [7]
//              [3]      minmei     [210]   Development   [9]
//              [4]      george     [220]     Service     [5]
//              [8]   Tom Rayburn   [200]      Admin      [3]
//              [8]   Tom Rayburn   [200]      Admin      [4]
//             [10]   Catalina Wei  [100]       CEO       [6]
//             [10]   Catalina Wei  [100]       CEO       [8]
//             TEST249; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { emp3, "minmei", dep210, "Development", emp1 },
                                 new Object[] { emp3, "minmei", dep210, "Development", emp2 },
                                 new Object[] { emp3, "minmei", dep210, "Development", emp7 },
                                 new Object[] { emp3, "minmei", dep210, "Development", emp9 },
                                 new Object[] { emp4, "george", dep220, "Service", emp5 },
                                 new Object[] { emp8, "Tom Rayburn", dep200, "Admin", emp3 },
                                 new Object[] { emp8, "Tom Rayburn", dep200, "Admin", emp4 },
                                 new Object[] { emp10, "Catalina Wei", dep100, "CEO", emp6 },
                                 new Object[] { emp10, "Catalina Wei", dep100, "CEO", emp8 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST250; select m.name from XMLEmpBean e join e.dept d join d.mgr m where e.name = 'Dave'
    public void testLoop250(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m.name from XMLEmpBean e join e.dept d join d.mgr m where e.name = 'Dave'";
            Query q = em.createQuery(qStr);

//            TEST250; select m.name from XMLEmpBean e join e.dept d join d.mgr m where e.name = 'Dave'
//                            TEST250; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST251; select m.name from XMLEmpBean e join e.dept.mgr m where e.name = 'Dave'
    public void testLoop251(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select m.name from XMLEmpBean e join e.dept.mgr m where e.name = 'Dave'";
            Query q = em.createQuery(qStr);

//            TEST251; select m.name from XMLEmpBean e join e.dept.mgr m where e.name = 'Dave'
//                            TEST251; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST252; select p from XMLEmpBean e left join e.dept.projects p
    public void testLoop252(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select p from XMLEmpBean e left join e.dept.projects p ";
            Query q = em.createQuery(qStr);

//            TEST252; select p from XMLEmpBean e left join e.dept.projects p
//            ProjectBean
//            ~~~~~~~~~~~
//               null
//               null
//               null
//               null
//               null
//              [1000]
//              [1000]
//              [1000]
//              [1000]
//              [2000]
//             TEST252; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { null, null, null, null, null,
                                 proj1000, proj1000, proj1000, proj1000, proj2000 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST253; select p.dept , e.dept.mgr from ProjectBean p, EmpBean e where e.name = 'john' and p.budget = e.salary
    public void testLoop253(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select p.dept , e.dept.mgr from ProjectBean p, EmpBean e where e.name = 'john' and p.budget = e.salary ";
            Query q = em.createQuery(qStr);

//            TEST253; select p.dept , e.dept.mgr from ProjectBean p, EmpBean e where e.name = 'john' and p.budget = e.salary
//                            TEST253; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST254; select p.dept ,e.dept.mgr, e.dept.name, e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and p.budget = e.salary
    public void testLoop254(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select p.dept ,e.dept.mgr, e.dept.name, e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and p.budget = e.salary";
            Query q = em.createQuery(qStr);

//            TEST254; select p.dept ,e.dept.mgr, e.dept.name, e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and p.budget = e.salary
//                            TEST254; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST255; select p.dept, e.dept.mgr from ProjectBean p, EmpBean e where e.name = 'john' and p.budget = e.salary
    public void testLoop255(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select p.dept, e.dept.mgr from ProjectBean p, EmpBean e where e.name = 'john' and p.budget = e.salary ";
            Query q = em.createQuery(qStr);

//          TEST255; select p.dept, e.dept.mgr from ProjectBean p, EmpBean e where e.name = 'john' and p.budget = e.salary
//            TEST255; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST256; select p.dept,e.dept.mgr, e.dept.name, e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and p.budget = e.salary
    public void testLoop256(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select p.dept,e.dept.mgr, e.dept.name, e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and p.budget = e.salary";
            Query q = em.createQuery(qStr);

//            TEST256; select p.dept,e.dept.mgr, e.dept.name, e.dept from XMLEmpBean e, ProjectBean p where e.name = 'john' and p.budget = e.salary
//                            TEST256; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST257; select p.dept,e.dept.mgr, e.dept.name, e.dept from ProjectBean p, EmpBean e where e.name = 'john' and p.budget = e.salary and e.dept.name = 'dept1'
    public void testLoop257(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select p.dept,e.dept.mgr, e.dept.name, e.dept from ProjectBean p, EmpBean e where e.name = 'john' and p.budget = e.salary and e.dept.name = 'dept1'";
            Query q = em.createQuery(qStr);

//            TEST257; select p.dept,e.dept.mgr, e.dept.name, e.dept from ProjectBean p, EmpBean e where e.name = 'john' and p.budget = e.salary and e.dept.name = 'dept1'
//                            TEST257; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST258; select p.projid, t9.cost from ProjectBean p join p.tasks t9 where p.projid= t9.taskid
    public void testLoop258(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select p.projid, t9.cost from ProjectBean p join p.tasks t9 where p.projid= t9.taskid ";
            Query q = em.createQuery(qStr);

//            TEST258; select p.projid, t9.cost from ProjectBean p join p.tasks t9 where p.projid= t9.taskid
//                            TEST258; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST259; select p.projid, t9.cost from ProjectBean p left join p.tasks t9 where p.projid= t9.taskid
    public void testLoop259(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select p.projid, t9.cost from ProjectBean p left join p.tasks t9 where p.projid= t9.taskid ";
            Query q = em.createQuery(qStr);

//          TEST259; select p.projid, t9.cost from ProjectBean p left join p.tasks t9 where p.projid= t9.taskid
//            TEST259; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST260; select pm, dm from ProjectBean p left join p.dept pm, EmpBean e left join e.dept d left join d.mgr dm where e.name = 'john' and p.budget = e.salary
    public void testLoop260(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select pm, dm from ProjectBean p left join p.dept pm, EmpBean e left join e.dept d left join d.mgr dm where e.name = 'john' and p.budget = e.salary ";
            Query q = em.createQuery(qStr);

//         TEST260; select pm, dm from ProjectBean p left join p.dept pm, EmpBean e left join e.dept d left join d.mgr dm where e.name = 'john' and p.budget = e.salary
//            TEST260; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST261; select pm, dm, d.name, d from ProjectBean p left join p.dept pm, EmpBean e left join e.dept d left join d.mgr dm where e.name = 'john' and p.budget = e.salary and dm.name = 'dept1'
    public void testLoop261(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select pm, dm, d.name, d from ProjectBean p left join p.dept pm, EmpBean e left join e.dept d left join d.mgr dm where e.name = 'john' and p.budget = e.salary and dm.name = 'dept1' ";
            Query q = em.createQuery(qStr);

//         TEST261; select pm, dm, d.name, d from ProjectBean p left join p.dept pm, EmpBean e left join e.dept d left join d.mgr dm where e.name = 'john' and p.budget = e.salary and dm.name = 'dept1'
//            TEST261; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //TEST262; select pm, dm, d.name, d from ProjectBean p left join p.dept.mgr pm, EmpBean e left join e.dept d left join d.mgr dm where e.name = 'john' and p.budget = e.salary and dm.name = 'dept1'
    public void testLoop262(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select pm, dm, d.name, d from ProjectBean p left join p.dept.mgr pm, EmpBean e left join e.dept d left join d.mgr dm where e.name = 'john' and p.budget = e.salary and dm.name = 'dept1' ";
            Query q = em.createQuery(qStr);

//         TEST262; select pm, dm, d.name, d from ProjectBean p left join p.dept.mgr pm, EmpBean e left join e.dept d left join d.mgr dm where e.name = 'john' and p.budget = e.salary and dm.name = 'dept1'
//            TEST262; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST263; select t9 from ProjectBean p join p.tasks t9
    public void testLoop263(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select t9 from ProjectBean p join p.tasks t9 ";
            Query q = em.createQuery(qStr);

//         TEST263; select t9 from ProjectBean p join p.tasks t9
//            TaskBean
//            ~~~~~~~~
//             [1010] [1020] [1030] [2010] [2020]
//             TEST263; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { task1010, task1020, task1030, task2010, task2020 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST264; select x.empid, x.name, x.dept.no from XMLEmpBean x where x.empid<17 order by x.empid
    public void testLoop264(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select x.empid, x.name, x.dept.no from XMLEmpBean x where x.empid<17 order by x.empid ";
            Query q = em.createQuery(qStr);

//         TEST264; select x.empid, x.name, x.dept.no from XMLEmpBean x where x.empid<17 order by x.empid
//            x.empid   x.name    x.dept.no
//            ~~~~~~~ ~~~~~~~~~~~ ~~~~~~~~~
//               1       david       210
//               2      andrew       210
//               3      minmei       200
//               4      george       200
//               5      ritika       220
//               6       ahmad       100
//               7     charlene      210
//               8    Tom Rayburn    100
//               9       harry       210
//             TEST264; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, "david", 210 },
                                 new Object[] { 2, "andrew", 210 },
                                 new Object[] { 3, "minmei", 200 },
                                 new Object[] { 4, "george", 200 },
                                 new Object[] { 5, "ritika", 220 },
                                 new Object[] { 6, "ahmad", 100 },
                                 new Object[] { 7, "charlene", 210 },
                                 new Object[] { 8, "Tom Rayburn", 100 },
                                 new Object[] { 9, "harry", 210 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST265; select x.empid, y.empid, x.name, x.dept.no from XMLEmpBean x, in (x.dept.emps) y where x.name=y.name  order by x.dept.no
    public void testLoop265(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select x.empid, y.empid, x.name, x.dept.no from XMLEmpBean x, in (x.dept.emps) y where x.name=y.name  order by x.dept.no";
            Query q = em.createQuery(qStr);

//          TEST265; select x.empid, y.empid, x.name, x.dept.no from XMLEmpBean x, in (x.dept.emps) y where x.name=y.name  order by x.dept.no
//            x.empid y.empid   x.name    x.dept.no
//            ~~~~~~~ ~~~~~~~ ~~~~~~~~~~~ ~~~~~~~~~
//               1       1       david       210
//               2       2      andrew       210
//               3       3      minmei       200
//               4       4      george       200
//               5       5      ritika       220
//               6       6       ahmad       100
//               7       7     charlene      210
//               8       8    Tom Rayburn    100
//               9       9       harry       210
//             TEST265; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, 1, "david", 210 },
                                 new Object[] { 2, 2, "andrew", 210 },
                                 new Object[] { 3, 3, "minmei", 200 },
                                 new Object[] { 4, 4, "george", 200 },
                                 new Object[] { 5, 5, "ritika", 220 },
                                 new Object[] { 6, 6, "ahmad", 100 },
                                 new Object[] { 7, 7, "charlene", 210 },
                                 new Object[] { 8, 8, "Tom Rayburn", 100 },
                                 new Object[] { 9, 9, "harry", 210 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST266; select x.name from XMLEmpBean x  order by x.name desc
    public void testLoop266(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select x.name from XMLEmpBean x  order by x.name desc ";
            Query q = em.createQuery(qStr);

//            TEST266; select x.name from XMLEmpBean x  order by x.name desc
//            x.name
//         ~~~~~~~~~~~~
//            ahmad
//            andrew
//            david
//            george
//            harry
//            minmei
//            ritika
//           charlene
//         Catalina Wei
//         Tom Rayburn
//          TEST266; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { "ahmad",
                                 "andrew",
                                 "david",
                                 "george",
                                 "harry",
                                 "minmei",
                                 "ritika",
                                 "charlene",
                                 "Catalina Wei",
                                 "Tom Rayburn"
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST267; select x.no, x.name, y.empid, y.name from XMLDeptBean x, in (x.emps) y where x.mgr.empid = y.empid
    public void testLoop267(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select x.no, x.name, y.empid, y.name from XMLDeptBean x, in (x.emps) y where x.mgr.empid = y.empid  ";
            Query q = em.createQuery(qStr);

//          TEST267; select x.no, x.name, y.empid, y.name from XMLDeptBean x, in (x.emps) y where x.mgr.empid = y.empid
//            TEST267; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST268; select y.empid, x.name from XMLDeptBean x, EmpBean y where x.no = y.dept.no
    public void testLoop268(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select y.empid, x.name from XMLDeptBean x, EmpBean y where x.no = y.dept.no ";
            Query q = em.createQuery(qStr);

//         TEST268; select y.empid, x.name from XMLDeptBean x, EmpBean y where x.no = y.dept.no
//            y.empid   x.name
//            ~~~~~~~ ~~~~~~~~~~~
//               1    Development
//               2    Development
//               3       Admin
//               4       Admin
//               5      Service
//               6        CEO
//               7    Development
//               8        CEO
//               9    Development
//             TEST268; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, "Development" },
                                 new Object[] { 2, "Development" },
                                 new Object[] { 3, "Admin" },
                                 new Object[] { 4, "Admin" },
                                 new Object[] { 5, "Service" },
                                 new Object[] { 6, "CEO" },
                                 new Object[] { 7, "Development" },
                                 new Object[] { 8, "CEO" },
                                 new Object[] { 9, "Development" },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST269; select d from  DeptBean AS d, IN (d.emps)e, IN (e.dept.reportsTo.emps) f where f.empid is null
    public void testLoop269(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from  DeptBean AS d, IN (d.emps)e, IN (e.dept.reportsTo.emps) f where f.empid is null ";
            Query q = em.createQuery(qStr);

//         TEST269; select d from  DeptBean AS d, IN (d.emps)e, IN (e.dept.reportsTo.emps) f where f.empid is null
//            TEST269; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST270; select d from XMLDeptBean d where d.reportsTo.mgr.dept is null or d.reportsTo.mgr.dept.no  is not null
    public void testLoop270(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where d.reportsTo.mgr.dept is null or d.reportsTo.mgr.dept.no  is not null ";
            Query q = em.createQuery(qStr);

//            TEST270; select d from XMLDeptBean d where d.reportsTo.mgr.dept is null or d.reportsTo.mgr.dept.no  is not null
//            DeptBean
//            ~~~~~~~~
//             [100] [200] [210] [220] [300]
//             TEST270; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { dep100, dep200, dep210, dep220, dep300 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST271; select d from XMLDeptBean d where d.reportsTo.mgr.dept is null
    public void testLoop271(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where d.reportsTo.mgr.dept is null ";
            Query q = em.createQuery(qStr);

//            TEST271; select d from XMLDeptBean d where d.reportsTo.mgr.dept is null
//            DeptBean
//            ~~~~~~~~
//             [100] [200] [300]
//             TEST271; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { dep100, dep200, dep300 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST272; select object(e) from XMLEmpBean e where e.empid is null
    public void testLoop272(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select object(e) from XMLEmpBean e where e.empid is null ";
            Query q = em.createQuery(qStr);

//            TEST272; select object(e) from XMLEmpBean e where e.empid is null
//            TEST272; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST273; select d from XMLDeptBean d where (d.mgr.empid between 10 and 20)
    public void testLoop273(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where (d.mgr.empid between 10 and 20) ";
            Query q = em.createQuery(qStr);

//            TEST273; select d from XMLDeptBean d where (d.mgr.empid between 10 and 20)
//            DeptBean
//            ~~~~~~~~
//             [100]
//             TEST273; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { dep100 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST274; select e from XMLEmpBean e where (e.dept between (select d from XMLDeptBean d where d.no = 100) and (select d from XMLDeptBean d where d.no = 200))
    public void testLoop274(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.dept between (select d from XMLDeptBean d where d.no = 100) and (select d from XMLDeptBean d where d.no = 200)) ";
            Query q = em.createQuery(qStr);

//            TEST274; select e from XMLEmpBean e where (e.dept between (select d from XMLDeptBean d where d.no = 100) and (select d from XMLDeptBean d where d.no = 200))
//            EmpBean
//            ~~~~~~~
//              [3] [4] [6] [8]
//             TEST274; 4 tuples

            List rList = q.getResultList();

            Object[] targets = { emp3, emp4, emp6, emp8 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST275; select e from XMLEmpBean e where (e.salary between 50000 and 60000)
    public void testLoop275(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary between 50000 and 60000) ";
            Query q = em.createQuery(qStr);

//            TEST275; select e from XMLEmpBean e where (e.salary between 50000 and 60000)
//            TEST275; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST276; select e from XMLEmpBean e where (e.salary between 56000 and 64000 or e.salary between 72000 and 80000)
    public void testLoop276(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary between 56000 and 64000 or e.salary between 72000 and 80000) ";
            Query q = em.createQuery(qStr);

//            TEST276; select e from XMLEmpBean e where (e.salary between 56000 and 64000 or e.salary between 72000 and 80000)
//            TEST276; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST277; select   e.empid, e.salary from XMLEmpBean e where e.empid+3 NOT between 4 and 8 OR NOT e.salary between 80 and 160
    public void testLoop277(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select   e.empid, e.salary from XMLEmpBean e where e.empid+3 NOT between 4 and 8 OR NOT e.salary between 80 and 160  ";
            Query q = em.createQuery(qStr);

//            TEST277; select   e.empid, e.salary from XMLEmpBean e where e.empid+3 NOT between 4 and 8 OR NOT e.salary between 80 and 160
//            e.empid e.salary
//            ~~~~~~~ ~~~~~~~~
//               1      12.1
//               2      13.1
//               3      15.5
//               4      0.0
//               5      0.0
//               6      0.0
//               7      0.0
//               8      0.0
//               9      0.0
//              10      0.0
//             TEST277; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, 12.1 },
                                 new Object[] { 2, 13.1 },
                                 new Object[] { 3, 15.5 },
                                 new Object[] { 4, 0.0 },
                                 new Object[] { 5, 0.0 },
                                 new Object[] { 6, 0.0 },
                                 new Object[] { 7, 0.0 },
                                 new Object[] { 8, 0.0 },
                                 new Object[] { 9, 0.0 },
                                 new Object[] { 10, 0.0 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST278; select   e.empid, e.salary from XMLEmpBean e where e.empid+3 between 4 and 8
    public void testLoop278(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select   e.empid, e.salary from XMLEmpBean e where e.empid+3 between 4 and 8 ";
            Query q = em.createQuery(qStr);

//            TEST278; select   e.empid, e.salary from XMLEmpBean e where e.empid+3 between 4 and 8
//            e.empid e.salary
//            ~~~~~~~ ~~~~~~~~
//               1      12.1
//               2      13.1
//               3      15.5
//               4      0.0
//               5      0.0
//             TEST278; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, 12.1 },
                                 new Object[] { 2, 13.1 },
                                 new Object[] { 3, 15.5 },
                                 new Object[] { 4, 0.0 },
                                 new Object[] { 5, 0.0 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST279; select   e.empid, e.salary from XMLEmpBean e where e.empid+3 between 4 and 8 and e.salary between 80 and 160
    public void testLoop279(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select   e.empid, e.salary from XMLEmpBean e where e.empid+3 between 4 and 8 and e.salary between 80 and 160 ";
            Query q = em.createQuery(qStr);

//            TEST279; select   e.empid, e.salary from XMLEmpBean e where e.empid+3 between 4 and 8 and e.salary between 80 and 160
//            TEST279; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST280; select   e.empid, e.salary from XMLEmpBean e where e.empid+3 between 4 and 8 and not e.salary between 80 and 160
    public void testLoop280(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select   e.empid, e.salary from XMLEmpBean e where e.empid+3 between 4 and 8 and not e.salary between 80 and 160  ";
            Query q = em.createQuery(qStr);

//            TEST280; select   e.empid, e.salary from XMLEmpBean e where e.empid+3 between 4 and 8 and not e.salary between 80 and 160
//            e.empid e.salary
//            ~~~~~~~ ~~~~~~~~
//               1      12.1
//               2      13.1
//               3      15.5
//               4      0.0
//               5      0.0
//             TEST280; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, 12.1 },
                                 new Object[] { 2, 13.1 },
                                 new Object[] { 3, 15.5 },
                                 new Object[] { 4, 0.0 },
                                 new Object[] { 5, 0.0 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST281; select   e.empid, e.salary from XMLEmpBean e where e.empid+3 not between 4 and 8
    public void testLoop281(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select   e.empid, e.salary from XMLEmpBean e where e.empid+3 not between 4 and 8 ";
            Query q = em.createQuery(qStr);

//            TEST281; select   e.empid, e.salary from XMLEmpBean e where e.empid+3 not between 4 and 8
//            e.empid e.salary
//            ~~~~~~~ ~~~~~~~~
//               6      0.0
//               7      0.0
//               8      0.0
//               9      0.0
//              10      0.0
//             TEST281; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 6, 0.0 },
                                 new Object[] { 7, 0.0 },
                                 new Object[] { 8, 0.0 },
                                 new Object[] { 9, 0.0 },
                                 new Object[] { 10, 0.0 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST282; select   e.empid, e.salary from XMLEmpBean e where e.empid+3 not between 4 and 8 and not e.salary between 500 and 1000
    public void testLoop282(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select   e.empid, e.salary from XMLEmpBean e where e.empid+3 not between 4 and 8 and not e.salary between 500 and 1000  ";
            Query q = em.createQuery(qStr);

//            TEST282; select   e.empid, e.salary from XMLEmpBean e where e.empid+3 not between 4 and 8 and not e.salary between 500 and 1000
//            e.empid e.salary
//            ~~~~~~~ ~~~~~~~~
//               6      0.0
//               7      0.0
//               8      0.0
//               9      0.0
//              10      0.0
//             TEST282; 5 tuples

            List rList = q.getResultList();

            Object[] targets = {
//                                 new Object[] { 1, 12.1 },
//                                 new Object[] { 2, 13.1 },
//                                 new Object[] { 3, 15.5 },
//                                 new Object[] { 4, 0.0 },
//                                 new Object[] { 5, 0.0 },
                                 new Object[] { 6, 0.0 },
                                 new Object[] { 7, 0.0 },
                                 new Object[] { 8, 0.0 },
                                 new Object[] { 9, 0.0 },
                                 new Object[] { 10, 0.0 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST283; select   e.empid, e.salary from XMLEmpBean e where e.salary between 560 and 640 or e.salary between 720 and 800
    public void testLoop283(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select   e.empid, e.salary from XMLEmpBean e where e.salary between 560 and 640 or e.salary between 720 and 800 ";
            Query q = em.createQuery(qStr);

//            TEST283; select   e.empid, e.salary from XMLEmpBean e where e.salary between 560 and 640 or e.salary between 720 and 800
//            TEST283; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST284; select   e.empid, e.salary from XMLEmpBean e where e.salary between 560 and 640 or not e.salary between 720 and 800
    public void testLoop284(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select   e.empid, e.salary from XMLEmpBean e where e.salary between 560 and 640 or not e.salary between 720 and 800  ";
            Query q = em.createQuery(qStr);

//            TEST284; select   e.empid, e.salary from XMLEmpBean e where e.salary between 560 and 640 or not e.salary between 720 and 800
//            e.empid e.salary
//            ~~~~~~~ ~~~~~~~~
//               1      12.1
//               2      13.1
//               3      15.5
//               4      0.0
//               5      0.0
//               6      0.0
//               7      0.0
//               8      0.0
//               9      0.0
//              10      0.0
//             TEST284; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, 12.1 },
                                 new Object[] { 2, 13.1 },
                                 new Object[] { 3, 15.5 },
                                 new Object[] { 4, 0.0 },
                                 new Object[] { 5, 0.0 },
                                 new Object[] { 6, 0.0 },
                                 new Object[] { 7, 0.0 },
                                 new Object[] { 8, 0.0 },
                                 new Object[] { 9, 0.0 },
                                 new Object[] { 10, 0.0 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST285; select   e.empid, e.salary from XMLEmpBean e where e.salary not between 560 and 640 or e.salary between 720 and 800
    public void testLoop285(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select   e.empid, e.salary from XMLEmpBean e where e.salary not between 560 and 640 or e.salary between 720 and 800 ";
            Query q = em.createQuery(qStr);

//            TEST285; select   e.empid, e.salary from XMLEmpBean e where e.salary not between 560 and 640 or e.salary between 720 and 800
//            e.empid e.salary
//            ~~~~~~~ ~~~~~~~~
//               1      12.1
//               2      13.1
//               3      15.5
//               4      0.0
//               5      0.0
//               6      0.0
//               7      0.0
//               8      0.0
//               9      0.0
//              10      0.0
//             TEST285; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, 12.1 },
                                 new Object[] { 2, 13.1 },
                                 new Object[] { 3, 15.5 },
                                 new Object[] { 4, 0.0 },
                                 new Object[] { 5, 0.0 },
                                 new Object[] { 6, 0.0 },
                                 new Object[] { 7, 0.0 },
                                 new Object[] { 8, 0.0 },
                                 new Object[] { 9, 0.0 },
                                 new Object[] { 10, 0.0 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST286; select   x.empid, x.name from XMLEmpBean x where x.empid between 10 and 20
    public void testLoop286(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select   x.empid, x.name from XMLEmpBean x where x.empid between 10 and 20  ";
            Query q = em.createQuery(qStr);

//            TEST286; select   x.empid, x.name from XMLEmpBean x where x.empid between 10 and 20
//            x.empid    x.name
//            ~~~~~~~ ~~~~~~~~~~~~
//              10    Catalina Wei
//             TEST286; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 10, "Catalina Wei" }, };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST287; select   x.empid, x.name from XMLEmpBean x where x.empid not between 10 and 20
    public void testLoop287(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select   x.empid, x.name from XMLEmpBean x where x.empid not between 10 and 20";
            Query q = em.createQuery(qStr);

//            TEST287; select   x.empid, x.name from XMLEmpBean x where x.empid not between 10 and 20
//            x.empid   x.name
//            ~~~~~~~ ~~~~~~~~~~~
//               1       david
//               2      andrew
//               3      minmei
//               4      george
//               5      ritika
//               6       ahmad
//               7     charlene
//               8    Tom Rayburn
//               9       harry
//             TEST287; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, "david" },
                                 new Object[] { 2, "andrew" },
                                 new Object[] { 3, "minmei" },
                                 new Object[] { 4, "george" },
                                 new Object[] { 5, "ritika" },
                                 new Object[] { 6, "ahmad" },
                                 new Object[] { 7, "charlene" },
                                 new Object[] { 8, "Tom Rayburn" },
                                 new Object[] { 9, "harry" },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST288; select e from XMLEmpBean e where e.empid+3 between 4 and 8 and not e.salary between 80 and 160
    public void testLoop288(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.empid+3 between 4 and 8 and not e.salary between 80 and 160 ";
            Query q = em.createQuery(qStr);

//            TEST288; select e from XMLEmpBean e where e.empid+3 between 4 and 8 and not e.salary between 80 and 160
//           EmpBean
//           ~~~~~~~
//             [1] [2] [3] [4] [5]
//            TEST288; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST289; select e from XMLEmpBean e where e.empid+3 between 4 and 8 and e.salary between 80 and 160
    public void testLoop289(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.empid+3 between 4 and 8 and e.salary between 80 and 160 ";
            Query q = em.createQuery(qStr);

//            TEST289; select e from XMLEmpBean e where e.empid+3 between 4 and 8 and e.salary between 80 and 160
//            TEST289; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST290; select e from XMLEmpBean e where e.empid+3 between 4 and 8
    public void testLoop290(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.empid+3 between 4 and 8 ";
            Query q = em.createQuery(qStr);

//            TEST290; select e from XMLEmpBean e where e.empid+3 between 4 and 8
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5]
//             TEST290; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST291; select e from XMLEmpBean e where e.empid+3 not between 4 and 8 and not e.salary between 0 and 1000
    public void testLoop291(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.empid+3 not between 4 and 8 and not e.salary between 0 and 1000 ";
            Query q = em.createQuery(qStr);

//            TEST291; select e from XMLEmpBean e where e.empid+3 not between 4 and 8 and not e.salary between 0 and 1000
//            TEST291; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST292; select e from XMLEmpBean e where e.empid+3 not between 4 and 8
    public void testLoop292(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.empid+3 not between 4 and 8 ";
            Query q = em.createQuery(qStr);

//            TEST292; select e from XMLEmpBean e where e.empid+3 not between 4 and 8
//            EmpBean
//            ~~~~~~~
//              [6] [7] [8] [9] [10]
//             TEST292; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { emp6, emp7, emp8, emp9, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST293; select e from XMLEmpBean e where e.salary between 560 and 640 or e.salary between 720 and 800
    public void testLoop293(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary between 560 and 640 or e.salary between 720 and 800 ";
            Query q = em.createQuery(qStr);

//            TEST293; select e from XMLEmpBean e where e.salary between 560 and 640 or e.salary between 720 and 800
//            TEST293; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST294; select e from XMLEmpBean e where e.salary between 560 and 640 or not e.salary between 720 and 800
    public void testLoop294(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary between 560 and 640 or not e.salary between 720 and 800 ";
            Query q = em.createQuery(qStr);

//            TEST294; select e from XMLEmpBean e where e.salary between 560 and 640 or not e.salary between 720 and 800
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [6] [7] [8] [9] [10]
//             TEST294; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST295; select e from XMLEmpBean e where e.salary not between 560 and 640 or e.salary between 720 and 800
    public void testLoop295(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary not between 560 and 640 or e.salary between 720 and 800  ";
            Query q = em.createQuery(qStr);

//            TEST295; select e from XMLEmpBean e where e.salary not between 560 and 640 or e.salary between 720 and 800
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [6] [7] [8] [9] [10]
//             TEST295; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST296; select e from XMLEmpBean e where e.salary+3 NOT between 4 and 8 OR NOT e.salary between 80 and 160
    public void testLoop296(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary+3 NOT between 4 and 8 OR NOT e.salary between 80 and 160 ";
            Query q = em.createQuery(qStr);

//            TEST296; select e from XMLEmpBean e where e.salary+3 NOT between 4 and 8 OR NOT e.salary between 80 and 160
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [6] [7] [8] [9] [10]
//             TEST296; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST297; select e.name from XMLEmpBean e where e.salary between 10 and 20
    public void testLoop297(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.salary between 10 and 20 ";
            Query q = em.createQuery(qStr);

//            TEST297; select e.name from XMLEmpBean e where e.salary between 10 and 20
//            e.name
//            ~~~~~~
//            andrew
//            david
//            minmei
//             TEST297; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { "andrew",
                                 "david",
                                 "minmei"
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST298; select e.name from XMLEmpBean e where e.salary between 10.0 and 20.0
    public void testLoop298(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.salary between 10.0 and 20.0 ";
            Query q = em.createQuery(qStr);

//            TEST298; select e.name from XMLEmpBean e where e.salary between 10.0 and 20.0
//            e.name
//            ~~~~~~
//            andrew
//            david
//            minmei
//             TEST298; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { "andrew",
                                 "david",
                                 "minmei"
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST299; select e.name from XMLEmpBean e where e.salary not between 10 and 20
    public void testLoop299(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.salary not between 10 and 20 ";
            Query q = em.createQuery(qStr);

//            TEST299; select e.name from XMLEmpBean e where e.salary not between 10 and 20
//            e.name
//         ~~~~~~~~~~~~
//            ahmad
//            george
//            harry
//            ritika
//           charlene
//         Catalina Wei
//         Tom Rayburn
//          TEST299; 7 tuples

            List rList = q.getResultList();

            Object[] targets = { "ahmad",
                                 "george",
                                 "harry",
                                 "ritika",
                                 "charlene",
                                 "Catalina Wei",
                                 "Tom Rayburn"
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST300; select e.name from XMLEmpBean e where not (e.salary  between 10 and 20 and e.salary = 5)
    public void testLoop300(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where not (e.salary  between 10 and 20 and e.salary = 5)  ";
            Query q = em.createQuery(qStr);

//            TEST300; select e.name from XMLEmpBean e where not (e.salary  between 10 and 20 and e.salary = 5)
//            e.name
//         ~~~~~~~~~~~~
//            ahmad
//            andrew
//            david
//            george
//            harry
//            minmei
//            ritika
//           charlene
//         Catalina Wei
//         Tom Rayburn
//          TEST300; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { "ahmad",
                                 "andrew",
                                 "david",
                                 "george",
                                 "harry",
                                 "minmei",
                                 "ritika",
                                 "charlene",
                                 "Catalina Wei",
                                 "Tom Rayburn"
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST301; select x from XMLEmpBean x where x.empid between 10 and 20
    public void testLoop301(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select x from XMLEmpBean x where x.empid between 10 and 20 ";
            Query q = em.createQuery(qStr);

//            TEST301; select x from XMLEmpBean x where x.empid between 10 and 20
//            EmpBean
//            ~~~~~~~
//             [10]
//             TEST301; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST302; select x from XMLEmpBean x where x.empid not between 10 and 20
    public void testLoop302(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select x from XMLEmpBean x where x.empid not between 10 and 20 ";
            Query q = em.createQuery(qStr);

//            TEST302; select x from XMLEmpBean x where x.empid not between 10 and 20
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [6] [7] [8] [9]
//             TEST302; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST303; select e from XMLEmpBean e where (e.empid+3 between 4 and 8)
    public void testLoop303(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.empid+3 between 4 and 8) ";
            Query q = em.createQuery(qStr);

//            TEST303; select e from XMLEmpBean e where (e.empid+3 between 4 and 8)
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5]
//             TEST303; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST304; select d from XMLDeptBean d where (d.name in ('dept1', 'dept2'))
    public void testLoop304(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where (d.name in ('dept1', 'dept2')) ";
            Query q = em.createQuery(qStr);

//            TEST304; select d from XMLDeptBean d where (d.name in ('dept1', 'dept2'))
//            TEST304; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST305; select e from XMLEmpBean e where (e.dept.no not in (1, 2, 3))
    public void testLoop305(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.dept.no not in (1, 2, 3)) ";
            Query q = em.createQuery(qStr);

//            TEST305; select e from XMLEmpBean e where (e.dept.no not in (1, 2, 3))
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [6] [7] [8] [9]
//             TEST305; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST306; select e from XMLEmpBean e where (e.salary in (10000, 15000))
    public void testLoop306(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary in (10000, 15000)) ";
            Query q = em.createQuery(qStr);

//            TEST306; select e from XMLEmpBean e where (e.salary in (10000, 15000))
//            TEST306; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST307; select e from XMLEmpBean e where ( e.salary in (10, 20) )
    public void testLoop307(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where ( e.salary in (10, 20) ) ";
            Query q = em.createQuery(qStr);

//            TEST307; select e from XMLEmpBean e where ( e.salary in (10, 20) )
//            TEST307; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST308; select e from XMLEmpBean e where ( e.salary not in (10, 20) )
    public void testLoop308(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where ( e.salary not in (10, 20) ) ";
            Query q = em.createQuery(qStr);

//            TEST308; select e from XMLEmpBean e where ( e.salary not in (10, 20) )
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [6] [7] [8] [9] [10]
//             TEST308; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST309; select e from XMLEmpBean e where e.salary in (10, 20)
    public void testLoop309(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary in (10, 20)";
            Query q = em.createQuery(qStr);

//            TEST309; select e from XMLEmpBean e where e.salary in (10, 20)
//            TEST309; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST310; select e from XMLEmpBean e where e.salary not in (10, 20)
    public void testLoop310(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary not in (10, 20)";
            Query q = em.createQuery(qStr);

//            TEST310; select e from XMLEmpBean e where e.salary not in (10, 20)
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [6] [7] [8] [9] [10]
//             TEST310; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST311; select d.name from XMLDeptBean d WHERE SQRT(d.no) = 2
    public void testLoop311(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name from XMLDeptBean d WHERE SQRT(d.no) = 2 ";
            Query q = em.createQuery(qStr);

//            TEST311; select d.name from XMLDeptBean d WHERE SQRT(d.no) = 2
//                            TEST311; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST312; select d.name from XMLDeptBean d WHERE (d.name) = 'Development1'
    public void testLoop312(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name from XMLDeptBean d WHERE (d.name) = 'Development1' ";
            Query q = em.createQuery(qStr);

//            TEST312; select d.name from XMLDeptBean d WHERE (d.name) = 'Development1'
//                            TEST312; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST313; select d.name from XMLDeptBean d where CONCAT('Name: ',d.name) like 'Firstname1%'
    public void testLoop313(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.name from XMLDeptBean d where CONCAT('Name: ',d.name) like 'Firstname1%'";
            Query q = em.createQuery(qStr);

//            TEST313; select d.name from XMLDeptBean d where CONCAT('Name: ',d.name) like 'Firstname1%'
//            TEST313; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST314; select e.name from XMLEmpBean e WHERE ABS(e.salary) > 100
    public void testLoop314(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e WHERE ABS(e.salary) > 100";
            Query q = em.createQuery(qStr);

//            TEST314; select e.name from XMLEmpBean e WHERE ABS(e.salary) > 100
//            TEST314; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST315; select e.name from XMLEmpBean e WHERE LENGTH(e.name) > 0
    public void testLoop315(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e WHERE LENGTH(e.name) > 0 ";
            Query q = em.createQuery(qStr);

//            TEST315; select e.name from XMLEmpBean e WHERE LENGTH(e.name) > 0
//            e.name
//         ~~~~~~~~~~~~
//            ahmad
//            andrew
//            david
//            george
//            harry
//            minmei
//            ritika
//           charlene
//         Catalina Wei
//         Tom Rayburn
//          TEST315; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { "ahmad",
                                 "andrew",
                                 "david",
                                 "george",
                                 "harry",
                                 "minmei",
                                 "ritika",
                                 "charlene",
                                 "Catalina Wei",
                                 "Tom Rayburn"
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST316; select e.name from XMLEmpBean e WHERE LOCATE('5',e.name) > 0
    public void testLoop316(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e WHERE LOCATE('5',e.name) > 0 ";
            Query q = em.createQuery(qStr);

//            TEST316; select e.name from XMLEmpBean e WHERE LOCATE('5',e.name) > 0
//            TEST316; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST317; select e.name from XMLEmpBean e WHERE SQRT(e.salary) > 100
    public void testLoop317(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e WHERE SQRT(e.salary) > 100";
            Query q = em.createQuery(qStr);

//            TEST317; select e.name from XMLEmpBean e WHERE SQRT(e.salary) > 100
//            TEST317; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST318; select e from XMLEmpBean e where LOCATE('name','Dave''s name') = 8
    public void testLoop318(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where LOCATE('name','Dave''s name') = 8 ";
            Query q = em.createQuery(qStr);

//            TEST318; select e from XMLEmpBean e where LOCATE('name','Dave''s name') = 8
//                            EmpBean
//                            ~~~~~~~
//                              [1] [2] [3] [4] [5] [6] [7] [8] [9] [10]
//                             TEST318; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST319; select e from XMLEmpBean e where LOCATE('name',e.name) = 6
    public void testLoop319(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where LOCATE('name',e.name) = 6 ";
            Query q = em.createQuery(qStr);

//            TEST319; select e from XMLEmpBean e where LOCATE('name',e.name) = 6
//                            TEST319; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST320; select e from XMLEmpBean e where LOCATE(e.name,'name') = 0
    public void testLoop320(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where LOCATE(e.name,'name') = 0 ";
            Query q = em.createQuery(qStr);

//            TEST320; select e from XMLEmpBean e where LOCATE(e.name,'name') = 0
//                            EmpBean
//                            ~~~~~~~
//                              [1] [2] [3] [4] [5] [6] [7] [8] [9] [10]
//                             TEST320; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST321; select e from XMLEmpBean e where LOCATE(e.name,e.name) = 1
    public void testLoop321(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where LOCATE(e.name,e.name) = 1 ";
            Query q = em.createQuery(qStr);

//            TEST321; select e from XMLEmpBean e where LOCATE(e.name,e.name) = 1
//                            EmpBean
//                            ~~~~~~~
//                              [1] [2] [3] [4] [5] [6] [7] [8] [9] [10]
//                             TEST321; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST322; select e from XMLEmpBean e where LOWER('NAME') = 'name'
    public void testLoop322(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where LOWER('NAME') = 'name'";
            Query q = em.createQuery(qStr);

//            TEST322; select e from XMLEmpBean e where LOWER('NAME') = 'name'
//                            EmpBean
//                            ~~~~~~~
//                              [1] [2] [3] [4] [5] [6] [7] [8] [9] [10]
//                             TEST322; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST323; select e from XMLEmpBean e where UPPER('name') = 'NAME'
    public void testLoop323(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where UPPER('name') = 'NAME' ";
            Query q = em.createQuery(qStr);

//            TEST323; select e from XMLEmpBean e where UPPER('name') = 'NAME'
//                            EmpBean
//                            ~~~~~~~
//                              [1] [2] [3] [4] [5] [6] [7] [8] [9] [10]
//                             TEST323; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST324; select p from ProjectBean p where size(p.dept) > 0
    public void testLoop324(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select p from ProjectBean p where size(p.dept) > 0 ";
            Query q = em.createQuery(qStr);

//            TEST324; select p from ProjectBean p where size(p.dept) > 0
//            ProjectBean
//            ~~~~~~~~~~~
//              [1000] [2000] [3000]
//             TEST324; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { proj1000, proj2000, proj3000 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST325; select p from ProjectBean p where size(p.dept.emps) > 0
    public void testLoop325(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select p from ProjectBean p where size(p.dept.emps) > 0 ";
            Query q = em.createQuery(qStr);

//            TEST325; select p from ProjectBean p where size(p.dept.emps) > 0
//            ProjectBean
//            ~~~~~~~~~~~
//              [1000]
//              [2000]
//             TEST325; 2 tuples

            List rList = q.getResultList();

            Object[] targets = { proj1000, proj2000 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST326; select p from ProjectBean p where size(p.dept.emps) > 0
    public void testLoop326(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select p from ProjectBean p where size(p.dept.emps) > 0 ";
            Query q = em.createQuery(qStr);

//            TEST326; select p from ProjectBean p where size(p.dept.emps) > 0
//            ProjectBean
//            ~~~~~~~~~~~
//              [1000] [2000]
//             TEST326; 2 tuples

            List rList = q.getResultList();

            Object[] targets = { proj1000, proj2000 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST327; select d from XMLDeptBean d where (size(d.emps) = 4)
    public void testLoop327(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where (size(d.emps) = 4) ";
            Query q = em.createQuery(qStr);

//            TEST327; select d from XMLDeptBean d where (size(d.emps) = 4)
//            DeptBean
//            ~~~~~~~~
//             [210]
//             TEST327; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { dep210 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST328; select d from XMLDeptBean d where size(d.emps) > 10
    public void testLoop328(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where size(d.emps) > 10 ";
            Query q = em.createQuery(qStr);

//            TEST328; select d from XMLDeptBean d where size(d.emps) > 10
//            TEST328; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST329; select e from XMLEmpBean e where e.salary > 7 and lower(e.name) = 'david'
    public void testLoop329(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary > 7 and lower(e.name) = 'david'";
            Query q = em.createQuery(qStr);

//            TEST329; select e from XMLEmpBean e where e.salary > 7 and lower(e.name) = 'david'
//                            EmpBean
//                            ~~~~~~~
//                              [1]
//                             TEST329; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { emp1 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST330; select e from XMLEmpBean e where e.salary < 15 and upper(e.name) = 'DAVID'
    public void testLoop330(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary < 15 and upper(e.name) = 'DAVID'";
            Query q = em.createQuery(qStr);

//            TEST330; select e from XMLEmpBean e where e.salary < 15 and upper(e.name) = 'DAVID'
//                            EmpBean
//                            ~~~~~~~
//                              [1]
//                             TEST330; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { emp1 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST331; select e from XMLEmpBean e where e.salary < 15 and lower(upper(e.name)) = 'david'
    public void testLoop331(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary < 15 and lower(upper(e.name)) = 'david'";
            Query q = em.createQuery(qStr);

//            TEST331; select e from XMLEmpBean e where e.salary < 15 and lower(upper(e.name)) = 'david'
//                            EmpBean
//                            ~~~~~~~
//                              [1]
//                             TEST331; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { emp1 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST332; select e from XMLEmpBean e where e.salary > 5 and (e.name) = 'david'
    public void testLoop332(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary > 5 and (e.name) = 'david'";
            Query q = em.createQuery(qStr);

//            TEST332; select e from XMLEmpBean e where e.salary > 5 and (e.name) = 'david'
//                            EmpBean
//                            ~~~~~~~
//                              [1]
//                             TEST332; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { emp1 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST333; select e from XMLEmpBean e where e.salary > 5 and e.name = 'mgr'
    public void testLoop333(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary > 5 and e.name = 'mgr'";
            Query q = em.createQuery(qStr);

//            TEST333; select e from XMLEmpBean e where e.salary > 5 and e.name = 'mgr'
//                            TEST333; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST334; select e from XMLEmpBean e where e.salary > 5 and locate('vid', e.name) > 0
    public void testLoop334(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary > 5 and locate('vid', e.name) > 0";
            Query q = em.createQuery(qStr);

//            TEST334; select e from XMLEmpBean e where e.salary > 5 and locate('vid', e.name) > 0
//            EmpBean
//            ~~~~~~~
//              [1]
//             TEST334; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { emp1 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST335; select e from XMLEmpBean e where e.salary > 5 and e.empid < 4
    public void testLoop335(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary > 5 and e.empid < 4 ";
            Query q = em.createQuery(qStr);

//            TEST335; select e from XMLEmpBean e where e.salary > 5 and e.empid < 4
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3]
//             TEST335; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST336; select e from XMLEmpBean e where e.salary > 5 and length(e.name) > 5
    public void testLoop336(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary > 5 and length(e.name) > 5";
            Query q = em.createQuery(qStr);

//            TEST336; select e from XMLEmpBean e where e.salary > 5 and length(e.name) > 5
//            EmpBean
//            ~~~~~~~
//              [2] [3]
//             TEST336; 2 tuples

            List rList = q.getResultList();

            Object[] targets = { emp2, emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST337; select d from XMLDeptBean d where d.mgr.salary > 7 and d.name = 'Sales' and size(d.emps) > 0
    public void testLoop337(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where d.mgr.salary > 7 and d.name = 'Sales' and size(d.emps) > 0";
            Query q = em.createQuery(qStr);

//            TEST337; select d from XMLDeptBean d where d.mgr.salary > 7 and d.name = 'Sales' and size(d.emps) > 0
//                            TEST337; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST338; select e from XMLEmpBean e where e.salary > 5 and size(e.dept) = 1
    public void testLoop338(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary > 5 and size(e.dept) = 1";
            Query q = em.createQuery(qStr);

//            TEST338; select e from XMLEmpBean e where e.salary > 5 and size(e.dept) = 1
//                            EmpBean
//                            ~~~~~~~
//                              [1] [2] [3]
//                             TEST338; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST339; select e from XMLEmpBean e where e.salary > 5 and sqrt(e.salary) > 2.5
    public void testLoop339(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary > 5 and sqrt(e.salary) > 2.5";
            Query q = em.createQuery(qStr);

//            TEST339; select e from XMLEmpBean e where e.salary > 5 and sqrt(e.salary) > 2.5
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3]
//             TEST339; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST340; select e from XMLEmpBean e where (e.dept <> all (select d from XMLDeptBean d where d.no = 100 or d.no = 300))
    public void testLoop340(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.dept <> all (select d from XMLDeptBean d where d.no = 100 or d.no = 300))";
            Query q = em.createQuery(qStr);

//            TEST340; select e from XMLEmpBean e where (e.dept <> all (select d from XMLDeptBean d where d.no = 100 or d.no = 300))
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [7] [9]
//             TEST340; 7 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp7, emp9 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST341; select e from XMLEmpBean e where (e.dept = any (select d from XMLDeptBean d where d.no = 100 or d.no = 300))
    public void testLoop341(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.dept = any (select d from XMLDeptBean d where d.no = 100 or d.no = 300)) ";
            Query q = em.createQuery(qStr);

//            TEST341; select e from XMLEmpBean e where (e.dept = any (select d from XMLDeptBean d where d.no = 100 or d.no = 300))
//            EmpBean
//            ~~~~~~~
//              [6] [8]
//             TEST341; 2 tuples

            List rList = q.getResultList();

            Object[] targets = { emp6, emp8 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST342; select e from XMLEmpBean e where e.dept = any (select d from XMLDeptBean d where d.no = 100 or d.no = 300)
    public void testLoop342(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.dept = any (select d from XMLDeptBean d where d.no = 100 or d.no = 300) ";
            Query q = em.createQuery(qStr);

//            TEST342; select e from XMLEmpBean e where e.dept = any (select d from XMLDeptBean d where d.no = 100 or d.no = 300)
//                            EmpBean
//                            ~~~~~~~
//                              [6] [8]
//                             TEST342; 2 tuples

            List rList = q.getResultList();

            Object[] targets = { emp6, emp8 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST343; select e from XMLEmpBean e where e.empid = any (select d.no from XMLDeptBean d)
    public void testLoop343(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.empid = any (select d.no from XMLDeptBean d) ";
            Query q = em.createQuery(qStr);

//            TEST343; select e from XMLEmpBean e where e.empid = any (select d.no from XMLDeptBean d)
//                            TEST343; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST344; select e.name from XMLEmpBean e where e.salary > 0 and e.salary = any (select f.salary from XMLEmpBean f where f.salary > 0)
    public void testLoop344(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.salary > 0 and e.salary = any (select f.salary from XMLEmpBean f where f.salary > 0) ";
            Query q = em.createQuery(qStr);

//            TEST344; select e.name from XMLEmpBean e where e.salary > 0 and e.salary = any (select f.salary from XMLEmpBean f where f.salary > 0)
//                            e.name
//                            ~~~~~~
//                            andrew
//                            david
//                            minmei
//                             TEST344; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { "andrew", "david", "minmei" };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST345; select object(e) from XMLEmpBean e where e.empid = any (select d.no from XMLDeptBean d)
    public void testLoop345(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select object(e) from XMLEmpBean e where e.empid = any (select d.no from XMLDeptBean d) ";
            Query q = em.createQuery(qStr);

//            TEST345; select object(e) from XMLEmpBean e where e.empid = any (select d.no from XMLDeptBean d)
//                            TEST345; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST346; select e from XMLEmpBean e where e.dept <> all (select d from XMLDeptBean d where d.no = 100 or d.no = 300)
    public void testLoop346(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.dept <> all (select d from XMLDeptBean d where d.no = 100 or d.no = 300) ";
            Query q = em.createQuery(qStr);

//            TEST346; select e from XMLEmpBean e where e.dept <> all (select d from XMLDeptBean d where d.no = 100 or d.no = 300)
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [7] [9]
//             TEST346; 7 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp7, emp9 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST347; select e.name, d.name from XMLEmpBean as e, DeptBean as d where e.name = any (select d.name from d.projects as d where d.name = 'DEV')
    public void testLoop347(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name, d.name from XMLEmpBean as e, DeptBean as d where e.name = any (select d.name from d.projects as d where d.name = 'DEV') ";
            Query q = em.createQuery(qStr);

//            TEST347; select e.name, d.name from XMLEmpBean as e, DeptBean as d where e.name = any (select d.name from d.projects as d where d.name = 'DEV')
//                            TEST347; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST348; select e, d from XMLEmpBean as e , DeptBean as d where e.name = 'harry' and  e.name = any (select d.name from d.projects as d where d.name = 'DEV' and d.name = e.name)
    public void testLoop348(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e, d from XMLEmpBean as e , DeptBean as d where e.name = 'harry' and  e.name = any (select d.name from d.projects as d where d.name = 'DEV' and d.name = e.name)";
            Query q = em.createQuery(qStr);

//            TEST348; select e, d from XMLEmpBean as e , DeptBean as d where e.name = 'harry' and  e.name = any (select d.name from d.projects as d where d.name = 'DEV' and d.name = e.name)
//                            TEST348; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST349; select object(f) from XMLDeptBean f where f.no not in(select e.empid from XMLEmpBean e where e.empid =f.no-1)
    public void testLoop349(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select object(f) from XMLDeptBean f where f.no not in(select e.empid from XMLEmpBean e where e.empid =f.no-1) ";
            Query q = em.createQuery(qStr);

//            TEST349; select object(f) from XMLDeptBean f where f.no not in(select e.empid from XMLEmpBean e where e.empid =f.no-1)
//            DeptBean
//            ~~~~~~~~
//             [100] [200] [210] [220] [300]
//             TEST349; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { dep100, dep200, dep210, dep220, dep300 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST350; select d from XMLDeptBean d where (d.mgr.name like '%')
    public void testLoop350(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where (d.mgr.name like '%') ";
            Query q = em.createQuery(qStr);

//            TEST350; select d from XMLDeptBean d where (d.mgr.name like '%')
//            DeptBean
//            ~~~~~~~~
//             [100] [200] [210] [220] [300]
//             TEST350; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { dep100, dep200, dep210, dep220, dep300 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST351; select d from XMLDeptBean d where (d.name like 'ss')
    public void testLoop351(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where (d.name like 'ss') ";
            Query q = em.createQuery(qStr);

//            TEST351; select d from XMLDeptBean d where (d.name like 'ss')
//            TEST351; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST352; select d from XMLDeptBean d where (d.name not like '_bbc')
    public void testLoop352(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where (d.name not like '_bbc')";
            Query q = em.createQuery(qStr);

//            TEST352; select d from XMLDeptBean d where (d.name not like '_bbc')
//            DeptBean
//            ~~~~~~~~
//             [100] [200] [210] [220] [300]
//             TEST352; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { dep100, dep200, dep210, dep220, dep300 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST353; select d from XMLDeptBean d where (d.name not like 'bbc')
    public void testLoop353(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where (d.name not like 'bbc') ";
            Query q = em.createQuery(qStr);

//            TEST353; select d from XMLDeptBean d where (d.name not like 'bbc')
//            DeptBean
//            ~~~~~~~~
//             [100] [200] [210] [220] [300]
//             TEST353; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { dep100, dep200, dep210, dep220, dep300 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST354; select e from XMLEmpBean e where (e.dept.name like '')
    public void testLoop354(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.dept.name like '') ";
            Query q = em.createQuery(qStr);

//            TEST354; select e from XMLEmpBean e where (e.dept.name like '')
//            TEST354; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST355; select e from XMLEmpBean e where (e.name like '%ac')
    public void testLoop355(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.name like '%ac') ";
            Query q = em.createQuery(qStr);

//            TEST355; select e from XMLEmpBean e where (e.name like '%ac')
//            TEST355; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST356; select e from XMLEmpBean e where (e.name like '///_foo' escape '/')
    public void testLoop356(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.name like '///_foo' escape '/') ";
            Query q = em.createQuery(qStr);

//            TEST356; select e from XMLEmpBean e where (e.name like '///_foo' escape '/')
//            TEST356; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST357; select e from XMLEmpBean e where (e.name like '//_foo' escape '/')
    public void testLoop357(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.name like '//_foo' escape '/') ";
            Query q = em.createQuery(qStr);

//            TEST357; select e from XMLEmpBean e where (e.name like '//_foo' escape '/')
//            TEST357; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST358; select e from XMLEmpBean e where (e.name like '/_foo' escape '/')
    public void testLoop358(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.name like '/_foo' escape '/') ";
            Query q = em.createQuery(qStr);

//            TEST358; select e from XMLEmpBean e where (e.name like '/_foo' escape '/')
//            TEST358; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST359; select e from XMLEmpBean e where (not e.dept.name like '')
    public void testLoop359(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        final String dbProductName = (testProps == null) ? "UNKNOWN" : ((testProps.get("dbProductName") == null) ? "UNKNOWN" : (String) testProps.get("dbProductName"));

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (not e.dept.name like '') ";
            Query q = em.createQuery(qStr);

//            TEST359; select e from XMLEmpBean e where (not e.dept.name like '')
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [6] [7] [8] [9]
//             TEST359; 9 tuples

// Oracle:
//            TEST359; select e from XMLEmpBean e where (not e.dept.name like '')
//            TEST359; 0 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9 };
            if (DatabaseVendor.checkDBProductName(dbProductName, DatabaseVendor.ORACLE)) {
                targets = new Object[] {};
            }
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST360; select d from XMLDeptBean d where LOWER(d.name) like '%AR%'
    public void testLoop360(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where LOWER(d.name) like '%AR%' ";
            Query q = em.createQuery(qStr);

//            TEST360; select d from XMLDeptBean d where LOWER(d.name) like '%AR%'
//            TEST360; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST361; select d from XMLDeptBean d where LOWER(d.name) like '%a%'
    public void testLoop361(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where LOWER(d.name) like '%a%' ";
            Query q = em.createQuery(qStr);

//            TEST361; select d from XMLDeptBean d where LOWER(d.name) like '%a%'
//            DeptBean
//            ~~~~~~~~
//             [200] [300]
//             TEST361; 2 tuples

            List rList = q.getResultList();

            Object[] targets = { dep200, dep300 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST362; select d from XMLDeptBean d where UPPER(d.name) like '%AR%'
    public void testLoop362(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where UPPER(d.name) like '%AR%'";
            Query q = em.createQuery(qStr);

//            TEST362; select d from XMLDeptBean d where UPPER(d.name) like '%AR%'
//            TEST362; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST363; select d from XMLDeptBean d where UPPER(d.name) like '%a%'
    public void testLoop363(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        final String dbProductName = (testProps == null) ? "UNKNOWN" : ((testProps.get("dbProductName") == null) ? "UNKNOWN" : (String) testProps.get("dbProductName"));

        // TODO: Fails on SQLServer, does not match the answer file:
        // Likely a server config regarding the processing of case sensitivity.

//        Validate Output for test "JULoopQueryAnoTest.testLoop363":
//            SQL: select d from XMLDeptBean d where UPPER(d.name) like '%a%'
//            Expected Output:
//
//            Generated Results:
//              0  ( DeptBean: no=200 name =Admin)
//              1  ( DeptBean: no=300 name =Sales)
        if (DatabaseVendor.checkDBProductName(dbProductName, DatabaseVendor.SQLSERVER)) {
            System.out.println("This test has been disabled on sqlserver.");
            return;
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where UPPER(d.name) like '%a%'";
            Query q = em.createQuery(qStr);

//            TEST363; select d from XMLDeptBean d where UPPER(d.name) like '%a%'
//            TEST363; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST364; select e from XMLDeptBean e where (e.name) like '%1%'
    public void testLoop364(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLDeptBean e where (e.name) like '%1%'";
            Query q = em.createQuery(qStr);

//            TEST364; select e from XMLDeptBean e where (e.name) like '%1%'
//            TEST364; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST365; select e from XMLEmpBean e where e = e.dept.mgr and e.name like 'dumb'
    public void testLoop365(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e = e.dept.mgr and e.name like 'dumb' ";
            Query q = em.createQuery(qStr);

//            TEST365; select e from XMLEmpBean e where e = e.dept.mgr and e.name like 'dumb'
//                            TEST365; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST366; select e from XMLEmpBean e where e.dept.name like '%1'
    public void testLoop366(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.dept.name like '%1' ";
            Query q = em.createQuery(qStr);

//            TEST366; select e from XMLEmpBean e where e.dept.name like '%1'
//            TEST366; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST367; select e from XMLEmpBean e where e.dept.name like '%1' or e.dept.name not like 'Departme_t2'
    public void testLoop367(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.dept.name like '%1' or e.dept.name not like 'Departme_t2' ";
            Query q = em.createQuery(qStr);

//            TEST367; select e from XMLEmpBean e where e.dept.name like '%1' or e.dept.name not like 'Departme_t2'
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [6] [7] [8] [9]
//             TEST367; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST368; select e from XMLEmpBean e where e.name like '%DEV'
    public void testLoop368(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.name like '%DEV'";
            Query q = em.createQuery(qStr);

//            TEST368; select e from XMLEmpBean e where e.name like '%DEV'
//            TEST368; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST369; select e from XMLEmpBean e where e.name not like '%DEV'
    public void testLoop369(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.name not like '%DEV' ";
            Query q = em.createQuery(qStr);

//            TEST369; select e from XMLEmpBean e where e.name not like '%DEV'
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [6] [7] [8] [9] [10]
//             TEST369; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST370; select e from XMLEmpBean e where e.name not like '\_D%' escape '\'
    public void testLoop370(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.name not like '\\_D%' escape '\\' ";
            Query q = em.createQuery(qStr);

//            TEST370; select e from XMLEmpBean e where e.name not like '\_D%' escape '\'
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [6] [7] [8] [9] [10]
//             TEST370; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST371; select e.hireDate, e.hireTime, e.hireTimestamp from XMLEmpBean e, DeptBean d where (d.no <= e.dept.no and e.dept.name like '%Department%')
    public void testLoop371(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.hireDate, e.hireTime, e.hireTimestamp from XMLEmpBean e, DeptBean d where (d.no <= e.dept.no and e.dept.name like '%Department%') ";
            Query q = em.createQuery(qStr);

//            TEST371; select e.hireDate, e.hireTime, e.hireTimestamp from XMLEmpBean e, DeptBean d where (d.no <= e.dept.no and e.dept.name like '%Department%')
//            TEST371; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST372; select distinct x.empid, x.name from XMLEmpBean x
    public void testLoop372(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select distinct x.empid, x.name from XMLEmpBean x ";
            Query q = em.createQuery(qStr);

//            TEST372; select distinct x.empid, x.name from XMLEmpBean x
//            distinct x.empid    x.name
//            ~~~~~~~~~~~~~~~~ ~~~~~~~~~~~~
//                   1            david
//                   10        Catalina Wei
//                   2            andrew
//                   3            minmei
//                   4            george
//                   5            ritika
//                   6            ahmad
//                   7           charlene
//                   8         Tom Rayburn
//                   9            harry
//             TEST372; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, "david" },
                                 new Object[] { 10, "Catalina Wei" },
                                 new Object[] { 2, "andrew" },
                                 new Object[] { 3, "minmei" },
                                 new Object[] { 4, "george" },
                                 new Object[] { 5, "ritika" },
                                 new Object[] { 6, "ahmad" },
                                 new Object[] { 7, "charlene" },
                                 new Object[] { 8, "Tom Rayburn" },
                                 new Object[] { 9, "harry" },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST373; select distinct x.no, x.name from XMLDeptBean x,   EmpBean y where x.no = y.dept.no
    public void testLoop373(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select distinct x.no, x.name from XMLDeptBean x,   EmpBean y where x.no = y.dept.no ";
            Query q = em.createQuery(qStr);

//            TEST373; select distinct x.no, x.name from XMLDeptBean x,   EmpBean y where x.no = y.dept.no
//                            distinct x.no   x.name
//                            ~~~~~~~~~~~~~ ~~~~~~~~~~~
//                                 100          CEO
//                                 200         Admin
//                                 210      Development
//                                 220        Service
//                             TEST373; 4 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 100, "CEO" },
                                 new Object[] { 200, "Admin" },
                                 new Object[] { 210, "Development" },
                                 new Object[] { 220, "Service" },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST374; select distinct x.name from XMLEmpBean x  order by x.name
    public void testLoop374(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select distinct x.name from XMLEmpBean x  order by x.name ";
            Query q = em.createQuery(qStr);

//            TEST374; select distinct x.name from XMLEmpBean x  order by x.name
//            distinct x.name
//            ~~~~~~~~~~~~~~~
//                 ahmad
//                 david
//                 harry
//                andrew
//                george
//                minmei
//                ritika
//               charlene
//              Tom Rayburn
//             Catalina Wei
//             TEST374; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { "ahmad",
                                 "david",
                                 "harry",
                                 "andrew",
                                 "george",
                                 "minmei",
                                 "ritika",
                                 "charlene",
                                 "Tom Rayburn",
                                 "Catalina Wei"
            };
            validateQueryResult(testName, qStr, rList, targets, true);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST375; select e from XMLEmpBean e where e.empid < (select max(f.no) from XMLDeptBean f )
    public void testLoop375(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.empid < (select max(f.no) from XMLDeptBean f ) ";
            Query q = em.createQuery(qStr);

//            TEST375; select e from XMLEmpBean e where e.empid < (select max(f.no) from XMLDeptBean f )
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [6] [7] [8] [9] [10]
//             TEST375; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST376; select e.empid from XMLEmpBean e where e.empid = (select max(e.no) from XMLDeptBean e)
    public void testLoop376(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid from XMLEmpBean e where e.empid = (select max(e.no) from XMLDeptBean e) ";
            Query q = em.createQuery(qStr);

//            TEST376; select e.empid from XMLEmpBean e where e.empid = (select max(e.no) from XMLDeptBean e)
//                            TEST376; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST377; select e.empid from XMLEmpBean e where e.empid > (select min(e.no) from XMLDeptBean e)
    public void testLoop377(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid from XMLEmpBean e where e.empid > (select min(e.no) from XMLDeptBean e)";
            Query q = em.createQuery(qStr);

//            TEST377; select e.empid from XMLEmpBean e where e.empid > (select min(e.no) from XMLDeptBean e)
//            TEST377; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST378; select e.empid, e.isManager, e.salary from XMLEmpBean e where e.empid = (select max(f.no) from XMLDeptBean f )
    public void testLoop378(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid, e.isManager, e.salary from XMLEmpBean e where e.empid = (select max(f.no) from XMLDeptBean f ) ";
            Query q = em.createQuery(qStr);

//            TEST378; select e.empid, e.isManager, e.salary from XMLEmpBean e where e.empid = (select max(f.no) from XMLDeptBean f )
//                            TEST378; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST379; select e from XMLEmpBean e where e.salary = (select max(e.salary) from XMLEmpBean e where e.salary > 0)
    public void testLoop379(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary = (select max(e.salary) from XMLEmpBean e where e.salary > 0) ";
            Query q = em.createQuery(qStr);

//            TEST379; select e from XMLEmpBean e where e.salary = (select max(e.salary) from XMLEmpBean e where e.salary > 0)
//                            EmpBean
//                            ~~~~~~~
//                              [3]
//                             TEST379; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST380; select e from XMLEmpBean e where e.salary > (select min(e.salary) from XMLEmpBean e where e.salary > 0)
    public void testLoop380(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.salary > (select min(e.salary) from XMLEmpBean e where e.salary > 0) ";
            Query q = em.createQuery(qStr);

//            TEST380; select e from XMLEmpBean e where e.salary > (select min(e.salary) from XMLEmpBean e where e.salary > 0)
//            EmpBean
//            ~~~~~~~
//              [2] [3]
//             TEST380; 2 tuples

            List rList = q.getResultList();

            Object[] targets = { emp2, emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST381; select e from XMLEmpBean e where (e.empid = (select max(e.empid) from XMLEmpBean e))
    public void testLoop381(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.empid = (select max(e.empid) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST381; select e from XMLEmpBean e where (e.empid = (select max(e.empid) from XMLEmpBean e))
//            EmpBean
//            ~~~~~~~
//             [10]
//             TEST381; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST382; select e from XMLEmpBean e where (e.hireDate = (select max(e.hireDate) from XMLEmpBean e))
    public void testLoop382(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.hireDate = (select max(e.hireDate) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST382; select e from XMLEmpBean e where (e.hireDate = (select max(e.hireDate) from XMLEmpBean e))
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [6] [7] [8] [9] [10]
//             TEST382; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST383; select e from XMLEmpBean e where (e.hireDate = (select min(e.hireDate) from XMLEmpBean e))
    public void testLoop383(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.hireDate = (select min(e.hireDate) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST383; select e from XMLEmpBean e where (e.hireDate = (select min(e.hireDate) from XMLEmpBean e))
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [6] [7] [8] [9] [10]
//             TEST383; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    //  TEST384; select e from XMLEmpBean e where (e.salary  = (select count(distinct e.name) from XMLEmpBean e))
    public void testLoop384(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary  = (select count(distinct e.name) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST384; select e from XMLEmpBean e where (e.salary  = (select count(distinct e.name) from XMLEmpBean e))
//            TEST384; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST385; select e from XMLEmpBean e where (e.salary  = (select count(e.hireDate) from XMLEmpBean e))
    public void testLoop385(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary  = (select count(e.hireDate) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST385; select e from XMLEmpBean e where (e.salary  = (select count(e.hireDate) from XMLEmpBean e))
//            TEST385; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST386; select e from XMLEmpBean e where (e.salary > (select avg(  e.salary) from XMLEmpBean e))
    public void testLoop386(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select avg(  e.salary) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST386; select e from XMLEmpBean e where (e.salary > (select avg(  e.salary) from XMLEmpBean e))
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3]
//             TEST386; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST387; select e from XMLEmpBean e where (e.salary > (select avg( distinct  e.salary) from XMLEmpBean e))
    public void testLoop387(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select avg( distinct  e.salary) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST387; select e from XMLEmpBean e where (e.salary > (select avg( distinct  e.salary) from XMLEmpBean e))
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3]
//             TEST387; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST388; select e from XMLEmpBean e where (e.salary > (select avg(e.salary) from XMLEmpBean e))
    public void testLoop388(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select avg(e.salary) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST388; select e from XMLEmpBean e where (e.salary > (select avg(e.salary) from XMLEmpBean e))
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3]
//             TEST388; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST389; select e from XMLEmpBean e where (e.salary > (select count(  e.salary) from XMLEmpBean e))
    public void testLoop389(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select count(  e.salary) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST389; select e from XMLEmpBean e where (e.salary > (select count(  e.salary) from XMLEmpBean e))
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3]
//             TEST389; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST390; select e from XMLEmpBean e where (e.salary > (select count( distinct  e.salary) from XMLEmpBean e))
    public void testLoop390(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select count( distinct  e.salary) from XMLEmpBean e))";
            Query q = em.createQuery(qStr);

//            TEST390; select e from XMLEmpBean e where (e.salary > (select count( distinct  e.salary) from XMLEmpBean e))
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3]
//             TEST390; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST391; select e from XMLEmpBean e where (e.salary > (select count(e.salary) from XMLEmpBean e))
    public void testLoop391(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select count(e.salary) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST391; select e from XMLEmpBean e where (e.salary > (select count(e.salary) from XMLEmpBean e))
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3]
//             TEST391; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST392; select e from XMLEmpBean e where (e.salary > (select max( e.salary) from XMLEmpBean e))
    public void testLoop392(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select max( e.salary) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST392; select e from XMLEmpBean e where (e.salary > (select max( e.salary) from XMLEmpBean e))
//            TEST392; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST393; select e from XMLEmpBean e where (e.salary > (select max( distinct  e.salary) from XMLEmpBean e))
    public void testLoop393(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select max( distinct  e.salary) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST393; select e from XMLEmpBean e where (e.salary > (select max( distinct  e.salary) from XMLEmpBean e))
//            TEST393; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST394; select e from XMLEmpBean e where (e.salary > (select max(e.salary) from XMLEmpBean e))
    public void testLoop394(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select max(e.salary) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST394; select e from XMLEmpBean e where (e.salary > (select max(e.salary) from XMLEmpBean e))
//            TEST394; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST395; select e from XMLEmpBean e where (e.salary > (select min( e.salary) from XMLEmpBean e))
    public void testLoop395(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select min( e.salary) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST395; select e from XMLEmpBean e where (e.salary > (select min( e.salary) from XMLEmpBean e))
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3]
//             TEST395; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST396; select e from XMLEmpBean e where (e.salary > (select min( distinct  e.salary) from XMLEmpBean e))
    public void testLoop396(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select min( distinct  e.salary) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST396; select e from XMLEmpBean e where (e.salary > (select min( distinct  e.salary) from XMLEmpBean e))
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3]
//             TEST396; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST397; select e from XMLEmpBean e where (e.salary > (select min(e.salary)  from XMLEmpBean e))
    public void testLoop397(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select min(e.salary)  from XMLEmpBean e))";
            Query q = em.createQuery(qStr);

//            TEST397; select e from XMLEmpBean e where (e.salary > (select min(e.salary)  from XMLEmpBean e))
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3]
//             TEST397; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST398; select e from XMLEmpBean e where (e.salary > (select min(e.salary) from XMLEmpBean e))
    public void testLoop398(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select min(e.salary) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST398; select e from XMLEmpBean e where (e.salary > (select min(e.salary) from XMLEmpBean e))
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3]
//             TEST398; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST399; select e from XMLEmpBean e where (e.salary > (select sum( e.salary) from XMLEmpBean e))
    public void testLoop399(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select sum( e.salary) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST399; select e from XMLEmpBean e where (e.salary > (select sum( e.salary) from XMLEmpBean e))
//            TEST399; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST400; select e from XMLEmpBean e where (e.salary > (select sum( distinct  e.salary) from XMLEmpBean e))
    public void testLoop400(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select sum( distinct  e.salary) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST400; select e from XMLEmpBean e where (e.salary > (select sum( distinct  e.salary) from XMLEmpBean e))
//            TEST400; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST401; select e from XMLEmpBean e where (e.salary > (select sum(e.salary) from XMLEmpBean e))
    public void testLoop401(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select sum(e.salary) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST401; select e from XMLEmpBean e where (e.salary > (select sum(e.salary) from XMLEmpBean e))
//            TEST401; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST402; select max( e.salary)  from XMLDeptBean d, in(d.emps) e
    public void testLoop402(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select max( e.salary)  from XMLDeptBean d, in(d.emps) e ";
            Query q = em.createQuery(qStr);

//            TEST402; select max( e.salary)  from XMLDeptBean d, in(d.emps) e
//            max( e.salary)
//            ~~~~~~~~~~~~~~
//                 15.5
//             TEST402; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { 15.5 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST403; select count(distinct e.isManager) from XMLEmpBean e
    public void testLoop403(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select count(distinct e.isManager) from XMLEmpBean e";
            Query q = em.createQuery(qStr);

//            TEST403; select count(distinct e.isManager) from XMLEmpBean e
//            count(distinct e.isManager)
//            ~~~~~~~~~~~~~~~~~~~~~~~~~~~
//                         2
//             TEST403; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { 2l };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST404; select x from XMLEmpBean x where x.salary < (select max(y.salary) from XMLEmpBean y where x.salary=y.salary)
    public void testLoop404(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select x from XMLEmpBean x where x.salary < (select max(y.salary) from XMLEmpBean y where x.salary=y.salary)";
            Query q = em.createQuery(qStr);

//            TEST404; select x from XMLEmpBean x where x.salary < (select max(y.salary) from XMLEmpBean y where x.salary=y.salary)
//            TEST404; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST405; select e from XMLEmpBean e where (e.salary > (select avg(e.salary) from XMLEmpBean e))
    public void testLoop405(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select avg(e.salary) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST405; select e from XMLEmpBean e where (e.salary > (select avg(e.salary) from XMLEmpBean e))
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3]
//             TEST405; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST406; select e from XMLEmpBean e where (e.salary > (select avg(e.salary) from XMLEmpBean e))
    public void testLoop406(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (e.salary > (select avg(e.salary) from XMLEmpBean e)) ";
            Query q = em.createQuery(qStr);

//            TEST406; select e from XMLEmpBean e where (e.salary > (select avg(e.salary) from XMLEmpBean e))
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3]
//             TEST406; 3 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST407; select count(e.isManager) from XMLEmpBean e
    public void testLoop407(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select count(e.isManager) from XMLEmpBean e  ";
            Query q = em.createQuery(qStr);

//            TEST407; select count(e.isManager) from XMLEmpBean e
//            count(e.isManager)
//            ~~~~~~~~~~~~~~~~~~
//                    10
//             TEST407; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { 10l };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//   TEST408; select e.empid from XMLEmpBean e where e.empid < (select avg(e.empid) from XMLEmpBean e)
    public void testLoop408(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        final String dbProductName = (testProps == null) ? "UNKNOWN" : ((testProps.get("dbProductName") == null) ? "UNKNOWN" : (String) testProps.get("dbProductName"));

        final boolean isOracle = DatabaseVendor.checkDBProductName(dbProductName, DatabaseVendor.ORACLE);
        final boolean isPostgres = DatabaseVendor.checkDBProductName(dbProductName, DatabaseVendor.POSTGRES);
        final boolean isSQLServer = DatabaseVendor.checkDBProductName(dbProductName, DatabaseVendor.SQLSERVER);

        // TODO: Address postgresql result
        /*
         * Excluding postgres because it returns different results, and tehre is no existing answer file
         * for postgresql
         *
         * Expected Output:
         * 0 1
         * 1 2
         * 2 3
         * 3 4
         *
         * Generated Results:
         * 0 3
         * 1 2
         * 2 4
         * 3 5
         * 4 1
         *
         */
        if (DatabaseVendor.checkDBProductName(dbProductName, DatabaseVendor.POSTGRES)) {
            System.out.println("Test is marked excluded against PostgreSQL.");
            return;
        }

        JPAPersistenceProvider provider = JPAPersistenceProvider.resolveJPAPersistenceProvider(jpaResource);

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid from XMLEmpBean e where e.empid < (select avg(e.empid) from XMLEmpBean e) ";
            Query q = em.createQuery(qStr);

//            TEST408; select e.empid from XMLEmpBean e where e.empid < (select avg(e.empid) from XMLEmpBean e)
//            e.empid
//            ~~~~~~~
//               1
//               2
//               3
//               4
//             TEST408; 4 tuples

// Oracle:
//            TEST408; select e.empid from XMLEmpBean e where e.empid < (select avg(e.empid) from XMLEmpBean e)
//            e.empid
//            ~~~~~~~
//               1
//               2
//               3
//               4
//               5
//             TEST408; 5 tuples
//
            List rList = q.getResultList();

            /*
             * TODO: investigate what the correct behavior should be.
             */
            Object[] targets = new Object[] { 1, 2, 3, 4 };
            if (DatabaseVendor.checkDBProductName(dbProductName, DatabaseVendor.ORACLE)) {
                targets = new Object[] { 1, 2, 3, 4, 5 };
            }

            if (JPAPersistenceProvider.HIBERNATE.equals(provider)) {
                if (isSQLServer) {
                    targets = new Object[] { 1, 2, 3, 4 };
                } else {
                    targets = new Object[] { 1, 2, 3, 4, 5 };
                }
            }

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST409; select SUM (distinct e.bonus ) from XMLEmpBean e where e.dept.no = 5 group by e.bonus, e.salary having e.salary > 1000 and e.bonus < 1000
    public void testLoop409(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select SUM (distinct e.bonus ) from XMLEmpBean e where e.dept.no = 5 group by e.bonus, e.salary having e.salary > 1000 and e.bonus < 1000 ";
            Query q = em.createQuery(qStr);

//           TEST409; select SUM (distinct e.bonus ) from XMLEmpBean e where e.dept.no = 5 group by e.bonus, e.salary having e.salary > 1000 and e.bonus < 1000
//                           TEST409; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST410; select SUM (e.bonus), SUM(e.salary) from XMLEmpBean e where e.dept.no = 5 group by e.bonus, e.salary
    public void testLoop410(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select SUM (e.bonus), SUM(e.salary) from XMLEmpBean e where e.dept.no = 5 group by e.bonus, e.salary ";
            Query q = em.createQuery(qStr);

//            TEST410; select SUM (e.bonus), SUM(e.salary) from XMLEmpBean e where e.dept.no = 5 group by e.bonus, e.salary
//                            TEST410; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST411; select avg(e.dept.no) from XMLEmpBean e  group by e.empid
    public void testLoop411(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        JPAPersistenceProvider provider = JPAPersistenceProvider.resolveJPAPersistenceProvider(jpaResource);

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select avg(e.dept.no) from XMLEmpBean e  group by e.empid ";
            Query q = em.createQuery(qStr);

//            TEST411; select avg(e.dept.no) from XMLEmpBean e  group by e.empid
//            avg(e.dept.no)
//            ~~~~~~~~~~~~~~
//                 100 100 200 200 210 210 210 210 220
//             TEST411; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { 100d, 100d, 200d, 200d, 210d, 210d, 210d, 210d, 220d };

            // OpenJPA returns Integer values instead of Double values...
            if (JPAPersistenceProvider.OPENJPA.equals(provider)) {
                targets = new Object[] { 100, 100, 200, 200, 210, 210, 210, 210, 220 };
            }
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST412; select avg(e.salary) from XMLEmpBean e
    public void testLoop412(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select avg(e.salary) from XMLEmpBean e";
            Query q = em.createQuery(qStr);

//            TEST412; select avg(e.salary) from XMLEmpBean e
//            avg(e.salary)
//            ~~~~~~~~~~~~~
//                4.07
//             TEST412; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { 4.07 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST413; select count(distinct e) from XMLDeptBean d left join d.emps e
    public void testLoop413(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select count(distinct e) from XMLDeptBean d left join d.emps e ";
            Query q = em.createQuery(qStr);

//            TEST413; select count(distinct e) from XMLDeptBean d left join d.emps e
//            count(distinct e)
//            ~~~~~~~~~~~~~~~~~
//                    9
//             TEST413; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { 9l };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST414; select count(distinct e.name) from XMLEmpBean e
    public void testLoop414(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select count(distinct e.name) from XMLEmpBean e ";
            Query q = em.createQuery(qStr);

//            TEST414; select count(distinct e.name) from XMLEmpBean e
//            count(distinct e.name)
//            ~~~~~~~~~~~~~~~~~~~~~~
//                      10
//             TEST414; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { 10l };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST415; select count(e) from XMLDeptBean d left join d.emps e
    public void testLoop415(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select count(e) from XMLDeptBean d left join d.emps e ";
            Query q = em.createQuery(qStr);

//            TEST415; select count(e) from XMLDeptBean d left join d.emps e
//            count(e)
//            ~~~~~~~~
//               9
//             TEST415; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { 9l };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST416; select count(e) from XMLEmpBean e
    public void testLoop416(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select count(e) from XMLEmpBean e ";
            Query q = em.createQuery(qStr);

//            TEST416; select count(e) from XMLEmpBean e
//            count(e)
//            ~~~~~~~~
//               10
//             TEST416; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { 10l };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST417; select d from XMLDeptBean d where d.no <> (select sum(e.salary) from XMLEmpBean e where e.salary > 0) and d.name = 'Sales'
    public void testLoop417(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where d.no <> (select sum(e.salary) from XMLEmpBean e where e.salary > 0) and d.name = 'Sales' ";
            Query q = em.createQuery(qStr);

//            TEST417; select d from XMLDeptBean d where d.no <> (select sum(e.salary) from XMLEmpBean e where e.salary > 0) and d.name = 'Sales'
//                            DeptBean
//                            ~~~~~~~~
//                             [300]
//                             TEST417; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { dep300 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST418; select e.empid from XMLEmpBean e where e.empid <> (select sum(e.no) from XMLDeptBean e)
    public void testLoop418(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.empid from XMLEmpBean e where e.empid <> (select sum(e.no) from XMLDeptBean e) ";
            Query q = em.createQuery(qStr);

//            TEST418; select e.empid from XMLEmpBean e where e.empid <> (select sum(e.no) from XMLDeptBean e)
//            e.empid
//            ~~~~~~~
//               1 2 3 4 5 6 7 8 9 10
//             TEST418; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { 1, 2, 3, 4, 5, 6, 7, 8, 9, 10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST419; select e.name from XMLEmpBean e where e.hireDate = (select min(f.hireDate) from XMLEmpBean f)
    public void testLoop419(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e where e.hireDate = (select min(f.hireDate) from XMLEmpBean f) ";
            Query q = em.createQuery(qStr);

//            TEST419; select e.name from XMLEmpBean e where e.hireDate = (select min(f.hireDate) from XMLEmpBean f)
//                            e.name
//                         ~~~~~~~~~~~~
//                            ahmad
//                            andrew
//                            david
//                            george
//                            harry
//                            minmei
//                            ritika
//                           charlene
//                         Catalina Wei
//                         Tom Rayburn
//                          TEST419; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { "ahmad", "andrew", "david", "george", "harry", "minmei", "ritika", "charlene", "Catalina Wei", "Tom Rayburn" };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST420; select e.dept.no, sum (distinct e.empid) from XMLEmpBean e  group by e.dept.no
    public void testLoop420(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.no, sum (distinct e.empid) from XMLEmpBean e  group by e.dept.no ";
            Query q = em.createQuery(qStr);

//            TEST420; select e.dept.no, sum (distinct e.empid) from XMLEmpBean e  group by e.dept.no
//            e.dept.no sum (distinct e.empid)
//            ~~~~~~~~~ ~~~~~~~~~~~~~~~~~~~~~~
//               100              14
//               200              7
//               210              19
//               220              5
//             TEST420; 4 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 100, 14l },
                                 new Object[] { 200, 7l },
                                 new Object[] { 210, 19l },
                                 new Object[] { 220, 5l },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST421; select e.dept.no, count (e), count (distinct e.name) from XMLEmpBean  e group by e.dept.no
    public void testLoop421(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.no, count (e), count (distinct e.name) from XMLEmpBean  e group by e.dept.no ";
            Query q = em.createQuery(qStr);

//            TEST421; select e.dept.no, count (e), count (distinct e.name) from XMLEmpBean  e group by e.dept.no
//            e.dept.no count (e) count (distinct e.name)
//            ~~~~~~~~~ ~~~~~~~~~ ~~~~~~~~~~~~~~~~~~~~~~~
//               100        2                2
//               200        2                2
//               210        4                4
//               220        1                1
//             TEST421; 4 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 100, 2l, 2l },
                                 new Object[] { 200, 2l, 2l },
                                 new Object[] { 210, 4l, 4l },
                                 new Object[] { 220, 1l, 1l },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST422; select e.dept.no, sum (e.empid), avg (e.empid),   count(e) from XMLEmpBean e  group by e.dept.no
    public void testLoop422(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        final String dbProductName = (testProps == null) ? "UNKNOWN" : ((testProps.get("dbProductName") == null) ? "UNKNOWN" : (String) testProps.get("dbProductName"));

        // TODO: Skip with Oracle, as it produces output that differs from the original answer file for Oracle:
        /*
         * Generated Results:
         * 0 100 14 7.0 2
         * 1 200 7 3.5 2
         * 2 210 19 4.75 4
         * 3 220 5 5.0 1
         */
        if (DatabaseVendor.checkDBProductName(dbProductName, DatabaseVendor.ORACLE)) {
            System.out.println("Skipping test on Oracle platform. ");
            return;
        }

        // TODO: Skip with Postgresql, as it produces output that differs from the original answer file for Oracle:
        /*
         * Generated Results:
         * 0 220 5 5.0 1
         * 1 100 14 7.0 2
         * 2 200 7 3.5 2
         * 3 210 19 4.75 4
         */
        if (DatabaseVendor.checkDBProductName(dbProductName, DatabaseVendor.POSTGRES)) {
            System.out.println("Skipping test on Postgresql platform. ");
            return;
        }

        JPAPersistenceProvider provider = JPAPersistenceProvider.resolveJPAPersistenceProvider(jpaResource);

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.no, sum (e.empid), avg (e.empid),   count(e) from XMLEmpBean e  group by e.dept.no ";
            Query q = em.createQuery(qStr);

//            TEST422; select e.dept.no, sum (e.empid), avg (e.empid),   count(e) from XMLEmpBean e  group by e.dept.no
//            e.dept.no sum (e.empid) avg (e.empid) count(e)
//            ~~~~~~~~~ ~~~~~~~~~~~~~ ~~~~~~~~~~~~~ ~~~~~~~~
//               100         14             7          2
//               200          7             3          2
//               210         19             4          4
//               220          5             5          1
//             TEST422; 4 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 100, 14l, 7d, 2l },
                                 new Object[] { 200, 7l, 3d, 2l },
                                 new Object[] { 210, 19l, 4d, 4l },
                                 new Object[] { 220, 5l, 5d, 1l },
            };

            // OpenJPA returns Integer values instead of Double values...
            if (JPAPersistenceProvider.OPENJPA.equals(provider)) {
                targets = new Object[] { new Object[] { 100, 14l, 7, 2l },
                                         new Object[] { 200, 7l, 3, 2l },
                                         new Object[] { 210, 19l, 4, 4l },
                                         new Object[] { 220, 5l, 5, 1l },
                };
            }
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST423; select e.dept.no, Avg(e.salary) from XMLEmpBean e  group by e.dept.no
    public void testLoop423(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.no, Avg(e.salary) from XMLEmpBean e  group by e.dept.no ";
            Query q = em.createQuery(qStr);

//            TEST423; select e.dept.no, Avg(e.salary) from XMLEmpBean e  group by e.dept.no
//            e.dept.no Avg(e.salary)
//            ~~~~~~~~~ ~~~~~~~~~~~~~
//               100         0.0
//               200        7.75
//               210         6.3
//               220         0.0
//             TEST423; 4 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 100, 0d },
                                 new Object[] { 200, 7.75d },
                                 new Object[] { 210, 6.3d },
                                 new Object[] { 220, 0.0d },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST424; select e.dept, avg(e.salary) from XMLEmpBean e group by e.dept
    public void testLoop424(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        JPAPersistenceProvider provider = JPAPersistenceProvider.resolveJPAPersistenceProvider(jpaResource);

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            TransactionJacket tj = jpaResource.getTj();
            em.clear();

            String qStr = "select e.dept, avg(e.salary) from XMLEmpBean e group by e.dept";

            try {
                // Begin new transaction
                System.out.println("Beginning new transaction...");
                tj.beginTransaction();
                if (tj.isApplicationManaged()) {
                    System.out.println("Joining entitymanager to JTA transaction...");
                    em.joinTransaction();
                }

                List rList = em.createQuery(qStr).getResultList();

//              TEST424; select e.dept, avg(e.salary) from XMLEmpBean e group by e.dept
//              DeptBean avg(e.salary)
//              ~~~~~~~~ ~~~~~~~~~~~~~
//               [100]        0.0
//               [200]       7.75
//               [210]        6.3
//               [220]        0.0
//               TEST424; 4 tuples

                if (JPAPersistenceProvider.HIBERNATE.equals(provider)) {
                    Assert.fail("Expected Hibernate to fail query execution");
                }

                Object[] targets = { new Object[] { dep100, 0d },
                                     new Object[] { dep200, 7.75d },
                                     new Object[] { dep210, 6.3d },
                                     new Object[] { dep220, 0.0d },
                };

                validateQueryResult(testName, qStr, rList, targets);

                System.out.println("Rollback transaction...");
                if (tj.isTransactionActive()) {
                    tj.rollbackTransaction();
                }

                // Clear persistence context
                System.out.println("Clearing persistence context...");
                em.clear();
            } catch (java.lang.AssertionError ae) {
                throw ae;
            } catch (Throwable t) {
                System.out.println("The transaction commit threw exception (" + t + ") for provider " + provider.name());

                /*
                 * TODO: Hibernate throws an exception
                 * https://hibernate.atlassian.net/browse/HHH-1615
                 *
                 * Derby:
                 * Caused by: ERROR 42Y36:
                 * Column reference 'DEPTBEAN1_.DEPTNO' is invalid, or is part of an invalid expression.
                 * For a SELECT list with a GROUP BY, the columns and expressions being selected may only contain valid grouping expressions and valid aggregate expressions.
                 *
                 * SQLServer:
                 * Caused by: com.microsoft.sqlserver.jdbc.SQLServerException:
                 * Column 'JPADeptBean.deptno' is invalid in the select list because it is not contained in either an aggregate function or the GROUP BY clause.
                 */
                if (JPAPersistenceProvider.HIBERNATE.equals(provider)) {
                    return;
                }
                throw t;
            }
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST425; select d.no , avg(e.salary) from XMLEmpBean e, DeptBean d where e member of d.emps group by d.no
    public void testLoop425(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.no , avg(e.salary) from XMLEmpBean e, DeptBean d where e member of d.emps group by d.no";
            Query q = em.createQuery(qStr);

//            TEST425; select d.no , avg(e.salary) from XMLEmpBean e, DeptBean d where e member of d.emps group by d.no
//            d.no avg(e.salary)
//            ~~~~ ~~~~~~~~~~~~~
//            100       0.0
//            200      7.75
//            210       6.3
//            220       0.0
//             TEST425; 4 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 100, 0d },
                                 new Object[] { 200, 7.75d },
                                 new Object[] { 210, 6.3d },
                                 new Object[] { 220, 0.0d },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST426; select d.no , avg(e.salary) from XMLDeptBean d left join d.emps e group by d.no
    public void testLoop426(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.no , avg(e.salary) from XMLDeptBean d left join d.emps e group by d.no";
            Query q = em.createQuery(qStr);

//            TEST426; select d.no , avg(e.salary) from XMLDeptBean d left join d.emps e group by d.no
//            d.no avg(e.salary)
//            ~~~~ ~~~~~~~~~~~~~
//            100       0.0
//            200      7.75
//            210       6.3
//            220       0.0
//            300       0.0
//             TEST426; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 100, 0d },
                                 new Object[] { 200, 7.75d },
                                 new Object[] { 210, 6.3d },
                                 new Object[] { 220, 0.0d },
                                 new Object[] { 300, 0.0d },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST427; select e.dept.no, e.bonus , e.salary from XMLEmpBean e  group by e.dept.no, e.bonus, e.salary
    public void testLoop427(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.dept.no, e.bonus , e.salary from XMLEmpBean e  group by e.dept.no, e.bonus, e.salary";
            Query q = em.createQuery(qStr);

//            TEST427; select e.dept.no, e.bonus , e.salary from XMLEmpBean e  group by e.dept.no, e.bonus, e.salary
//            e.dept.no e.bonus e.salary
//            ~~~~~~~~~ ~~~~~~~ ~~~~~~~~
//               100      0.0     0.0
//               200      0.0     0.0
//               200      0.0     15.5
//               210      0.0     0.0
//               210      0.0     12.1
//               210      0.0     13.1
//               220      0.0     0.0
//             TEST427; 7 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 100, 0.0, 0.0 },
                                 new Object[] { 200, 0.0, 0.0 },
                                 new Object[] { 200, 0.0, 15.5 },
                                 new Object[] { 210, 0.0, 0.0 },
                                 new Object[] { 210, 0.0, 12.1 },
                                 new Object[] { 210, 0.0, 13.1 },
                                 new Object[] { 220, 0.0, 0.0 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST428; select e.name, min(e.name) from XMLEmpBean e  group  by e.name order by e.name
    public void testLoop428(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name, min(e.name) from XMLEmpBean e  group  by e.name order by e.name ";
            Query q = em.createQuery(qStr);

//            TEST428; select e.name, min(e.name) from XMLEmpBean e  group  by e.name order by e.name
//            e.name    min(e.name)
//         ~~~~~~~~~~~~ ~~~~~~~~~~~~
//            ahmad        ahmad
//            andrew       andrew
//            david        david
//            george       george
//            harry        harry
//            minmei       minmei
//            ritika       ritika
//           charlene     charlene
//         Catalina Wei Catalina Wei
//         Tom Rayburn  Tom Rayburn
//          TEST428; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { "ahmad", "ahmad" },
                                 new Object[] { "andrew", "andrew" },
                                 new Object[] { "david", "david" },
                                 new Object[] { "george", "george" },
                                 new Object[] { "harry", "harry" },
                                 new Object[] { "minmei", "minmei" },
                                 new Object[] { "ritika", "ritika" },
                                 new Object[] { "charlene", "charlene" },
                                 new Object[] { "Catalina Wei", "Catalina Wei" },
                                 new Object[] { "Tom Rayburn", "Tom Rayburn" },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST429; select e.name, sum(e.salary ) from XMLEmpBean e group by e.bonus, e.salary, e.name
    public void testLoop429(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name, sum(e.salary ) from XMLEmpBean e group by e.bonus, e.salary, e.name ";
            Query q = em.createQuery(qStr);

//            TEST429; select e.name, sum(e.salary ) from XMLEmpBean e group by e.bonus, e.salary, e.name
//            e.name    sum(e.salary )
//         ~~~~~~~~~~~~ ~~~~~~~~~~~~~~
//            ahmad          0.0
//            andrew         13.1
//            david          12.1
//            george         0.0
//            harry          0.0
//            minmei         15.5
//            ritika         0.0
//           charlene        0.0
//         Catalina Wei      0.0
//         Tom Rayburn       0.0
//          TEST429; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { "ahmad", 0.0d },
                                 new Object[] { "andrew", 13.1d },
                                 new Object[] { "david", 12.1d },
                                 new Object[] { "george", 0.0d },
                                 new Object[] { "harry", 0.0d },
                                 new Object[] { "minmei", 15.5d },
                                 new Object[] { "ritika", 0.0d },
                                 new Object[] { "charlene", 0.0d },
                                 new Object[] { "Catalina Wei", 0.0d },
                                 new Object[] { "Tom Rayburn", 0.0d },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST430; select e.name, sum(e.salary ) from XMLEmpBean e group by e.name
    public void testLoop430(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name, sum(e.salary ) from XMLEmpBean e group by e.name ";
            Query q = em.createQuery(qStr);

//            TEST430; select e.name, sum(e.salary ) from XMLEmpBean e group by e.name
//            e.name    sum(e.salary )
//         ~~~~~~~~~~~~ ~~~~~~~~~~~~~~
//            ahmad          0.0
//            andrew         13.1
//            david          12.1
//            george         0.0
//            harry          0.0
//            minmei         15.5
//            ritika         0.0
//           charlene        0.0
//         Catalina Wei      0.0
//         Tom Rayburn       0.0
//          TEST430; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { "ahmad", 0.0d },
                                 new Object[] { "andrew", 13.1d },
                                 new Object[] { "david", 12.1d },
                                 new Object[] { "george", 0.0d },
                                 new Object[] { "harry", 0.0d },
                                 new Object[] { "minmei", 15.5d },
                                 new Object[] { "ritika", 0.0d },
                                 new Object[] { "charlene", 0.0d },
                                 new Object[] { "Catalina Wei", 0.0d },
                                 new Object[] { "Tom Rayburn", 0.0d },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST431; select e.salary, e.name from XMLEmpBean e group by e.salary,e.name having e.name like 'Firstname3%'
    public void testLoop431(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.salary, e.name from XMLEmpBean e group by e.salary,e.name having e.name like 'Firstname3%' ";
            Query q = em.createQuery(qStr);

//            TEST431; select e.salary, e.name from XMLEmpBean e group by e.salary,e.name having e.name like 'Firstname3%'
//            TEST431; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST432; select e.salary, e.name from XMLEmpBean e group by e.salary,e.name
    public void testLoop432(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.salary, e.name from XMLEmpBean e group by e.salary,e.name ";
            Query q = em.createQuery(qStr);

//            TEST432; select e.salary, e.name from XMLEmpBean e group by e.salary,e.name
//            e.salary    e.name
//            ~~~~~~~~ ~~~~~~~~~~~~
//              0.0       ahmad
//              0.0       george
//              0.0       harry
//              0.0       ritika
//              0.0      charlene
//              0.0    Catalina Wei
//              0.0    Tom Rayburn
//              12.1      david
//              13.1      andrew
//              15.5      minmei
//             TEST432; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 0.0, "ahmad" },
                                 new Object[] { 0.0, "george" },
                                 new Object[] { 0.0, "harry" },
                                 new Object[] { 0.0, "ritika" },
                                 new Object[] { 0.0, "charlene" },
                                 new Object[] { 0.0, "Catalina Wei" },
                                 new Object[] { 0.0, "Tom Rayburn" },
                                 new Object[] { 12.1, "david" },
                                 new Object[] { 13.1, "andrew" },
                                 new Object[] { 15.5, "minmei" },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST433; select e.isManager  from XMLEmpBean e group by e.isManager
    public void testLoop433(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.isManager  from XMLEmpBean e group by e.isManager ";
            Query q = em.createQuery(qStr);
//
//            TEST433; select e.isManager  from XMLEmpBean e group by e.isManager
//            e.isManager
//            ~~~~~~~~~~~
//               false
//               true
//             TEST433; 2 tuples

            List rList = q.getResultList();

            Object[] targets = { false, true };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST434; select d from XMLDeptBean d where (d.emps is empty)
    public void testLoop434(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where (d.emps is empty) ";
            Query q = em.createQuery(qStr);

//            TEST434; select d from XMLDeptBean d where (d.emps is empty)
//            DeptBean
//            ~~~~~~~~
//             [300]
//             TEST434; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { dep300 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST435; select d from XMLDeptBean d where d.emps is not empty and d.name = 'Sales'
    public void testLoop435(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where d.emps is not empty and d.name = 'Sales' ";
            Query q = em.createQuery(qStr);

//            TEST435; select d from XMLDeptBean d where d.emps is not empty and d.name = 'Sales'
//                            TEST435; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST436; select e, d from XMLEmpBean as e , DeptBean as d where   e MEMBER OF d.emps
    public void testLoop436(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e, d from XMLEmpBean as e , DeptBean as d where   e MEMBER OF d.emps ";
            Query q = em.createQuery(qStr);

//            TEST436; select e, d from XMLEmpBean as e , DeptBean as d where   e MEMBER OF d.emps
//            EmpBean DeptBean
//            ~~~~~~~ ~~~~~~~~
//              [1]    [210]
//              [2]    [210]
//              [3]    [200]
//              [4]    [200]
//              [5]    [220]
//              [6]    [100]
//              [7]    [210]
//              [8]    [100]
//              [9]    [210]
//             TEST436; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { emp1, dep210 },
                                 new Object[] { emp2, dep210 },
                                 new Object[] { emp3, dep200 },
                                 new Object[] { emp4, dep200 },
                                 new Object[] { emp5, dep220 },
                                 new Object[] { emp6, dep100 },
                                 new Object[] { emp7, dep210 },
                                 new Object[] { emp8, dep100 },
                                 new Object[] { emp9, dep210 },
            };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST437; select d from XMLDeptBean d where d.mgr member d.emps
    public void testLoop437(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where d.mgr member d.emps ";
            Query q = em.createQuery(qStr);

//            TEST437; select d from XMLDeptBean d where d.mgr member d.emps
//            TEST437; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST438; select d from XMLDeptBean d where d.mgr not member d.emps
    public void testLoop438(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where d.mgr not member d.emps ";
            Query q = em.createQuery(qStr);

//            TEST438; select d from XMLDeptBean d where d.mgr not member d.emps
//            DeptBean
//            ~~~~~~~~
//             [100] [200] [210] [220] [300]
//             TEST438; 5 tuples

            List rList = q.getResultList();

            Object[] targets = { dep100, dep200, dep210, dep220, dep300 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST439; select d from XMLDeptBean d where d.mgr.dept.mgr member of d.emps
    public void testLoop439(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d from XMLDeptBean d where d.mgr.dept.mgr member of d.emps ";
            Query q = em.createQuery(qStr);

//            TEST439; select d from XMLDeptBean d where d.mgr.dept.mgr member of d.emps
//            TEST439; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST440; select d.no from XMLDeptBean d where d.mgr member d.emps
    public void testLoop440(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select d.no from XMLDeptBean d where d.mgr member d.emps ";
            Query q = em.createQuery(qStr);

//            TEST440; select d.no from XMLDeptBean d where d.mgr member d.emps
//            TEST440; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST441; select e from XMLEmpBean e where e member e.dept.emps
    public void testLoop441(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e member e.dept.emps ";
            Query q = em.createQuery(qStr);

//            TEST441; select e from XMLEmpBean e where e member e.dept.emps
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [6] [7] [8] [9]
//             TEST441; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST442; select e from XMLEmpBean e where e member of e.dept.emps and e.empid = 6
    public void testLoop442(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e member of e.dept.emps and e.empid = 6 ";
            Query q = em.createQuery(qStr);

//            TEST442; select e from XMLEmpBean e where e member of e.dept.emps and e.empid = 6
//                            EmpBean
//                            ~~~~~~~
//                              [6]
//                             TEST442; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { emp6 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST443; select e from XMLEmpBean e where e member of e.dept.emps or e.empid = 6
    public void testLoop443(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e member of e.dept.emps or e.empid = 6 ";
            Query q = em.createQuery(qStr);

//            TEST443; select e from XMLEmpBean e where e member of e.dept.emps or e.empid = 6
//                            EmpBean
//                            ~~~~~~~
//                              [1] [2] [3] [4] [5] [6] [6] [7] [8] [9]
//                             TEST443; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp6, emp7, emp8, emp9 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST444; select e from XMLEmpBean e where e member of e.dept.emps
    public void testLoop444(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e member of e.dept.emps";
            Query q = em.createQuery(qStr);

//            TEST444; select e from XMLEmpBean e where e member of e.dept.emps
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [6] [7] [8] [9]
//             TEST444; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST445; select e from XMLEmpBean e where e not member of e.dept.emps
    public void testLoop445(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e not member of e.dept.emps";
            Query q = em.createQuery(qStr);

//            TEST445; select e from XMLEmpBean e where e not member of e.dept.emps
//            EmpBean
//            ~~~~~~~
//             [10]
//             TEST445; 1 tuple

            List rList = q.getResultList();

            Object[] targets = { emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST446; select e from XMLEmpBean e where e.dept.mgr member of e.dept.emps
    public void testLoop446(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.dept.mgr member of e.dept.emps ";
            Query q = em.createQuery(qStr);

//            TEST446; select e from XMLEmpBean e where e.dept.mgr member of e.dept.emps
//            TEST446; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST447; select e from XMLEmpBean e where e.dept.mgr not member e.dept.emps
    public void testLoop447(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where e.dept.mgr not member e.dept.emps ";
            Query q = em.createQuery(qStr);

//            TEST447; select e from XMLEmpBean e where e.dept.mgr not member e.dept.emps
//            EmpBean
//            ~~~~~~~
//              [1] [2] [3] [4] [5] [6] [7] [8] [9] [10]
//             TEST447; 10 tuples

            List rList = q.getResultList();

            Object[] targets = { emp1, emp2, emp3, emp4, emp5, emp6, emp7, emp8, emp9, emp10 };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST448; select e.name from XMLEmpBean e, DeptBean d where d.mgr not member of e.dept.emps
    public void testLoop448(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e.name from XMLEmpBean e, DeptBean d where d.mgr not member of e.dept.emps ";
            Query q = em.createQuery(qStr);

//            TEST448; select e.name from XMLEmpBean e, DeptBean d where d.mgr not member of e.dept.emps
//            e.name
//         ~~~~~~~~~~~~
//            andrew
//            david
//            harry
//            ritika
//           charlene
//         Catalina Wei
//          TEST448; 6 tuples

            List rList = q.getResultList();

            Object[] targets = { "andrew", "david", "harry", "ritika", "charlene", "Catalina Wei" };
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST449; select e from XMLEmpBean e where (exists (select d from XMLDeptBean d where d.no = e.empid))
    public void testLoop449(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select e from XMLEmpBean e where (exists (select d from XMLDeptBean d where d.no = e.empid)) ";
            Query q = em.createQuery(qStr);

//            TEST449; select e from XMLEmpBean e where (exists (select d from XMLDeptBean d where d.no = e.empid))
//            TEST449; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST450; select x.empid, x.dept.no from XMLEmpBean x where exists (select d from XMLDeptBean d where (d.no <1 and d.no >99999) and d.no=x.empid )
    public void testLoop450(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select x.empid, x.dept.no from XMLEmpBean x where exists (select d from XMLDeptBean d where (d.no <1 and d.no >99999) and d.no=x.empid )   ";
            Query q = em.createQuery(qStr);

//            TEST450; select x.empid, x.dept.no from XMLEmpBean x where exists (select d from XMLDeptBean d where (d.no <1 and d.no >99999) and d.no=x.empid )
//            TEST450; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

//  TEST451; select x.empid, x.dept.no from XMLEmpBean x where exists (select d from XMLDeptBean d where d.no = 1)
    public void testLoop451(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select x.empid, x.dept.no from XMLEmpBean x where exists (select d from XMLDeptBean d where d.no = 1)  ";
            Query q = em.createQuery(qStr);

//            TEST451; select x.empid, x.dept.no from XMLEmpBean x where exists (select d from XMLDeptBean d where d.no = 1)
//            TEST451; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

// TEST452; select x.empid, x.dept.no from XMLEmpBean x where exists (select d from XMLDeptBean d where d.no = x.empid)
    public void testLoop452(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select x.empid, x.dept.no from XMLEmpBean x where exists (select d from XMLDeptBean d where d.no = x.empid)";
            Query q = em.createQuery(qStr);

//            TEST452; select x.empid, x.dept.no from XMLEmpBean x where exists (select d from XMLDeptBean d where d.no = x.empid)
//            TEST452; 0 tuples

            List rList = q.getResultList();

            Object[] targets = {};
            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST453; select x.empid, x.dept.no from XMLEmpBean x where not exists (select d from XMLDeptBean d where (d.no <1 and d.no >99999) and d.no=x.empid )
    public void testLoop453(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select x.empid, x.dept.no from XMLEmpBean x where not exists (select d from XMLDeptBean d where (d.no <1 and d.no >99999) and d.no=x.empid )";
            Query q = em.createQuery(qStr);

//            TEST453; select x.empid, x.dept.no from XMLEmpBean x where not exists (select d from XMLDeptBean d where (d.no <1 and d.no >99999) and d.no=x.empid )
//            x.empid x.dept.no
//            ~~~~~~~ ~~~~~~~~~
//               1       210
//               2       210
//               3       200
//               4       200
//               5       220
//               6       100
//               7       210
//               8       100
//               9       210
//             TEST453; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, 210 },
                                 new Object[] { 2, 210 },
                                 new Object[] { 3, 200 },
                                 new Object[] { 4, 200 },
                                 new Object[] { 5, 220 },
                                 new Object[] { 6, 100 },
                                 new Object[] { 7, 210 },
                                 new Object[] { 8, 100 },
                                 new Object[] { 9, 210 },
            };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST454; select x.empid, x.dept.no from XMLEmpBean x where not exists (select d from XMLDeptBean d where d.no = x.empid*x.empid)
    public void testLoop454(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            for (String key : testProps.keySet()) {
                System.out.println("Test Property: " + key + " = " + testProps.get(key));
            }
        }

        // Execute Test Case
        try {
            EntityManager em = jpaResource.getEm();
            em.clear();

            String qStr = "select x.empid, x.dept.no from XMLEmpBean x where not exists (select d from XMLDeptBean d where d.no = x.empid*x.empid) ";
            Query q = em.createQuery(qStr);

//            TEST454; select x.empid, x.dept.no from XMLEmpBean x where not exists (select d from XMLDeptBean d where d.no = x.empid*x.empid)
//            x.empid x.dept.no
//            ~~~~~~~ ~~~~~~~~~
//               1       210
//               2       210
//               3       200
//               4       200
//               5       220
//               6       100
//               7       210
//               8       100
//               9       210
//             TEST454; 9 tuples

            List rList = q.getResultList();

            Object[] targets = { new Object[] { 1, 210 },
                                 new Object[] { 2, 210 },
                                 new Object[] { 3, 200 },
                                 new Object[] { 4, 200 },
                                 new Object[] { 5, 220 },
                                 new Object[] { 6, 100 },
                                 new Object[] { 7, 210 },
                                 new Object[] { 8, 100 },
                                 new Object[] { 9, 210 },
            };

            validateQueryResult(testName, qStr, rList, targets);
            em.clear();
        } catch (java.lang.AssertionError ae) {
            throw ae;
        } finally {
            System.out.println(testName + ": End");
        }
    }

    // TEST455; select x.empid, x.dept.no from XMLEmpBean x where x.empid > 10 or exists (select d from XMLDeptBean d where d.no = x.empid*x.empid-d.no)
    public void testLoop455(TestExecutionContext testExecCtx, TestExecutionResources testExecResources,
                            Object managedComponentObject) {
        final String testName = getTestName();

        // Verify parameters
        if (testExecCtx == null || testExecResources == null) {
            Assert.fail(testName + ": Missing context and/or resources.  Cannot execute the test.");
            return;
        }

        final JPAResource jpaResource = testExecResources.getJpaResourceMap().get("test-jpa-resource");
        if (jpaResource == null) {
            Assert.fail("Missing JPAResource 'test-jpa-resource').  Cannot execute the test.");
            return;
        }

        // Process Test Properties
        final Map<String, Serializable> testProps = testExecCtx.getProperties();
        if (testProps != null) {
            fo