#!/bin/sh

# Functions

abort()
{
   echo >&2
   for ARG in "$@"; do
      echo "$ARG" >&2
   done
   exit 1
}

usage() 
{   
   abort "Usage:   $EXE RELEASE_VERSION DEVELOPMENT_VERSION RHQ_VERSION SVN_USERNAME SVN_PASSWORD test|production DEBUG" "Example: $EXE 2.4.0.GA_QA 2.4.0-SNAPSHOT 3.0.0.GA_QA jshaughn hereiam test false"   
}

# Validate command line args.
if [ "$#" -ne 7 ]; then
   usage
fi  

# If DEBUG=true the SVN password will get echoed in the console output!
DEBUG="$7" 
if [ "$DEBUG" != "true" ]; then
   set +x
else
   set -x
fi

EXE=`basename $0`
PROJECT_NAME="jon"
PROJECT_DISPLAY_NAME="JBossON"
PROJECT_SVN_URL="https://svn.devel.redhat.com/repos/jboss-jon"
TAG_PREFIX="JBossON"
MINIMUM_MAVEN_VERSION="2.1.0"

# Process command line args.
RELEASE_VERSION="$1"
TAG_VERSION=`echo $RELEASE_VERSION | sed 's/\./_/g'`
RELEASE_TAG="${TAG_PREFIX}_${TAG_VERSION}"
DEVELOPMENT_VERSION="$2"
RHQ_VERSION="$3"
SVN_USERNAME="$4"
SVN_PASSWORD="$5"
MODE="$6"
if [ "$MODE" != "test" ] && [ "$MODE" != "production" ]; then
   usage "Invalid mode: $MODE (valid modes are 'test' or 'production')"
fi


# Make sure SUBVERSION_HOME points to a valid Subversion install.

if [ -z "$SUBVERSION_HOME" ]; then
   abort "SUBVERSION_HOME environment variable is not set." >&2
fi

if [ ! -d "$SUBVERSION_HOME" ]; then
   abort "SUBVERSION_HOME ($SUBVERSION_HOME) does not exist or is not a directory."
fi

echo "Prepending $SUBVERSION_HOME/bin to PATH..."
PATH="$SUBVERSION_HOME/bin:$PATH"

if ! which svn >/dev/null 2>&1; then
   abort "svn not found in PATH ($PATH) - SUBVERSION_HOME must point to an SVN install dir."
fi

echo "Prepending $SUBVERSION_HOME/lib to LD_LIBRARY_PATH..."
LD_LIBRARY_PATH="$SUBVERSION_HOME/lib:$LD_LIBRARY_PATH"
export LD_LIBRARY_PATH


# Make sure JAVA_HOME points to a valid JDK 1.6+ install.

if [ -z "$JAVA_HOME" ]; then
   abort "JAVA_HOME environment variable is not set - JAVA_HOME must point to a JDK (not JRE) 6 install dir."
fi

if [ ! -d "$JAVA_HOME" ]; then
   abort "JAVA_HOME ($JAVA_HOME) does not exist or is not a directory - JAVA_HOME must point to a JDK (not JRE) 6 install dir."
fi

echo "Prepending $JAVA_HOME/bin to PATH..."
PATH="$JAVA_HOME/bin:$PATH"

if ! which java >/dev/null 2>&1; then
   abort "java not found in PATH ($PATH) - JAVA_HOME must point to a JDK (not JRE) 6 install dir."
fi

if ! which javac >/dev/null 2>&1; then
   abort "javac not found in PATH ($PATH) - JAVA_HOME must point to a JDK (not JRE) 6 install dir."
fi

if ! javap java.util.Deque >/dev/null 2>&1; then
   abort "java.util.Deque not found - Java version appears to be less than 1.6 - Jave version must be 1.6 or later."
fi


# Make sure M2_HOME points to a valid Maven 2 install.

if [ -z "$M2_HOME" ]; then
   abort "M2_HOME environment variable is not set - M2_HOME must point to a Maven, $MINIMUM_MAVEN_VERSION or later, install dir."
fi

if [ ! -d "$M2_HOME" ]; then
   abort "M2_HOME ($M2_HOME) does not exist or is not a directory - M2_HOME must point to a Maven, $MINIMUM_MAVEN_VERSION or later, install dir."
fi

echo "Prepending $M2_HOME/bin to PATH..."
PATH="$M2_HOME/bin:$PATH"

if ! which mvn >/dev/null 2>&1; then
   abort "mvn not found in PATH ($PATH) - M2_HOME must point to a Maven, $MINIMUM_MAVEN_VERSION or later, install dir."
fi


# Set various shell variables to be used by this script.

if [ -n "$HUDSON_URL" ] && [ -n "$WORKSPACE" ]; then
   echo "We appear to be running in a Hudson job." 
   WORK_DIR="$WORKSPACE"
   MAVEN_LOCAL_REPO_DIR="$HOME/.m2/hudson-release-enterprise-repository"   
elif [ -z "$WORK_DIR" ]; then
   WORK_DIR="$HOME/release/jon"
   MAVEN_LOCAL_REPO_DIR="$BASE_DIR/m2-repository"   
fi
cd "$WORK_DIR"

RELEASE_BRANCH_CHECKOUT_DIR="$WORK_DIR/jon/branch"
RELEASE_TAG_CHECKOUT_DIR="$WORK_DIR/jon/tag"
if [ -n "$RELEASE_BRANCH" ]; then
   RELEASE_BRANCH_SVN_URL="$PROJECT_SVN_URL/branches/$RELEASE_BRANCH"
else
   RELEASE_BRANCH_SVN_URL="$PROJECT_SVN_URL/trunk"
fi
RELEASE_TAG_SVN_URL="$PROJECT_SVN_URL/tags/$RELEASE_TAG"


MAVEN_SETTINGS_FILE="$WORK_DIR/m2-settings.xml"
MAVEN_ARGS="--settings $MAVEN_SETTINGS_FILE --errors --batch-mode"
if [ -z "$RHQ_RELEASE_QUIET" ]; then
   MAVEN_ARGS="$MAVEN_ARGS --debug"
fi

# password is added below
SVN_ARGS="--non-interactive --username $SVN_USERNAME"


# Set the system character encoding to ISO-8859-1 to ensure i18log reads its 
# messages and writes its resource bundle properties files in that encoding, 
# since that is how the German and French I18NMessage annotation values are
# encoded and the encoding used by i18nlog to read in resource bundle
# property files.
LANG=en_US.iso8859
export LANG


# Print out summary of environment.

echo
echo "========================= Environment Variables =============================="
echo "JAVA_HOME=$JAVA_HOME"
echo "M2_HOME=$M2_HOME"
echo "MAVEN_OPTS=$MAVEN_OPTS"
echo "SUBVERSION_HOME=$SUBVERSION_HOME"
echo "PATH=$PATH"
echo "LD_LIBRARY_PATH=$LD_LIBRARY_PATH"
echo "LANG=$LANG"
echo "============================ Local Variables ================================="
echo "WORK_DIR=$WORK_DIR"
echo "PROJECT_NAME=$PROJECT_NAME"
echo "RELEASE_VERSION=$RELEASE_VERSION"
echo "DEVELOPMENT_VERSION=$DEVELOPMENT_VERSION"
echo "RELEASE_BRANCH=$RELEASE_BRANCH"
echo "RELEASE_BRANCH_SVN_URL=$RELEASE_BRANCH_SVN_URL"
echo "RELEASE_BRANCH_CHECKOUT_DIR=$RELEASE_BRANCH_CHECKOUT_DIR"
echo "RELEASE_TAG=$RELEASE_TAG"
echo "RELEASE_TAG_SVN_URL=$RELEASE_TAG_SVN_URL"
echo "RELEASE_TAG_CHECKOUT_DIR=$RELEASE_TAG_CHECKOUT_DIR"
echo "MODE=$MODE"
echo "MAVEN_LOCAL_REPO_DIR=$MAVEN_LOCAL_REPO_DIR"
echo "MAVEN_LOCAL_REPO_PURGE_INTERVAL_HOURS=$MAVEN_LOCAL_REPO_PURGE_INTERVAL_HOURS"
echo "MAVEN_SETTINGS_FILE=$MAVEN_SETTINGS_FILE"
echo "MAVEN_ARGS=$MAVEN_ARGS"
echo "SVN_ARGS=$SVN_ARGS --password ******"
echo "============================ Program Versions ================================"
svn --version | head -2
echo
java -version
echo
mvn --version | head -1
echo "=============================================================================="
echo

# now that it's been echoed, add the password to SVN_ARGS for use in the rest of the script.
SVN_ARGS="$SVN_ARGS --password $SVN_PASSWORD"

# Create the Maven settings file.
cat <<EOF >"${MAVEN_SETTINGS_FILE}"
<settings>
   <localRepository>$MAVEN_LOCAL_REPO_DIR</localRepository>   
</settings>
EOF


# Check out the branch, update the RHQ version in the root pom, then run a test build before tagging.

if [ -d "$RELEASE_BRANCH_CHECKOUT_DIR" ]; then
   echo "Purging contents of RELEASE_BRANCH_CHECKOUT_DIR ($RELEASE_BRANCH_CHECKOUT_DIR)..."
   rm -rf "$RELEASE_BRANCH_CHECKOUT_DIR"
fi
mkdir -p "$RELEASE_BRANCH_CHECKOUT_DIR"

echo "Checking out branch source from $RELEASE_BRANCH_SVN_URL to $RELEASE_BRANCH_CHECKOUT_DIR (this will take about 5-10 minutes)..."
# We only need pom.xml, jon/**, and etc/product_connectors/**. Save some time by not checking out everything else (doc/** etc.).
svn co $SVN_ARGS -N $RELEASE_BRANCH_SVN_URL "$RELEASE_BRANCH_CHECKOUT_DIR"

cd "$RELEASE_BRANCH_CHECKOUT_DIR"
svn co $SVN_ARGS $RELEASE_BRANCH_SVN_URL/jon
mkdir etc
cd etc
svn co $SVN_ARGS $RELEASE_BRANCH_SVN_URL/etc/product_connectors
cd ..

# If the specified tag already exists remotely and we're in production mode, then abort. 
# If it exists and we're in test mode, then we'll delete it after we've had a successful dry run of
# release:prepare and are ready to tag.
svn info https://svn.devel.redhat.com/repos/jboss-jon/tags/$RELEASE_TAG >/dev/null 2>&1
if [ "$?" -eq 0 ]; then
   RELEASE_TAG_EXISTS=1
else
   unset RELEASE_TAG_EXISTS
fi
# TODO: Uncomment this after 2.4.0.GA release.
#if [ -n "$RELEASE_TAG_EXISTS" ] && [ "$MODE" = "production" ]; then
#   abort "A tag named $RELEASE_TAG already exists - aborting, since we are in production mode..."
#fi

echo
echo "Updating RHQ version in root pom..."
# Save the previous RHQ version.
RHQ_DEVELOPMENT_VERSION=`grep "<rhq.\version>.*</rhq\.version>" "pom.xml" | sed "s|\s*<rhq.version>\(.*\)</rhq.version>|\1|"`
echo "RHQ Development Version: " $RHQ_DEVELOPMENT_VERSION
echo "RHQ Build Version      : " $RHQ_VERSION
# Replace the rhq.version
sed -i -e "s|\(<rhq.version>\).*\(</rhq.version>\)|\1$RHQ_VERSION\2|" "pom.xml"
# Carefully replace the <version> tag in the parent section
sed -i -e "/rhq-parent/{N;s|\(rhq-parent.*\n.*<version>\).*\(</version>\)|\1$RHQ_VERSION\2|}" "pom.xml"
# don't need to commit this yet, do the bootstrap/test build first. If it fails, no commit issue

echo
echo "Building project to ensure tests pass and to bootstrap local Maven repo (this will take about 5-10 minutes)..."
# This will build everything, enforcing Java 5 APIs.
mvn install $MAVEN_ARGS
[ "$?" -ne 0 ] && abort "Test build failed. Please see above Maven output for details, fix any issues, then try again."

echo
echo "Test build succeeded!"


# Tag the release.

echo "Committing updated root pom to SVN..."
svn commit $SVN_ARGS -m "$PROJECT_DISPLAY_NAME Build $RELEASE_VERSION (in progress)"
[ "$?" -ne 0 ] && abort "Failed to commit updated root pom to SVN. Please see above svn output for details, fix any issues, then try again."

echo "Tagging the release (dry-run)..."
cd "$RELEASE_BRANCH_CHECKOUT_DIR"
mvn -Dusername=$SVN_USERNAME -Dpassword=$SVN_PASSWORD release:prepare $MAVEN_ARGS -DreleaseVersion=$RELEASE_VERSION -DdevelopmentVersion=$DEVELOPMENT_VERSION -Dresume=false -Dtag=$RELEASE_TAG -DdryRun=true
[ "$?" -ne 0 ] && abort "Tagging failed (dry-run). Please see above Maven output for details, fix any issues, then try again."
echo
echo "Tagging succeeded (dry-run)!"

echo "Clean anything left over by the release plugin (dry-run)"
mvn release:clean $MAVEN_ARGS


# If the specified tag already exists and we're in test mode, then delete the tag so it can be recreated. 
if [ -n "$RELEASE_TAG_EXISTS" ] && [ "$MODE" = "test" ]; then
   echo "Deleting existing release tag $RELEASE_TAG (test mode)..."
   svn delete $SVN_ARGS https://svn.devel.redhat.com/repos/jboss-jon/tags/$RELEASE_TAG --message "$EXE: delete test release tag, so it can be recreated"
   [ "$?" -ne 0 ] && abort "Failed to delete existing release tag. Please see above svn output for details, fix any issues, then try again."
fi

echo "Tagging the release..."
cd "$RELEASE_BRANCH_CHECKOUT_DIR"
mvn -Dusername=$SVN_USERNAME -Dpassword=$SVN_PASSWORD release:prepare $MAVEN_ARGS -DreleaseVersion=$RELEASE_VERSION -DdevelopmentVersion=$DEVELOPMENT_VERSION -Dresume=false -Dtag=$RELEASE_TAG -DdryRun=false
[ "$?" -ne 0 ] && abort "Tagging failed. Please see above Maven output for details, fix any issues, then try again."
echo
echo "Tagging succeeded!"

echo "Clean anything left over by the release plugin"
mvn release:clean $MAVEN_ARGS

echo "Resetting RHQ version in root pom to: " $RHQ_DEVELOPMENT_VERSION
# Save the previous RHQ version.
# Replace the rhq.version
sed -i -e "s|\(<rhq.version>\).*\(</rhq.version>\)|\1$RHQ_DEVELOPMENT_VERSION\2|" "pom.xml"
# Carefully replace the <version> tag in the parent section
sed -i -e "/rhq-parent/{N;s|\(rhq-parent.*\n.*<version>\).*\(</version>\)|\1$RHQ_DEVELOPMENT_VERSION\2|}" "pom.xml"

echo "Committing updated root pom to SVN..."
svn commit $SVN_ARGS -m "$PROJECT_DISPLAY_NAME Build $RELEASE_VERSION (done)"


# Checkout the tag and build it.

if [ -d "$RELEASE_TAG_CHECKOUT_DIR" ]; then
   echo "Purging contents of RELEASE_TAG_CHECKOUT_DIR ($RELEASE_TAG_CHECKOUT_DIR)..."
   rm -rf "$RELEASE_TAG_CHECKOUT_DIR"
fi
mkdir -p "$RELEASE_TAG_CHECKOUT_DIR"

echo "Checking out tag source from $RELEASE_TAG_SVN_URL to $RELEASE_TAG_CHECKOUT_DIR (this will take about 5-10 minutes)..."
# We only need pom.xml, jon/**, and etc/product_connectors/**. Save some time by not checking out everything else (doc/** etc.).
svn co $SVN_ARGS -N $RELEASE_TAG_SVN_URL "$RELEASE_TAG_CHECKOUT_DIR"
cd "$RELEASE_TAG_CHECKOUT_DIR"
svn co $SVN_ARGS $RELEASE_TAG_SVN_URL/jon
mkdir etc
cd etc
svn co $SVN_ARGS $RELEASE_BRANCH_SVN_URL/etc/product_connectors
cd ..

echo "Building release from tag (this will take about 10-15 minutes)..."
mvn install $MAVEN_ARGS -Dmaven.test.skip=true
[ "$?" -ne 0 ] && abort "Build failed. Please see above Maven output for details, fix any issues, then try again."
echo
echo "Release build succeeded!"

echo "=========================== Release Info ==============================="
echo "Version: $RELEASE_VERSION"
echo "Branch SVN URL: $RELEASE_BRANCH_SVN_URL"
echo "Tag SVN URL: $RELEASE_TAG_SVN_URL"
echo "========================================================================"

