#!/bin/sh

#
# The script that takes the following arguments(in any order):
# i)  [REQUIRED]--ga-brew-id Ex. --ga-brew-id="318824" See Public JON release history
# ii) [REQUIRED]--update-brew-id Ex. --update-brew-id="400000" See JON build engineer.
# iii)[REQUIRED]--update-type Ex. [server|agent|plugins]
# iv) [OPTIONAL]--update-id Ex. 01 
#  v) [OPTIONAL]--update-plugin Ex. jdg 
# and outputs cumulative patch binary Ex. jon-server-3.2.0.GA-update-01.zip with the following
# content structure:
# [jon-server-3.2.0.GA-update-01.zip]
#  - [jon-server-3.2.0.GA] : cumulative patch contents
#  - apply-update.sh/bat
#  - UPDATE_README.txt
#  - [old](contains moved/legacy contents from before patch)
#
# This script should be run from the root directory of {JON_REPO}.
#
# Assumed that customer/administrator will:
# 1)Unzip the update zip contents
# 2)run 'apply-update'.[sh|bat]
# 3)Get feedback about a)[success|failure] b)next steps

#include the utility library
source ../../../rhq_bash.lib

## Collect directory information for source generation
# Stores off the module directory
MODULE_DIR=`pwd`;
# Stores off the JON repo root directory in relative location format.
REPO_PWD=$(pwd)/../../..;
REPO_REMOVED_LIST="$REPO_PWD/patches/removed-list.txt";
# convert and store off directories to generate the absolute paths.
cd "$REPO_PWD";
GIT_REPO_DIR=`pwd`;
# return to original directory.
cd "$MODULE_DIR"
# Store off absolute path to target folder
TARGET=$(pwd)
TARGET="$TARGET/target"

# Constants/Variables/etc.
UPDATE_OPTIONS="[server|agent|plugins]"
UPDATE_TYPE="server" # default to 'server' update
UPDATE_PLUGIN=""
REPO_ROOT="$GIT_REPO_DIR"
UPDATE_ROOT="$TARGET/update_artifacts"
BUILD=$UPDATE_ROOT/BUILD
ZIP="$UPDATE_ROOT/ZIP"
PATCHED_DIR=$BUILD/.patched
SCRIPTS_DIR="misc"
UPDATE_SRC_ROOT="$UPDATE_ROOT/UPDATE"
EXTRACTED="EXTRACTED"
UPDATE_ARTIFACT_ID="" # Ex. jon-plugin-pack-jdg-3.2.0.GA-update-01
UPDATE_ARTIFACT="" #Ex. jon-plugin-pack-jdg-3.2.0.GA-update-01.zip
PLUGIN_ARTIFACT="" #Ex. jon-plugin-pack-jdg-3.2.0.GA.zip
SERVER_ARTIFACT="" #Ex. jon-server-3.2.0.GA.zip
VERSION="3.3.0.GA"
RHQ_VERSION="4.12.0.JON330GA";
REDHAT_INCREMENT="-redhat";
AGENT="rhq-enterprise-agent-$RHQ_VERSION";
AGENT_ARTIFACT="$AGENT.jar" #Ex. rhq-enterprise-agent-4.9.0.JON320GA.jar
AGENT_ZIP="$AGENT.zip" #Ex. rhq-enterprise-agent-4.9.0.JON320GA.zip
CLI_ARTIFACT="rhq-remoting-cli-$RHQ_VERSION.zip" #Ex. rhq-remoting-cli-4.9.0.JON320GA.jar
EAR_REL_LOCATION="modules/org/rhq/server-startup/main/deployments/rhq.ear"
DOWNLOADS_REL_LOCATION="$EAR_REL_LOCATION/rhq-downloads"
AGENT_REL_LOCATION="$DOWNLOADS_REL_LOCATION/rhq-agent"
CLI_REL_LOCATION="$DOWNLOADS_REL_LOCATION/rhq-client"
CONNECTORS_REL_LOCATION="$DOWNLOADS_REL_LOCATION/connectors"
PRODUCT_INFO="org/rhq/enterprise/server/core/ProductInfo.properties"
UPDATE_SCRIPT="apply-updates"
UPDATE_README="UPDATE_README.txt"
REMOVED_LIST="removed-list.txt"
JBOSS_CLI_COMMANDS="jboss-cli-commands.txt"
BREW_PREFIX="https://brewweb.devel.redhat.com//buildinfo?buildID="
CURRENT_UPDATE=12;
#Populate list of previous release artifacts in order(earliest[first]->newest[last])
#UPDATE_URLS=('http://porkchop.redhat.com/released/JBossON/3.3.0/jon-server-3.3.0.GA.zip');
#UPDATE_URLS=("${UPDATE_URLS[@]}" "http://porkchop.redhat.com/released/JBossON/3.3.1/jon-server-3.3-update-01.zip");
#UPDATE_URLS=("${UPDATE_URLS[@]}" "http://porkchop.redhat.com/released/JBossON/3.3.2/jon-server-3.3-update-02.zip");
UPDATE_URLS=("${UPDATE_URLS[@]}" "http://porkchop.redhat.com/released/JBossON/3.3.3/jon-server-3.3-update-03.zip");
#UPDATE_URLS=("${UPDATE_URLS[@]}" "http://porkchop.redhat.com/released/JBossON/3.3.4/jon-server-3.3-update-04.zip");
#UPDATE_URLS=("${UPDATE_URLS[@]}" "http://porkchop.redhat.com/released/JBossON/3.3.5/jon-server-3.3-update-05.zip");
#UPDATE_URLS=("${UPDATE_URLS[@]}" "http://porkchop.redhat.com/released/JBossON/3.3.6/jon-server-3.3-update-06.zip");
#UPDATE_URLS=("${UPDATE_URLS[@]}" "http://porkchop.redhat.com/released/JBossON/3.3.7/jon-server-3.3-update-07.zip");
#UPDATE_URLS=("${UPDATE_URLS[@]}" "http://porkchop.redhat.com/released/JBossON/3.3.8/jon-server-3.3-update-08.zip");
#UPDATE_URLS=("${UPDATE_URLS[@]}" "http://porkchop.redhat.com/released/JBossON/3.3.9/jon-server-3.3-update-09.zip");
# Storage node constants
RHQ_STORAGE_DIR="rhq-storage"
# *NOTE* if there are additional storage node changes for JON 3.3.x beyond -redhat-1 or to either the rhq-storage-wrapper.conf or cassandra-jvm.properties, both will need to include the previous content as defined by BZs 1272239 and/or 1274425 as is relevant. *Note*
CASSANDRA_CCM_CORE_VERSION="$RHQ_VERSION"-redhat-1
CON_FILTER="connector-rtfilter.zip"
CON_FILTER_MODULE="rhq-rtfilter-module.zip"
CON_FILTER_SUBSYS="rhq-rtfilter-subsystem-module.zip"
#License params
UPDATE_BUILD_DOCS_DIR="$UPDATE_SRC_ROOT/DOCS"
EAP_64_LICENSE_URL="http://download.eng.bos.redhat.com/brewroot/packages/org.jboss.as-jboss-as-parent/7.5.16.Final_redhat_1/1/maven/org/jboss/as/jboss-as-dist/7.5.16.Final-redhat-1/jboss-as-dist-7.5.16.Final-redhat-1.zip"
EAP_64_LICENSE_PATH="jboss-eap-6.4/docs"
JON_LICENSE_FILE="$REPO_PWD/patches/jon-licenses/licenses_6-8-18.zip"
JON_LICENSE_PATH="jon-licenses"
RHQ_STORAGE_PROPERTIES="rhq-storage.properties"

#DEBUG_ON="true"
# Enable DEBUG_ON if DEBUG environment variable is set. # Defaults to false/off.
if [ -z "$DEBUG" ];
then
  DEBUG_ON="false";
else
  DEBUG_ON="true";
fi
#========================================================================================
# Description: Initialize. Creates clean UPDATE_ROOT directory.
#========================================================================================
initialize()
{
 echo 
 echo "Listing environment variables used in generating ${UPDATE_OPTIONS} update artifact:"
 environment_variables=( "UPDATE_ROOT" "UPDATE_TYPE" "GA_BREW_ID" "UPDATE_BREW_ID" "UPDATE_ID" "DEBUG_ON" "UPDATE_PLUGIN" )
 print_variables "${environment_variables[@]}"
 echo 

 ## clean root artifact directory
 echo "Setting up the directories to build update artifacts in..."
 # clean
 echo "Cleaning out old $UPDATE_ROOT contents."
 debug "rm -rf $UPDATE_ROOT"
 rm -rf $UPDATE_ROOT

 # create update infrastructure
 debug "mkdir -p $UPDATE_ROOT"
 mkdir -p $UPDATE_ROOT
 debug "mkdir -p $BUILD"
 mkdir -p $BUILD
 debug "mkdir -p $PATCHED_DIR"
 mkdir -p $PATCHED_DIR

}

#========================================================================================
# Description: Validate and parse input arguments
#========================================================================================
parse_and_validate_options()
{
   print_function_information $FUNCNAME

   GA_BREW_ID=
   UPDATE_BREW_ID=
   UPDATE_TYPE=
   UPDATE_PLUGIN=
   UPDATE_ID=
   UPDATE_ARTIFACT=
   USE_SCRATCH_JON=
   PLUGIN_ARTIFACT=

   short_options="h"
   long_options="help,ga-brew-id:,update-brew-id:,update-type:,update-id:,update-debug:,use-scratch-jon:,update-plugin:"

   PROGNAME=${0##*/}
   ARGS=$(getopt -s bash --options $short_options --longoptions $long_options --name $PROGNAME -- "$@" )
   eval set -- "$ARGS"

   while true; do
	   case $1 in
         -h|--help)
            usage
            ;;
         --ga-brew-id)
            shift
            GA_BREW_ID="$1"
            shift
            ;;
         --update-brew-id)
            shift
            UPDATE_BREW_ID="$1"
            shift
            ;;
         --update-type)
            shift
            UPDATE_TYPE="$1"
            shift
            ;;
         --update-plugin)
            shift
            UPDATE_PLUGIN="$1"
            shift
            ;;
         --update-id)
            shift
            UPDATE_ID="$1"
            shift
            ;;
         --update-debug)
            shift
            DEBUG_ON="$1"
            shift
            ;;
         --use-scratch-jon)
            shift
            USE_SCRATCH_JON="$1"
            shift
            ;;
         --)
            shift
            break
            ;;
         *)
            usage
            ;;
	   esac
   done

   if [ -z "$GA_BREW_ID" ];
   then
      usage "GA brew build identifier not specified! Ex. '318824' -> https://brewweb.devel.redhat.com//buildinfo?buildID=318824"
   fi

   if [ -z "$UPDATE_BREW_ID" ];
   then
      usage "Update brew build identifier not specified! Ex. '418824' -> https://brewweb.devel.redhat.com//buildinfo?buildID=418824"
   fi

   if [ -z "$UPDATE_TYPE" ];
   then
      usage "Update type [server|agent|plugins] not specified! "
   fi

   # TODO: spinder, if UPDATE_TYPE=plugins then conditionally require UPDATE_PLUGIN

   if [ -z "$UPDATE_ID" ];
   then
      UPDATE_ID="$CURRENT_UPDATE";
      #Parse update location to see what's already available, then increment if necessary.
      #TODO: spinder
      echo "No update specifier defined. Using '${UPDATE_ID}' as the update version."
   fi

   print_centered "Script Options"
   script_options=( "GA_BREW_ID" "UPDATE_BREW_ID" "UPDATE_TYPE" "UPDATE_ID" "DEBUG_ON" "USE_SCRATCH_JON" "UPDATE_PLUGIN")
   print_variables "${script_options[@]}"
}

#========================================================================================
# Description: Display usage information then abort the script.
#========================================================================================
usage() 
{
   USAGE=$(
cat << EOF
USAGE:   generate-jon-update.sh OPTIONS

   --ga-brew-id=number              [REQUIRED]
      The GA brew build identifier used by this script. Ex. 318824

   --update-brew-id=number          [REQUIRED]
      The 'update' brew build identifier used by this script. Ex. 418824

   --update-type=server             [REQUIRED]
      The type of update binary being generated by this script.

   --update-id=01                   [OPTIONAL]
      The update number for this update artifact. Ex. jon-server-3.2.0.GA-update-01.zip

   --update-debug=true              [OPTIONAL]
      Whether to enable debug logging for this build. Ex. true||false

   --use-scratch-jon=url            [OPTIONAL]
      The brew url to retrieve the scratch jon build contents from. Ex. http://download.devel.redhat.com/brewroot/work/tasks/...

   --update-plugin=jdg              [OPTIONAL](not implemented)
      The name of the plugin that update binary is being generated for by this script.
EOF
)

   EXAMPLE="generate-jon-update.sh --ga-brew-id=\"318824\" --update-brew-id=\"418824\" --update-type=\"server\"  --update-id=\"01\""

   abort "$@" "$USAGE" "$EXAMPLE"
}

#========================================================================================
# Description: Pulls the specific brew build down to the specified or default location. 
#========================================================================================
get_brew_artifacts(){
 #Assume defaults
 CURRENT_EXTRACT_LOCATION=$UPDATE_SRC_ROOT
 debug "CURRENT_EXTRACT_LOCATION" "$CURRENT_EXTRACT_LOCATION"

 CURRENT_BREW_ID=$UPDATE_BREW_ID
 debug "CURRENT_BREW_ID" "$CURRENT_BREW_ID"

 #Update the variables if necessary
 if [ $# -eq 2 ];
    then
     debug "Updating the BREW location variables"
     CURRENT_EXTRACT_LOCATION=$1
     debug "CURRENT_EXTRACT_LOCATION" "$CURRENT_EXTRACT_LOCATION"
     CURRENT_BREW_ID=$2 
     debug "CURRENT_BREW_ID" "$CURRENT_BREW_ID"
 fi 

 # -Pull down the plugin brew build to $CURRENT_EXTRACT_LOCATION
 echo
 echo "mkdir $CURRENT_EXTRACT_LOCATION"
 mkdir $CURRENT_EXTRACT_LOCATION

 # -retrieve the artifact
 cd $CURRENT_EXTRACT_LOCATION
 pwd
 echo "Pulling down update artifacts for brew build: $BREW_PREFIX$CURRENT_BREW_ID"
 #echo "$REPO_ROOT/pull-down-jon-artifacts.sh --jon-brew-url=\"$BREW_PREFIX$CURRENT_BREW_ID\""
 #$REPO_ROOT/pull-down-jon-artifacts.sh --jon-brew-url="$BREW_PREFIX$CURRENT_BREW_ID"
 echo "$REPO_ROOT/pull-down-jon-artifacts.sh --jon-brew-url=\"$BREW_PREFIX$CURRENT_BREW_ID\" --jon-git-root=\"$REPO_ROOT\"" 
 $REPO_ROOT/pull-down-jon-artifacts.sh --jon-brew-url="$BREW_PREFIX$CURRENT_BREW_ID" --jon-git-root="$REPO_ROOT"

}

## Some files on the installed agents should never be blanket replaced with a 
## cumulative patch. What follows is an explicit list of agent files that 
## the customer is NOT allowed to modify in their environment and is
## safe to be replaced/patched with a cumulative patch. See GSS for more details
load_replaceable_artifacts()
{
 # Exclude Agent modules 
 ReplaceableOnAgent=('bin/plugin-validator.sh');
 ReplaceableOnAgent=("${ReplaceableOnAgent[@]}" 'bin/rhq-agent.bat');
 ReplaceableOnAgent=("${ReplaceableOnAgent[@]}" 'bin/rhq-agent.sh');
 ReplaceableOnAgent=("${ReplaceableOnAgent[@]}" 'bin/rhq-agent-wrapper.bat');
 ReplaceableOnAgent=("${ReplaceableOnAgent[@]}" 'bin/rhq-agent-wrapper.sh');
 ReplaceableOnAgent=("${ReplaceableOnAgent[@]}" 'bin/standalone-pc.bat');
 ReplaceableOnAgent=("${ReplaceableOnAgent[@]}" 'bin/standalone-pc.sh');
 ReplaceableOnAgent=("${ReplaceableOnAgent[@]}" 'bin/wrapper/windows-x86_32/wrapper.dll');
 ReplaceableOnAgent=("${ReplaceableOnAgent[@]}" 'bin/wrapper/windows-x86_32/wrapper.exe');
 ReplaceableOnAgent=("${ReplaceableOnAgent[@]}" 'bin/wrapper/windows-x86_32/wrapper.jar');

 # Exclude Server modules 
 ReplaceableOnServer=('bin/internal/*');
 ReplaceableOnServer=("${ReplaceableOnServer[@]}" 'bin/rhq-data-migration.bat');
 ReplaceableOnServer=("${ReplaceableOnServer[@]}" 'bin/rhq-data-migration.sh');
 ReplaceableOnServer=("${ReplaceableOnServer[@]}" 'bin/rhq-encode-password.bat');
 ReplaceableOnServer=("${ReplaceableOnServer[@]}" 'bin/rhq-encode-password.sh');
 ReplaceableOnServer=("${ReplaceableOnServer[@]}" 'bin/wrapper/windows-x86_32/wrapper.dll');
 ReplaceableOnServer=("${ReplaceableOnServer[@]}" 'bin/wrapper/windows-x86_32/wrapper.exe');
 ReplaceableOnServer=("${ReplaceableOnServer[@]}" 'bin/wrapper/windows-x86_32/wrapper.jar');

 echo -e "Replaceable on Agent \n"
 echo ${ReplaceableOnAgent[@]};
 echo -e "\nReplaceable on Server \n"
 echo ${ReplaceableOnServer[@]};
}
is_replaceable()
{
  PARAM=$1;
  LIST=$2; # take second argument AGENT|SERVER
  if [[ "$LIST" == *"AGENT"* ]];
  then
    if [[ " ${ReplaceableOnAgent[*]} " == *" $PARAM "* ]];
    then
      echo "true";
    else
      echo "false";
    fi
  else
    if [[ "$PARAM" == *"bin/internal/"* ]];
    then
       echo "true";
    elif [[ " ${ReplaceableOnServer[*]} " == *" $PARAM "* ]];
    then
      echo "true";
    else
      echo "false";
    fi
  fi
}

#========================================================================================
# Description: Select the type of 'update' an then generate update binary.
#========================================================================================
execute_type_of_update(){
 #populate replaceable artifact definition
 load_replaceable_artifacts

 ## generate the name of the artifact
 # convert to lower case
 debug "UPDATE_TYPE=\"$(echo $UPDATE_TYPE | tr '[A-Z]' '[a-z]')\""
 UPDATE_TYPE="$(echo $UPDATE_TYPE | tr '[A-Z]' '[a-z]')"
 debug "UPDATE_TYPE" "$UPDATE_TYPE"

 ############################################
 #generate_server_update
 if [ "$UPDATE_TYPE" == "server" ]
 then
   UPDATE_ARTIFACT_ID="jon-server-$VERSION-update-$UPDATE_ID"
   debug "UPDATE_ARTIFACT_ID" "$UPDATE_ARTIFACT_ID"

   PREVIOUS_UPDATE_ARTIFACT_ID="jon-server-$VERSION-update-01"
   debug "PREVIOUS_UPDATE_ARTIFACT_ID" "$PREVIOUS_UPDATE_ARTIFACT_ID"

   UPDATE_ARTIFACT="$UPDATE_ARTIFACT_ID.zip"
   debug "UPDATE_ARTIFACT" "$UPDATE_ARTIFACT"

   SERVER_ARTIFACT="jon-server-$VERSION.zip"
   debug "SERVER_ARTIFACT" "$SERVER_ARTIFACT"

   generate_server_update
 fi

 ############################################
 #generate_agent_update 
 if [ "$UPDATE_TYPE" == "agent" ]
 then
   UPDATE_ARTIFACT="jon-agent-$VERSION-update-$UPDATE_ID.zip";
   debug "UPDATE_ARTIFACT" "$UPDATE_ARTIFACT"

   #Launch agent update logic
   generate_agent_update
 fi

 # UPDATE_TYPE=plugins && UPDATE_PLUGIN=(non-empty)
 # plugins
 if [ "$UPDATE_TYPE" == "plugins" ]
 then
   UPDATE_ARTIFACT_ID="jon-plugin-pack-$UPDATE_PLUGIN-$VERSION-update-$UPDATE_ID"
   debug "UPDATE_ARTIFACT_ID" "$UPDATE_ARTIFACT_ID"

   UPDATE_ARTIFACT="$UPDATE_ARTIFACT_ID.zip"
   debug "UPDATE_ARTIFACT" "$UPDATE_ARTIFACT"

   PLUGIN_ARTIFACT="jon-plugin-pack-$UPDATE_PLUGIN-$VERSION.zip"
   debug "PLUGIN_ARTIFACT" "$PLUGIN_ARTIFACT"

   #Launch plugin update logic
   generate_plugins_update
 fi
 echo "Generating update zip contents for: $UPDATE_ARTIFACT ..."
}

#========================================================================================
# Description: Builds the update artifact binary.
#========================================================================================
generate_update_artifact(){

 # -collect remaining artifacts to be included in the binary
 echo
 echo "Collecting remaining contents for the update binary."
 cp "$REPO_ROOT/patches/$UPDATE_SCRIPT.sh" "$BUILD" 
 cp "$REPO_ROOT/patches/$UPDATE_SCRIPT.bat" "$BUILD"
 cp "$REPO_ROOT/patches/$REMOVED_LIST" "$PATCHED_DIR/$REMOVED_LIST"
 cp "$REPO_ROOT/patches/$JBOSS_CLI_COMMANDS" "$PATCHED_DIR/$JBOSS_CLI_COMMANDS" 
 chmod 755 "$BUILD/$UPDATE_SCRIPT.sh"
 cp "$REPO_ROOT/patches/UPDATE_README.txt" "$BUILD/UPDATE_README.txt" 

 if [ "$UPDATE_TYPE" == "server" ]
 then
   # remove extra zip
   debug "rm \"$BUILD/$SERVER_ARTIFACT\"";
   rm "$BUILD/$SERVER_ARTIFACT";
 fi

 # -generate the update binary
 echo 
 echo "Generating the update binary now..."
 mkdir $BUILD/$UPDATE_ARTIFACT_ID
 cp -r -p  "$BUILD/$SERVER_OVERLAY" $BUILD/$UPDATE_ARTIFACT_ID
 cp -r -p "$BUILD/$AGENT_OVERLAY" $BUILD/$UPDATE_ARTIFACT_ID
 cp -p "$BUILD/$UPDATE_SCRIPT.bat" $BUILD/$UPDATE_ARTIFACT_ID
 cp -p "$BUILD/$UPDATE_SCRIPT.sh" $BUILD/$UPDATE_ARTIFACT_ID
 cp -p "$BUILD/$UPDATE_README" $BUILD/$UPDATE_ARTIFACT_ID

 #Add supporting-scripts content.
 debug "mkdir $BUILD/$SCRIPTS_DIR";
 mkdir "$BUILD/$SCRIPTS_DIR";
 # recursively copy all content into place.
 debug "cp -r -p \"$REPO_ROOT/patches/supporting-scripts/*\" \"$BUILD/$SCRIPTS_DIR/\"";
 cp -r -p "$REPO_ROOT"/patches/supporting-scripts/* "$BUILD/$SCRIPTS_DIR/"
 cp -r -p "$BUILD/$SCRIPTS_DIR" $BUILD/$UPDATE_ARTIFACT_ID 

 debug "rm -rf \"$BUILD/$UPDATE_ARTIFACT_ID/$UPDATE_ARTIFACT_ID\"";
 rm -rf "$BUILD/$UPDATE_ARTIFACT_ID/$UPDATE_ARTIFACT_ID";

 chmod 755 $BUILD/$UPDATE_ARTIFACT_ID/*.sh
 cp -r -p $BUILD/.patched $BUILD/$UPDATE_ARTIFACT_ID

 debug "cd $BUILD"
 cd $BUILD

 debug "rm -rf .patched";
 rm -rf .patched # remove extra .patched dir visible

 LARGE_PATH_DIR="$SERVER_OVERLAY";
 debug "cd \"$UPDATE_ARTIFACT_ID/$LARGE_PATH_DIR\"";

 #BZ 1484900: include necessary license files
 ##make docs unpack dir aka $UPDATE_BUILD_DOCS_DIR
 debug "mkdir -p \"$UPDATE_BUILD_DOCS_DIR\"";
 mkdir -p "$UPDATE_BUILD_DOCS_DIR";
 ## copy $JON_LICENSE_FILE to $UPDATE_BUILD_DOCS_DIR
 debug "cp \"$JON_LICENSE_FILE\" \"$UPDATE_BUILD_DOCS_DIR\""
 cp "$JON_LICENSE_FILE" "$UPDATE_BUILD_DOCS_DIR"
 debug "FILE_HASH=\$(openssl dgst -md5 \"$UPDATE_BUILD_DOCS_DIR/licenses_\"*.zip\");"
 FILE_HASH=$(openssl dgst -md5 "$UPDATE_BUILD_DOCS_DIR/licenses_"*.zip);
 debug "FILE_HASH" "$FILE_HASH";
 ##make docs directory in update artifact. Ex. make "$BUILD/$UPDATE_ARTIFACT_ID/$LARGE_PATH_DIR/docs"
 debug "mkdir -p \"$BUILD/$UPDATE_ARTIFACT_ID/$LARGE_PATH_DIR/docs\"";
 mkdir -p "$BUILD/$UPDATE_ARTIFACT_ID/$LARGE_PATH_DIR/docs"
 ##wget contents and echo digest
 debug "wget \"$EAP_64_LICENSE_URL\" -P \"$UPDATE_BUILD_DOCS_DIR\"";
 wget "$EAP_64_LICENSE_URL" -P "$UPDATE_BUILD_DOCS_DIR"
 ##echo digest local license file
 debug "FILE_HASH=\$(openssl dgst -md5 \"$UPDATE_BUILD_DOCS_DIR/jboss-as-dist-\"*.zip\");";
 FILE_HASH=$(openssl dgst -md5 "$UPDATE_BUILD_DOCS_DIR/jboss-as-dist-"*.zip);
 debug "FILE_HASH" "$FILE_HASH";
 ##extract both to $UPDATE_BUILD_DOCS_DIR
 debug "unzip \"$UPDATE_BUILD_DOCS_DIR/licenses_\"*.zip -d \"$UPDATE_BUILD_DOCS_DIR/\""
 unzip "$UPDATE_BUILD_DOCS_DIR/licenses_"*.zip -d "$UPDATE_BUILD_DOCS_DIR/"
 debug "unzip \"$UPDATE_BUILD_DOCS_DIR/jboss-as-dist-\"*.zip -d \"$UPDATE_BUILD_DOCS_DIR/\""
 unzip "$UPDATE_BUILD_DOCS_DIR/jboss-as-dist-"*.zip -d "$UPDATE_BUILD_DOCS_DIR/"

 ##copy content from both locations into "$BUILD/$UPDATE_ARTIFACT_ID/docs"
 debug "cp -r \"$UPDATE_BUILD_DOCS_DIR/$EAP_64_LICENSE_PATH/licenses\" \"$BUILD/$UPDATE_ARTIFACT_ID/$LARGE_PATH_DIR/docs/\";"
 cp -r "$UPDATE_BUILD_DOCS_DIR/$EAP_64_LICENSE_PATH/licenses" "$BUILD/$UPDATE_ARTIFACT_ID/$LARGE_PATH_DIR/docs/";
 debug "cp -r \"$UPDATE_BUILD_DOCS_DIR/$JON_LICENSE_PATH\" \"$BUILD/$UPDATE_ARTIFACT_ID/$LARGE_PATH_DIR/docs/\";"
 cp -r "$UPDATE_BUILD_DOCS_DIR/$JON_LICENSE_PATH" "$BUILD/$UPDATE_ARTIFACT_ID/$LARGE_PATH_DIR/docs/";

 # BZ 1294626 : repackage long path contents into subzip to be unpackaged via script accordingly to work around path length issues.
 ## zip up large path contents in it's own zip. Ex. jon-server-3.3-update-08.zip(includes 'jon-server-updates.zip' and empty dir 'jon-server-updates')
 cd "$UPDATE_ARTIFACT_ID/$LARGE_PATH_DIR";
 pwd
 ls -al
 debug "zip -r \"$BUILD/$UPDATE_ARTIFACT_ID/$LARGE_PATH_DIR.zip\" \"*\""; 
 zip -r "$BUILD/$UPDATE_ARTIFACT_ID/$LARGE_PATH_DIR.zip" *

 debug "cd $BUILD";
 cd "$BUILD"
 pwd
 debug "rm -rf \"$BUILD/$UPDATE_ARTIFACT_ID/$LARGE_PATH_DIR\"";
 rm -rf "$BUILD/$UPDATE_ARTIFACT_ID/$LARGE_PATH_DIR";

 debug "mkdir \"$BUILD/$UPDATE_ARTIFACT_ID/$LARGE_PATH_DIR\"";
 mkdir "$BUILD/$UPDATE_ARTIFACT_ID/$LARGE_PATH_DIR"
 ls -al "$UPDATE_ARTIFACT_ID"
 
 # Create release zip.
 debug "zip -r \"$BUILD/$UPDATE_ARTIFACT\" \"$UPDATE_ARTIFACT_ID\"";
 zip -r "$BUILD/$UPDATE_ARTIFACT" "$UPDATE_ARTIFACT_ID" 
 
 # -display success/failure.
 echo "The '$UPDATE_TYPE' update artifact for JON is complete. You should upload $BUILD/$UPDATE_ARTIFACT to jon01 and forward
  to QE for testing/verification."

}
#========================================================================================
# Description: Builds the specific plugin pack update binary.
#========================================================================================
generate_plugins_update(){

 #Copy the specific artifact for expansion
 debug "cp \"$UPDATE_SRC_ROOT/$PLUGIN_ARTIFACT\" \"$BUILD/$PLUGIN_ARTIFACT\""
 cp "$UPDATE_SRC_ROOT/$PLUGIN_ARTIFACT" "$BUILD/$PLUGIN_ARTIFACT"

 # -generate metadata for the update.
 echo -e "Name: \t\t $UPDATE_ARTIFACT_ID" > "$PATCHED_DIR/$UPDATE_ARTIFACT_ID.txt"
 echo -e "Update File: \t $PLUGIN_ARTIFACT"
 echo -e "ID: \t\t (Change me)" >> "$PATCHED_DIR/$UPDATE_ARTIFACT_ID.txt"
 echo -e "Version: \t $VERSION" >> "$PATCHED_DIR/$UPDATE_ARTIFACT_ID.txt"
 echo -e "Build ID: \t (Change me)" >> "$PATCHED_DIR/$UPDATE_ARTIFACT_ID.txt"
 echo -e "Build Time: \t (Change me)" >> "$PATCHED_DIR/$UPDATE_ARTIFACT_ID.txt"
 echo -e "" >> "$PATCHED_DIR/$UPDATE_ARTIFACT_ID.txt"
 openssl dgst -md5 $BUILD/*.zip >> "$PATCHED_DIR/$UPDATE_ARTIFACT_ID.txt"
 #TODO: i)insert hash of patch file
 #      ii)expand patch and layout hashes for individual values?

}

#========================================================================================
# Description: Builds the specific server update binary.
#========================================================================================
generate_server_update(){
 GA="GA"
 #Copy the specific artifact for expansion
 debug "cp \"$UPDATE_SRC_ROOT/$SERVER_ARTIFACT\" \"$BUILD/$SERVER_ARTIFACT\""
 cp "$UPDATE_SRC_ROOT/$SERVER_ARTIFACT" "$BUILD/$SERVER_ARTIFACT"

 ## Process the artifact to build the update artifact.
 # Pull down original GA build
 debug "get_brew_artifacts \"$GA\" \"$GA_BREW_ID\""
 get_brew_artifacts "$GA" "$GA_BREW_ID"

 ## Test to see whether the we're being asked to 
 if [[ $USE_SCRATCH_JON == *cp.use.scratch.jon* ]];
 then
     echo "Proceeding with normal(no scratch content) brew build...."
     :; # Normal brew build. All brew urls are actual.
 else
   debug "openssl dgst -md5 $UPDATE_SRC_ROOT/$SERVER_ARTIFACT";
   openssl dgst -md5 "$UPDATE_SRC_ROOT/$SERVER_ARTIFACT"; 
   # TODO: should we be paranoid and only process from brew urls? Rel-eng might want that.
   echo "@@@@ WARNING: This option should ONLY be used for scratch builds. NOT for production use. @@@@"
   # pull down the scratch artifact && replace the JON build as expected.
   debug "wget --no-check-certificate $USE_SCRATCH_JON -O $UPDATE_SRC_ROOT/$SERVER_ARTIFACT";
   wget --no-check-certificate "$USE_SCRATCH_JON" -O "$UPDATE_SRC_ROOT/$SERVER_ARTIFACT";
   
   # Check for wget failure and exit if so.
   debug "GET_RESULT=\"$?\"";
   GET_RESULT="$?";
   debug "GET_RESULT" "$GET_RESULT"; 
   if [ $? -ne 0 ];
   then
      echo "Attempt to retrieve scratch content [$USE_SCRATCH_JON] has failed [$GET_RESULT]. Exiting...";
      exit 1;
   fi

   debug "openssl dgst -md5 $UPDATE_SRC_ROOT/$SERVER_ARTIFACT";
   openssl dgst -md5 "$UPDATE_SRC_ROOT/$SERVER_ARTIFACT"   
 fi

 # Check for file presence, if not exit hard/fail.
 if [ ! -f "$UPDATE_SRC_ROOT/$SERVER_ARTIFACT" ]; 
 then 
   echo "[ERROR] JON Server file was NOT retrieved for some reason. Exiting...."; 
   exit 1; 
 fi

 # Extract the POST_GA and GA
 debug "unzip -q \"$UPDATE_SRC_ROOT/$SERVER_ARTIFACT\" -d \"$UPDATE_SRC_ROOT\"";
 unzip -q "$UPDATE_SRC_ROOT/$SERVER_ARTIFACT" -d "$UPDATE_SRC_ROOT"; # unpack zip to [EXTRACTED] 
 
 # Copy in additional/storage/files before diff. 
 debug "Processing extra storage node updates for cumulative patch."
 ## Extract latest cassandra version
 ZIP_STORAGE="$UPDATE_SRC_ROOT/ZIP_STORAGE";
 debug "ZIP_STORAGE" "$ZIP_STORAGE"
 debug "mkdir -p \"$ZIP_STORAGE\"";
 mkdir -p "$ZIP_STORAGE"
 CASSANDRA_CCM_CORE="rhq-cassandra-ccm-core-$CASSANDRA_CCM_CORE_VERSION"
 debug "CASSANDRA_CCM_CORE" "$CASSANDRA_CCM_CORE"
 debug "cp '$UPDATE_SRC_ROOT/jon-server-$VERSION/modules/org/rhq/rhq-cassandra-installer/main/$CASSANDRA_CCM_CORE.jar' '$ZIP_STORAGE/$CASSANDRA_CCM_CORE.jar'";
 cp "$UPDATE_SRC_ROOT/jon-server-$VERSION/modules/org/rhq/rhq-cassandra-installer/main/$CASSANDRA_CCM_CORE.jar" "$ZIP_STORAGE/$CASSANDRA_CCM_CORE.jar"; 
 debug "mkdir -p '$ZIP_STORAGE/$CASSANDRA_CCM_CORE/cassandra'";
 mkdir -p "$ZIP_STORAGE/$CASSANDRA_CCM_CORE/cassandra"
 debug "unzip '$ZIP_STORAGE/$CASSANDRA_CCM_CORE.jar' -d '$ZIP_STORAGE/$CASSANDRA_CCM_CORE'";
 unzip "$ZIP_STORAGE/$CASSANDRA_CCM_CORE.jar" -d "$ZIP_STORAGE/$CASSANDRA_CCM_CORE" 
 debug "unzip '$ZIP_STORAGE/$CASSANDRA_CCM_CORE/cassandra.zip' -d '$ZIP_STORAGE/$CASSANDRA_CCM_CORE/cassandra'";
 unzip "$ZIP_STORAGE/$CASSANDRA_CCM_CORE/cassandra.zip" -d "$ZIP_STORAGE/$CASSANDRA_CCM_CORE/cassandra" 
 debug "mkdir -p '$UPDATE_SRC_ROOT/jon-server-$VERSION/$RHQ_STORAGE_DIR'";
 mkdir -p "$UPDATE_SRC_ROOT/jon-server-$VERSION/$RHQ_STORAGE_DIR"
 ## BZ 1088046: 
 debug "mkdir -p '$UPDATE_SRC_ROOT/jon-server-$VERSION/$RHQ_STORAGE_DIR/bin'";
 mkdir -p "$UPDATE_SRC_ROOT/jon-server-$VERSION/$RHQ_STORAGE_DIR/bin"
 debug "cp '$ZIP_STORAGE/$CASSANDRA_CCM_CORE/cassandra/bin/cassandra' '$UPDATE_SRC_ROOT/jon-server-$VERSION/$RHQ_STORAGE_DIR/bin'";
 cp "$ZIP_STORAGE/$CASSANDRA_CCM_CORE/cassandra/bin/cassandra" "$UPDATE_SRC_ROOT/jon-server-$VERSION/$RHQ_STORAGE_DIR/bin"
 chmod 764 "$UPDATE_SRC_ROOT/jon-server-$VERSION/$RHQ_STORAGE_DIR/bin/cassandra"
 ## BZ 1229978: 
 #copy additional storage node files
 debug "mkdir -p '$UPDATE_SRC_ROOT/jon-server-$VERSION/$RHQ_STORAGE_DIR/conf'"
 mkdir -p "$UPDATE_SRC_ROOT/jon-server-$VERSION/$RHQ_STORAGE_DIR/conf"
 debug "mkdir -p '$UPDATE_SRC_ROOT/jon-server-$VERSION/$RHQ_STORAGE_DIR/lib'"
 mkdir -p "$UPDATE_SRC_ROOT/jon-server-$VERSION/$RHQ_STORAGE_DIR/lib"
 debug "cp '$ZIP_STORAGE/$CASSANDRA_CCM_CORE/cassandra/lib/apache-cassandra*' '$UPDATE_SRC_ROOT/jon-server-$VERSION/$RHQ_STORAGE_DIR/lib/'"
 cp "$ZIP_STORAGE/$CASSANDRA_CCM_CORE/cassandra/lib/apache-cassandra"* "$UPDATE_SRC_ROOT/jon-server-$VERSION/$RHQ_STORAGE_DIR/lib/"
 # BZ 1272239: 
 STORAGE_NODE_NEW_PROPERTIES="$REPO_ROOT/jon/dist/server/src/main/resources-binary/BZ_1272239";
 debug "cp '$STORAGE_NODE_NEW_PROPERTIES/cassandra-jvm.properties.new' '$UPDATE_SRC_ROOT/jon-server-$VERSION/$RHQ_STORAGE_DIR/conf/cassandra-jvm.properties.new'"
 cp "$STORAGE_NODE_NEW_PROPERTIES/cassandra-jvm.properties.new" "$UPDATE_SRC_ROOT/jon-server-$VERSION/$RHQ_STORAGE_DIR/conf/cassandra-jvm.properties.new"
 # BZ 1274425
 STORAGE_WRAPPER_CONF_NEW="$REPO_ROOT/jon/dist/server/src/main/resources-binary/BZ_1274425";
 #copy additional storage node files
 debug "mkdir -p '$UPDATE_SRC_ROOT/jon-server-$VERSION/bin/wrapper'"
 mkdir -p "$UPDATE_SRC_ROOT/jon-server-$VERSION/bin/wrapper"
 debug "cp '$STORAGE_WRAPPER_CONF_NEW/rhq-storage-wrapper.conf.new' '$UPDATE_SRC_ROOT/jon-server-$VERSION/bin/wrapper/rhq-storage-wrapper.conf.new'"
 cp "$STORAGE_WRAPPER_CONF_NEW/rhq-storage-wrapper.conf.new" "$UPDATE_SRC_ROOT/jon-server-$VERSION/bin/wrapper/rhq-storage-wrapper.conf.new"
 # *NOTE* if there are additional storage node changes for JON 3.3.x to either rhq-storage-wrapper.conf or cassandra-jvm.properties, both will need to include the previous content as defined by BZs 1272239 and/or 1274425 as is relevant.

 # BZ_1517717: See BZ for more details, but we're storing a copy of the storage patch values for the case where we can patch the storage node.  
 STORAGE_PROPERTIES_PREINSTALL="$REPO_ROOT/jon/dist/server/src/main/resources-binary/BZ_1517717";
 ## -generate the rhq-storage.properties.preinstall file
 # copy rhq-storage.properties -> .patched\rhq-storage.properties.preinstall
 debug "cp '$STORAGE_PROPERTIES_PREINSTALL/$RHQ_STORAGE_PROPERTIES' '$PATCHED_DIR/$RHQ_STORAGE_PROPERTIES.preinstall'"
 cp "$STORAGE_PROPERTIES_PREINSTALL/$RHQ_STORAGE_PROPERTIES" "$PATCHED_DIR/$RHQ_STORAGE_PROPERTIES.preinstall"
 # end of storage node edits

 debug "unzip -q \"$UPDATE_SRC_ROOT/$GA/$SERVER_ARTIFACT\" -d \"$UPDATE_SRC_ROOT/$GA\"";
 unzip -q "$UPDATE_SRC_ROOT/$GA/$SERVER_ARTIFACT" -d "$UPDATE_SRC_ROOT/$GA"; # unpack zip to [EXTRACTED] 

 ##Decision to rebundle agent+cli+server with every cumulative patch. Copy agent and cli into update root
 #Update the agent for download
 mkdir -p "$ZIP/jon-server-$VERSION/$AGENT_REL_LOCATION";
 debug "cp \"$UPDATE_SRC_ROOT/jon-server-$VERSION/$AGENT_REL_LOCATION/$AGENT_ARTIFACT\" \"$ZIP/jon-server-$VERSION/$AGENT_REL_LOCATION/$AGENT_ARTIFACT\"";
 cp "$UPDATE_SRC_ROOT/jon-server-$VERSION/$AGENT_REL_LOCATION/$AGENT_ARTIFACT" "$ZIP/jon-server-$VERSION/$AGENT_REL_LOCATION/$AGENT_ARTIFACT";
 #Update the CLI for download
 mkdir -p "$ZIP/jon-server-$VERSION/$CLI_REL_LOCATION";
 debug "cp \"$UPDATE_SRC_ROOT/jon-server-$VERSION/$CLI_REL_LOCATION/$CLI_ARTIFACT\" \"$ZIP/jon-server-$VERSION/$CLI_REL_LOCATION/$CLI_ARTIFACT\"";
 cp "$UPDATE_SRC_ROOT/jon-server-$VERSION/$CLI_REL_LOCATION/$CLI_ARTIFACT" "$ZIP/jon-server-$VERSION/$CLI_REL_LOCATION/$CLI_ARTIFACT";
 #[BZ 1206641] Update the Storage node artifact to the latest to patch all future storage node installations.
 debug "Adding [rhq-cassandra-installer] and [rhq-installer-util] folders....";
 debug "mkdir -p [$ZIP/jon-server-$VERSION/modules/org/rhq/rhq-cassandra-installer/main]";
 mkdir -p "$ZIP/jon-server-$VERSION/modules/org/rhq/rhq-cassandra-installer/main";
 debug "mkdir -p $ZIP/jon-server-$VERSION/modules/org/rhq/rhq-installer-util/main]";
 mkdir -p "$ZIP/jon-server-$VERSION/modules/org/rhq/rhq-installer-util/main";

 debug "cp [$UPDATE_SRC_ROOT/jon-server-$VERSION/modules/org/rhq/rhq-cassandra-installer/main/$CASSANDRA_CCM_CORE.jar] [$ZIP/jon-server-$VERSION/modules/org/rhq/rhq-cassandra-installer/main/$CASSANDRA_CCM_CORE.jar];"
 cp "$UPDATE_SRC_ROOT/jon-server-$VERSION/modules/org/rhq/rhq-cassandra-installer/main/$CASSANDRA_CCM_CORE.jar" "$ZIP/jon-server-$VERSION/modules/org/rhq/rhq-cassandra-installer/main/$CASSANDRA_CCM_CORE.jar";
 debug "cp [$UPDATE_SRC_ROOT/jon-server-$VERSION/modules/org/rhq/rhq-cassandra-installer/main/$CASSANDRA_CCM_CORE.jar] [$ZIP/jon-server-$VERSION/modules/org/rhq/rhq-installer-util/main/$CASSANDRA_CCM_CORE.jar];"
 cp "$UPDATE_SRC_ROOT/jon-server-$VERSION/modules/org/rhq/rhq-cassandra-installer/main/$CASSANDRA_CCM_CORE.jar" "$ZIP/jon-server-$VERSION/modules/org/rhq/rhq-installer-util/main/$CASSANDRA_CCM_CORE.jar";

 #[BZ 1209558] Update *rtfilter*.zip content with latest version.
 # Update the connectors content for download. [spinder 7/9/15]connector-apache.zip NOT added now. No change and 7MB. 
 mkdir -p "$ZIP/jon-server-$VERSION/$CONNECTORS_REL_LOCATION";
 ## filter
 debug "cp \"$UPDATE_SRC_ROOT/jon-server-$VERSION/$CONNECTORS_REL_LOCATION/$CON_FILTER\" \"$ZIP/jon-server-$VERSION/$CONNECTORS_REL_LOCATION/$CON_FILTER\"";
 cp "$UPDATE_SRC_ROOT/jon-server-$VERSION/$CONNECTORS_REL_LOCATION/$CON_FILTER" "$ZIP/jon-server-$VERSION/$CONNECTORS_REL_LOCATION/$CON_FILTER";
 ## filter module
 debug "cp \"$UPDATE_SRC_ROOT/jon-server-$VERSION/$CONNECTORS_REL_LOCATION/$CON_FILTER_MODULE\" \"$ZIP/jon-server-$VERSION/$CONNECTORS_REL_LOCATION/$CON_FILTER_MODULE\"";
 cp "$UPDATE_SRC_ROOT/jon-server-$VERSION/$CONNECTORS_REL_LOCATION/$CON_FILTER_MODULE" "$ZIP/jon-server-$VERSION/$CONNECTORS_REL_LOCATION/$CON_FILTER_MODULE";
 ## filter subsystem
 debug "cp \"$UPDATE_SRC_ROOT/jon-server-$VERSION/$CONNECTORS_REL_LOCATION/$CON_FILTER_SUBSYS\" \"$ZIP/jon-server-$VERSION/$CONNECTORS_REL_LOCATION/$CON_FILTER_SUBSYS\"";
 cp "$UPDATE_SRC_ROOT/jon-server-$VERSION/$CONNECTORS_REL_LOCATION/$CON_FILTER_SUBSYS" "$ZIP/jon-server-$VERSION/$CONNECTORS_REL_LOCATION/$CON_FILTER_SUBSYS";

 # Calculate diff GA then Post-GA
 debug "diff -r \"$UPDATE_SRC_ROOT/jon-server-$VERSION\" \"$UPDATE_SRC_ROOT/$GA/jon-server-$VERSION\" > \"$UPDATE_SRC_ROOT/diff.txt\""
 diff -r "$UPDATE_SRC_ROOT/jon-server-$VERSION" "$UPDATE_SRC_ROOT/$GA/jon-server-$VERSION" > "$UPDATE_SRC_ROOT/diff.txt"

 ## create and get in location for difference detection.
 debug "mkdir $ZIP"
 mkdir $ZIP
 echo "cd \"$UPDATE_SRC_ROOT\""
 cd "$UPDATE_SRC_ROOT"

 ## Start special handling for patch content
 # The following CoreGUI.* files need to be removed to accomodate the latest build
 CORE_GUI_DIR="$EAR_REL_LOCATION/coregui.war"
 CORE_GUI="org.rhq.coregui.CoreGUI"
 DOMAIN="org.rhq.core.RHQDomain"
 CLASSES="classes"
 debug "mkdir -p \"$ZIP/jon-server-$VERSION/$CORE_GUI_DIR/WEB-INF/$CLASSES\""
 mkdir -p "$ZIP/jon-server-$VERSION/$CORE_GUI_DIR/WEB-INF/$CLASSES"

 debug "mv \"jon-server-$VERSION/$CORE_GUI_DIR/$DOMAIN\" \"$ZIP/jon-server-$VERSION/$CORE_GUI_DIR/$DOMAIN\""
 mv "jon-server-$VERSION/$CORE_GUI_DIR/$DOMAIN" "$ZIP/jon-server-$VERSION/$CORE_GUI_DIR/$DOMAIN"
 debug "mv \"jon-server-$VERSION/$CORE_GUI_DIR/$CORE_GUI\" \"$ZIP/jon-server-$VERSION/$CORE_GUI_DIR/$CORE_GUI\""
 mv "jon-server-$VERSION/$CORE_GUI_DIR/$CORE_GUI" "$ZIP/jon-server-$VERSION/$CORE_GUI_DIR/$CORE_GUI"   
 debug " mv \"jon-server-$VERSION/$CORE_GUI_DIR/CoreGUI.html\" \"$ZIP/jon-server-$VERSION/$CORE_GUI_DIR/CoreGUI.html\""
 mv "jon-server-$VERSION/$CORE_GUI_DIR/CoreGUI.html" "$ZIP/jon-server-$VERSION/$CORE_GUI_DIR/CoreGUI.html"
 # spinder: This is rare on patch branch. If we edit the .cs or .js files again we need to increment from redhat-1 -> redhat-2.
 # [1/20/15] removing below as no more css in root coregui folder. Patternfly?
 #debug "mv jon-server-$VERSION/$CORE_GUI_DIR/CoreGUI-$RHQ_VERSION$REDHAT_INCREMENT-1.css $ZIP/jon-server-$VERSION/$CORE_GUI_DIR/CoreGUI-$RHQ_VERSION$REDHAT_INCREMENT-1.css"
 #mv "jon-server-$VERSION/$CORE_GUI_DIR/CoreGUI-$RHQ_VERSION$REDHAT_INCREMENT-1.css" "$ZIP/jon-server-$VERSION/$CORE_GUI_DIR/CoreGUI-$RHQ_VERSION$REDHAT_INCREMENT-1.css"
 debug "mv \"jon-server-$VERSION/$CORE_GUI_DIR/WEB-INF/$CLASSES\" \"$ZIP/jon-server-$VERSION/$CORE_GUI_DIR/WEB-INF\""
 mv "jon-server-$VERSION/$CORE_GUI_DIR/WEB-INF/$CLASSES" "$ZIP/jon-server-$VERSION/$CORE_GUI_DIR/WEB-INF"

 # Store off files for the removed-list/backup.
 REMOVED_LIST_ADDITIONS="removed-list.txt";
 echo "" > $REMOVED_LIST_ADDITIONS;

 ## iterate over binary differences. Including: .keystore,.jar,.zip,.class
 debug "\n\n Beginning to parse binary differences."
 # Policy: i)Updated classes in newer build should be taken ii)patched jars ignored iii)zips special cases
 for new_bin in `diff -r "jon-server-$VERSION" "$GA/jon-server-$VERSION" | grep -i "Binary files.*differ" | cut -d' ' -f3`
 do
   #echo "$new_bin" >> "$BUILD/bin-diff-only.txt";  # uncomment for detailed diff content
   TRIMMED=${new_bin#jon-server-$VERSION/};
   debug "TRIMMED" "$TRIMMED"
   if [[ $new_bin == *rhq.keystore* ]] || [[ $new_bin == *ojdbc6-11.2.0.3.0* ]] ; # ignore the jbossas/keystore and oracle jar files
    then
     :; # ignore
   # spinder: 7-15-14: handle this class as a special cases. Odd brew behavior at times with RHQLParser.
   #   modules/org/rhq/server-startup/main/deployments/rhq.ear/rhq-server.jar/org/rhq/enterprise/server/search/RHQLParser.class
   # NOTE: Internal classes may legitimately be re-generated even when there are no relevant git changes to the enclosing class source.
   #   See BZ 1110462 and following class for more details:   
   #   modules/org/rhq/server-startup/main/deployments/rhq.ear/rhq-server.jar/org/rhq/enterprise/server/rest/MetricHandlerBean$RawNumericStreamingOutput.class
   elif [[ $new_bin == *RHQLParser.class ]] 
    then
      :; # ignore completely. This class is completely generated.
      # this happens during RHQ brew build. Another option is to rebuild RHQ in brew again.
   elif [[ $new_bin == bin/wrapper/windows-x86_32/wrapper.jar ]]; # safe to take updates to this jar.
     then
       debug "Found server binary update for [$new_bin], needs to be applied.";
       CURRENT_DIR=$(dirname "${new_bin}"); # get current
       debug "CURRENT_DIR" "$CURRENT_DIR";
       debug "mkdir -p \"$ZIP/$CURRENT_DIR\"";
       mkdir -p "$ZIP/$CURRENT_DIR";
       debug "mv $new_bin $ZIP/$CURRENT_DIR"
       mv $new_bin $ZIP/$CURRENT_DIR
       # optionally, store off change for backup/removed-list
       if grep -Fxq "$TRIMMED" "$REPO_REMOVED_LIST"
       then
         # ignore. Already captured in the removed list file.
         debug "Skipping... [$TRIMMED] already in $REPO_REMOVED_LIST";
       else
         debug "echo $TRIMMED >> $REMOVED_LIST_ADDITIONS";
         echo "$TRIMMED" >> "$REMOVED_LIST_ADDITIONS";  
       fi 
   else
     if [[ $new_bin == *.jar ]] || [[ $new_bin == *.zip ]] ; 
     then 
      :;
      #Add support for special eap jar that is unversioned but affected during rebases.
      if [[ $new_bin == *jboss-modules.jar ]];
      then
        # detect rebase/fix changes 
        ORIGINAL_MODULE=$(openssl dgst -md5 $GA/jon-server-$VERSION/jbossas/jboss-modules.jar | cut -d'=' -f2);
        NEW_MODULE=$(openssl dgst -md5 jon-server-$VERSION/jbossas/jboss-modules.jar | cut -d'=' -f2);
        if [[ $ORIGINAL_MODULE == $NEW_MODULE ]];
        then
          :; # ignore, no update needed.
        else
          # detected rebase related update/patch of jboss-modules.jar. Unversioned in EAP 6.3.x
          debug "Detected rebase related update/patch of jboss-modules.jar...";
          debug "Processing binary difference [$new_bin]";
          CURRENT_DIR=$(dirname "${new_bin}"); # get current
          debug "CURRENT_DIR" "$CURRENT_DIR";
          debug "mkdir -p \"$ZIP/$CURRENT_DIR\"";
          mkdir -p "$ZIP/$CURRENT_DIR";
          debug "mv $new_bin $ZIP/$CURRENT_DIR"
          mv $new_bin $ZIP/$CURRENT_DIR
          # optionally, store off change for backup/removed-list
          if grep -Fxq "$TRIMMED" "$REPO_REMOVED_LIST"
          then
            # ignore. Already captured in the removed list file.
            debug "Skipping... [$TRIMMED] already in $REPO_REMOVED_LIST";
          else
            debug "echo $TRIMMED >> $REMOVED_LIST_ADDITIONS";
            echo "$TRIMMED" >> "$REMOVED_LIST_ADDITIONS";  
          fi 
        fi
      else
        debug "Ignoring $new_bin as is .jar/.zip";
      fi
     else # does not end in .jar or .zip, so new content for patch.
       debug "Processing binary difference [$new_bin]";
       CURRENT_DIR=$(dirname "${new_bin}"); # get current
       debug "CURRENT_DIR" "$CURRENT_DIR";
       debug "mkdir -p \"$ZIP/$CURRENT_DIR\"";
       mkdir -p "$ZIP/$CURRENT_DIR";
       debug "mv $new_bin $ZIP/$CURRENT_DIR"
       mv $new_bin $ZIP/$CURRENT_DIR
       # optionally, store off change for backup/removed-list
       if grep -Fxq "$TRIMMED" "$REPO_REMOVED_LIST"
       then
         # ignore. Already captured in the removed list file.
         debug "Skipping... [$TRIMMED] already in $REPO_REMOVED_LIST";
       else
         debug "echo $TRIMMED >> $REMOVED_LIST_ADDITIONS";
         echo "$TRIMMED" >> "$REMOVED_LIST_ADDITIONS";  
       fi 
     fi
   fi
 done

 # iterate over text differences. Including: .xml,.js,.MF,.properties,.java
 debug "\n\n Iterating over text only differences...";
 # Policy: i)All content outside[alert-scripts|bin|etc] dirs is valid content or updated metadata about library increments
 #         ii)Content in [alert-scripts|bin|etc] dirs must be check for replaceable, otherwise .new 
 for new_txt in `diff -r "jon-server-$VERSION" "$GA/jon-server-$VERSION" | grep -i "diff -r jon-server-$VERSION" | cut -d' ' -f3`
 do
   debug "File" "$new_txt";
   #echo "$new_txt" >> "$BUILD/text-diff-only.txt"; # uncomment for detailed diff content
   CURRENT_DIR=$(dirname "${new_txt}"); # get current
   debug "mkdir -p \"$ZIP/$CURRENT_DIR\"";
   mkdir -p "$ZIP/$CURRENT_DIR";
   TRIMMED=${new_txt#jon-server-$VERSION/};
   debug "TRIMMED" "$TRIMMED"

   if [[ $new_txt == jon-server-$VERSION/alert-scripts* ]] || [[ $new_txt == jon-server-$VERSION/bin* ]] || [[ $new_txt == jon-server-$VERSION/etc* ]]; #
   then # test for replaceability of files
     #Test to see if we're allowed to replace this file. Strip 'jon-server-$VERSION/' first.
     REPLACEABLE=`is_replaceable "$TRIMMED" "SERVER"`;
     debug "REPLACEABLE" "$REPLACEABLE";
     if [[ "$REPLACEABLE" == "true" ]];
     then
      debug "Server file $new_txt IS replaceable as is. Updating with patch.";
      debug "mv $new_txt $ZIP/$CURRENT_DIR"
      mv "$new_txt" "$ZIP/$CURRENT_DIR"
      # optionally, store off change for backup/removed-list
      if grep -Fxq "$TRIMMED" "$REPO_REMOVED_LIST"
      then
         # ignore. Already captured in the removed list file.
         debug "Skipping... [$TRIMMED] already in $REPO_REMOVED_LIST";
      else
         debug "echo $TRIMMED >> $REMOVED_LIST_ADDITIONS";
         echo "$TRIMMED" >> "$REMOVED_LIST_ADDITIONS";  
      fi 
     else
      FILE_NAME=$(basename $new_txt);
      debug "FILE_NAME" "$FILE_NAME"
      debug "Server file is NOT replaceable as customer could have modified.";
      debug "mv $new_txt $ZIP/$CURRENT_DIR/$FILE_NAME.new"
      mv "$new_txt" "$ZIP/$CURRENT_DIR/$FILE_NAME.new";
     fi
   else # take the updates.
     # Add special treatment for 6.4.x standalone.conf, standalone.conf.bat and standalone-full.xml
     if [[ $new_txt == jon-server-$VERSION/jbossas/bin/standalone.conf ]] || [[ $new_txt == jon-server-$VERSION/jbossas/bin/standalone.conf.bat ]] || [[ $new_txt == jon-server-$VERSION/jbossas/standalone/configuration/standalone-full.xml ]]; #
     then
         # spinder: 9-14-15: ignore changes in 6.4.x standalone.conf, standalone.conf.bat and standalone-full.xml 
         debug "Skipping... [$new_txt] as is 6.4 configuration update. JON 3.3.0 was released on EAP 6.3.x so using those versions instead.";
         echo "Parsing... [$new_txt]...";
     else 
         debug "mv $new_txt $ZIP/$CURRENT_DIR"
         mv $new_txt $ZIP/$CURRENT_DIR
         # optionally, store off change for backup/removed-list
         if grep -Fxq "$TRIMMED" "$REPO_REMOVED_LIST"
         then
             # ignore. Already captured in the removed list file.
             debug "Skipping... [$TRIMMED] already in $REPO_REMOVED_LIST";
         else
             debug "echo $TRIMMED >> $REMOVED_LIST_ADDITIONS";
             echo "$TRIMMED" >> "$REMOVED_LIST_ADDITIONS";  
         fi
     fi 
   fi
 done

 ## iterate over new files/dirs. Including: .jar
 #Stop for loops from splitting on spaces
 IFS=$'\n';
 debug "\n Iterating over new files/dirs..."

 # Policy: i)All the updated modules.
 for new_file in `diff -r "jon-server-$VERSION" "$GA/jon-server-$VERSION" | grep -i "Only in jon-server-$VERSION" | cut -d' ' -f3-`
 do
   #echo "$new_file" >> "$BUILD/post-ga-only.txt"; # uncomment for detailed diff content
   unset IFS; # re-enable default splitting
   #select components
   #debug "PARENT_DIR=\$(echo $new_file | cut -d':' -f1);"
   PARENT_DIR=$(echo $new_file | cut -d':' -f1);
   debug "PARENT_DIR" "$PARENT_DIR";
   #debug "FILE_PORTION=\$(echo $new_file | cut -d':' -f2)"
   FILE_PORTION=$(echo $new_file | cut -d':' -f2); 
   debug "FILE_PORTION" "$FILE_PORTION"
   FILE_NAME="${FILE_PORTION// }";
   debug "FILE_NAME" "$FILE_NAME";
   #debug "CURRENT_DIR=\$(dirname "$PARENT_DIR/$FILE_NAME");"
   CURRENT_DIR=$(dirname "$PARENT_DIR/$FILE_NAME"); # get current
   debug "CURRENT_DIR" "$CURRENT_DIR";
   debug "mkdir -p \"$ZIP/$CURRENT_DIR\"";
   mkdir -p "$ZIP/$CURRENT_DIR";
   debug "mv $PARENT_DIR/$FILE_NAME $ZIP/$CURRENT_DIR"
   mv $PARENT_DIR/$FILE_NAME $ZIP/$CURRENT_DIR
   #record files new to this release.
   echo "$PARENT_DIR/$FILE_NAME" >> "$BUILD/new-only.txt"

   IFS=$'\n'; # disable space splitting for next iteration
 done

 # iterate over old files/dirs. Including: .xml,.MF,.jar, 
 #Stop for loops from splitting on spaces
 IFS=$'\n';

 # Policy: i).jar got patched, remove. ii)coregui*, ignore. iii).xml,.MF,.properties,*.* is cruft, ignore.
 #         iv).class. Should not have much of this. Minor compile differences not referenced by updated class. Brew build stable.  
 for new_file in `diff -r "jon-server-$VERSION" "$GA/jon-server-$VERSION" | grep -i "Only in $GA/jon-server-$VERSION" | cut -d' ' -f3-`
 do
   #echo "$new_file" >> "$BUILD/ga-only.txt"; # uncomment for detailed diff content
   unset IFS; # re-enable default splitting
   #select components
   debug "PARENT_DIR=\$(echo $new_file | cut -d':' -f1);"
   PARENT_DIR=$(echo $new_file | cut -d':' -f1);
   debug "PARENT_DIR" "$PARENT_DIR";
   debug "FILE_PORTION=\$(echo $new_file | cut -d':' -f2)"
   FILE_PORTION=$(echo $new_file | cut -d':' -f2); 
   debug "FILE_PORTION" "$FILE_PORTION";
   FILE_NAME="${FILE_PORTION// }"; 
   debug "FILE_NAME" "$FILE_NAME";
   #remove prefix. Ex. GA/jon-server-3.2.0.GA/
   REMOVED="${PARENT_DIR#$GA/jon-server-$VERSION/}";
   debug "REMOVED" "$REMOVED";

   if [[ $FILE_NAME == "" ]] || [[ $FILE_NAME == "/" ]] || [[ $PARENT_DIR == *coregui.war* ]];
   then
     :; # ignore 
   else
     if [[ $FILE_NAME == *.jar ]];
     then
       if grep -Fxq "$REMOVED/$FILE_NAME" "$REPO_REMOVED_LIST"
       then
         # ignore. Already captured in the removed list file.
         debug "Skipping... [$REMOVED/$FILE_NAME] already in $REPO_REMOVED_LIST";
       else
         debug "echo $REMOVED/$FILE_NAME >> $REMOVED_LIST_ADDITIONS";
         echo "$REMOVED/$FILE_NAME" >> "$REMOVED_LIST_ADDITIONS";  
       fi 
     else
       :; # ignore the rest. No need to remove them. No real risk and is customer content.
     fi
   fi
   IFS=$'\n'; # disable space splitting for next iteration
 done

 #Allow bash to split on spaces again.
 unset IFS; # re-enable default splitting

 ##Apply CVE cumulative patches that need to be applied to EAP.
 #Need i)full jboss path to old version file ii)new brew url to new file iii)module file to update.
 CVE_LIST_FILE="$REPO_ROOT/patches/cve-list.txt";
 debug "CVE_LIST_FILE" "$CVE_LIST_FILE"

 echo -e "\n)(if applicable)Applying CVE changes."
 LINES_TO_REMOVE=$(wc -l < "$CVE_LIST_FILE");
 debug "LINES_TO_REMOVE" "$LINES_TO_REMOVE";

 if [ "$LINES_TO_REMOVE" -eq 0 ]; 
 then
   echo "No CVE's to apply.";
 else
   echo "Iterating over the following CVE file changes... ";
   # iterate over lines in CVE list and i)add to remove list
   #    ii)prepare to update content iii)apply and update eap content.
   debug " parsing $CVE_LIST_FILE instructions...";
   while read p; do
    if [[ "$p" == "" || "$p" == \#* ]]; 
     then
      # skip over comments and empty lines
      :;
     else
       debug "Parsing server CVE change[$p]..."
       # split line into appropriate elements
       OLD_FILE_PATH=`echo $p | cut -d' ' -f1`;
       NEW_FILE=`echo $p | cut -d' ' -f2`;
       NEW_FILE_IDENTIFIER=`echo $p | cut -d' ' -f3`;
       MODULE_FILE=`echo $p | cut -d' ' -f4`;
       debug "\n\tOLD_FILE_PATH=>[$OLD_FILE_PATH]";
       debug "\n\tNEW_FILE=>[$NEW_FILE]";
       debug "\n\tNEW_FILE_IDENTIFIER=>[$NEW_FILE_IDENTIFIER]";
       debug "\n\tMODULE_FILE=>[$MODULE_FILE]";
       # optionally, adding OLD_FILE_PATH to removed-list
       if grep -Fxq "$OLD_FILE_PATH" "$REPO_REMOVED_LIST"
       then
         # ignore. Already captured in the removed list file.
         debug "Skipping... [$OLD_FILE_PATH] already in $REPO_REMOVED_LIST";
       else
         debug "echo $OLD_FILE_PATH >> $REMOVED_LIST_ADDITIONS";
         echo "$OLD_FILE_PATH" >> "$REMOVED_LIST_ADDITIONS";  
       fi  
       #get OLD_FILE_PATH directory and duplicate dir structure in jon-server-$VERSION
       OLD_FILE_DIR=$(dirname "${OLD_FILE_PATH}"); # get current
       OLD_FILE_NAME=$(basename "${OLD_FILE_PATH}"); # get current
       debug "OLD_FILE_DIR" "$OLD_FILE_DIR"
       debug "OLD_FILE_NAME" "$OLD_FILE_NAME"
       debug "mkdir -p \"$ZIP/jon-server-$VERSION/$OLD_FILE_DIR\""
       mkdir -p "$ZIP/jon-server-$VERSION/$OLD_FILE_DIR";
       # check for http url and if so download, otherwise copy
       if [[ $NEW_FILE == http* ]]; 
       then
         debug "wget --no-check-certificate \"$NEW_FILE\" -O \"$ZIP/jon-server-$VERSION/$OLD_FILE_DIR/$NEW_FILE_IDENTIFIER\";"  
         wget --no-check-certificate "$NEW_FILE" -O "$ZIP/jon-server-$VERSION/$OLD_FILE_DIR/$NEW_FILE_IDENTIFIER";
       else
	 debug "New file source is SCM. Updating content..."
         debug "cp \"$REPO_ROOT/$NEW_FILE\" \"$ZIP/jon-server-$VERSION/$OLD_FILE_DIR/$NEW_FILE_IDENTIFIER\"";
         cp "$REPO_ROOT/$NEW_FILE" "$ZIP/jon-server-$VERSION/$OLD_FILE_DIR/$NEW_FILE_IDENTIFIER"; 
       fi

       ## If EAP CVE, then process module file, otherwise direct update.
       if [[ -z "$MODULE_FILE" ]]; # no module file passed in
       then ## NON EAP file replacement
         debug "Simple file replace for CVE. No module.xml to update."
       else ## EAP CVE modification. 
         #update MODULE_FILE to reference newest file.
         debug "Updating the EAP module to reference patched content."
         #debug "cp \"$GA/jon-server-$VERSION/$OLD_FILE_DIR/$MODULE_FILE\" \"$ZIP/jon-server-$VERSION/$OLD_FILE_DIR/$MODULE_FILE\""
         #cp "$GA/jon-server-$VERSION/$OLD_FILE_DIR/$MODULE_FILE" "$ZIP/jon-server-$VERSION/$OLD_FILE_DIR/$MODULE_FILE"
         #Ex. sed -i "s/jbossweb-7.2.2.Final-redhat-1.jar/jbossweb-7.2.2.Final-redhat-2.jar/" jon-server-3.2.0.GA/jbossas/modules/system/layers/base/org/jboss/as/web/main/module.xml 
         #debug "sed -i \"s/$OLD_FILE_NAME/$NEW_FILE_IDENTIFIER/\" \"$ZIP/jon-server-$VERSION/$OLD_FILE_DIR/$MODULE_FILE\""
         #sed -i "s/$OLD_FILE_NAME/$NEW_FILE_IDENTIFIER/" "$ZIP/jon-server-$VERSION/$OLD_FILE_DIR/$MODULE_FILE"
       fi
       debug "finished applying patch and updating [$OLD_FILE_NAME] with [$NEW_FILE_IDENTIFIER]";
    fi
   done < "$CVE_LIST_FILE"
 fi

 # echo out the removed-list contents
 ## server removed list contents
 echo "Newly detected removed list entries for server includes:"
 cat "$REMOVED_LIST_ADDITIONS";
 echo "Newly detected removed list entries for server concluded."

 # build server overlay zip
 debug "cd $ZIP"
 cd $ZIP

 SERVER_OVERLAY="jon-server-updates";
 mkdir "$BUILD/$SERVER_OVERLAY";
 for file in "jon-server-$VERSION/"*
 do
  debug "mv \"$file\" \"$BUILD/$SERVER_OVERLAY\"";
  mv "$file" "$BUILD/$SERVER_OVERLAY";  
 done

 #[BZ 1301053] Copy latest INSTALL_README.txt and UPGRADE_README.txt at patch build time. 
 #[spinder 1/27/16] For some reason our static docs links keep changing despite assurances to the contrary.
 #                  This change also allows us to patch/fix without having to rebuild JON and redo RPM builds/testing.
 GA_README_ROOT="$REPO_ROOT/jon/dist/server/src/main/resources-binary";
 UPGRADE_README="$GA_README_ROOT/UPGRADE_README.txt";
 INSTALL_README="$GA_README_ROOT/INSTALL_README.txt";
 debug "cp \"$UPGRADE_README\" \"$BUILD/$SERVER_OVERLAY/\"";
 cp "$UPGRADE_README" "$BUILD/$SERVER_OVERLAY/";
 debug "cp \"$INSTALL_README\" \"$BUILD/$SERVER_OVERLAY/\"";
 cp "$INSTALL_README" "$BUILD/$SERVER_OVERLAY/";

 ##Do custom library pull in as well. Agent and *.zip.
 #Agent patch application
 UPDATE_AGENT="$UPDATE_SRC_ROOT/UPDATE_AGENT";
 mkdir -p "$UPDATE_AGENT/$GA";
 ZIP_AGENT="$UPDATE_SRC_ROOT/ZIP_AGENT";
 # create AGENT zip dir
 mkdir "$ZIP_AGENT"

 # cp updated agent
 debug "cp \"$UPDATE_SRC_ROOT/jon-server-$VERSION/$AGENT_REL_LOCATION/$AGENT_ARTIFACT\" \"$UPDATE_AGENT/$AGENT_ARTIFACT\"";
 cp "$UPDATE_SRC_ROOT/jon-server-$VERSION/$AGENT_REL_LOCATION/$AGENT_ARTIFACT" "$UPDATE_AGENT/$AGENT_ARTIFACT";
 # cp GA agent
 debug "cp \"$UPDATE_SRC_ROOT/$GA/jon-server-$VERSION/$AGENT_REL_LOCATION/$AGENT_ARTIFACT\" \"$UPDATE_AGENT/$GA/$AGENT_ARTIFACT\"";
 cp "$UPDATE_SRC_ROOT/$GA/jon-server-$VERSION/$AGENT_REL_LOCATION/$AGENT_ARTIFACT" "$UPDATE_AGENT/$GA/$AGENT_ARTIFACT";
 # expand both
 unzip "$UPDATE_AGENT/$AGENT_ARTIFACT" -d "$UPDATE_AGENT";
 unzip "$UPDATE_AGENT/$AGENT_ZIP" -d "$UPDATE_AGENT";
 unzip "$UPDATE_AGENT/$GA/$AGENT_ARTIFACT" -d "$UPDATE_AGENT/$GA"; 
 unzip "$UPDATE_AGENT/$GA/$AGENT_ZIP" -d "$UPDATE_AGENT/$GA";

 # do recursive diff.
 cd "$UPDATE_AGENT"

 # Calculate diff GA then Post-GA
 debug "diff -r \"rhq-agent\" \"$GA/rhq-agent\" > \"$UPDATE_AGENT/diff-agent.txt\""
 diff -r "rhq-agent" "$GA/rhq-agent" > "$UPDATE_AGENT/diff-agent.txt"

 AGENT_REMOVED_LIST_ADDITIONS="agent-removed-list.txt";
 echo "" > $AGENT_REMOVED_LIST_ADDITIONS;

 ## iterate over binary differences. Including: .jar,.dll,.so,.exe
 debug "\n\n Beginning to parse agent binary differences."
 # Policy: i)Take all binary updates except jars ii)patched jars ignored 
 # *NOTE: if we do update a .jar but do not increment it's version and name we will have to change this logic.
 for new_bin in `diff -r "rhq-agent" "$GA/rhq-agent" | grep -i "Binary files.*differ" | cut -d' ' -f3`
 do
   #echo "$new_bin" >> "$BUILD/agent-bin-diff-only.txt";  # uncomment for detailed diff content

     if [[ $new_bin == bin/wrapper/windows-x86_32/wrapper.jar ]]; # safe to take updates to this jar.
     then
       debug "Found agent binary update for [$new_bin], needs to be applied.";
       CURRENT_DIR=$(dirname "${new_bin}"); # get current
       debug "CURRENT_DIR" "$CURRENT_DIR";
       debug "mkdir -p \"$ZIP_AGENT/$CURRENT_DIR\"";
       mkdir -p "$ZIP_AGENT/$CURRENT_DIR";
       debug "mv $new_bin $AGENT_ZIP/$CURRENT_DIR"
       mv $new_bin $ZIP_AGENT/$CURRENT_DIR
       #optionally, track change for backup/remove-list.
       if grep -Fxq "../rhq-agent/$new_bin" "$REPO_REMOVED_LIST"
       then
         # ignore. Already captured in the removed list file.
         debug "Skipping... [../rhq-agent/$new_bin] already in $REPO_REMOVED_LIST";
       else
         debug "echo ../rhq-agent/$new_bin >> $AGENT_REMOVED_LIST_ADDITIONS";
         echo "../rhq-agent/$new_bin" >> "$AGENT_REMOVED_LIST_ADDITIONS";  
       fi
      
     elif [[ $new_bin == *.jar ]]; 
     then 
      :;
      debug "Ignoring $new_bin as is .jar";
     else # does not end in .jar so new content for patch.
       debug "Processing agent binary difference [$new_bin]";
       CURRENT_DIR=$(dirname "${new_bin}"); # get current
       debug "CURRENT_DIR" "$CURRENT_DIR";
       debug "mkdir -p \"$ZIP_AGENT/$CURRENT_DIR\"";
       mkdir -p "$ZIP_AGENT/$CURRENT_DIR";
       debug "mv $new_bin $AGENT_ZIP/$CURRENT_DIR"
       mv $new_bin $ZIP_AGENT/$CURRENT_DIR
       #optionally, track change for backup/remove-list.
       if grep -Fxq "../$new_bin" "$REPO_REMOVED_LIST"
       then
         # ignore. Already captured in the removed list file.
         debug "Skipping... [../$new_bin] already in $REPO_REMOVED_LIST";
       else
         debug "echo ../$new_bin >> $AGENT_REMOVED_LIST_ADDITIONS";
         echo "../$new_bin" >> "$AGENT_REMOVED_LIST_ADDITIONS";  
       fi 

     fi
 done

 #update the agent jar anyway since cumulative patched agent must match server.
 mkdir -p "$ZIP_AGENT/rhq-agent/lib";
 cp "rhq-agent/lib/$AGENT_ARTIFACT" "$ZIP_AGENT/rhq-agent/lib/$AGENT_ARTIFACT"

 # iterate over text differences. Including: *.sh,*.bat,.xml
 debug "\n\n Iterating over agent text only differences...";
 # Policy: i)All valid content or updated metadata about library increments
 for new_txt in `diff -r "rhq-agent" "$GA/rhq-agent" | grep -i "diff -r rhq-agent" | cut -d' ' -f3`
 do
   #echo "$new_txt" >> "$BUILD/agent-text-diff-only.txt"; # uncomment for detailed diff content
   CURRENT_DIR=$(dirname "${new_txt}"); # get current
   debug "mkdir -p \"$ZIP_AGENT/$CURRENT_DIR\"";
   mkdir -p "$ZIP_AGENT/$CURRENT_DIR";
   #Test to see if we're allowed to replace this file. Strip 'rhq-agent/' first.
    TRIMMED=${new_txt#rhq-agent/};
    debug "TRIMMED" "$TRIMMED"
    REPLACEABLE=`is_replaceable "$TRIMMED" "AGENT"`;
    debug "REPLACEABLE" "$REPLACEABLE";
    if [[ "$REPLACEABLE" == "true" ]];
    then
     debug "Agent file $new_txt IS replaceable as is. Updating with patch.";
     debug "mv $new_txt $ZIP_AGENT/$CURRENT_DIR"
     mv "$new_txt" "$ZIP_AGENT/$CURRENT_DIR"
     #optionally, track change for backup/remove-list.
     if grep -Fxq "../$new_txt" "$REPO_REMOVED_LIST"
     then
         # ignore. Already captured in the removed list file.
         debug "Skipping... ../$new_txt] already in $REPO_REMOVED_LIST";
     else
         debug "echo ../$new_txt >> $AGENT_REMOVED_LIST_ADDITIONS";
         echo "../$new_txt" >> "$AGENT_REMOVED_LIST_ADDITIONS";  
     fi 

    else
     FILE_NAME=$(basename $new_txt);
     debug "FILE_NAME" "$FILE_NAME"
     debug "Agent file is NOT replaceable as customer could have modified.";
     debug "mv $new_txt $ZIP_AGENT/$CURRENT_DIR/$FILE_NAME.new"
     mv "$new_txt" "$ZIP_AGENT/$CURRENT_DIR/$FILE_NAME.new";
    fi
 done

 ## iterate over new agent files/dirs. Including: .jar
 #Stop for loops from splitting on spaces
 IFS=$'\n';
 debug "\n Iterating over new agent files/dirs..."

 # Policy: i)All the updated modules/libraries.
 for new_file in `diff -r "rhq-agent" "$GA/rhq-agent" | grep -i "Only in rhq-agent" | cut -d' ' -f3-`
 do
   #echo "$new_file" >> "$BUILD/agent-post-ga-only.txt"; # uncomment for detailed diff content
   unset IFS; # re-enable default splitting
   #select components
   #debug "PARENT_DIR=\$(echo $new_file | cut -d':' -f1);"
   PARENT_DIR=$(echo $new_file | cut -d':' -f1);
   debug "PARENT_DIR" "$PARENT_DIR";
   #debug "FILE_PORTION=\$(echo $new_file | cut -d':' -f2)"
   FILE_PORTION=$(echo $new_file | cut -d':' -f2); 
   debug "FILE_PORTION" "$FILE_PORTION"
   FILE_NAME="${FILE_PORTION// }";
   debug "FILE_NAME" "$FILE_NAME";
   #debug "CURRENT_DIR=\$(dirname "$PARENT_DIR/$FILE_NAME");"
   CURRENT_DIR=$(dirname "$PARENT_DIR/$FILE_NAME"); # get current
   debug "CURRENT_DIR" "$CURRENT_DIR";
   debug "mkdir -p \"$ZIP_AGENT/$CURRENT_DIR\"";
   mkdir -p "$ZIP_AGENT/$CURRENT_DIR";
   debug "mv $PARENT_DIR/$FILE_NAME $ZIP_AGENT/$CURRENT_DIR"
   mv "$PARENT_DIR/$FILE_NAME" "$ZIP_AGENT/$CURRENT_DIR";
   #record files new to this release.
   echo "$PARENT_DIR/$FILE_NAME" >> "$BUILD/new-only.txt"

   IFS=$'\n'; # disable space splitting for next iteration
 done

 # iterate over old files/dirs. Including: .jar, 
 #Stop for loops from splitting on spaces
 IFS=$'\n';

 # Policy: i).jar got patched, remove. 
 for new_file in `diff -r "rhq-agent" "$GA/rhq-agent" | grep -i "Only in $GA/rhq-agent" | cut -d' ' -f3-`
 do
   #echo "$new_file" >> "$BUILD/agent-ga-only.txt"; # uncomment for detailed diff content
   unset IFS; # re-enable default splitting
   #select components
   debug "PARENT_DIR=\$(echo $new_file | cut -d':' -f1);"
   PARENT_DIR=$(echo $new_file | cut -d':' -f1);
   debug "PARENT_DIR" "$PARENT_DIR";
   debug "FILE_PORTION=\$(echo $new_file | cut -d':' -f2)"
   FILE_PORTION=$(echo $new_file | cut -d':' -f2); 
   debug "FILE_PORTION" "$FILE_PORTION";
   FILE_NAME="${FILE_PORTION// }"; 
   debug "FILE_NAME" "$FILE_NAME";
   #remove prefix. Ex. GA/rhq-agent/
   REMOVED="${PARENT_DIR#$GA/rhq-agent/}";
   debug "REMOVED" "$REMOVED";

   if [[ $FILE_NAME == "" ]] || [[ $FILE_NAME == "/" ]];
   then
     :; # ignore 
   else
     if [[ $FILE_NAME == *.jar ]];
     then
       if grep -Fxq "../rhq-agent/$REMOVED/$FILE_NAME" "$REPO_REMOVED_LIST"
       then
         # ignore. Already captured in the removed list file.
         debug "Skipping... [../rhq-agent/$REMOVED/$FILE_NAME] already in $REPO_REMOVED_LIST";
       else
         debug "echo ../rhq-agent/$REMOVED/$FILE_NAME >> $AGENT_REMOVED_LIST_ADDITIONS";
         echo "../rhq-agent/$REMOVED/$FILE_NAME" >> "$AGENT_REMOVED_LIST_ADDITIONS";  
       fi 

     else
       :; # ignore the rest. No need to remove them. No real risk and is customer content.
     fi
   fi
   IFS=$'\n'; # disable space splitting for next iteration
 done

 #Allow bash to split on spaces again.
 unset IFS; # re-enable default splitting

 # echo out the removed-list contents
 ## agent removed list contents
 echo "Newly detected removed list entries for agent includes:"
 cat "$AGENT_REMOVED_LIST_ADDITIONS";
 echo "Newly detected removed list entries for agent concluded."

 # build agent overlay zip
 debug "cd $ZIP_AGENT"
 cd $ZIP_AGENT
 AGENT_OVERLAY="jon-agent-updates"
 mkdir "$BUILD/$AGENT_OVERLAY";
 # copy artifacts to $AGENT_OVERLAY location
 for file in "rhq-agent/"*
 do
  debug "mv \"$file\" \"$BUILD/$AGENT_OVERLAY\"";
  mv "$file" "$BUILD/$AGENT_OVERLAY"  
 done

 # -generate and include the EarlyCopy and UpdateProperties binaries
 #mkdir .bin
 CUMULATIVE_BIN="$PATCHED_DIR/.bin"
 mkdir "$CUMULATIVE_BIN" 
 #javac src to dest
 javac -cp "$BUILD/$AGENT_OVERLAY/lib/*" -source 1.6 -target 1.6 -s "$REPO_ROOT/patches" -d "$CUMULATIVE_BIN" $REPO_ROOT/patches/EarlyCopy.java
 javac -cp "$BUILD/$AGENT_OVERLAY/lib/*" -source 1.6 -target 1.6 -s "$REPO_ROOT/patches" -d "$CUMULATIVE_BIN" $REPO_ROOT/patches/UpdateProperties.java
 #BZ 1301575 generate TimeStamp binaries.
 javac -cp "$BUILD/$AGENT_OVERLAY/lib/*" -source 1.6 -target 1.6 -s "$REPO_ROOT/patches" -d "$CUMULATIVE_BIN" $REPO_ROOT/patches/TimeStamp.java
 #BZ 1294626 generate LongPathExtract binaries.
 javac -cp "$BUILD/$AGENT_OVERLAY/lib/*" -source 1.6 -target 1.6 -s "$REPO_ROOT/patches" -d "$CUMULATIVE_BIN" $REPO_ROOT/patches/LongPathExtract.java


 # -generate metadata for the update.
 BUILD_ID=$(more "$BUILD/$SERVER_OVERLAY/$EAR_REL_LOCATION/rhq-server.jar/$PRODUCT_INFO" | grep buildNumber | cut -d'=' -f 2 | sed 's/[\]//');
 NOW=$(date +%s);
 BUILD_TIME=$(date -d @$NOW); 
 echo -e "Name: \t\t $UPDATE_ARTIFACT_ID" > "$PATCHED_DIR/$UPDATE_ARTIFACT_ID.txt"
 echo -e "Update File: \t $UPDATE_ARTIFACT_ID.zip" >> "$PATCHED_DIR/$UPDATE_ARTIFACT_ID.txt"
 echo -e "ID: \t\t $UPDATE_ID" >> "$PATCHED_DIR/$UPDATE_ARTIFACT_ID.txt"
 echo -e "Version: \t $VERSION Update $UPDATE_ID" >> "$PATCHED_DIR/$UPDATE_ARTIFACT_ID.txt"
 echo -e "Build ID: \t $BUILD_ID" >> "$PATCHED_DIR/$UPDATE_ARTIFACT_ID.txt"
 echo -e "Build Time: \t $BUILD_TIME" >> "$PATCHED_DIR/$UPDATE_ARTIFACT_ID.txt"
 echo -e "" >> "$PATCHED_DIR/$UPDATE_ARTIFACT_ID.txt"

 # Execute update steps if greater than (Update 01)
 if [ "$UPDATE_ID" -gt 1 ]; 
 then
   # -generate earlier metadata marker files so that Updates (N-1) not applicable as this supersedes previous updates.
   echo -e "(Patch metadata generated by $UPDATE_ARTIFACT_ID)" > "$PATCHED_DIR/$PREVIOUS_UPDATE_ARTIFACT_ID.txt"
   echo -e "Update $UPDATE_ARTIFACT_ID is cumulative and supersedes all updates released prior to this." >> "$PATCHED_DIR/$PREVIOUS_UPDATE_ARTIFACT_ID.txt"

   # This logic is run during productized builds for Update 02 and later.
   # Every published build is stored in the UPDATE_URLS array and for the current Update, the previous Update 
   # is downloaded and if the same file(*-redhat-*.jar) is being generated for the current Update, 
   # then the version from the previous CP is copied into place.  As the list is cumulative
   # the same binaries continue into later Update builds when applicable.

   # Location of the prepared files
   LOADED_URLS_COUNT=${#UPDATE_URLS[@]}
   debug "LOADED_URLS_COUNT" "$LOADED_URLS_COUNT"
   # Detect which previous patch location/url to use. i.e. (Update Id - 1) # 03 - 1 = 2. Zero based bash array index.
   PREVIOUS_BINARY_INDEX=$(( 10#$UPDATE_ID - 1));  # Ex. Update 03 -> 03 -> url at index 2.
   debug "UPDATE_ID" "$UPDATE_ID"
   debug "PREVIOUS_BINARY_INDEX" "$PREVIOUS_BINARY_INDEX"

   # Load the url of the previous cumulative patch build. Ex. jon01 or brew build artifact url.
   URL=${UPDATE_URLS[$PREVIOUS_BINARY_INDEX]};
   debug "URL" "$URL"

   # extract local file name
   UPDATE_FILE=${URL##*/}
   debug "UPDATE_FILE" "$UPDATE_FILE"

   # extract previous update.
   PREVIOUS="$UPDATE_SRC_ROOT/PREVIOUS"
   CURRENT="$UPDATE_SRC_ROOT/CURRENT"
   mkdir "$PREVIOUS"
   mkdir "$CURRENT"

   # Lazily download previous update zip if not already there.
   if [ -e "$UPDATE_SRC_ROOT/$UPDATE_FILE" ]
   then
     debug "File [$UPDATE_SRC_ROOT/$UPDATE_FILE] already exists."
   else
     debug "wget -O $UPDATE_SRC_ROOT/$UPDATE_FILE $URL";
     wget -O "$UPDATE_SRC_ROOT/$UPDATE_FILE" "$URL";
   fi

   #setup up files for comparison
   ## unzip the previous update
   unzip -q "$UPDATE_SRC_ROOT/$UPDATE_FILE" -d "$PREVIOUS"
   ## copy the content that is about to be used for this new update
   cp -r "$BUILD/$SERVER_OVERLAY" "$CURRENT"
   cp -r "$BUILD/$AGENT_OVERLAY" "$CURRENT"

   # save off current directory
   CURRENT_FOLDER=`pwd`;
   # cd into current update payload to generate file hashes
   cd "$CURRENT";

   # generate lists of and hashes for all redhat-N.jar content.
   find . -name "*redhat*.jar" > "$CURRENT/current.txt"
   find . -name "*redhat*.jar" -exec openssl dgst -md5 {} \; > "$CURRENT/current-hashes.txt" 

   # 
   cd "$CURRENT_FOLDER"
   SUFFIX=".zip"
   #detect unpacked dir name. Trip .zip. Ex.jon-server-3.2.0.GA-update-02.zip -> jon-server-3.2.0.GA-update-02 
   UNPACKED=${UPDATE_FILE%$SUFFIX}
   FULL_GA="3.3.0.GA"
   UNPACKED="${UNPACKED/3.3/$FULL_GA}"
   debug "UNPACKED" "$UNPACKED"

   cd "$PREVIOUS/$UNPACKED" 
   # generate lists of and hashes for all redhat-N.jar content.
   find . -name "*redhat*.jar" > "$PREVIOUS/previous.txt"
   find . -name "*redhat*.jar" -exec openssl dgst -md5 {} \; > "$PREVIOUS/previous-hashes.txt" 
   cd "$CURRENT_FOLDER"

   # Iterate over previous Update jar list and if also in current update, then copy into place.
   echo "" > "$PREVIOUS/to-verify-list.txt"; # reset-verify list.
   while read p; 
   do
     # file from previous Update is still in the new Update, therefore this did not change and original should be used.
     if grep -q $p "$CURRENT/current.txt"
     then
        debug "Detected that [$p] should be copied to CURRENT..."
        debug "cp -p \"$PREVIOUS/$UNPACKED/$p\" \"$CURRENT/$p\"";
        cp -p "$PREVIOUS/$UNPACKED/$p" "$CURRENT/$p"
        # also copy to $BUILD as $p contains both server and agent OVERLAY details already.
        debug "cp -p \"$PREVIOUS/$UNPACKED/$p\" \"$BUILD/$p\"";
        cp -p "$PREVIOUS/$UNPACKED/$p" "$BUILD/$p" 

        # saving this off to verification list
        echo "$p" >> "$PREVIOUS/to-verify-list.txt";
     fi
    done < "$PREVIOUS/previous.txt"

    cd "$CURRENT"
    find . -name "*redhat*.jar" -exec openssl dgst -md5 {} \; > "$CURRENT/current-updated-hashes.txt" 

    # Test that all relevant hashes from previous Update are also now in current Update 
    while read v;
    do
       debug "For verification [$v] with $CURRENT/current-updated-hashes.txt";
       # check that 
       ORIGINAL_HASH=`grep "$v" "$PREVIOUS/previous-hashes.txt"`
       if grep "$ORIGINAL_HASH" "$CURRENT/current-updated-hashes.txt"
       then
         debug "Success. Verified that [$v] copied to current Update correctly."
       else
         debug "WARNING: Alert productization. It appears that [$v] is not being used in the current Update. Is this correct?";
       fi
    done < "$PREVIOUS/to-verify-list.txt";

    cd "$CURRENT_FOLDER";

 fi
}

debug()
{
 if [ "$DEBUG_ON" == "true" ];
  then
   if [ $# -eq 1 ];
    then
     echo -e "[DEBUG] Executing... [$1]"; # Log next steps
    else
     echo -e "[DEBUG] Variable $1 has value [$2]."; # display variable and value
   fi
 fi
}


############ MAIN SCRIPT ############

parse_and_validate_options $@

initialize

get_brew_artifacts

# Select and execute the specific type update to build
execute_type_of_update

# Create the update artifact
generate_update_artifact
