/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.box;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import com.box.sdk.BoxFile;
import com.box.sdk.BoxFolder;
import org.apache.camel.CamelContext;
import org.apache.camel.CamelExecutionException;
import org.apache.camel.RuntimeCamelException;
import org.apache.camel.support.PropertyBindingSupport;
import org.apache.camel.test.junit5.CamelTestSupport;
import org.junit.jupiter.api.TestInstance;

/**
 * Abstract base class for Box Integration tests generated by Camel API component maven plugin.
 */
@TestInstance(TestInstance.Lifecycle.PER_CLASS)
public class AbstractBoxITSupport extends CamelTestSupport {

    private static final String TEST_OPTIONS_PROPERTIES = "/test-options.properties";
    private static Properties properties = new Properties();

    protected BoxFolder testFolder;
    protected BoxFile testFile;
    protected boolean jwtAuthentication;
    protected Map<String, Object> options;

    private static void loadProperties() {
        // read Box component configuration from TEST_OPTIONS_PROPERTIES
        try {
            properties.load(AbstractBoxITSupport.class.getResourceAsStream(TEST_OPTIONS_PROPERTIES));
        } catch (Exception e) {
            throw new RuntimeCamelException(
                    String.format("%s could not be loaded: %s", TEST_OPTIONS_PROPERTIES, e.getMessage()),
                    e);
        }
    }

    private static boolean hasCredentials() {
        if (properties.isEmpty()) {
            loadProperties();
        }

        return !properties.getProperty("userName", "").isEmpty()
                && !properties.getProperty("userPassword", "").isEmpty();
    }

    @Override
    protected CamelContext createCamelContext() throws Exception {

        final CamelContext context = super.createCamelContext();

        options = new HashMap<>();
        for (Map.Entry<Object, Object> entry : properties.entrySet()) {
            options.put(entry.getKey().toString(), entry.getValue());
        }

        final BoxConfiguration configuration = new BoxConfiguration();
        PropertyBindingSupport.bindProperties(context, configuration, options);

        // add BoxComponent to Camel context
        final BoxComponent component = new BoxComponent(context);
        component.setConfiguration(configuration);
        context.addComponent("box", component);

        //initialize flag, whether is box configuration JWT
        String authenticationType = properties.getProperty("authenticationType");
        jwtAuthentication = !BoxConfiguration.STANDARD_AUTHENTICATION.equals(authenticationType);

        return context;
    }

    @SuppressWarnings("unchecked")
    protected <T> T requestBodyAndHeaders(String endpointUri, Object body, Map<String, Object> headers)
            throws CamelExecutionException {
        return (T) template().requestBodyAndHeaders(endpointUri, body, headers);
    }

    @SuppressWarnings("unchecked")
    protected <T> T requestBody(String endpoint, Object body) throws CamelExecutionException {
        return (T) template().requestBody(endpoint, body);
    }

    protected void deleteTestFolder() {
        try {
            testFolder.delete(true);
        } catch (Throwable t) {
        }
        testFolder = null;
    }

    protected void deleteTestFile() {
        try {
            testFile.delete();
        } catch (Throwable t) {
        }
        testFile = null;
    }
}
