---
name: "Gradle :: Plugin"
description: |
  Example to use a JKube Plugin in a Java Application.
  The plugin generates a build timestamp file that is later used by an application to return its version in and endpoint. It also downloads the latest Spring Boot Reference Documentation PDF to be exposed by the application as its home landing page.
---

This project demonstrates how to use JKube Plugins to perform additional actions during the different stages of the JKube processes.

The project is divided into two modules:
- `buildSrc` (build plugin dir which defines plugin)
- `app`      (consumer module which uses plugin)

Let's start by analyzing the `buildSrc` module.

## BuildSrc

`buildSrc` is a standard directory which gradle automatically compiles and puts it in classpath of your build script. We have defined
our plugin in this directory which would be automatically added to `app` module build script.

The files generated by this plugin are then consumed by the `app` module. Let's analyze it.

## app

This module contains a very simple Spring Boot application that exposes 2 endpoints that can be consumed through a web browser.

The first endpoint `/version` returns the contents of the file containing the build timestamp that was generated by the custom plugin.

The second endpoint `/` is the home landing page of the web application. It will automatically load and present the **latest** version of the Spring Boot documentation in PDF, that was downloaded by the custom plugin.

### Gradle configuration
Note how the plugin module is declared in `buildSrc/` for Kubernetes Gradle Plugin dependency:
```
buildSrc/
├── build.gradle
└── src
    └── main
        ├── java
        │ └── org
        │     └── eclipse
        │         └── jkube
        │             └── quickstart
        │                 └── plugin
        │                     └── SimpleJKubePlugin.java
        └── resources
            └── META-INF
                └── jkube
                    └── plugin
```

### Dockerfile
The Dockerfile is tuned to copy the generated and downloaded files to the container image so that they can be consumed by the application.

```dockerfile
COPY maven/build/classes/java/main/jkube-extra/* files/
```

The application will attempt to load these files:
```java
@GetMapping("/version")
public String version() throws IOException {
  if (isProd()) {
    return Files.readString(Path.of("/deployments", "files", "build-timestamp"));
  }
  return Instant.now().toString();
}
```

## Running the example

If you have Minikube, it's as simple as running the following commands from the main quickstart project directory:

```shell
eval $(minikube docker-env)
./gradlew k8sBuild k8sResource k8sApply
```

You need to run `minikube service app` from the command line, and a web browser window will open showing the latest Spring Boot Reference Documentation in PDF.
