#!/bin/bash

. scripts/read_var.sh

function error() {
    echo $*
    exit 1
}

echo "Skupper operator update preparation tool"
echo "========================================"
echo
read_var NEW_VERSION "New skupper-operator version" true ""
MAJOR_MIN_VERSION=`echo ${NEW_VERSION} | sed -re 's/(.*)\.[0-9]+.*/\1/g'`
read_var UPSTREAM_VERSION_TAG "Upstream version tag that corresponds with this release" true "${NEW_VERSION}"

# Retrieve upstream repository commit SHA for tag == $NEW_VERSION
upstream_sha=`curl -s https://api.github.com/repos/skupperproject/skupper-operator/tags | jq -r ".[] | select (.name == \"${UPSTREAM_VERSION_TAG}\") | .commit.sha"`
[[ -z "${upstream_sha}" ]] && error "Upstream version tag ${UPSTREAM_VERSION_TAG} does not correspond to a TAG on github.com/skupperproject/skupper-operator repository"

prev_default=`ls -1 manifests/*clusterserviceversion.yaml | sed -re 's#manifests/skupper-operator.v(.*).clusterserviceversion.yaml#\1#g'`
read_var PREV_VERSION "Previous skupper-operator version to be replaced" true "${prev_default}"
SKIP_VERSIONS=()
while true; do
    read_var SKIP_VERSION "Enter version(s) to be skipped (or empty when done)" false ""
    [[ -z "${SKIP_VERSION}" ]] && break
    SKIP_VERSIONS+=("${SKIP_VERSION}")
done

echo
echo Enter image SHAs
echo

read_var SITE_CONTROLLER_SHA "Site controller SHA" true ""
read_var SERVICE_CONTROLLER_SHA "Service controller SHA" true ""
read_var SKUPPER_ROUTER_SHA "Skupper router SHA" true ""

echo
echo Summary
echo
printf "%-25s: %s\n" "New version" "${NEW_VERSION}"
printf "%-25s: %s\n" "Previous version" "${PREV_VERSION}"
printf "%-25s: " "Versions to skip"
SKIP_VERSIONS_YAML=""
[[ ${#SKIP_VERSIONS[@]} -eq 0 ]] && echo "none" || echo ""
for skip_version in ${SKIP_VERSIONS[@]}; do
    [[ -z "${SKIP_VERSIONS_YAML}" ]] && SKIP_VERSIONS_YAML="  skips:\n"
    printf "     - %s\n" "${skip_version}"
    SKIP_VERSIONS_YAML+="    - skupper-operator.v${skip_version}\n"
done
printf "%-25s: %s\n" "Site Controller SHA" "${SITE_CONTROLLER_SHA}"
printf "%-25s: %s\n" "Service Controller SHA" "${SERVICE_CONTROLLER_SHA}"
printf "%-25s: %s\n" "Skupper Router SHA" "${SKUPPER_ROUTER_SHA}"

echo
read_var CONTINUE "Continue?" true "yes" "yes" "no"
[[ "${CONTINUE,,}" = "no" ]] && exit 0
echo

echo "- Updating container.yaml"
sed -i "s/ref: .*/ref: ${upstream_sha}/g" container.yaml || error "Error updating container.yaml"

echo "- Updating Dockerfile"
sed -ri "s/ version=\"[^\"]+\"/ version=\"${NEW_VERSION}\"/g" Dockerfile || error "Error updating Dockerfile"

echo "- Renaming CSV file"
old_csv="manifests/skupper-operator.v${PREV_VERSION}.clusterserviceversion.yaml"
new_csv="manifests/skupper-operator.v${NEW_VERSION}.clusterserviceversion.yaml"
if [[ ! -f ${old_csv} ]]; then
    error "CSV file for ${PREV_VERSION} does not exist"
fi
if [[ -f ${new_csv} ]]; then
    error "CSV file for ${NEW_VERSION} already exists"
fi
mv ${old_csv} ${new_csv} || error "Error renaming CSV file from ${old_csv} to ${new_csv}"

echo "- Updating CSV file"
sed -i "s/name: skupper-operator.v${PREV_VERSION}/name: skupper-operator.v${NEW_VERSION}/g" ${new_csv} || error "Error updating skupper-operator version"
sed -i "s/version: ${PREV_VERSION}/version: ${NEW_VERSION}/g" ${new_csv} || error "Error updating version"

# removing replaces till apiservicedefinition range
sed -i '/  replaces:.*/{:a;N;/  apiservicedefinitions: {}/!ba; d}' ${new_csv} || error "Error removing range between replaces and apiservicedefinitions"

# inserting replaces and apiservicedefinitions back
sed -i "s/^spec:$/spec:\n  replaces: skupper-operator.v${PREV_VERSION}\n${SKIP_VERSIONS_YAML}  apiservicedefinitions: {}/g" ${new_csv} || error "Error inserting back replaces, skipVersions and apiservicedefinitions"

# updating rht.prod_ver and rht.comp_ver
sed -i "s/rht.prod_ver: .*/rht.prod_ver: \"${MAJOR_MIN_VERSION}\"/g" ${new_csv} || error "Error updating rht.prod_ver"
sed -i "s/rht.comp_ver: .*/rht.comp_ver: \"${MAJOR_MIN_VERSION}\"/g" ${new_csv} || error "Error updating rht.comp_ver"

# updatig image SHAs
sed -ri "s#(registry.redhat.io/application-interconnect/site-controller-rhel8@sha256:).*#\1${SITE_CONTROLLER_SHA}#g" ${new_csv} || error "Error updating site-controller image SHA"
sed -ri "s#(registry.redhat.io/application-interconnect/skupper-router-rhel8@sha256:).*#\1${SKUPPER_ROUTER_SHA}#g" ${new_csv} || error "Error updating skupper-router image SHA"
sed -ri "s#(registry.redhat.io/application-interconnect/service-controller-rhel8@sha256:).*#\1${SERVICE_CONTROLLER_SHA}#g" ${new_csv} || error "Error updating service-controller image SHA"

echo
echo "- Validating changes"
operator-sdk bundle validate . || error "Operator-sdk bundle validation failed"

echo
echo "- Git changes"
git status
