package com.redhat.installer.validator;


import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.installer.DataValidator;
import com.izforge.izpack.panels.ProcessingClient;
import com.izforge.izpack.panels.Validator;

import javax.naming.Context;
import javax.naming.NameNotFoundException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.SearchControls;
import java.util.Hashtable;

/** Validate that the ldap base DN is correct
 *  NOTE: PLEASE USE THE LDAP VALIDATOR TO FIGURE OUT IF CONNECTION IS VALID FIRST.
 *        We need a connection to check if we have the correct base DN.
 *        If you use this as a validator in install.xml it will check that both,
 *        LDAP connection and BaseDn are valid.
 *        We split the two validators because of our LDAP Test button.
 */
public class BaseDnValidator implements DataValidator, Validator {

    String warning;

    public static DirContext makeConnection(String dnServer, String dn, String dnPassword)  {
        /** Ldap Explanation
         * 1. Indicate you're using the LDAP service provider
         * 2. Specify where ldap is running
         * 3. Specify name of the user/program that is doing the authentication
         * 4. Specify the credentials of the user/program doing the authentication.
         *
         * Note:
         * You can specify SECURITY_AUTHENTICATION with "none" or "simple"
         * If unspecified behaviour is determined by the service provider
         */
        Hashtable<String, Object> env = new Hashtable<String, Object>();
        env.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
        env.put(Context.PROVIDER_URL, dnServer);
        env.put(Context.SECURITY_PRINCIPAL, dn);
        env.put(Context.SECURITY_CREDENTIALS, dnPassword);
        DirContext ctx = null;

        try { ctx = new InitialDirContext(env); }
        catch (NameNotFoundException e) { return ctx;}
        catch (NamingException e) { return ctx;}
        return ctx;
    }

    public static boolean validateBaseDn(DirContext ctx, String baseDn, SearchControls ctls){
        try {
            String filter = baseDn;
            NamingEnumeration answer = ctx.search(baseDn, filter, ctls);
            if (answer.hasMore()) return true;
            else return false;
        } catch (NamingException e) {
            return false;
        }
    }

    public Status validateData(AutomatedInstallData adata) {
        String dnServer = adata.getVariable("ldap.url");
        String dn = adata.getVariable("ldap.dn");
        String dnPassword = adata.getVariable("ldap.password");
        String baseDn = adata.getVariable("ldap.basedn" );

        DirContext ctx = makeConnection(dnServer, dn, dnPassword);
        if (ctx == null) return Status.WARNING; //Should be verified with ldapValidator first

        setWarningMessageId(adata.langpack.getString("Ldap.error2"));
        SearchControls ctls = new SearchControls();
        ctls.setSearchScope(SearchControls.SUBTREE_SCOPE);

        boolean result = validateBaseDn(ctx, baseDn, ctls);
        if (!result) return Status.WARNING;
        return Status.OK;
    }

    //Used through test button of userInputPanel
    public boolean validate(ProcessingClient client)
    {
        Status status =  validateData(AutomatedInstallData.getInstance());
        if (status == Status.OK)
            return true;
        else
            return false;
    }
    public String getErrorMessageId() {
        return null;
    }

    public String getWarningMessageId() {
        return warning;
    }

    private void setWarningMessageId(String warning) {
        this.warning = warning;
    }

    public boolean getDefaultAnswer() {
        return false;
    }
}
