package com.redhat.installer.tests;

import com.izforge.izpack.LocaleDatabase;
import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.util.AbstractUIProcessHandler;
import org.apache.commons.io.FileUtils;

import java.io.*;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.security.KeyStore;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

/**
 * Collection of various actions involving creating files / directories and more for testing purposes
 * Created by thauser on 1/29/14.
 */
public class TestUtils {
    public static final String TEST_RESOURCE_DIR = "src/test/resources";
    public static final String workingDir = System.getProperty("user.dir");
    public static final String INSTALL_PATH = "junit-installdir";
    public static final String INSTALL_SUBPATH = "junit-subdir";
    public static final String baseTestDir = INSTALL_PATH + File.separator + INSTALL_SUBPATH;
    public static final File INSTALL_PATH_FILE = new File(INSTALL_PATH);
    public static final String ldapWorkingDir = "/ldap-server";
    public static final String ldapWorkingPath = baseTestDir + ldapWorkingDir;
    public static final File ldapWorkingFile = new File(ldapWorkingPath);
    public static final String ldapTestUrl =  "ldap://localhost:10389";
    public static final String ldapAdminDn = "uid=admin,ou=system";
    public static final String ldapAdminPassword = "secret";
    public static final String mockLangpackFilename = "/mock-langpack.xml";
    public static final String mockLangpackPath = baseTestDir + mockLangpackFilename;
    public static final String mockKeystoreFilename = "/mock-keystore.keystore";
    public static final String mockKeystorePath = baseTestDir + mockKeystoreFilename;
    public static final String mockKeystorePassword = "testpass";
    public static final String mockKeystoreAlias = "testalias";
    public static final String mockSettingsFilename = "/mock-settings.xml";
    public static final String mockSettingsPath = baseTestDir + mockSettingsFilename;
    public static final String testLogFilename = "/test-installationlog.txt";
    public static final String testLogPath = INSTALL_PATH +"/"+ testLogFilename;
    public static final String testPropertiesFilename = "/test-properties.properties";
    public static final String testPropertiesPath = baseTestDir + testPropertiesFilename;
    public static final String layersConfPath = baseTestDir + "/modules/layers.conf";
    public static final String productConfPath = baseTestDir + "/bin/product.conf";


    /**************************************************************************
     *  Helper methods below
     **************************************************************************/


    public static void createMockInstallDir() throws Exception{
        File dir = new File(TestUtils.baseTestDir);

        dir.mkdirs();
    }

    /**
     * Usage: Every string following the first parameter is taken as a new line of text in the
     * produced file. Creates the new file in baseTestDir/path
     * @param path
     * @param contents
     * @throws Exception
     */
    public static void createNewFileWithContent(String path, String ... contents){
        File layersConf = new File(baseTestDir +"/"+ path);
        layersConf.getParentFile().mkdirs();
        try {
            layersConf.createNewFile();
        } catch (IOException e) {
            e.printStackTrace();
            System.out.println("createNewFileWithContent(): failed to createNewFile at : " + layersConf.getPath());
        }
        PrintWriter fw = null;
        try {
            fw = new PrintWriter(new FileWriter(layersConf));
        } catch (IOException e) {
            e.printStackTrace();
            System.out.println("createNewFileWithContent(): failed to instantiate PrintWriter");
        }
        for (String line : contents){
            fw.println(line);
        }
        fw.close();
    }

    /**
     * Copy a file from the given path in resources to the generated test dir at the given path.
     * @param resourcePath
     * @param testPath
     */
    public static void copyFileToTestDir(String resourcePath, String testPath) {
        File targetFolder = new File(baseTestDir + "/" + testPath);
        targetFolder.getParentFile().mkdirs();

        try {
            FileUtils.copyFile(new File(TestUtils.TEST_RESOURCE_DIR + resourcePath), new File(baseTestDir + testPath));
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Creates a new file at the given path within baseTestDir
     * @param path
     * @throws Exception
     */
    public static void createNewEmptyFile(String path) throws Exception{
        File file = new File(baseTestDir + "/" + path);
        file.getParentFile().mkdirs();
        file.createNewFile();
    }

    /**
     * Creates a new file with no write, read, or executable permissions
     */
    public static void createNewUnreadableFile(String path) throws Exception {
        File file = new File(baseTestDir +"/"+path);
        file.mkdirs();
        file.createNewFile();
        file.setWritable(false);
        file.setReadable(false);
        file.setExecutable(false);
    }

    /**
     * Creates a new empty zip file at the given path within baseTestDir
     */
    public static void createNewEmptyZipFile(String path) throws Exception {
        new File(baseTestDir + "/"+path).getParentFile().mkdirs();
        new ZipOutputStream(new FileOutputStream(baseTestDir+"/"+path)).close();
    }

    /**
     * Creates a new zip file with minimal content
     */
    public static void createNewZipFile(String path) throws Exception {
        new File(baseTestDir + "/"+path).getParentFile().mkdirs();
        ZipOutputStream zos = new ZipOutputStream(new FileOutputStream(baseTestDir+"/"+path));
        zos.putNextEntry(new ZipEntry(path));
        zos.write(100);
        zos.close();
    }


    /**
     * makes the given file readable again
     * @param path
     * @throws Exception
     */
    public static void revertUnreadableFile(String path) throws Exception {
        File file = new File(baseTestDir + "/"+path);
        file.setReadable(true);
        file.setWritable(true);
        file.setExecutable(true);
    }

    /**
     * Convenience method to create a product.conf in the appropriate location
     */
    public static void createProductConf(String products){
        createNewFileWithContent("/bin/product.conf", "slot=" + products);
    }

    /**
     * Convenience method to create a layers.conf in the appropriate location
     */
    public static void createLayersConf(String layers){
        createNewFileWithContent("/modules/layers.conf", "layers="+layers);
    }

    /**
     * Convenience method to create a version.txt in the appropriate location
     */
    public static void createVersionTxt(String version){
        createNewFileWithContent("/version.txt",version);
    }


    /**
     * Convenience method to create required files for EAP to exist
     */
    public static void createEAPScripts() throws Exception{
        createNewEmptyFile("/standalone/configuration/standalone.xml");
        createNewEmptyFile("/bin/standalone.sh");
        createNewEmptyFile("/bin/domain.sh");
        createNewEmptyFile("/domain/configuration/domain.xml");
        createNewEmptyFile("/domain/configuration/host.xml");
    }

    /**
     * Hack to destroy the internal reference to the idata object
     * @throws Exception
     */
    public static void destroyIdataSingleton() throws Exception{
        Class<?> clazz = AutomatedInstallData.class;
        Field self = clazz.getDeclaredField("self");
        self.setAccessible(true);
        self.set(null, null);
    }

/*
    public static void destroyProcessPanelHelperIdata() throws Exception {
        Class<?> clazz = ProcessPanelHelper.class;
        Field
    }
*/

    /**
     * Creates a mock langpack in which keys == values. Use to set idata's langpack field.
     * @param strings
     * @return
     * @throws Exception
     */

    public static LocaleDatabase createMockLangpack(String ... strings) throws Exception {
        ArrayList<String> langpack = new ArrayList<String>(10);
        langpack.add("<langpack>");
        for (String s : strings) {
            langpack.add("<str id=\""+s+"\" txt=\""+s+"\"/>");
        }
        langpack.add("</langpack>");

        createNewFileWithContent(TestUtils.mockLangpackFilename, langpack.toArray(new String[langpack.size()]));
        LocaleDatabase ld = new LocaleDatabase(new FileInputStream(TestUtils.mockLangpackPath));
        return ld;
    }

    /**
     * Creates a mock keystore of the given type under baseTestDir/mock-keystore.keystore
     */
    public static void createMockKeystore(String type) throws Exception{
        KeyStore ks = KeyStore.getInstance(type);
        ks.load(null, TestUtils.mockKeystorePassword.toCharArray());
        KeyStore.ProtectionParameter prot = new KeyStore.PasswordProtection(TestUtils.mockKeystorePassword.toCharArray());

        if (!INSTALL_PATH_FILE.exists()){
            createMockInstallDir();
        }
        FileOutputStream fos = new FileOutputStream(TestUtils.mockKeystorePath);
        ks.store (fos, TestUtils.mockKeystorePassword.toCharArray());
    }

    /**
     * convenience method to call createMockKeystore("JKS")
     * @throws Exception
     */
    public static void createMockKeystore() throws Exception {
        createMockKeystore("jks");
    }

    /**
     * Convenience method to set vault keystore variables to correct values
     * @param idata
     */
    public static void setCorrectVaultKeystoreVariables(AutomatedInstallData idata){
        idata.setVariable("vault.keystoreloc", TestUtils.mockKeystorePath);
        idata.setVariable("vault.keystorepwd", TestUtils.mockKeystorePassword);
    }

    /**
     * Convenience method to set ssl keystore variables to correct values
     * @param idata
     */
    public static void setCorrectSSLKeystoreVariables(AutomatedInstallData idata){
        idata.setVariable("ssl.path", TestUtils.mockKeystorePath);
        idata.setVariable("ssl.password", TestUtils.mockKeystorePassword);
    }

    public static Set<String> getFileLinesAsSet(String filePath) throws Exception {
        Set<String> set = new HashSet<String>();
        BufferedReader r = new BufferedReader(new FileReader(new File(filePath)));
        String line ="";
        while ((line = r.readLine()) != null){
            set.add(line);
        }

        return set;
    }

    public static void instantiateClassThroughReflection(Class<?> clazz, String myClassName) throws NoSuchMethodException, InstantiationException, IllegalAccessException, ExceptionInInitializerError, ClassNotFoundException{
        boolean result;
        ClassLoader loader = clazz.getClassLoader();
        Class procClass = loader.loadClass(myClassName);

        Object o = procClass.newInstance();
        Method m = procClass.getMethod("run", new Class[]{AbstractUIProcessHandler.class,
                String[].class});


    }
}
