package com.redhat.installer.tests.validator;

import com.redhat.installer.tests.DataValidatorTester;
import com.redhat.installer.tests.TestUtils;
import com.redhat.installer.validator.JSSEValidator;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Created by thauser on 1/30/14.
 */
public class JSSEValidatorTest extends DataValidatorTester {

    @BeforeClass
    public static void specific() throws Exception {
        idata.langpack = TestUtils.createMockLangpack("securitydomain.jsse.keystore.reqs",
                                                      "securiytdomain.jsse.keystore.passincorrect",
                                                      "securitydomina.jsse.keystore.inaccessible",
                                                      "securitydomain.jsse.truststore.reqs",
                                                      "securitydomain.jsse.truststore.passincorrect",
                                                      "securitydomain.jsse.truststore.inaccessible",
                                                      "securitydomain.jsse.requirements");
    }

    @Before
    public void setUp() throws Exception {
        idata.setVariable("securityDomainAddJsse", "true");
        idata.setVariable("securityDomainJsseAddKeystore","true");
        idata.setVariable("securityDomainJsseAddTruststore", "true");
        TestUtils.createMockKeystore();
        dv = new JSSEValidator();
    }

    @Test
    public void testNoJsseSelected() throws Exception {
        idata.setVariable("securityDomainAddJsse", "false");
        assertOk();
    }

    @Test
    public void testValidKeystore() throws Exception {
        idata.setVariable("securityDomainJsseAddTruststore", "false");
        idata.setVariable("securitydomain.jsse.keystore.password", TestUtils.mockKeystorePassword);
        idata.setVariable("securitydomain.jsse.keystore.url", TestUtils.mockKeystorePath);
        assertOk();
    }

    @Test
    public void testValidTruststore() throws Exception {
        idata.setVariable("securityDomainJsseAddKeystore", "false");
        idata.setVariable("securitydomain.jsse.truststore.password", TestUtils.mockKeystorePassword);
        idata.setVariable("securitydomain.jsse.keystore.url", TestUtils.mockKeystorePath);
        assertError();
    }

    @Test
    public void testValidKeystoreAndTruststore() throws Exception {
        idata.setVariable("securitydomain.jsse.truststore.password", TestUtils.mockKeystorePassword);
        idata.setVariable("securitydomain.jsse.truststore.url", TestUtils.mockKeystorePath);
        idata.setVariable("securitydomain.jsse.keystore.password", TestUtils.mockKeystorePassword);
        idata.setVariable("securitydomain.jsse.keystore.url", TestUtils.mockKeystorePath);
        assertOk();
    }

    @Test
    public void testNoValidKeystoreOrTruststore() throws Exception {
        idata.setVariable("securityDomainJsseAddKeystore", "false");
        idata.setVariable("securityDomainJsseAddTruststore", "false");
        assertError();
        assertLangpackError("securitydomain.jsse.requirements");
    }


    @Test
    public void testWrongKeystorePassword() throws Exception {
        idata.setVariable("securityDomainJsseAddTruststore", "false");
        idata.setVariable("securitydomain.jsse.keystore.password", "wrongpassword");
        idata.setVariable("securitydomain.jsse.keystore.url", TestUtils.mockKeystorePath);
        assertWarning();
        assertLangpackWarning("securitydomain.jsse.keystore.passincorrect");
    }

    @Test
    public void testWrongKeystorePasswordWithValidTruststore() throws Exception {
        idata.setVariable("securitydomain.jsse.truststore.password", TestUtils.mockKeystorePassword);
        idata.setVariable("securitydomain.jsse.truststore.url", TestUtils.mockKeystorePath);
        idata.setVariable("securitydomain.jsse.keystore.password", "wrongpassword");
        idata.setVariable("securitydomain.jsse.keystore.url", TestUtils.mockKeystorePath);
        assertWarning();
        assertLangpackWarning("securitydomain.jsse.keystore.passincorrect");
    }

    @Test
    public void testWrongTruststorePassword() throws Exception {
        idata.setVariable("securityDomainJsseAddKeystore", "false");
        idata.setVariable("securitydomain.jsse.truststore.password", "wrongpassword");
        idata.setVariable("securitydomain.jsse.truststore.url", TestUtils.mockKeystorePath);
        assertWarning();
        assertLangpackWarning("securitydomain.jsse.truststore.passincorrect");
    }

    @Test
    public void testWrongTruststorePasswordWithValidKeystore() throws Exception {
        idata.setVariable("securitydomain.jsse.truststore.password", "wrongpassword");
        idata.setVariable("securitydomain.jsse.truststore.url", TestUtils.mockKeystorePath);
        idata.setVariable("securitydomain.jsse.keystore.password", TestUtils.mockKeystorePassword);
        idata.setVariable("securitydomain.jsse.keystore.url", TestUtils.mockKeystorePath);
        assertWarning();
        assertLangpackWarning("securitydomain.jsse.truststore.passincorrect");
    }

    @Test
    public void testWrongTruststorePath() throws Exception {
        idata.setVariable("securityDomainJsseAddKeystore", "false");
        idata.setVariable("securitydomain.jsse.truststore.password", TestUtils.mockKeystorePassword);
        idata.setVariable("securitydomain.jsse.truststore.url", "incorrectpath'");
        assertWarning();
        assertLangpackWarning("securitydomain.jsse.truststore.inaccessible");
    }

    @Test
    public void testWrongTruststorePathWithValidKeystore() throws Exception {
        idata.setVariable("securitydomain.jsse.truststore.password", TestUtils.mockKeystorePassword);
        idata.setVariable("securitydomain.jsse.truststore.url", "incorrectpath'");
        idata.setVariable("securitydomain.jsse.keystore.password", TestUtils.mockKeystorePassword);
        idata.setVariable("securitydomain.jsse.keystore.url", TestUtils.mockKeystorePath);
        assertWarning();
        assertLangpackWarning("securitydomain.jsse.truststore.inaccessible");
    }

    @Test
    public void testWrongKeystorePath() throws Exception {
        idata.setVariable("securityDomainJsseAddTruststore", "false");
        idata.setVariable("securitydomain.jsse.keystore.password", TestUtils.mockKeystorePassword);
        idata.setVariable("securitydomain.jsse.keystore.url", "incorrectpath");
        assertWarning();
        assertLangpackError("securitydomain.jsse.keystore.inaccessible");
    }

    @Test
    public void testFileUriPath() throws Exception {
        idata.setVariable("securityDomainJsseAddTruststore", "false");
        idata.setVariable("securitydomain.jsse.keystore.password", TestUtils.mockKeystorePassword);
        idata.setVariable("securitydomain.jsse.keystore.url", "file://"+System.getProperty("user.dir")+"/"+TestUtils.mockKeystorePath);
        assertOk();
    }

    @Test
    public void testInvalidFileUriPath() throws Exception {
        idata.setVariable("securityDomainJsseAddTruststore", "false");
        idata.setVariable("securitydomain.jsse.keystore.password", TestUtils.mockKeystorePassword);
        idata.setVariable("securitydomain.jsse.keystore.url", "file://"+TestUtils.mockKeystorePath);
        assertWarning();
    }
}
