package com.redhat.installer.asconfiguration.jdbc.validator;

import com.izforge.izpack.installer.AutomatedInstallData;
import com.izforge.izpack.installer.DataValidator;
import com.izforge.izpack.panels.ProcessingClient;
import com.izforge.izpack.panels.Validator;
import com.redhat.installer.asconfiguration.jdbc.constant.JBossJDBCConstants;

import java.sql.Connection;
import java.sql.Driver;
import java.sql.SQLException;
import java.util.HashSet;
import java.util.Set;

/**
 * Validator used by the Verify Database Connection button on DV's dblogging panel
 * Created by thauser on 7/10/14.
 */
public class DatabaseLoggingConnectionValidator implements DataValidator, Validator {

    private AutomatedInstallData idata;
    private String message;
    private String error;

    @Override
    public Status validateData(AutomatedInstallData idata) {
        this.idata = idata;

        if (!isDriverNameValid()){
            setErrorMessageId("DatabaseLoggingConnectionValidator.missing.using.h2");
            setFormattedMessage(idata.langpack.getString(getErrorMessageId()));
            return Status.ERROR; // can't test with H2
        }
        if (!connectToDatabaseLoggingDatabase()){
            return Status.ERROR;
        }
        return Status.OK;
    }

    private boolean connectToDatabaseLoggingDatabase() {
        Object conn = getDatabaseLoggingDatabaseConnection();

        if (conn != null) {
            if (conn.getClass().equals(String.class)){
                setErrorMessageId("DatabaseLoggingConnectionValidator.connection.failed");
                setFormattedMessage(idata.langpack.getString(getErrorMessageId()));
                return false;
            }
            try {
                ((Connection) conn).close();
            } catch (SQLException e){
                e.printStackTrace();
            }
            setErrorMessageId("DatabaseLoggingConnectionValidator.connection.success");
            setFormattedMessage(idata.langpack.getString(getErrorMessageId()));
            return true;
        } else {
            // error messages are set correctly by previous methods
            return false;
        }
    }

    private Object getDatabaseLoggingDatabaseConnection() {
        Driver driver = getDriverInstance();
        String dbUser = idata.getVariable("dblogging.username");
        String dbPassword = idata.getVariable("dblogging.password");
        String dbUrl = idata.getVariable("dblogging.url");
        if (driver == null){
            setErrorMessageId("DatabaseLoggingConnectionValidator.driver.instantiation.failed");
            setFormattedMessage(String.format(idata.langpack.getString(getErrorMessageId()),driver.getClass().getName()));
            return null;
        }
        // note: this getDatabaseConnection method returns either a String or a Connection object; this allows passing the failure string from
        // the database along.
        return JDBCConnectionUtils.getDatabaseConnection(driver, dbUser, dbPassword, dbUrl);
    }

    private Driver getDriverInstance() {
        Class driverClass = findDriverClass();
        if (driverClass == null) {
            return null;
        }
        Driver driver;
        try {
            driver = (Driver) driverClass.newInstance();
        } catch (InstantiationException e){
            e.printStackTrace();
            return null;
        } catch (IllegalAccessException e) {
            e.printStackTrace();
            return null;
        }
        return driver;
    }

    /**
     * Small wrapper around the loadDriverClass in JDBCConnectionUtils
     * @return
     */
    private Class findDriverClass() {
        Set<String> jarPaths = loadJarPaths();
        String driverClassName = JBossJDBCConstants.classnameMap.get(getDbLoggingDriverName());
        Class returnClass = null;
        if (jarPaths.size() > 0){
            returnClass = JDBCConnectionUtils.findDriverClass(driverClassName, JDBCConnectionUtils.convertToUrlArray(jarPaths.toArray()));
        }
        return returnClass;
    }

    /**
     * A valid name means that the variable has non null contents and is not "h2"
     * @return
     */
    private boolean isDriverNameValid() {
        String driverName = getDbLoggingDriverName();
        if (driverName == null){
            setErrorMessageId("DatabaseLoggingConnectionValidator.missing.driver.name");
            return false;
        }
        if (driverName.equals(JBossJDBCConstants.h2JdbcName)){
            return false; // no tests performed
        }
        return true;
    }

    private String getDbLoggingDriverName() {
        String driverName = idata.getVariable("dblogging.driver");
        if (driverName.equals("installed")){
            driverName = idata.getVariable("jdbc.driver.name");
        }
        return driverName;
    }

    /**
     * Loads the jar paths from the JDBC panel
     */
    private Set<String> loadJarPaths() {
        Set<String> jars = new HashSet<String>();
        for (int i = 1; ; i++){
            // format for the variable name is dictated by the DynamicComponentsPanel.serialize method.
            String driverPath = idata.getVariable("jdbc.driver.jar-"+i+"-path");
            if (driverPath==null){
                break;
            } else {
                jars.add(driverPath);
            }
        }
        return jars;
    }


    public void setFormattedMessage(String message){
        this.message = message;
    }

    public void setErrorMessageId(String id){
        error = id;
    }

    @Override
    public String getErrorMessageId() {
        return error;
    }

    @Override
    public String getWarningMessageId() {
        return error;
    }

    @Override
    public boolean getDefaultAnswer() {
        return true;
    }

    @Override
    public String getFormattedMessage() {
        return message;
    }

    @Override
    public boolean validate(ProcessingClient client) {
        Status status = validateData(AutomatedInstallData.getInstance());
        if (status == Status.OK){
            return true;
        } else {
            return false;
        }
    }


}
