package org.jboss.brmsbpmsuite.patching.client;

import org.apache.commons.io.FileUtils;
import org.osgi.framework.Version;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;

/**
 * Patches a distribution. The distribution root is the specified directory, so the relative paths start from inside that dir.
 */
public class GeneralDistributionPatcher implements DistributionPatcher {
    private static final Logger logger = LoggerFactory.getLogger(GeneralDistributionPatcher.class);

    private final File rootDir;
    private final DistributionChecker distributionChecker;
    private final VersionChecker versionChecker;
    private final Patcher delegatePatcher;

    public GeneralDistributionPatcher(File rootDir, DistributionChecker distributionChecker,
                                      VersionChecker versionChecker, Patcher delegatePatcher) {
        this.rootDir = rootDir;
        this.distributionChecker = distributionChecker;
        this.versionChecker = versionChecker;
        this.delegatePatcher = delegatePatcher;
    }

    @Override
    public void checkDistro() {
        // fail fast if the distribution root or patch root does not exist
        if (!rootDir.exists()) {
            throw new InvalidDistributionRootException("Distribution root " + rootDir + " does not exist!");
        }
        if (!distributionChecker.check(rootDir)) {
            throw new InvalidDistributionRootException("Distribution root " + rootDir + " is not valid " +
                    "for the specified distribution type! Please double-check that (1) the specified directory or file " +
                    "exists and (2) it matches the specified distribution type.");

        }
    }

    @Override
    public void checkVersion() throws IOException {
        Version version = Version.parseVersion(PatchingUtils.implementationVersionFromManifest(rootDir));
        if (!versionChecker.isValidForVersion(version)) {
            throw new InvalidVersionException("The patch is not valid for the distribution and version!");
        }
    }

    @Override
    public void backup(File backupDir) throws IOException {
        // backup the whole distribution
        // ordinary war file is valid distribution root in some cases (e.g. individual WAS8 wars)
        if (rootDir.isFile()) {
            logger.info("Backing-up (coping) file {} to {}", rootDir, backupDir);
            FileUtils.copyFileToDirectory(rootDir, backupDir);
        } else {
            delegatePatcher.backup(backupDir);
        }
    }

    @Override
    public void explodeWar() throws IOException {
        PatchingUtils.unzipWarFileIntoDirWithSameName(rootDir);
    }

    @Override
    public void zipWar() throws IOException {
        PatchingUtils.zipExploadedWar(rootDir);
    }

    @Override
    public void apply() throws IOException {
        delegatePatcher.apply();
    }

}
