/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.listeners.war;

import org.jboss.soa.esb.util.DeploymentArchive;
import org.jboss.soa.esb.util.FreeMarkerTemplate;
import org.jboss.soa.esb.ConfigurationException;

import java.io.InputStream;
import java.io.IOException;
import java.io.ByteArrayInputStream;
import java.util.Map;
import java.util.HashMap;

import freemarker.template.TemplateException;

/**
 * Web deployment archive.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class WebDeploymentArchive extends DeploymentArchive {

    private WebModel webModel;
    private boolean archiveFinalized = false;

    /**
     * Public constructor.
     *
     * @param archiveName The archive name of the deployment.
     */
    public WebDeploymentArchive(String archiveName) {
        super(archiveName);
        webModel = new WebModel(archiveName);
    }

    public WebModel getWebModel() {
        assertNotFinalized();
        return webModel;
    }

    public void finalizeArchive() throws ConfigurationException {
        try {
            // Add the web.xml...
            addWebXML();

            // Add the jboss-web.xml...
            addJBossWebXML();
        } finally {
            // Mark as finalized...
            archiveFinalized = true;
        }
    }

    private void addWebXML() throws ConfigurationException {
        try {
            String webXml = getWebXML();
            addEntry("WEB-INF/web.xml", new ByteArrayInputStream(webXml.getBytes("UTF-8")));
        } catch (Exception e) {
            throw new ConfigurationException("Error adding web.xml to web deployment.", e);
        }
    }

    String getWebXML() throws TemplateException {
        FreeMarkerTemplate template = new FreeMarkerTemplate("templates/web.xml.ftl", WebDeploymentArchive.class);
        Map contextObj = new HashMap();

        contextObj.put("webModel", webModel);
        
        return template.apply(contextObj);
    }

    private void addJBossWebXML() throws ConfigurationException {
        try {
            String jbossWebXml = getJBossWebXML();
            addEntry("WEB-INF/jboss-web.xml", new ByteArrayInputStream(jbossWebXml.getBytes("UTF-8")));
        } catch (Exception e) {
            throw new ConfigurationException("Error adding web.xml to web deployment.", e);
        }
    }

    String getJBossWebXML() throws TemplateException {
        FreeMarkerTemplate template = new FreeMarkerTemplate("templates/jboss-web.xml.ftl", WebDeploymentArchive.class);
        Map contextObj = new HashMap();

        contextObj.put("webModel", webModel);

        return template.apply(contextObj);
    }

    public final void addEntry(String path, InputStream data) {
        assertNotFinalized();
        super.addEntry(path, data);
    }

    public final void addEntry(String path, byte[] data) {
        assertNotFinalized();
        super.addEntry(path, data);
    }

    public final void addEntry(String path) {
        super.addEntry(path);
    }

    public final void addEntry(String path, String resource) throws IOException {
        super.addEntry(path, resource);
    }

    public void addEntry(Class clazz) throws IOException {
        super.addEntry(clazz);
    }

    private void assertNotFinalized() {
        if(archiveFinalized) {
            throw new IllegalStateException("Cannot update archive after it has been finalized!");
        }
    }
}
