package org.jboss.internal.soa.esb.message.format.serialized.body.content;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import org.jboss.internal.soa.esb.message.format.serialized.BodyImpl;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.message.body.content.IncompatibleModeException;
import org.jboss.soa.esb.message.body.content.InvalidPayloadException;
import org.jboss.soa.esb.message.body.content.Payload;
import org.jboss.soa.esb.util.ContextObjectInputStream;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

/**
 * The Message payload contains an arbitrary byte array that can be interpreted
 * by the application as basic Java types.
 * 
 * The instance must be placed into either a read mode or write mode. You can
 * switch between the two modes per instance, but each time you do it will reset
 * the buffer.
 * 
 * When finished, you <i>must</i> flush the instance to the underlying payload
 * implementation.
 */

public class BytesBodyImpl extends BodyImpl implements BytesBody
{
	private static final long serialVersionUID = 0xdeadbeef;

	private static final int READ_MODE = 0;
	private static final int WRITE_MODE = 1;
	private static final int UNSET_MODE = 2;
	
	public BytesBodyImpl(BodyImpl payload) throws IOException
	{
		super(payload);
		
		super.add(Payload.CONTENT_TYPE, Payload.BYTES_BODY);
	}
	
	public boolean readBoolean() throws IncompatibleModeException, IOException,
			InvalidPayloadException
	{
		if (_mode != READ_MODE)
			throw new IncompatibleModeException();

		valid();

		return _inputStream.readBoolean();
	}

	public int readInt() throws IncompatibleModeException, IOException,
			InvalidPayloadException
	{
		if (_mode != READ_MODE)
			throw new IncompatibleModeException();

		valid();

		return _inputStream.readInt();
	}

	public long readLong() throws IncompatibleModeException, IOException,
			InvalidPayloadException
	{
		if (_mode != READ_MODE)
			throw new IncompatibleModeException();

		valid();

		return _inputStream.readLong();
	}

	public char readChar() throws IncompatibleModeException, IOException,
			InvalidPayloadException
	{
		if (_mode != READ_MODE)
			throw new IncompatibleModeException();

		valid();

		return _inputStream.readChar();
	}

	public double readDouble() throws IncompatibleModeException, IOException,
			InvalidPayloadException
	{
		if (_mode != READ_MODE)
			throw new IncompatibleModeException();

		valid();

		return _inputStream.readDouble();
	}

	public float readFloat() throws IncompatibleModeException, IOException,
			InvalidPayloadException
	{
		if (_mode != READ_MODE)
			throw new IncompatibleModeException();

		valid();

		return _inputStream.readFloat();
	}

	public short readShort() throws IncompatibleModeException, IOException,
			InvalidPayloadException
	{
		if (_mode != READ_MODE)
			throw new IncompatibleModeException();

		valid();

		return _inputStream.readShort();
	}

	public String readUTFString() throws IncompatibleModeException,
			IOException, InvalidPayloadException
	{
		if (_mode != READ_MODE)
			throw new IncompatibleModeException();

		valid();

		return _inputStream.readUTF();
	}

	public void writeBoolean(boolean b) throws IncompatibleModeException,
			IOException
	{
		if (_mode != WRITE_MODE)
			throw new IncompatibleModeException();

		_outputStream.writeBoolean(b);
	}

	public void writeInt(int i) throws IncompatibleModeException, IOException
	{
		if (_mode != WRITE_MODE)
			throw new IncompatibleModeException();

		_outputStream.writeInt(i);
	}

	public void writeLong(long l) throws IncompatibleModeException, IOException
	{
		if (_mode != WRITE_MODE)
			throw new IncompatibleModeException();

		_outputStream.writeLong(l);
	}

	public void writeChar(char c) throws IncompatibleModeException, IOException
	{
		if (_mode != WRITE_MODE)
			throw new IncompatibleModeException();

		_outputStream.writeChar(c);
	}

	public void writeDouble(double d) throws IncompatibleModeException,
			IOException
	{
		if (_mode != WRITE_MODE)
			throw new IncompatibleModeException();
		
		_outputStream.writeDouble(d);
	}

	public void writeFloat(float f) throws IncompatibleModeException,
			IOException
	{
		if (_mode != WRITE_MODE)
			throw new IncompatibleModeException();

		_outputStream.writeFloat(f);
	}

	public void writeShort(short s) throws IncompatibleModeException,
			IOException
	{
		if (_mode != WRITE_MODE)
			throw new IncompatibleModeException();

		_outputStream.writeShort(s);
	}

	public void writeUTFString(String s) throws IncompatibleModeException,
			IOException
	{
		if (_mode != WRITE_MODE)
			throw new IncompatibleModeException();

		_outputStream.writeUTF(s);
	}

	/**
	 * Place the instance into read mode, so that the information within the
	 * byte stream can be retrieved.
	 * 
	 * @throws IOException
	 */

	public void readMode() throws IOException
	{
		if (super.get(BytesBody.BYTES_LOCATION) == null)
			throw new IOException();
		
		_mode = READ_MODE;
		_inputStream = new ContextObjectInputStream(new ByteArrayInputStream((byte[]) super
				.get(BytesBody.BYTES_LOCATION)));
		_outputStream = null;
		_byteStream = null;
	}

	/**
	 * Plce the instance into write mode, so that information may be placed
	 * within the byte stream.
	 * 
	 * @throws IOException
	 */

	public void writeMode() throws IOException
	{
		_mode = WRITE_MODE;
		_byteStream = new ByteArrayOutputStream();
		_outputStream = new ObjectOutputStream(_byteStream);
		_inputStream = null;
	}

	/**
	 * Reset the internal buffer. The mode will remain the same.
	 * 
	 * @throws IOException
	 */

	public void reset() throws IOException
	{
		if (_mode == READ_MODE)
			readMode();
		else
		{
			if (_mode == WRITE_MODE)
				writeMode();
		}
	}

	/**
	 * Flush the internal byte array through to the payload.
	 * 
	 * @throws IOException
	 */

	public void flush() throws IOException
	{
		if (_outputStream != null)
		{
			_outputStream.flush();
			_outputStream.close();

			super.add(BytesBody.BYTES_LOCATION, _byteStream.toByteArray());
		}
		else
		{
			if (_inputStream != null)
				_inputStream.close();
		}
	}

	private final void valid() throws InvalidPayloadException
	{
		String type = (String) super.get(Payload.CONTENT_TYPE);
		
		if (type == null)
			throw new InvalidPayloadException();
		else
		{
			if (!type.equals(Payload.BYTES_BODY))
				throw new InvalidPayloadException();
		}
	}

	private int _mode = UNSET_MODE;

	private transient ObjectOutputStream _outputStream;

	private transient ObjectInputStream _inputStream;

	private transient ByteArrayOutputStream _byteStream;

}