/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.config;

import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.config.model.ModelAdapter;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.internal.soa.esb.listeners.war.WebModel;
import org.jboss.internal.soa.esb.listeners.war.WebGatewayDeploymentFactory;
import org.w3c.dom.Document;
import org.w3c.dom.NodeList;
import org.w3c.dom.Node;
import org.w3c.dom.Element;

import java.util.*;

/**
 * ESB configuration model utility methods.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public abstract class ModelUtil {

    /**
     * Get the lister groups (by busIdRef) in the supplied configuration model.
     * <p/>
     * Only looks at gateways at the moment.
     *
     * @param model The configuration model.
     * @return A {@link java.util.Map} containing Lists of listener configurations, by busIdRef.
     * @throws org.jboss.soa.esb.ConfigurationException Unable to convert the model to a DOM.
     */
    public static Map<String, List<ConfigTree>> getListenerGroups(ModelAdapter model) throws ConfigurationException {
        Map<String, List<ConfigTree>> listenerGroups = new LinkedHashMap<String, List<ConfigTree>>();
        Document gatewayConfig = model.generateGatewayConfig();
        NodeList nodeList = gatewayConfig.getDocumentElement().getChildNodes();
        int count = nodeList.getLength();

        for(int i = 0; i < count; i++) {
            Node child = nodeList.item(i);

            if(child.getNodeType() == Node.ELEMENT_NODE) {
                Element listenerElement = (Element) child;
                String busIdRef = listenerElement.getAttribute(ListenerTagNames.BUSIDREF_ATTRIBUTE_TAG);

                List<ConfigTree> listenerGroup = listenerGroups.get(busIdRef);

                if(listenerGroup == null) {
                    listenerGroup = new ArrayList<ConfigTree>();
                    listenerGroups.put(busIdRef, listenerGroup);
                }
                listenerGroup.add(ConfigTree.fromElement(listenerElement));
            }
        }

        return listenerGroups;
    }

    /**
     * Update the supplied {@link org.jboss.internal.soa.esb.listeners.war.WebModel} based on the supplied listener group
     * configurations.
     * @param listenerGroups Listener group configurations, keyed by busrefid.
     * @param webModel The webmodel to be updated.
     * @throws org.jboss.soa.esb.ConfigurationException A Configuration exception occured mapping data onto
     * the WebModel.
     */
    public static void updateWebModel(Map<String, List<ConfigTree>> listenerGroups, WebModel webModel) throws ConfigurationException {
        Set<Map.Entry<String, List<ConfigTree>>> listenerGroupEntries = listenerGroups.entrySet();
        for(Map.Entry<String, List<ConfigTree>> listenerGroup : listenerGroupEntries) {
            List<ConfigTree> listeners = listenerGroup.getValue();

            String gatewayClass = listeners.get(0).getAttribute(ListenerTagNames.GATEWAY_CLASS_TAG);
            if(gatewayClass != null) {
                WebGatewayDeploymentFactory deploymentFactory = WebGatewayDeploymentFactory.FactoryBuilder.createInstance(gatewayClass);

                if(deploymentFactory != null) {
                    // For each listener...
                    for(ConfigTree listenerConfig : listeners) {
                        deploymentFactory.updateWebModel(listenerConfig, webModel);
                    }
                }
            }
        }
    }
}
