/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.deployers.mc;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.spi.deployer.DeploymentStages;
import org.jboss.deployers.vfs.spi.deployer.AbstractSimpleVFSRealDeployer;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.internal.soa.esb.webservice.ESBContractGenerator;
import org.jboss.internal.soa.esb.webservice.ESBServiceEndpointInfo;
import org.jboss.soa.esb.listeners.config.WebserviceInfo;
import org.jboss.soa.esb.listeners.deployers.mc.util.VfsUtil;
import org.jboss.virtual.MemoryFileFactory;
import org.jboss.virtual.VFS;
import org.jboss.virtual.VirtualFile;
import org.jboss.virtual.VirtualFileFilter;

/**
 * MC deployer that generates a wsdl and then adds this wsdl to 
 * a virtual memory file system.
 * <p/>
 * This deployer accepts an EsbMetaData instance and looks for the existence of any WebService (WS) 
 * information. If WS information is found a wsdl will be generated and it will be added
 * to the deployment unit as a metadata location. The wsdl file is added as a VirtualFile 
 * to an in-memory virtual file system.
 * <p/>
 * 
 * This deployer is set at the POST_CLASSLOADER deployment stage as it requries
 * access to classloaders for generating the wsdl.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 */
public class EsbWsdlDeployer extends AbstractSimpleVFSRealDeployer<EsbMetaData>
{
    private Logger log = Logger.getLogger(EsbWsdlDeployer.class);
    
    /**
     * Key used for attaching a list of VirtualFile representing generated wsdl files.
     */
    static final String WSDL_LOCATIONS_KEY = "WSDL-LOCATIONS-KEY";
    
    /**
     * Key used for attaching a URL representing a ref to 
     * a dynamically created in-memory virtual file system.
     */
    static final String DYNAMIC_ROOT_URL_KEY = "WSDL-DYNAMIC-ROOT-URL";
    
    /**
     * Key used for attaching a URL representing a ref to 
     * a dynamically created in-memory virtual file system.
     */
    static final String DYNAMIC_ROOT_DIR_KEY = "WSDL-DYNAMIC-ROOT-DIR";
    
    /**
     * {@link VirtualFileFilter} that selects '.xsd'.
     */
    private static final VirtualFileFilter schemaFileFilter = new SchemaFileFilter();
    
    /**
     * No-args constructor.
     */
    public EsbWsdlDeployer()
    {
        super(EsbMetaData.class);
        // Need access to classloaders.
        setStage(DeploymentStages.POST_CLASSLOADER);
    }
    
    /**
     * Generates a wsdl for all web services specified in the {@link EsbMetaData}s model and makes it available
     * as deployment unit metadata.
     * <p/>
     * Deploy will inspect the {@link EsbMetaData} and see if its model contains web services information and 
     * if so generates the wsdl for them, and will add any additional schemas in the deployment unit to an in-memory
     * virtual file system.
     * <p/>
     * 
     * @param unit The {@link VFSDeploymentUnit} for this deploment.
     * @param esbMetaData The {@link EsbMetaData} for this deployment.
     */
    @Override
    public void deploy(final VFSDeploymentUnit unit, final EsbMetaData esbMetaData) throws DeploymentException
    {
        final List<WebserviceInfo> webServices = esbMetaData.getModel().getWebserviceServices();
        if (webServices != null && webServices.size() > 0)
        {
            try
            {
                final String inMemRootName = esbMetaData.getDeploymentName() + "_WSDL";
                final URL inMemRootUrl = VfsUtil.createInMemUrl(inMemRootName);
                final VirtualFile inMemRootDir = VfsUtil.createInMemoryFs(inMemRootUrl);
                // Attach the in-memory root url so we can remove it in undeploy.
                unit.addAttachment(DYNAMIC_ROOT_URL_KEY, inMemRootUrl);
                // Attach the in-memory root dir so we can remove it in undeploy.
                unit.addAttachment(DYNAMIC_ROOT_DIR_KEY, inMemRootDir);
                
                final List<URL> wsdls = new ArrayList<URL>();
                
                if (webServices.size() > 0)
                {
                    for (WebserviceInfo wsInfo : webServices)
                    {
                        // Generate the wsdl for the web service.
                        final ESBServiceEndpointInfo serviceInfo = new ESBServiceEndpointInfo(wsInfo);
                        final String wsdl = ESBContractGenerator.generateWSDL(wsInfo, serviceInfo, unit.getClassLoader());
                        final URL wsdlUrl = VfsUtil.createInMemUrl(inMemRootUrl, serviceInfo.getWSDLFileName());
                        log.info("Generating wsdl url : " + wsdlUrl);
                        
                        // Add the wsdl in the virtual file system
                        VirtualFile wsdlFile = VfsUtil.addFile(inMemRootUrl, serviceInfo.getWSDLFileName(), wsdl.getBytes());
                        wsdls.add(wsdlUrl);
                        
                        // Add any additional schemas in the deployment unit to the in-memory vfs.
                        addSchemasToVfs(getSchemas(unit.getRoot()), wsdlFile.getParent(), esbMetaData.getArchiveName());
                    }
                    
                    // Add the root of the virtual file system as a meta data location.
                    unit.appendMetaDataLocation(inMemRootDir);
                }
                
                // Attach all the wsdl locations so that we can remove them in undeploy.
                unit.addAttachment(WSDL_LOCATIONS_KEY, wsdls);
            }
            catch (final Exception e)
            {
                throw new DeploymentException("Failed to create webservice artifact", e);
            }
        }
    }
    
    /**
     * Retrieves all the schemas (.xsd) from the virtual file root passed in.
     * 
     * @param root The virtual file that should be searched recursively for schemas.
     * @return List<VirtualFile> List of virtual files that are schemas.
     * @throws IOException If an exception in the search occurs.
     */
    List<VirtualFile> getSchemas(final VirtualFile root) throws IOException
    {
        return root.getChildrenRecursively(schemaFileFilter);
    }

    /**
     * Will add all the schemas to the virtual file system specified by toVfs.
     * <p/>
     * For example, the following vfs file:
     * "Quickstart_publish_as_webservice.esb/request.xsd"
     * will be added to the virtual toVfs with the following url:
     * "vfsmemory://Quickstart_publish_as_webservice_WSDL/WEB-INF/wsdl/ESBServiceSample/request.xsd"
     * <p/>
     * 
     * @param schemas The schemas to add.
     * @param toVfs The destination vfs to which the schemas should be added.
     * @param archiveName Is the name of this archive deployment. 
     *          
     * @throws MalformedURLException
     * @throws IOException
     * @throws URISyntaxException
     */
    void addSchemasToVfs(final List<VirtualFile> schemas, VirtualFile toVfs, String archiveName) throws MalformedURLException, IOException, URISyntaxException
    {
        VfsUtil.addFiles(schemas, toVfs.toURL(), archiveName, "WEB-INF/wsdl/" + toVfs.getName());
    }

    /**
     * Undeploy will inspect the deployment unit to see if the deploy method attached a
     * {@link VirtualFile} representing an in-memory file system. If one exists it will
     * be removed.
     * <p/>
     * 
     * @param unit The {@link VFSDeploymentUnit} for this deploment.
     * @param esbMetaData The {@link EsbMetaData} for this deployment.
     */
    @Override
    public void undeploy(final VFSDeploymentUnit unit, final EsbMetaData esbMetaData) 
    {
        try
        {
            final List<URL> locations = unit.removeAttachment(WSDL_LOCATIONS_KEY, List.class);
            if (locations != null)
            {
                for (URL url : locations)
                {
                    MemoryFileFactory.delete(url);
                }
            }
        }
        finally
        {
            try
            {
                final URL rootUrl = unit.removeAttachment(DYNAMIC_ROOT_URL_KEY, URL.class);
                if (rootUrl != null)
                {
                    MemoryFileFactory.deleteRoot(rootUrl);
                    // Also need remove the root from the metadata locations.
                    unit.removeMetaDataLocation(VFS.getVFS(rootUrl).getRoot());
                }
                
            }
            catch (final Exception e)
            {
                log.warn("Error deleting dynamic class root for " + unit.getName(), e);
            }
        }
    }
    
    /**
     * A {@link VirtualFileFilter} that selects all files ending with '.xsd'.
     * 
     * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
     *
     */
    private static class SchemaFileFilter implements VirtualFileFilter
    {
        public boolean accepts(final VirtualFile file)
        {
            return file.getName().endsWith(".xsd");
        }
    }
}
